/******************************************************************
 * NOTICE: The information contained in this file is proprietary  *
 * to Gateway Design Automation Corp. and is being made available *
 * to Gateway's customers under strict non-disclosure agreements. *
 * Use or disclosure of this information is permissible only      *
 * under the terms of the existing non-disclosure agreement.      *
 ******************************************************************/

/****************************************************************
 *                   MVS DEMONSTRATION                          *
 *  Stochastic process example:                                 *
 *    Model of an interactive system with multiprogramming      *
 *    in virtual storage (4 disks, and 10 terminals)            *
 ****************************************************************/

module system;
    parameter
        debug = 0,
        num_terms = 10,
        sched_q = 100;
    integer
        done_job,
        sched_q_count,
        jobcount,
        status;
    reg
        reset;
      
    integer seed1,seed2,seed3,seed4,seed5;
    initial begin :bars
        integer
             // mean arrival times for queues
             sched_q_mat, dispatch_q_mat, disk_q_mat[1:4],
             // average wait times for queues
             sched_q_awt, dispatch_q_awt, disk_q_awt[1:4],
             // longest wait times for queues
             sched_q_lwt, dispatch_q_lwt, disk_q_lwt[1:4],
             sched_q_maxlwt, dispatch_q_maxlwt, disk_q_maxlwt[1:4];

        sched_q_maxlwt = 0;
        dispatch_q_maxlwt = 0;
        disk_q_maxlwt[1] = 0;
        disk_q_maxlwt[2] = 0;
        disk_q_maxlwt[3] = 0;
        disk_q_maxlwt[4] = 0;

        $gr_bars(
             "queue lengths", 10,
             "Schq", "Dpchq", "Dkq1", "Dkq2", "Dkq3", "Dkq4",
                  sched_q_count, s1.dispatch_q_count,
                  s1.diskq_count[1], s1.diskq_count[2],
                  s1.diskq_count[3], s1.diskq_count[4]);
        $gr_bars(
             "mean inter-arrival times", 100,
             "Schq", "Dpchq", "Dkq1", "Dkq2", "Dkq3", "Dkq4",
                  sched_q_mat, dispatch_q_mat,
                  disk_q_mat[1], disk_q_mat[2],
                  disk_q_mat[3], disk_q_mat[4]);
        $gr_bars(
             "average wait times", 100,
             "Schq", "Dpchq", "Dkq1", "Dkq2", "Dkq3", "Dkq4",
                  sched_q_awt, dispatch_q_awt,
                  disk_q_awt[1], disk_q_awt[2],
                  disk_q_awt[3], disk_q_awt[4]);
        $gr_bars(
             "longest wait times currently in queues", 50,
             "Schq", "Dpchq", "Dkq1", "Dkq2", "Dkq3", "Dkq4",
                  sched_q_lwt, dispatch_q_lwt,
                  disk_q_lwt[1], disk_q_lwt[2],
                  disk_q_lwt[3], disk_q_lwt[4]);
        $gr_bars(
             "cpu busy flag and cpu utilization", 100,
             "busy", "util",
                  s1.c1.cpubusy*100, (s1.c1.totalbusy*100)/$time);

         // setup a register screen display
        $gr_regs(

"     MULTIPROGRAMMING VIRTUAL STORAGE SYSTEM           ",
"                                                       ",
"  disks %d                    terminal %d              ",
"                                                       ",
"  cpu utilization %d          jobcount %d              ",
"                                                       ",
"    disk     track seek times   record search time     ",
"      1         %d                 %d                  ",
"      2         %d                 %d                  ",
"      3         %d                 %d                  ",
"      4         %d                 %d                  ",
"                                                       ",
"    terminal mean user delay    %d                     ",
"    terminal mean cpu job time  %d                     ",
"                                                       ",
"                                                       ",
"    maximum wait times in queues                       ",
"                                                       ",
"        Schq=%d                 Dpchq=%d               ",
"                                                       ",
"        Dkq1=%d                  Dkq2=%d                ",
"        Dkq3=%d                  Dkq4=%d                ",
"                                                       ",

             s1.numdisks, num_terms,
             (s1.c1.totalbusy*100)/$time, jobcount,
             s1.d1.maxseek, s1.d1.maxsearch,
             s1.d2.maxseek, s1.d2.maxsearch,
             s1.d3.maxseek, s1.d3.maxsearch,
             s1.d4.maxseek, s1.d4.maxsearch,
             t1.mean_cpu_time, t1.user_delay,
             sched_q_maxlwt, dispatch_q_maxlwt,
             disk_q_maxlwt[1], disk_q_maxlwt[2],
             disk_q_maxlwt[3], disk_q_maxlwt[4]); 
        forever #3 begin
           
             $q_exam(sched_q,       2, sched_q_mat,    status);
             $q_exam(s1.dispatch_q, 2, dispatch_q_mat, status);
             $q_exam(s1.diskq[1],   2, disk_q_mat[1],  status);
             $q_exam(s1.diskq[2],   2, disk_q_mat[2],  status);
             $q_exam(s1.diskq[3],   2, disk_q_mat[3],  status);
             $q_exam(s1.diskq[4],   2, disk_q_mat[4],  status);

             $q_exam(sched_q,       6, sched_q_awt,    status);
             $q_exam(s1.dispatch_q, 6, dispatch_q_awt, status);
             $q_exam(s1.diskq[1],   6, disk_q_awt[1],  status);
             $q_exam(s1.diskq[2],   6, disk_q_awt[2],  status);
             $q_exam(s1.diskq[3],   6, disk_q_awt[3],  status);
             $q_exam(s1.diskq[4],   6, disk_q_awt[4],  status);

             $q_exam(sched_q,       5, sched_q_lwt,    status);
             $q_exam(s1.dispatch_q, 5, dispatch_q_lwt, status);
             $q_exam(s1.diskq[1],   5, disk_q_lwt[1],  status);
             $q_exam(s1.diskq[2],   5, disk_q_lwt[2],  status);
             $q_exam(s1.diskq[3],   5, disk_q_lwt[3],  status);
             $q_exam(s1.diskq[4],   5, disk_q_lwt[4],  status);

             if(sched_q_lwt > sched_q_maxlwt)
                 sched_q_maxlwt = sched_q_lwt;
             if(dispatch_q_lwt > dispatch_q_maxlwt)
                 dispatch_q_maxlwt = dispatch_q_lwt;
             if(disk_q_lwt[1] > disk_q_maxlwt[1])
                 disk_q_maxlwt[1] = disk_q_lwt[1];
             if(disk_q_lwt[2] > disk_q_maxlwt[2])
                 disk_q_maxlwt[2] = disk_q_lwt[2];
             if(disk_q_lwt[3] > disk_q_maxlwt[3])
                 disk_q_maxlwt[3] = disk_q_lwt[3];
             if(disk_q_lwt[4] > disk_q_maxlwt[4])
                 disk_q_maxlwt[4] = disk_q_lwt[4];

             end
     end

     initial
            forever #100
                $display($time,,"av. resp. = %d jobs = %d",
                     (t1.avresp+t2.avresp+t3.avresp+t4.avresp+
                      t5.avresp+t6.avresp+t7.avresp+t8.avresp+
                      t9.avresp+t10.avresp)/10,jobcount);

     always @sched_q_count
           if(debug)
               $display("time = %d sched_q_count changed to",
                   $time, sched_q_count);

    terminal
        t1(), t2(), t3(), t4(), t5(),
        t6(), t7(), t8(), t9(), t10();
    defparam
           t1.termid = 1,
           t2.termid = 2,
           t3.termid = 3,
           t4.termid = 4,
           t5.termid = 5,
           t6.termid = 6,
           t7.termid = 7,
           t8.termid = 8,
           t9.termid = 9,
           t10.termid = 10;

    server
        s1();

    initial begin
        jobcount = 0;
        sched_q_count = 0;
        done_job = 0;
        $q_initialize(sched_q, 1, num_terms, status);
        #1 reset = 0;              
        seed1=1;
        seed2=1;
        seed3=1;
        seed4=1;
        seed5=1;
    end

task qlengths;
  integer i;
  begin
    $display("sched_q =",sched_q_count);
    $display("server_q =",s1.server_count);
    for (i=1;i<=s1.numdisks;i=i+1) begin
      $display("disk_q number_",i," =",s1.diskq_count[i]);
      $display("disk_q_busy_",i," =",s1.diskq_count[i]);
    end
    $display("cpubusy=",s1.c1.cpubusy);
    $display("dispatch_q =",s1.dispatch_q_count);
  end
endtask

task checkqlengths;
  integer i;
  forever begin
      @(s1.diskq_count[1] or s1.diskq_count[2] or
          s1.diskq_count[3] or s1.diskq_count[4] or s1.server_count)
      i=s1.dispatch_q_count + s1.diskq_count[1] + s1.diskq_count[2] 
           + s1.diskq_count[3] + s1.diskq_count[4]
           + s1.c1.cpubusy 
           + s1.diskq_busy[1]
           + s1.diskq_busy[2]
           + s1.diskq_busy[3]
           + s1.diskq_busy[4];

      if(i !=  s1.server_count) begin
           qlengths;
           $stop;
      end
  end
endtask
endmodule


module server;
    parameter
        numdisks = 4,
        max_diskq = 3,
        maxjobs = 6,
        dispatch_q = 101;
    integer
        i,
        diskq[numdisks:1],
        diskq_count[numdisks:1],
        diskq_busy[numdisks:1],
        dispatch_q_count,
        server_count,
        jobid,
        jobinfo,
        status;

    disk
        d1(), d2(), d3(), d4();
    defparam
        d1.diskid = 1, // fast disk
        d2.diskid = 2, // fast disk
        d3.diskid = 3, d3.maxseek = 25, d3.maxsearch = 10, // slow disk
        d4.diskid = 4, d4.maxseek = 25, d4.maxsearch = 10; // slow disk

    cpu
        c1();

    always @dispatch_q_count
        if(system.debug)
             $display("time = %d dispatch_q_count changed to",
                  $time, dispatch_q_count);

    initial
        @system.reset
            forever begin
                wait(system.sched_q_count > 0);
                wait(server_count < maxjobs);
                $q_remove(system.sched_q, jobid, jobinfo, status);
                system.sched_q_count = system.sched_q_count - 1;
                     if(system.debug)
                         $display("time = %d dispatching job",
                             $time, jobid);
                $q_add(dispatch_q, jobid, jobinfo, status);
                dispatch_q_count = dispatch_q_count + 1;
                     server_count=server_count+1;
            end

    initial begin
        $q_initialize(dispatch_q, 1, maxjobs, status);
        dispatch_q_count = 0;
           server_count=0;
        for(i = 1; i <= numdisks; i = i+1) begin
            diskq[i] = i;
            $q_initialize(diskq[i], 1, max_diskq, status);
            diskq_count[i] = 0;
        end
    end
endmodule


module cpu;
    integer
        cpubusy,
        jobid,
        jobinfo,
        status;
    time
        temptime,
        extime,
        startbusy,
        totalbusy;
    parameter
        pagefaultslice = 9,
        timeslice = 3,
        debug = 0;

    initial begin
        cpubusy=0;
        totalbusy = 0;
    end

    initial @system.reset
        forever begin
            wait(s1.dispatch_q_count != 0);
            $q_remove(s1.dispatch_q, jobid, jobinfo, status);
                startbusy=$time;
                cpubusy=1;
                s1.dispatch_q_count=s1.dispatch_q_count-1;
                if(system.debug)
                     $display("time = %d starting job execution for job",
                         $time, jobid);
            fork
                begin :pf
                    temptime = $dist_uniform(system.seed1,1,pagefaultslice);
                    if(jobinfo > temptime) begin
                        #temptime disable ex;
                        jobinfo = jobinfo - temptime;
                        cpubusy=0;
                        totalbusy = totalbusy + ($time - startbusy);
                        pagefault;
                    end
                end
                begin :ex
                    if(jobinfo < timeslice)
                        extime = jobinfo;
                    else
                        extime = timeslice;
                    #extime disable pf;
                    jobinfo = jobinfo - extime;
                          cpubusy=0;
                          totalbusy = totalbusy + ($time -startbusy);
                    jobswitch;
                end
            join
        end
           
      integer i;
      initial
            for (i=1; i<=4; i=i+1)
                 begin
                 pagefault.numaccesses[i] = 0;
                 pagefault.lat[i] = 0;
                 pagefault.mat[i] = 0;
                 end

    task pagefault;
        integer
            disknum,
                numaccesses[4:1],
                lat[4:1],
                mat[4:1];
        begin
            if(system.debug)
                $display("time = %d pagefault on job",
                     $time, jobid);
            disknum = (($random>>10) % s1.numdisks) + 1;      
            numaccesses[disknum] = numaccesses[disknum] + 1;
            mat[disknum] = $time/numaccesses[disknum];
            if (debug)
                $display($time,,"%d accesses: %d  lat: %d  mat: %d",
                    disknum,numaccesses[disknum],
                    lat[disknum],mat[disknum]);
            lat[disknum]=$time;
            $q_add(s1.diskq[disknum], jobid, jobinfo, status);
            s1.diskq_count[disknum] = s1.diskq_count[disknum] + 1;
        end
    endtask

    task jobswitch;
    begin
        if(jobinfo == 0) begin
            if(system.debug)
                 $display("time = %d completed job",
                      $time, jobid);
            system.done_job = jobid;
        end
        else begin
            if(system.debug)
                 $display("time = %d timesliced on job",
                      $time, jobid);
            $q_add(system.sched_q, jobid, jobinfo, status);
            system.sched_q_count = system.sched_q_count + 1;
        end
        s1.server_count = s1.server_count - 1;
    end
    endtask
endmodule



module disk;
    parameter
        diskid = 1,
        servicetime = 1,
        maxseek = 4,
        maxsearch = 2;
    integer
        jobid,
        jobinfo,
        status;

    initial  begin
        s1.diskq_count[diskid] = 0;
        s1.diskq_busy[diskid] = 0;
    end


    initial @system.reset
        forever begin
            wait(s1.diskq_count[diskid] != 0);
            //transaction available - get transaction
            $q_remove(s1.diskq[diskid], jobid, jobinfo, status);
                s1.diskq_busy[diskid]=1;
            s1.diskq_count[diskid] = s1.diskq_count[diskid] - 1;
                if(system.debug)
                     $display(
                         "time = %d job", $time, jobid,
                         "      accessing disk", diskid);

            // track seek time
            #($dist_exponential(system.seed4,maxseek));

            // record search time
            #($dist_uniform(system.seed5, 0, maxsearch));

            // time to read/write a page
            #servicetime;
                if(system.debug)
                     $display(
                         "time = %d job", $time, jobid,
                         "      finished accessing disk", diskid);

            $q_add(s1.dispatch_q, jobid, jobinfo, status);
                s1.diskq_busy[diskid]=0;
                s1.dispatch_q_count=s1.dispatch_q_count+1;
        end
endmodule



module terminal; 
    time
        jobtime;
    parameter
        termid = 1,
        mean_cpu_time = 20,
        user_delay = 5,
        debug = 0;
    integer
        jobid,
        status,
        job_start,
        response,
        numjobs,
        avresp;

    initial @system.reset begin
        numjobs = 0; 
        avresp = 0;
        forever begin
            #($dist_poisson(system.seed3,user_delay))
            system.jobcount = system.jobcount + 1;
                        numjobs = numjobs + 1;
            jobtime = $dist_exponential(system.seed2, mean_cpu_time);
            jobid = system.jobcount;
            if(system.debug)
                 $display(
                     "time = %d scheduling user job", $time, jobid,
                     "      from terminal", termid);
            $q_add(system.sched_q, jobid, jobtime, status); 
            job_start = $time;
            system.sched_q_count = system.sched_q_count + 1;
            wait(jobid == system.done_job)
                response = $time - job_start;
            if (debug)
                $display($time,,"term %d jobsize= %d  response= %d",
                     termid,jobtime,response);
            avresp = ((avresp*(numjobs-1))+response)/numjobs;
        end
    end
endmodule

