/* toggle_pli.c
 *
 * This file contains the top-level routines associated with
 * the toggle test application.
 *
 */

#include "veriuser.h"
#include "acc_user.h"
#include "tt_toggle.h"
#include "tt_routines.h"


/* tg_check()
 *
 * This is the 'checktf' routine associated with the toggle test
 * application.  It checks to make sure that the associated
 * system task has been called with one or more parameters, and that
 * each parameter identifies a module instance.
 *
 */
int tg_check()
{
   int num_params;
   int i;
   handle param;

   /* initialize and configure access routines */
   acc_initialize();
#if 0
   acc_configure(accDevelopmentVersion,"1.5c");
#endif

   /* check for at least one parameter */
   if ((num_params = tf_nump()) == 0)
   {
      tf_error("$toggle system tasks must be called with at least one parameter");
      return 1;
   }

   /* make sure that all parameters are module instances */
   for (i = 1; i <= num_params; i++)
   {
      /* get handle to the current parameter and check it */
      if ((param = acc_handle_tfarg(i)) == null)
      {
         tf_error("could not get handle to $toggle parameter #%d",i);
         continue;
      }

      /* check parameter type */
      if (acc_fetch_type(param) != accModule)
         tf_error("parameter #%d to $toggle illegal - must be module instance");
   }

   /* close access routines */
   acc_close();

   return 0;
}


/* tg_call()
 *
 * This is the 'calltf' routine associated with the toggle test
 * application.  It sets up the toggle data structures
 * and initiates the VCL toggle monitoring.  The processing of the
 * monitored data is handled in the VCL consumer routine.
 *
 */
int tg_call()
{
   int num_params;
   int i;
   handle module;
   handle curr_cell;
   handle curr_port;
   handle curr_net;
   handle sim_net;
   p_tg_net net_node;

   /* initialize and configure access routines */
   acc_initialize();
   acc_configure(accDevelopmentVersion,"1.5c");

   /* initialize counts */
   tg_num_mon_nets = 0;
   tg_num_mon_port_bits = 0;
   tg_toggled_nets = 0;
   tg_toggled_port_bits = 0;

   /* initialize current buffer location */
   tg_curr_buff_ele = 0;

   /* get number of parameters */
   num_params = tf_nump();

   /* scan all cells under each module instance parameter passed
      to the system task */
   for (i = 1; i <= num_params; i++)
   {
      /* get handle to the module */
      module = acc_handle_tfarg(i);

      /* scan cells under this module */
      curr_cell = null;
      while ((curr_cell = acc_next_cell(module,curr_cell)) != null)
      {
         /* scan cell ports */
         curr_port = null;
         while ((curr_port = acc_next_port(curr_cell,curr_port)) != null)
         {
            /* scan the higher-level nets connected to this port */
            curr_net = null;
            while ((curr_net = acc_next_hiconn(curr_port,curr_net)) != null)
            {
               /* get simulated net */
               sim_net = acc_handle_simulated_net(curr_net);

               /* make an entry for this net in the hash table if one
                  does not already exist - start monitoring if this
                  is a new net */
               if (tg_hash_enter(sim_net,curr_port,&net_node))
               {
                  acc_vcl_add(sim_net,tg_process_toggle,
                                       (char *) net_node,vcl_verilog_logic);

                  /* update monitored net count */
                  tg_num_mon_nets++;
               }

               /* update monitored port bit count */
               tg_num_mon_port_bits++;
            }
         }
      }
   }

   /* close access routines */
   acc_close();

   return 0;
}


/* tg_misc()
 *
 * This is the miscellaneous routine for the toggle test application.  It
 * performs save/restart processing when the user executes '$save' or
 * '$restart'.
 *
 */
int tg_misc(data,reason)
int data, reason;
{
   int i;
   int j;
   int index;
   int element_count;
   int net_count;
   p_tg_net curr_net;

   /* only take action on '$save' or '$restart' calls */
   switch (reason)
   {
      /* take action on '$save' */
      case reason_save:
         /* process the buffer */
         tg_process_buffer();

         /* save global variables */
         tf_write_save((char *)&toggle_test_invoked,sizeof(int));
         tf_write_save((char *)&tg_num_mon_nets,sizeof(int)); 
         tf_write_save((char *)&tg_num_mon_port_bits,sizeof(int));  
         tf_write_save((char *)&tg_toggled_nets,sizeof(int));
         tf_write_save((char *)&tg_toggled_port_bits,sizeof(int)); 

         /* count number of array elements that will be stored */
         element_count = 0;
         for (i = 0; i < TABLE_SIZE; i++)
         {
            if (tg_nets[i].used)
               element_count++;
         }

         /* store number of array elements */
         tf_write_save((char *)&element_count,sizeof(int));
         
         /* traverse hash table and save information */
         for (i = 0; i < TABLE_SIZE; i++)
         {
            /* only store information if this array element
               is used */
            if (tg_nets[i].used)
            {
               /* store array element number */
               tf_write_save((char *)&i,sizeof(int));

               /* count number of nets at this location */
               net_count = 1;
               curr_net = tg_nets[i].next;
               while (curr_net != null)
               {
                  net_count++;
                  curr_net = curr_net->next;
               }

               /* store number of nets at this location */
               tf_write_save((char *)&net_count,sizeof(int));

               /* store the information for each net */
               curr_net = &tg_nets[i];
               while (curr_net != null)
               {
                  tg_store_net_info(curr_net);
                  curr_net = curr_net->next;
               }
            }
         }

         /* end processing on '$save' */
         break;

      /* take action on '$restart' */
      case reason_restart:
         /* initialize and configure access routines */
         acc_initialize();
         acc_configure(accDevelopmentVersion,"1.5c");

         /* restore global variables */
         tf_read_restart((char *)&toggle_test_invoked,sizeof(int));
         tf_read_restart((char *)&tg_num_mon_nets,sizeof(int));
         tf_read_restart((char *)&tg_num_mon_port_bits,sizeof(int));
         tf_read_restart((char *)&tg_toggled_nets,sizeof(int));
         tf_read_restart((char *)&tg_toggled_port_bits,sizeof(int));

         /* retrieve number of array elements that were written to
            the save file */
         tf_read_restart((char *)&element_count,sizeof(int));

         /* restore all of the array elements and their related data */
         for (i = 0; i < element_count; i++)
         {
            /* retrieve array element number */
            tf_read_restart((char *)&index,sizeof(int));

            /* retrieve number of nets at this location */
            tf_read_restart((char *)&net_count,sizeof(int));

            /* restore information for first net and start VCL
               monitoring if the net has not toggled */
            tg_restore_net_info(&tg_nets[index]);

            /* if there are other nets at this location, restore
               their information and start VCL monitoring if the
               given net has not toggled */
            if (net_count > 1)
            {
               /* restore information for all other nets at this location */
               for (j = 0; j < (net_count - 1); j++)
               {
                  /* allocate structure for net information */
                  curr_net = (p_tg_net) malloc(sizeof(s_tg_net));

                  /* restore net information */
                  tg_restore_net_info(curr_net);

                  /* add this net to linked list of nets at this location */
                  curr_net->next = tg_nets[index].next;
                  tg_nets[index].next = curr_net;
               }
            }
         }

         /* reset current buffer element */
         tg_curr_buff_ele = 0;
      
         /* close access routines */
         acc_close();

         /* end processing on '$restart' */
         break;
   } 

   return 0;
}



/* tg_store_net_info()
 *
 * This routine stores the hash table information relevant to one net
 * to the save file.
 *
 */
int tg_store_net_info(net)
p_tg_net net;
{
   int num_chars;
   p_tg_name curr_port;

   /* store net structure information */
   tf_write_save((char *)net,sizeof(s_tg_net));
 
   /* store number of characters in name of net and name
      of net */
   num_chars = strlen(net->full_name);
   tf_write_save((char *)&num_chars,sizeof(int));  
   tf_write_save(net->full_name,num_chars);
 
   /* store port names */
   curr_port = net->ports;
   while (curr_port != null)   
   {
      /* store number of characters in name of port and 
         name of port */
      num_chars = strlen(curr_port->name);
      tf_write_save((char *)&num_chars,sizeof(int));
      tf_write_save(curr_port->name,num_chars);
 
      /* get next port */
      curr_port = curr_port->next;
   }

   return 0;
}



/* tg_restore_net_info()
 *
 * This routine restores information for a net that was saved in a
 * save file when a '$save' was performed.  It also restarts VCL
 * monitoring of the net if the net has not toggled.
 *
 */
int tg_restore_net_info(net)
p_tg_net net;
{
   int num_chars;
   int i;
   p_tg_name curr_port;

   /* restore net structure information */
   tf_read_restart((char *)net,sizeof(s_tg_net));

   /* retrieve number of characters in name of net, allocate
      storage for name, and restore name of net */
   tf_read_restart((char *)&num_chars,sizeof(int));
   net->full_name = (char *) malloc((num_chars + 1)*sizeof(char));
   tf_read_restart(net->full_name,num_chars);
   (net->full_name)[num_chars] = '\0';

   /* get handle to the net */
   net->net_handle = acc_handle_object(net->full_name);

   /* initialize pointers to ports and other nets */
   net->ports = null;
   net->next = null;

   /* restore port information */
   for (i = 0; i < net->num_ports; i++)
   {
      /* allocate port structure */
      curr_port = (p_tg_name) malloc(sizeof(s_tg_name));

      /* retrieve number of characters in name of port, allocate
         storage for name, and restore name of port */
      tf_read_restart((char *)&num_chars,sizeof(int));
      curr_port->name = (char *) malloc((num_chars + 1)*sizeof(char));
      tf_read_restart(curr_port->name,num_chars);
      (curr_port->name)[num_chars] = '\0';

      /* add this port to the linked list of ports for this net */
      curr_port->next = net->ports;
      net->ports = curr_port;
   }

   /* restart VCL monitoring on this net if the net has not toggled */
   if ((!(net->one_to_zero)) || (!(net->zero_to_one)))
      acc_vcl_add(net->net_handle,tg_process_toggle,(char *) net,
                                                       vcl_verilog_logic);
   return 0;
}
      
      






                  
               

               
         
         


/* tg_process_toggle()
 *
 * This is the VCL consumer routine for the toggle test application.  It is
 * called whenever one of the monitored nets changes value.  This routine
 * stores the value change in the buffer and calls the buffer processing
 * routine if the buffer is full.
 *
 */
int tg_process_toggle(change_data)
p_vc_record change_data;
{
   /* store pointer to net structure and value in buffer */
   tg_buffer[tg_curr_buff_ele].net = (p_tg_net) change_data->user_data;
   tg_buffer[tg_curr_buff_ele].value = change_data->out_value.logic_value;

   /* update current buffer element - if buffer is full, call routine
      to process buffer */
   tg_curr_buff_ele++;

   if (tg_curr_buff_ele == BUFFER_SIZE)
      tg_process_buffer();

   return 0;
}


/* tg_process_buffer()
 *
 * This routine processes the data stored in the buffer.  It records the
 * toggle information in the net data structure and turns VCL monitoring
 * of the net off if the net has met the toggle criteria.
 *
 */
int tg_process_buffer()
{
   int i;
   p_tg_net net;

   /* initialize and configure access routines */
   acc_initialize();
   acc_configure(accDevelopmentVersion,"1.5c");

   /* process all of the data stored in the buffer */
   i = 0;
   while (i < tg_curr_buff_ele) 
   {
      /* set pointer to net structure */
      net = tg_buffer[i].net;

      /* only process data if net is being monitored */
      if ((!net->one_to_zero) || (!net->zero_to_one))
      {
         /* take action based on logic value - only changes to zero
            or one are recognized as toggles */
         switch(tg_buffer[i].value)
         {
            case vcl0:
               net->one_to_zero = TRUE;
               break;

            case vcl1:
               net->zero_to_one = TRUE;
               break;
         }

         /* turn off VCL monitoring of the net and increment counts
            if net has toggled */
         if (net->zero_to_one && net->one_to_zero)
         {
            acc_vcl_delete(net->net_handle,tg_process_toggle,
                                           (char *) net,vcl_verilog_logic);
            tg_toggled_nets++; 
            tg_toggled_port_bits = tg_toggled_port_bits + net->num_ports;
         }
      }

      i++;
   }

   /* reset current buffer element */
   tg_curr_buff_ele = 0;

   /* close access routines */
   acc_close();

   return 0;
}
