/* toggle_pli.c
 *
 * This file contains the top-level routines associated with
 * the toggle count application.
 *
 */

#include "veriuser.h"
#include "acc_user.h"
#include "tc_toggle.h"
#include "tc_routines.h"


/* tg_check()
 *
 * This is the 'checktf' routine associated with the toggle count
 * application.  It checks to make sure that the associated
 * system task has been called with one or more parameters, and that
 * each parameter identifies a module instance.
 *
 */
int tg_check()
{
   int num_params;
   int i;
   handle param;

   /* initialize and configure access routines */
   acc_initialize();
#if 0
   acc_configure(accDevelopmentVersion,"1.5c");
#endif

   /* check for at least one parameter */
   if ((num_params = tf_nump()) == 0)
   {
      tf_error("$toggle system tasks must be called with at least one parameter");
      return 1;
   }

   /* make sure that all parameters are module instances */
   for (i = 1; i <= num_params; i++)
   {
      /* get handle to the current parameter and check it */
      if ((param = acc_handle_tfarg(i)) == null)
      {
         tf_error("could not get handle to $toggle parameter #%d",i);
         continue;
      }

      /* check parameter type */
      if (acc_fetch_type(param) != accModule)
         tf_error("parameter #%d to $toggle illegal - must be module instance");
   }

   /* close access routines */
   acc_close();

   return 0;
}


/* tg_call()
 *
 * This is the 'calltf' routine associated with the toggle count
 * application.  It sets up the toggle data structures
 * and initiates the VCL toggle monitoring.  The processing of the
 * monitored data is handled in the VCL consumer routine.
 *
 */
int tg_call()
{
   int num_params;
   int i;
   handle module;
   handle curr_cell;
   handle curr_port;
   handle curr_net;
   handle sim_net;
   p_tg_net net_node;

   /* initialize and configure access routines */
   acc_initialize();
   acc_configure(accDevelopmentVersion,"1.5c");

   /* initialize counts */
   tg_num_mon_nets = 0;
   tg_num_mon_port_bits = 0;

   /* initialize current buffer location */
   tg_curr_buff_ele = 0;

   /* get number of parameters */
   num_params = tf_nump();

   /* scan all cells under each module instance parameter passed
      to the system task */
   for (i = 1; i <= num_params; i++)
   {
      /* get handle to the module */
      module = acc_handle_tfarg(i);

      /* scan cells under this module */
      curr_cell = null;
      while ((curr_cell = acc_next_cell(module,curr_cell)) != null)
      {
         /* scan cell ports */
         curr_port = null;
         while ((curr_port = acc_next_port(curr_cell,curr_port)) != null)
         {
            /* scan the higher-level nets connected to this port */
            curr_net = null;
            while ((curr_net = acc_next_hiconn(curr_port,curr_net)) != null)
            {
               /* get simulated net */
               sim_net = acc_handle_simulated_net(curr_net);

               /* make an entry for this net in the hash table if one
                  does not already exist - start monitoring if this
                  is a new net */
               if (tg_hash_enter(sim_net,curr_port,&net_node))
               {
                  acc_vcl_add(sim_net,tg_process_toggle,
                                       (char *) net_node,vcl_verilog_logic);

                  /* update monitored net count */
                  tg_num_mon_nets++;
               }

               /* update monitored port bit count */
               tg_num_mon_port_bits++;
            }
         }
      }
   }

   /* close access routines */
   acc_close();

   return 0;
}


/* tg_process_toggle()
 *
 * This is the VCL consumer routine for the toggle count application.  It is
 * called whenever one of the monitored nets changes value.  This routine
 * stores the value change in the buffer and calls the buffer processing
 * routine if the buffer is full.
 *
 */
int tg_process_toggle(change_data)
p_vc_record change_data;
{
   /* store pointer to net structure and value in buffer */
   tg_buffer[tg_curr_buff_ele].net = (p_tg_net) change_data->user_data;
   tg_buffer[tg_curr_buff_ele].value = change_data->out_value.logic_value;

   /* update current buffer element - if buffer is full, call routine
      to process buffer */
   tg_curr_buff_ele++;

   if (tg_curr_buff_ele == BUFFER_SIZE)
      tg_process_buffer();

   return 0;
}


/* tg_process_buffer()
 *
 * This routine processes the data stored in the buffer.  It records the
 * toggle information in the net data structure and turns VCL monitoring
 * of the net off if the net has met the toggle criteria.
 *
 */
int tg_process_buffer()
{
   int i;
   p_tg_net net;

   /* initialize and configure access routines */
   acc_initialize();
   acc_configure(accDevelopmentVersion,"1.5c");

   /* process all of the data stored in the buffer */
   i = 0;
   while (i < tg_curr_buff_ele) 
   {
      /* set pointer to net structure */
      net = tg_buffer[i].net;

      /* take action based on logic value - only changes to zero
         or one are recognized as toggles */
      switch(tg_buffer[i].value)
      {
         case vcl0:
            net->one_to_zero++;
            break;

         case vcl1:
            net->zero_to_one++;
            break;
      }

      i++;
   }

   /* reset current buffer element */
   tg_curr_buff_ele = 0;

   /* close access routines */
   acc_close();

   return 0;
}


          
      


 
               

      
   
      

   
 

   
 
