#define DL_BACKAN_C     1

/*
   This file contains routines which are usd by delay calculators
   (most notably those in dl_calc.c, in this case) to read files which
   contain interconnect capacitance data, place the information in
   a data structure, and retrieve that information in a timely fashion.
   This code uses common hashing techniques.
   The back annotation file must contain single line entries composed
   of a full hierarchical object name followed by a real value :

               top.m1.dffa.dffa3.qbar 4.7

   The input format may be altered to your liking by adjusting the 
   routine dl_make_backan_table appropriately.

   Although these routines are not access routines applications, because
   of their necessity for delay calculations they will be refined, and
   probably documented within an applications notes.
*/ 

#include <stdio.h>

#include "veriuser.h"
#include "acc_user.h"
#include "dl.h"

#if defined(__STDC__) || defined(__cplusplus)
typedef size_t SIZE_T;
#else
typedef unsigned long SIZE_T;
#endif

static void 	dl_init_backan_table PROTO_PARAMS((int net_count));
static void 	dl_backan_insert PROTO_PARAMS((handle addr, double val));
static int 	dl_hash PROTO_PARAMS((handle addr));
static int 	dl_get_numlines PROTO_PARAMS((char *filename));

/******************************************************************************/
/* Back annotation routines */

/* Back annotation hash table control */
global int  dl_make_backan_table();
static void dl_init_backan_table();
global void dl_free_backan_table();

/* Back annotation hash table insertion and search */
static void dl_backan_insert();
global double dl_backan_fetch();

/* Auxiliary routines */
static int  dl_hash();
global int  dl_next_prime();
static int  dl_get_numlines();

/******************************************************************************/
/* header items*/

#define no_backan_val -1.0

/* Table of prime values, 0 must be last entry  */
int primes[] = {
    20011, 25013, 30011, 35023, 40009, 45007,
    50021, 80021, 100003, 150001, 200003,
    1000003, 5000011,
    0
};
	
/* jjp011989 saber resolve prob - made these static */
static int  backan_table_size;

static struct table{
    handle addr;
    double val;
} *backan_table;


/******************************************************************************/

/*R DL_MAKE_BACKAN_TABLE
    The backan table is set up and the contents on the file
    referenced by "filename" are read into the backan table.
*/
exfunc int dl_make_backan_table(filename)
char *filename;
{
    FILE *net_file;
    double val;
    handle net_handle;
    char name[100];
    int net_count;
    int i;

    /***jjp01289 : < to == ***/
    if ((net_file = fopen(filename,"r")) == 0) {
        io_printf("Warning: Back-annotate file \"%s\" not found.\n",filename);
        return(FALSE);
    }

    net_count = dl_get_numlines(filename);
    dl_init_backan_table(net_count);

    /*acc_initialize();*/ /*assume initialization elsewhere (dcalc routine)*/
    for (i=0; i<net_count; i++)
    {
        if (fscanf(net_file,"%s%lf\n",name,&val) == EOF) {
            io_printf("Warning: \"%s\", unexpected end of file.\n",filename);
            break;
        }

        net_handle = acc_handle_object(name);
        if (net_handle == null)
            io_printf("%s%s%s",
                "Warning from file back annotation :  net name \"",
                name,"\" not found\n");
        else
            dl_backan_insert(net_handle,val);
    }

    fflush(stdout);
    fclose(net_file);
    return(TRUE);
}


/*R DL_INIT_BACKAN_TABLE
    Memory is allocated for the backan table, given the number
    of nets that have to be stored in it.  The values are
    all initialized to the defined constant "no_backan_val".
*/
static void dl_init_backan_table(net_count)
int net_count;
{
    int i;

    backan_table_size = dl_next_prime(net_count * 10);
    if (backan_table_size == 0) {
        io_printf("%s%s\n","Back-annotation error : ",
                  "required backan table too large; must enhance primes array");
        return;
    }
    else
    {
        backan_table = (struct table *)
            malloc((SIZE_T)(backan_table_size * sizeof(struct table)));
    }

    for (i=0; i<backan_table_size; i++)
        backan_table[i].val = no_backan_val;
}


/*R DL_FREE_BACK_TABLE
    Free the memory used by the backan table.
*/
exfunc void dl_free_backan_table()
{
    free(backan_table);
}


/******************************************************************************/


/*R DL_BACKAN_INSERT
    This adds an address and associated value into the backan table,
    using the hashing function on the address.
*/
static void dl_backan_insert(addr,val)
handle addr;
double val;
{
    int h = dl_hash(addr);

    /* find empty backan table slot */
    while (backan_table[h].val != no_backan_val)
    {
        if (backan_table[h].addr == addr) {
            /*
            io_printf("Warning: hash value for net \"%s\" reassigned.\n",
                       acc_fetch_pathname(addr));
            */
	    break;
        }
	h = (h+1) % backan_table_size;
    }

    /* assign entry to backan table slot */
    backan_table[h].val = val;
    backan_table[h].addr = addr;
}


/*R DL_BACKAN_FETCH
    Finds the backan table entry that coincides with the handle
    argument.  The value of the entry is returned.
*/
exfunc double dl_backan_fetch(handle_val)
handle handle_val;
{
    int h = dl_hash(handle_val);
    int dup = 0;

    while (backan_table[h].addr != handle_val  &&
           dup <= backan_table_size  &&
           backan_table[h].val != no_backan_val)
    {
        dup++;
        h = (h +1) % backan_table_size; /* Why not call dl_hash? */
    }

    /* if every slot in the backan table was checked and the correct */
    /* handle_val was not found, notify user of error */
    if (dup > backan_table_size) {
        io_printf("Internal Hashing error, unknown net handle %x\n",handle_val);
        return(no_backan_val);
    }

    return(backan_table[h].val);
}


/******************************************************************************/


/*R DL_HASH
    Returns the mod (remainder) of the address and the 
    backan table size.  This gives us a number in the range
    0..(backan_table_size - 1), which will be used as an
    index into the backan table array.
*/
static int dl_hash(addr)
handle addr;
{
    return(((int)addr) % backan_table_size);
}


/*R DL_NEXT_PRIME
    Returns first prime number higher than number passed in.
    (uses prime numbers found in the previously defined
    array of prime numbers).
*/
exfunc int dl_next_prime(num)
int num;
{
    int i;

    for (i=0; primes[i]!=0; i++)
        if (primes[i] > num)
            return(primes[i]);
    
    return(0); /* Exception value */
}


/*R DL_GET_NUMLINES
    Returns the number of lines in a file (which should be nearly
    the same as the number of nets contained in the file).
*/
static int dl_get_numlines(filename)
char *filename; /***jjp01289 was int***/
{
    int line = 0;
    double val;
    FILE *net_file;
    char name[256];

    /***jjp01289 : < to == ***/
    if ((net_file = fopen(filename,"r")) == 0) {
        io_printf("Warning: Back-annotate file \"%s\" not found.\n",filename);
        return(0);
    }

    while (fscanf(net_file,"%s%lf\n",name,&val) != EOF)
        line++;

    /*printf("file has %d lines\n",line);*/

    fclose(net_file);
    return(line);
}


/******************************************************************************/


