/*
	initram.c -- initialize dynamic allocated RAM from data file.
*/

#include	<stdio.h>
#include	<ctype.h>
#include	"veriuser.h"

#include	"common.h"

static void	setup_init_adrs PROTO_PARAMS((_mem_desc *mdp));
static int 	bound_check PROTO_PARAMS((_bignum *adrs, _bignum *bound1, _bignum *bound2));
static int 	write_mem PROTO_PARAMS((_mem_desc *mdp, _bignum *adrs, s_tfexprinfo *ip));
static void 	mem_init PROTO_PARAMS((_mem_desc *mdp, s_tfexprinfo *(*func)()));
static void 	get_adrs_pair PROTO_PARAMS((_bignum *from, _bignum *to));
static void	expected PROTO_PARAMS((char *str));
static int 	skipspaces PROTO_PARAMS((void));
static int 	gettoken PROTO_PARAMS((void));

/*
	gettoken() staff
*/
#define		SY_STR		0
#define		SY_SYM		1

static char token[IDSIZE];
static int tok_type = 0;

static _bignum *init_from, *init_to;
static int direction;
static FILE *ifp;
static char *filename;

static int cat = 1;	/* concatenate string flag */

int damem_initb_checktf()
{

	/* $damem_initb("mem99", "bin.dat", 0, 16); */

	int type, anyerr = 0;
#ifdef DEBUG
	io_printf("damem_initb_checktf() here.\n");
#endif

	if (tf_typep(1) != tf_string)
		anyerr = 1;
	if (tf_typep(2) != tf_string)
		anyerr = 1;
	switch (tf_nump()) {
	case 4:
		if ((type = tf_typep(4)) != tf_readonly && type != tf_readwrite)
			anyerr = 1;
	case 3:
		if ((type = tf_typep(3)) != tf_readonly && type != tf_readwrite)
			anyerr = 1;
		break;
	}
	if (anyerr)
		tf_error("illegal arguments for $damem_initb user task");

        return 0;
}

int damem_initb()
{
	s_tfexprinfo info, *ip = &info;
	char *id;	/* memory id */
	_mem_desc *mdp;

	/* $damem_initb("mem99", "bin.dat", 0, 16); */

	tf_exprinfo(2, ip);	/* file name */
	filename = strsave(ip->expr_string);
	if ((ifp = fopen(ip->expr_string, "r")) == NULL) {
		io_printf("Could not openread file \"%s\"\n", ip->expr_string);
		exit(1);
	}
	tf_exprinfo(1, ip);	/* ID */
	id = make_fullname(ip);
	if ((mdp = mem_desc_find(id)) == NULL) {
		tf_error("damem_initb(): Not found memory ID '%s' in this scope.\n", id);
		exit(1);
	}
	setup_init_adrs(mdp);
	mem_init(mdp, mvl_strb_to_mvl);
	fclose(ifp);

        return 0;
}

int damem_inith_checktf()
{
	/* $damem_inith("mem99", "hex.dat", 0, 16); */

	int type, anyerr = 0;
#ifdef DEBUG
	io_printf("damem_inith_checktf() here.\n");
#endif

	if (tf_typep(1) != tf_string)
		anyerr = 1;
	if (tf_typep(2) != tf_string)
		anyerr = 1;
	switch (tf_nump()) {
	case 4:
		if ((type = tf_typep(4)) != tf_readonly && type != tf_readwrite)
			anyerr = 1;
	case 3:
		if ((type = tf_typep(3)) != tf_readonly && type != tf_readwrite)
			anyerr = 1;
		break;
	}
	if (anyerr)
		tf_error("illegal arguments for $damem_inith user task");

        return 0;
}

int damem_inith()
{
	s_tfexprinfo info, *ip = &info;
	char *id;	/* memory id */
	_mem_desc *mdp;

	/* $damem_inith("mem99", "hex.dat", 0, 16); */

	tf_exprinfo(2, ip);	/* file name */
	filename = strsave(ip->expr_string);
	if ((ifp = fopen(ip->expr_string, "r")) == NULL) {
		io_printf("Could not openread file \"%s\"\n", ip->expr_string);
		exit(1);
	}
	tf_exprinfo(1, ip);	/* ID */
	id = make_fullname(ip);
	if ((mdp = mem_desc_find(id)) == NULL) {
		tf_error("damem_inith(): Not found memory ID '%s' in this scope.\n", id);
		exit(1);
	}
	setup_init_adrs(mdp);
	mem_init(mdp, mvl_strh_to_mvl);
	fclose(ifp);
        
        return 0;
        
}

static void setup_init_adrs(mdp)
_mem_desc *mdp;
{
	int n;
	s_tfexprinfo info, *ip = &info;

	n = tf_nump();
	init_from = init_to = NULL;
	
	if (n < 3) {	/* no address specified */
		init_from = bn_dup(mdp->adrs_from_bound);
		init_to   = bn_dup(mdp->adrs_to_bound);
	} else if (n == 3) {
		tf_exprinfo(3, ip);
		init_from = bn_dup(bn_expr_to_bignum(ip));
		init_to   = bn_dup(mdp->adrs_to_bound);
	} else if (n == 4) {
		tf_exprinfo(3, ip);
		init_from = bn_dup(bn_expr_to_bignum(ip));
		tf_exprinfo(4, ip);
		init_to   = bn_dup(bn_expr_to_bignum(ip));
	}
	if (bn_cmp(init_from, init_to) <= 0)
		direction = DIR_INCR;
	else
		direction = DIR_DECR;
}
		
static int lines = 1;
static int c = 0;
static int fatal = 0;

static int bound_check(adrs, bound1, bound2)
_bignum *adrs, *bound1, *bound2;
{
	_bignum *low, *high;

	if (bn_cmp(bound1, bound2) > 0) {
		low = bound2;
		high = bound1;
	} else {
		low = bound1;
		high = bound2;
	}
	if (bn_cmp(adrs, low) < 0 || bn_cmp(adrs, high) > 0)
		return (1);
	else
		return (0);
}

static int write_mem(mdp, adrs, ip)
_mem_desc *mdp;
_bignum *adrs;
s_tfexprinfo *ip;
{
	/*
		before write to memory, checks address bound.
	*/
	if (ip->expr_vec_size != mdp->bit_width) {
		io_printf("Bit width differ in \"%s\" at line %d\n",
			filename, lines);
	} else if (bound_check(adrs, mdp->adrs_from_bound, mdp->adrs_to_bound)) {
		io_printf("Address out of bound in \"%s\" at line %d\n",
			filename, lines);
	} else if (bound_check(adrs, init_from, init_to)) {
		io_printf("Address out of bound in \"%s\" at line %d\n",
			filename, lines);
	} else {
		mem_word_enter(mdp, adrs, (unsigned *)(ip->expr_value_p));
		return (1);	/* indicates memory is altered */
	}
	return (0);	/* not altered */
}

static void mem_init(mdp, func)
_mem_desc *mdp;
s_tfexprinfo *(*func)();
{
	_bignum *cur_adr;
	_bignum *from, *to;
	int altered = 0;

	lines = 1;
	cur_adr = bn_dup(init_from);
	from = bn_new(mdp->ngroups);
	to   = bn_new(mdp->ngroups);

	while (!fatal && gettoken() != EOF) {
		if (tok_type == SY_STR) {
			/* normal pattern */
			if (write_mem(mdp, cur_adr, func(token, mdp->bit_width)))
				altered = 1;
			if (direction == DIR_INCR)
				bn_incr(cur_adr);
			else
				bn_decr(cur_adr);
		} else if (*token == '@') {
			/* address */
			cat = 0;	/* not allow concatenation */
			gettoken();
			if (*token == '(') {
				/* range specified */
				get_adrs_pair(from, to);
				if (bound_check(from, mdp->adrs_from_bound, mdp->adrs_to_bound) ||
					bound_check(to,   mdp->adrs_from_bound, mdp->adrs_to_bound) ||
					bound_check(from, init_from, init_to) ||
					bound_check(to,   init_from, init_to)) {
					/* address bound error */
					io_printf("Address given in \"%s\" at line %d is out of bounds\n",
						filename, lines);
				} else {
					int dir;
					bn_copy(from, cur_adr);
					gettoken();	/* data */
					if (tok_type != SY_STR)
						expected("DATA");
					if (bn_cmp(from, to) <= 0)
						dir = DIR_INCR;
					else
						dir = DIR_DECR;
					for ( ; ; ) {
						int res;
						if (write_mem(mdp, cur_adr, func(token, mdp->bit_width)))
							altered = 1;
						if ((res = bn_cmp(cur_adr, to)) == 0)
							break;
						else {
							if (dir == DIR_INCR)
								bn_incr(cur_adr);
							else
								bn_decr(cur_adr);
						}
					}
				}
			} else if (tok_type == SY_STR) {
				_bignum *adrs;
				/* simple address */
				adrs = bn_strh_to_bignum(token);
				if (bound_check(adrs, mdp->adrs_from_bound, mdp->adrs_to_bound)||
					bound_check(adrs, init_from, init_to)) {
					io_printf("Address given in \"%s\" at line %d is out of bounds\n",
						filename, lines);
				} else {
					bn_copy(adrs, cur_adr);
				}
			} else {
				expected("DATA or '('");
			}
			cat = 1;	/* allow concatenation for data field */
		}
	}
	if (altered)
		io_printf("Warning! The contents of memory (%s) have been altered\n",
			mdp->id);
	else
		io_printf("Memory (%s) has not been altered\n", mdp->id);
}

static void get_adrs_pair(from, to)
_bignum *from, *to;
{
	gettoken();
	if (tok_type != SY_STR) {
		expected("DATA");
	} else {
		bn_copy(bn_strh_to_bignum(token), from);
		gettoken();
		if (*token != ',') {
			expected("','");
		} else {
			gettoken();
			if (tok_type != SY_STR) {
				expected("DATA");
			} else {
				bn_copy(bn_strh_to_bignum(token), to);
				gettoken();
				if (*token != ')')
					expected("')'");
			}
		}
	}
}

static void expected(str)
char *str;
{
	io_printf("Error found in '%s' at line %d; '%s' expected.\n",
		filename, lines, str);
	fatal = 1;
}

static int skipspaces()
{
	int c, skiptoEOL = 0, incomment = 0;
loop:
	c = getc(ifp);
	if (c == EOF) {
		return (EOF);
	} else if (incomment) {
		if (c == '*' && (c = getc(ifp)) == '/')
			incomment = 0;
		else if (c == '*')
			ungetc(c, ifp);
		goto loop;
	} else if (skiptoEOL) {
		if (c == '\n') {
			skiptoEOL = 0;
		}
		goto loop;
	} else if (c == '\n') {
		++lines;
		goto loop;
	} else if (c == ' ' || c == '\t') {
		/* ordinary spaces */
		goto loop;
	} else if (c == '/') {
		if ((c = getc(ifp)) == '/') {
			skiptoEOL = 1;
			goto loop;
		} else if (c == '*') {
			incomment = 1;
			goto loop;
		} else {
			ungetc(c, ifp);
			return ('/');	/* simple '/' */
		}
	} else {
		return (c);
	}
}

static int gettoken()
{
	int c;
	char *p = token;

	c = skipspaces();
	if (c == EOF)
		return (tok_type = EOF);
	else if (isalnum(c)) {
		while (isalnum(c) || (cat && index(" _", c))) {
			if (index(" _", c))
				;	/* skip it. */
			else
				*p++ = c;
			c = getc(ifp);
		}
		*p = '\0';
		ungetc(c, ifp);
		return (tok_type = SY_STR);
	} else {
		*p++ = c;
		*p = '\0';
		return (tok_type = SY_SYM);
	}
}
