/* File dl_net_conl.c 
 *
 * This file contains the routines that make up a simple delay
 * back annotator.  This back annotator reads a file that contains
 * cell output net names and their associated incremental rise and
 * fall delays.  For each net/delay set, the back annotator adds
 * the delays to the primitives connected to the given cell output net
 * within the appropriate cell.  This back annotator is meant to be
 * used with designs containing cells described with distributed
 * delay timing.  The following shows example entries in a delay file:
 *
 *      top.m1.m1.y 10 12
 *      top.m1.m2.y 10 12
 *      top.m1.m3.q 25 27
 *      top.m1.m3.qb 30 35
 *      top.m1.m4.q 25 27
 *      top.m1.m4.qb 30 35
 *
 * To associate these routines with a Verilog-XL system task,
 * place the following entry into the 'veriusertfs' data
 * structure in the 'veriuser.c' file:
 *
 *      s_tfcell veriusertfs[] = {
 *                                  {usertask,0,
 *                                   0,0,
 *                                   dl_read_net_delays,0,
 *                                   "$ba_delays",0},
 *
 *                                  {0}
 *                               };
 *
 * Once this file has been compiled and linked with the Verilog-XL
 * object files provided with the release, the delay back annotator
 * may be called from the Verilog-XL source description to perform
 * delay back annotation.  The system task must be invoked with one
 * argument - the name of the delay back annotation file.  The
 * following shows an example partial source description that invokes
 * the delay back annotator:
 *
 *      module top;
 *         .
 *         .
 *         .
 *         initial
 *            $ba_delays("test_conp.dat");
 *         .
 *         .
 *         .
 *      endmodule
 *
 */

#include <stdio.h>
#include "acc_user.h"
#include "veriuser.h"

#define dlArgsPerLine 3

/********************************************************************************
*  DL_READ_NET_DELAYS
*  This routine reads a file which contains one line entries composed
*  of the following data :
*      > a full hierarchical net name
*      > a rise delay
*      > a fall delay
*  Example file entry :
*          top.m1.dff3.qbar  42.6  38.4
*  This routine then finds all primitives in the same scope as the
*  net which drive the net, and adds the delay specified in the file
*  to the delay which currently exists on the primitives.
*
********************************************************************************/
dl_read_net_delays()
{
    FILE *fdin;
    char *file_name, net_name[256];
    int num_args, line_num = 0;
    double rise, fall;
    handle net, prim, driver;
    bool dl_verbose = false;

    /**********************************/
    /*** Initialize access routines ***/
    /**********************************/
    acc_initialize();
#if 0
    acc_configure(accDevelopmentVersion,"1.5a");
#endif

    /**********************************************************************/
    /*** Configure so that that if acc_replace_delays needs a hiz delay ***/
    /*** it makes it the maximum of the input rise and fall delays      ***/
    /**********************************************************************/
    acc_configure(accToHiZDelay,"max");

    /********************************************/
    /*** Check for +dlverbose on command line ***/
    /********************************************/
    if (mc_scan_plusargs("dlverbose"))
        dl_verbose = true;

    /**********************************/
    /*** Open delay annotation file ***/ 
    /**********************************/
    file_name = tf_strgetp(1,'b');
    if ((fdin = fopen(file_name,"r")) == null) {
        io_printf("file: <%s> not found\n", file_name);
        return;
    }

    /**********************************************/
    /*** Read and process each line in the file ***/
    /**********************************************/
    while ((num_args=fscanf(fdin,"%s %lf %lf",net_name,&rise,&fall)) != EOF)
    {
        /*****************************************************/
        /*** Check for correct number of arguments read in ***/
        /*****************************************************/
        line_num++;
        if (num_args != dlArgsPerLine) {
            io_printf("Input error in file %s at line %d\n",
                file_name, line_num);
            continue;
        }

        /****************************************************/
        /*** Find handle to input net, make sure it is OK ***/
        /****************************************************/
        net = acc_handle_object(net_name) ;
        if ((net == null) || (acc_fetch_type(net) != accNet)) {
            io_printf("Warning from dl_ann : net %s not found\n",net_name);
            continue;
        }

        /*******************************************************************/
        /*** Replace delays on all net drivers in same module as the net ***/
        /*******************************************************************/
        driver = null;
        while (driver = acc_next_driver(net,driver))
        {
            /*** check if driver and net are in same module ***/
            prim = acc_handle_parent(driver);
            if (acc_handle_parent(prim) != acc_handle_parent(net))
                continue;

            if (dl_verbose) {
                double newrise, newfall;
                acc_fetch_delays(prim,&newrise,&newfall);
                io_printf("Primitive %s had added rise fall delays : %d %d\n",
                    acc_fetch_fullname(prim), (int)rise, (int)fall);
            }

            acc_append_delays(driver, rise, fall);

            if (dl_verbose) {
                double newrise, newfall;
                acc_fetch_delays(prim,&newrise,&newfall);
                io_printf("New delays are                          : %d %d\n",
                    (int)newrise, (int)newfall);
            }
        }
    }

    /* clean up access routines */
    acc_close();
}
