/*
 *  This PLI routine is designed to work from inside a module 
 *  definition and change a specified path delay.
 *
 *  USAGE:  $initDelay(in, out, d1, d2, d3, d4, d5, d6);
 *          
 *      in, out:  are the names of the input and output pins that
 *                the pathdelay is associated with.
 *
 *      d1-d6:    Delay values for the path delay.  (d2-d6) are
 *                optional, the routine will just work with the
 *                delays it is given.
 *              
 *                The mapping of the delay inputs are:
 *
 *                  d1   01
 *                  d2   10
 *                  d3   0z  or turn off delay if no others given
 *                  d4   z1
 *                  d5   1z
 *                  d6   z0
 *
 *  To run this task from verilog you need to add the following lines
 *  to your veriuser.c file then recompile the executable:
 *
 *    to the top of the file add:

    extern initDelay();

 *    to the veriusertfs array add:

    {usertask, 0, 0, 0, initDelay, 0, "$initDelay", 1},

*/

#include <stdio.h>
#include "veriuser.h"
#include "acc_user.h"

/*
 *  In an attempt to save scan time for the pulse args
 *  +pulse_r/ and +pulse_e/
 */
static checkPulse = 0;
static double pulseR = 1.0,     /* Default R of 100 % */
              pulseE = 1.0;     /* Default E of 100 % */

int initDelay()
   {
   int    numParam;		/* Number of parameters to PLI routine */
   double delay[6];		/* Hold the different delay values     */
   handle parent, pathDelay;
   handle in, out;		/* Input and output pins for instance  */
   int cnt;
   char *c;

   /*   acc_initialize(); */
   
   /* Get the input and output pins for the path delay */
   in = acc_handle_tfarg(1);	/* Get the input pin */
   out = acc_handle_tfarg(2);	/* Get the output pin */

   /* is OK ? */
   if (in == NULL) {		/* Check for a valid input pin handle */
      tf_error("arg1 is not a valid input pin");
      return 1;
      }

   if (out == NULL) {		/* Check for a valid output pin handle */
      tf_error("arg2 is not a valid output pin");
      return 1;
      }
  
   /* Get the handle to the PLI call's parent */
   parent = acc_handle_parent(in);
   
   /* Get the handle to the path delay */
   acc_configure(accEnableArgs, "acc_handle_modpath");
   pathDelay = acc_handle_modpath(parent, NULL, NULL, in, out);
   if (pathDelay == NULL) {
      tf_error("Could not find path delay");
      return 1;
      }

   /* 
    * Get the new delay value and expand the Z value if needed.
    */
   numParam = tf_nump();
   for (cnt = 0; cnt <= (numParam-2); cnt++) 
     delay[cnt] = acc_fetch_tfarg_real(cnt+3);

   /* Expand out the delay information to any unset Z values */
   switch(numParam) {
    case 2:			/* No delays need to be changed */
      return 0;
      break;

    case 3:
      delay[1] = delay[0];
      acc_configure(accPathDelayCount, "2");
      break;

    case 4:
      acc_configure(accPathDelayCount, "2");
      break;
      
    case 5:
      acc_configure(accPathDelayCount, "3");
      break;

    case 6:
      delay[4] = delay[3];
      delay[5] = delay[3];
      acc_configure(accPathDelayCount, "6");
      break;

    case 7:
      delay[5] = delay[3];
      acc_configure(accPathDelayCount, "6");
      break;

    case 8:
      acc_configure(accPathDelayCount, "6");
      break;

    default:
      tf_error("Too many args only need a max of 8");
      return 1;
      break;
      }

   acc_replace_delays(pathDelay, delay[0], delay[1], delay[2], 
		      delay[3], delay[4], delay[5]);
   

   /* Reset the pulse r,e rejions */
   if (!checkPulse) {
      c = mc_scan_plusargs("pulse_r/");
      if (c != NULL)
	pulseR = atof(c)/100.0;

      c = mc_scan_plusargs("pulse_e/");
      if (c != NULL)
	pulseE = atof(c)/100.0;
      }

   acc_set_pulsere(pathDelay, pulseR, pulseE);


   /*
    *  Thats all folks
    */
   return 0;
   }
