/*************************************************************/
/* file: micro.c                                             */
/* abstract:  This file contains the main function           */
/*            call for reading in a file from a prom         */
/*            and pumping the JTAG ports.                    */
/*                                                           */
/* Notes: There is a compiler switch called DEBUG_MODE.      */
/*        If DEBUG_MODE is defined, the compiler will read   */
/*        the xsvf file from a file called "prom.bit".      */
/*        It will also enable debugging of the code          */
/*        by printing the TDI and TMS values on the          */
/*        rising edge of TCLK.                               */
/*************************************************************/

#include "lenval.h"
#include "ports.h"


#define CLOCK_RATE 150    /* set to be the clock rate of the system in kHz */

/* encodings of xsvf instructions */

#define XCOMPLETE        0 
#define XTDOMASK         1 
#define XSIR             2 
#define XSDR             3 
#define XRUNTEST         4        
#define XREPEAT          7
#define XSDRSIZE         8
#define XSDRTDO          9
#define XSETSDRMASKS     10
#define XSDRINC          11

/* return number of bytes necessary for "num" bits */
#define BYTES(num) \
        (((num%8)==0) ? (num/8) : (num/8+1))

extern void doSDRMasking(lenVal *dataVal, lenVal *nextData, 
			 lenVal *addressMask, lenVal *dataMask);
extern short loadSDR(int numBits, lenVal *dataVal, lenVal *TDOExpected, 
	      lenVal *TDOMask);
extern void clockOutLenVal(lenVal *lv,long numBits,lenVal *tdoStore);
extern void gotoIdle();

lenVal TDOMask;    /* last TDOMask received */
lenVal maxRepeat;  /* max times tdo can fail before ISP considered failed */
lenVal runTestTimes; /* value of last XRUNTEST instruction */


#ifdef DEBUG_MODE
#include <stdio.h>
FILE *in;          /* for debugging */
#endif


/* clock out the bit onto a particular port */
void clockOutBit(short p, short val)
{
    setPort(p,val);  /* change the value of TMS or TDI */
    pulseClock();    /* set TCK to Low->High->Low      */
}

/* clock out numBits from a lenVal;  the least significant bits are */
/* output on the TDI line first; exit into the exit(DR/IR) state.   */ 
/* if tdoStore!=0, store the TDO bits clocked out into tdoStore.    */
void clockOutLenVal(lenVal *lv,long numBits,lenVal *tdoStore)
{
    int i;
    short j,k;
    
    /* if tdoStore is not null set it up to store the tdoValue */
    if (tdoStore)
	tdoStore->len=lv->len;
    
    for (i=0;i<lv->len;i++)
    {
	/* nextByte contains the next byte of lenVal to be shifted out */ 
	/* into the TDI port                                           */
	unsigned char nextByte=lv->val[lv->len-i-1]; 
	unsigned char nextReadByte=0;
	unsigned char tdoBit;
	/* on the last bit, set TMS to 1 so that we go to the EXIT DR */
	/* or to the EXIT IR state */
	for (j=0;j<8;j++)
	{
	    /* send in 1 byte at a time */
	    /* on last bit, exit SHIFT SDR */
	    if (numBits==1)
		setPort(TMS,1);  
	    
	    if (numBits>0)
	    {
		tdoBit=readTDOBit();  /* read the TDO port into tdoBit */
		clockOutBit(TDI,nextByte & 0x1);  /* set TDI to last bit */
		nextByte=nextByte>>1;
		numBits--;
		/* first tdoBit of the byte goes to 0x00000001   */
		/* second tdoBit goes to 0x00000010, etc.        */
		/* Shift the TDO bit to the right location below */
		for (k=0;k<j;k++)
		    tdoBit=tdoBit<<1; 
		
		/* store the TDO value in the nextReadByte */
		nextReadByte|=tdoBit;
	    }
	}
	/* if storing the TDO value, store it in the correct place */
	if (tdoStore)
	    tdoStore->val[tdoStore->len-i-1]=nextReadByte;
    }
}


/* parse the xsvf file and pump the bits */
int main()
{
    lenVal inst; /* instruction */
    lenVal bitLengths; /* hold the length of the arguments to read in */
    lenVal dataVal,TDOExpected;
    lenVal SDRSize,addressMask,dataMask;
    lenVal sdrInstructs;
    long i;
    
#ifdef DEBUG_MODE
    /* read from the file "prom.bit" instead of a real prom */
    in=fopen("prom.bit","rb");
#endif
    gotoIdle();
    while (1)
    {
	readVal(&inst,1);  /* read 1 byte for the instruction */
	switch (value(&inst))  
	{
	case XTDOMASK:
	    /* read in new TDOMask */
	    readVal(&TDOMask,BYTES(value(&SDRSize)));  
	    break;
	case XREPEAT:
	    /* read in the new XREPEAT value */
	    readVal(&maxRepeat,1);
	    break;
	case XRUNTEST:
	    /* read in the new RUNTEST value */
	    readVal(&runTestTimes,4);
	    break;
	case XSIR:
	    /* load a value into the instruction register */
	    clockOutBit(TMS,1);  /* Select-DR-Scan state  */
	    clockOutBit(TMS,1);  /* Select-IR-Scan state  */
	    clockOutBit(TMS,0);  /* Capture-IR state      */
	    clockOutBit(TMS,0);  /* Shift-IR state        */
	    readVal(&bitLengths,1); /* get number of bits to shift in */
	    /* store instruction to shift in */
	    readVal(&dataVal,BYTES(value(&bitLengths))); 
	    /* send the instruction through the TDI port and end up   */
	    /* dumped in the Exit-IR state                            */
	    clockOutLenVal(&dataVal,value(&bitLengths),0);
	    clockOutBit(TMS,1);  /* Update-IR state       */
	    clockOutBit(TMS,0);  /* Run-Test/Idle state   */
	    break;
	case XSDRTDO:
	    /* get the data value to be shifted in */
	    readVal(&dataVal,BYTES(value(&SDRSize)));
	    /* store the TDOExpected value    */
	    readVal(&TDOExpected,BYTES(value(&SDRSize)));
	    /* shift in the data value and verify the TDO value against */
	    /* the expected value                                       */
	    if (!loadSDR(value(&SDRSize), &dataVal, &TDOExpected, &TDOMask))
	    {
		/* The ISP operation TDOs failed to match expected */
		return 0;  
	    }
	    break; 
	case XSDR:    
	    readVal(&dataVal,BYTES(value(&SDRSize)));
	    /* use TDOExpected from last XSDRTDO instruction */
	    if (!loadSDR(value(&SDRSize), &dataVal, &TDOExpected, &TDOMask))
		return 0;  /* TDOs failed to match expected */
	    break; 
	case XSDRINC:
	    readVal(&dataVal,BYTES(value(&SDRSize)));
	    if (!loadSDR(value(&SDRSize), &dataVal, &TDOExpected, &TDOMask))
		return 0;  /* TDOs failed to match expected */
	    readVal(&sdrInstructs,1);
	    for (i=0;i<value(&sdrInstructs);i++)
	    {
		lenVal nextData;
		int dataLength=8; /* fix to be number of 1's in dataMask */
		readVal(&nextData,BYTES(dataLength));
		doSDRMasking(&dataVal, &nextData, &addressMask, &dataMask);
		if (!loadSDR(value(&SDRSize), &dataVal, 
			     &TDOExpected, &TDOMask))
		    return 0;  /* TDOs failed to match expected */
	    }
	    break;
	case XSETSDRMASKS:
	    /* read the addressMask */
	    readVal(&addressMask,BYTES(value(&SDRSize)));
	    /* read the dataMask    */
	    readVal(&dataMask,BYTES(value(&SDRSize)));
	    break;
	case XCOMPLETE:
	    /* return from subroutine */
	    return 1;
	    break;   
        case XSDRSIZE:
	    /* set the SDRSize value */
	    readVal(&SDRSize,4);
	    break;
	    
	}
    }
}



/* determine the next data value from the XSDRINC instruction and store     */
/* it in dataVal.                                                           */
/* Example:  dataVal=0x01ff, nextData=0xab, addressMask=0x0100,             */
/*           dataMask=0x00ff, should set dataVal to 0x02ab                  */
void doSDRMasking(lenVal *dataVal, lenVal *nextData, lenVal *addressMask,
		  lenVal *dataMask)
{
    int i,j,count=0;
    /* add the address Mask to dataVal and return as a new dataVal */
    addVal(dataVal, dataVal, addressMask);  
    for (i=0;i<dataMask->len;i++)
    {
	/* look through each bit of the dataMask. If the bit is    */
	/* 1, then it is data and we must replace the corresponding*/
	/* bit of dataVal with the appropriate bit of nextData     */
	for (j=0;j<8;j++)
	    if (RetBit(dataMask,i,j))  /* this bit is data */
	    {
		/* replace the bit of dataVal with a bit from nextData */
		SetBit(dataVal,i,j,RetBit(nextData,count/8,count%8));
		count++;  /* count how many bits have been replaced */
	    }
    }
}

/* goto the idle state by setting TMS to 1 for 5 clocks, followed by TMS */
/* equal to 0 */
void gotoIdle()
{
    int i;
    setPort(TMS,1);
    for (i=0;i<5;i++)
	pulseClock();
    setPort(TMS,0);
    pulseClock();
}
    

/* return 0 iff the TDO doesn't match what is expected */
short loadSDR(int numBits, lenVal *dataVal, lenVal *TDOExpected, 
	      lenVal *TDOMask)
{
    int failTimes=0;
    while (1)
    {
	lenVal actualTDO;
	int repeat;  /* to do RUNTESTS */
	clockOutBit(TMS,1);  /* Select-DR-Scan state  */
	clockOutBit(TMS,0);  /* Capture-DR state      */
	clockOutBit(TMS,0);  /* Shift-DR state        */
	/* output dataVal onto the TDI ports; store the TDO value returned */
	clockOutLenVal(dataVal,numBits,&actualTDO);
	/* compare the TDO value against the expected TDO value */
	if (EqualLenVal(TDOExpected,&actualTDO,TDOMask))
	{
	    /* TDO matched what was expected */
	    clockOutBit(TMS,1);  /* Update-DR state    */
	    clockOutBit(TMS,0);  /* Run-Test/Idle state*/

	    /* wait in Run-Test/Idle state */
	    waitTime(value(&runTestTimes));
	    break;
	}
	else 
	{
	    /* TDO did not match the value expected */
	    failTimes++;      /* update failure count */
	    if (failTimes>value(&maxRepeat))
		return   0;   /* ISP failed */
	    clockOutBit(TMS,0); /* Pause-DR state      */
	    clockOutBit(TMS,1); /* Exit2-DR state      */
	    clockOutBit(TMS,0); /* Shift-DR state      */
	    clockOutBit(TMS,1); /* Exit1-DR state      */
	    clockOutBit(TMS,1); /* Update-DR state     */
	    clockOutBit(TMS,0); /* Run-Test/Idle state */
	    /* wait in Run-Test/Idle state */
	    waitTime(value(&runTestTimes));
	}
    }
    return 1;  
}
