/*******************************************************/
/* file: ports.c                                       */
/* abstract:  This file contains the routines to       */
/*            output values on the JTAG ports, to read */
/*            the TDO bit, and to read a byte of data  */
/*            from the prom                            */
/*                                                     */
/* Notes: See the notes for micro.c for explanation of */
/*        the compiler switch  "DEBUG_MODE".           */
/*******************************************************/
#include "ports.h"

#ifdef DEBUG_MODE
#include "stdio.h"
extern FILE *in;
#endif

#ifdef DEBUG_MODE
/* if in debugging mode, use variables instead of setting the ports */
short pTCK,pTMS,pTDI;
#endif


/* if in debugging mode, then just set the variables */
void setPort(short p,short val)
{
#ifdef DEBUG_MODE
    if (p==TCK)
	pTCK=val;
    if (p==TMS)
	pTMS=val;
    if (p==TDI)
	pTDI=val;
#endif
}

#ifdef DEBUG_MODE
void printPorts()
{
    printf("%d   %d\n",pTMS,pTDI);
}
#endif

/* toggle tck LHL */
void pulseClock()
{
    setPort(TCK,0);  /* set the TCK port to low  */
    setPort(TCK,1);  /* set the TCK port to high */
#ifdef DEBUG_MODE
    /* if in debugging mode, print the ports on the rising clock edge */
    printPorts();
#endif;
    setPort(TCK,0);  /* set the TCK port to low  */

}


/* read in a byte of data from the prom */
void readByte(unsigned char *data)
{
#ifdef DEBUG_MODE
    /* pretend reading using a file */
    fscanf(in,"%c",data);
#endif
}

/* read the TDO bit from port */
unsigned char readTDOBit()
{
#ifdef DEBUG_MODE
    return 1;  /* garbage value for now; replace with real port read. */
#endif
}


/* Wait at least the specified number of microsec.                           */
/* Use a timer if possible; otherwise estimate the number of instructions    */
/* necessary to be run based on the microcontroller speed.  For this example */
/* we pulse the TCK port a number of times based on the processor speed.     */
void waitTime(long microsec)
{
    int repeat;
#define CLOCK_RATE 150 /* set to be the clock rate of the system in kHz */
    long clockRunTests=microsec*CLOCK_RATE/1000;
    for (repeat=0;repeat<clockRunTests;repeat++)
                pulseClock();
}
