//this module contains all the logic of the design
module pci_tar(CLK, BCLK, FRAME_REGn, FRAME_INTn, IRDY_REGn,
	       IRDY_INTn, IDSEL_REG, TRDY_Dn, STOP_Dn, 
               DEVSEL_Dn, AD_REG, CBE_REG, PAR_D, 
               ADDR_REG, SR_REG, SRCS_INTn, SROE_INTn, SRWE_INTn, 
	       LOC_REQ, LOC_GNT_D, RSTn, ACKOE, PAROE, ADDR_CNT_D,
		SRD_OE, AD_OE, PCI_GNT);	
        
input  CLK, BCLK, FRAME_REGn, FRAME_INTn, IRDY_REGn, IRDY_INTn;
input  IDSEL_REG, RSTn; 
output TRDY_Dn, STOP_Dn, DEVSEL_Dn, PAR_D, PAROE;
input  [31:0] AD_REG, SR_REG; 
input  [3:0] CBE_REG; //PCI command/byte enable    
output [23:6] ADDR_REG;  //SRAM interface Declarations
output [3:0]  SRWE_INTn, ADDR_CNT_D;
output SRCS_INTn, SROE_INTn, LOC_GNT_D, ACKOE, PCI_GNT;
output SRD_OE, AD_OE;
input  LOC_REQ;

reg PAROE, WRrd, SRD_OE, HIT, BAREG_SELQ, SRCS_INTn;
reg AVALID, CONFIG_SPACE;
reg [7:0] BA_AD_REG;
reg [23:6] ADDR_REG;//sram address outputs
reg [3:0] ADDR_CNT, CBEDY;
reg ENDEDQn, HLD2END, HLD2ENDQ, PCI_REQ, ACKOE, AD_OE;
wire BA_WR, INC_ADDR, FRAME_REGn, IRDY_REGn, IDSEL_REG, PCI_REQ_D, NOMORE;  
wire SROE_INTn;    
wire [3:0] ADDR_CNT_D, SRWE_INTn;

//this module is a two way round-robin arbiter controlling access to the sram
arbiter ARB (CLK, RSTn, LOC_REQ, PCI_REQ, PCI_GNT, LOC_GNT_D, PCI_GNT_D);

//this module is a high level control state machine
// it setups up the datapath for the burst
pipcon PPC(CLK, RSTn, PCI_REQ, PCI_GNT, WRrd, ENDEDQn,  
	   TRDY_SM, INC_ADDR, WRGATE, FRSTWR);

//this module contains the same-cycle response handshake logic
//it works by gating pipcon outputs
critpath CRIT(CLK, RSTn, FRAME_INTn, IRDY_INTn, ENDEDn, NOMORE,
	 WRrd, TRDY_SM, STOP_Dn, TRDY_Dn, DEVSEL_Dn, PCI_REQ, IRDY_REGn);

parity PTYG(CLK, CBE_REG, SR_REG, PAR_D); //generate parity on read

//ENDEDn asserts at cycle end
always @(posedge CLK) ENDEDQn <=  ENDEDn;
//HLD2END holds signals from assertion until cycle end
always @(posedge CLK or negedge ENDEDn or negedge RSTn)
begin
         if (~RSTn)   HLD2END = 1'b0;
    else if (~ENDEDn) HLD2END = 1'b0;
    else if (~FRAME_REGn) HLD2END = 1'b1;
end
always @(posedge CLK) HLD2ENDQ <= HLD2END;

//PCI_REQ is the synchronized address decoder output
// it indicates selection for a PCI target access
always @(posedge CLK or negedge RSTn) begin
	 if(~RSTn)               PCI_REQ <= 1'b0;
    else if(~ENDEDn)             PCI_REQ <= 1'b0;
    else if (HLD2END & ~HLD2ENDQ) PCI_REQ <= PCI_REQ_D;
end

//ACKOE controls the STOP#, DEVSEL#, & TRDY# outputs
always @(posedge CLK or negedge RSTn)
begin
    if        (~RSTn) ACKOE = 1'b0;
    else if (~ENDEDn) ACKOE = 1'b0;
    else if (HLD2END & ~HLD2ENDQ) ACKOE = PCI_REQ_D;
end

//AD_OE controls the PCI_AD outputs
always @(posedge CLK or negedge ENDEDn)
begin
	if (~ENDEDn) AD_OE = 1'b0;
	else if (HLD2END & ~HLD2ENDQ) AD_OE = PCI_REQ_D & ~WRrd;
end

// for the SRAM data outputs
always @(posedge CLK) SRD_OE = PCI_GNT_D & WRrd;

//sram address register and counter
always @(posedge CLK)	// latch the address and hold until end    
 if (~FRAME_REGn & ~HLD2END) ADDR_REG <= AD_REG[23:6];

assign NOMORE = PCI_REQ & (ADDR_CNT[3:1] == 3'b111);//addr cntr ovrflw
//this is the mux of a ff-mux cntr structure
//the mux outputs also go to OUTFFT buffers to optimize timing
// increment is inhibited when IRDY_REGn is high -> initiator wait states
assign ADDR_CNT_D = (INC_ADDR & ~IRDY_REGn)? ADDR_CNT + 1: ADDR_CNT;
always @(posedge CLK or negedge RSTn)
   begin
	if (~RSTn)            ADDR_CNT = 4'h0;
	else if (~FRAME_REGn & ~HLD2END & !NOMORE) ADDR_CNT = AD_REG[5:2];
	else  ADDR_CNT = ADDR_CNT_D;
   end

reg NOMOREQ, WR_INHIB;
always @(posedge CLK) begin	//instantiate D ffs
	CBEDY    <= CBE_REG;
	NOMOREQ  <= PCI_REQ & (NOMOREQ | NOMORE);
	WR_INHIB <= IRDY_REGn | NOMOREQ;//for init waits or cntr overflow
end

// SRAM control outputs
wire SRWE3n = WR_INHIB | ~(~CBEDY[3] && WRGATE);
wire SRWE2n = WR_INHIB | ~(~CBEDY[2] && WRGATE);
wire SRWE1n = WR_INHIB | ~(~CBEDY[1] && WRGATE);
wire SRWE0n = WR_INHIB | ~(~CBEDY[0] && WRGATE);
assign SRWE_INTn = {SRWE3n, SRWE2n, SRWE1n, SRWE0n};

always @(WRrd or FRSTWR or WR_INHIB or WRGATE or BCLK)
    casex({WRrd, FRSTWR, WR_INHIB, WRGATE, BCLK})
	5'b0xxxx: SRCS_INTn = 1'b0;
	5'b11xx1: SRCS_INTn = 1'b0;
	5'b10011: SRCS_INTn = 1'b0;
	default:  SRCS_INTn = 1'b1;
    endcase

assign SROE_INTn = WRrd;// the sram output enable signal, internal version

//detect a configuration cycle access and hold resulting signal to end of cycle
always @(posedge CLK or negedge ENDEDn) begin
    	 if (~ENDEDn) CONFIG_SPACE <= 1'b0;
    else if (~FRAME_REGn & ~HLD2END) 
            CONFIG_SPACE <= (CBE_REG[3:1] == 3'b101) & IDSEL_REG;
end

//write/rd is encoded on CBE[0] during address phase
always @(posedge CLK or negedge RSTn) begin
         if (~RSTn)                  WRrd =  1'b0;
    else if (~FRAME_REGn & ~HLD2END) WRrd = CBE_REG[0];
end

//decode valid command during address phase and hold until cycle end
always @(posedge CLK or negedge ENDEDn) begin	
   	if (~ENDEDn) AVALID = 1'b0;
   else if (~FRAME_REGn & ~HLD2END)
   	casex(CBE_REG[3:1])         //respond to config & memory
/*	    3'b001: AVALID = 1'b1; IO */
	    3'b011: AVALID = 1'b1; //Memory
/*	    3'b101: AVALID = 1'b1; Config */
 	    3'b11x: AVALID = 1'b1; //Memory- mult, line, invalidate
	    default: AVALID = 1'b0;
   	endcase
end

// Configuration base address register and address comparator equations
always @(posedge CLK) BAREG_SELQ = (AD_REG[5:0] == 6'b01_0000);
assign BA_WR = CONFIG_SPACE & BAREG_SELQ & WRrd & HLD2END & ~HLD2ENDQ; 

//update the base addres register on BA_WR assertion           
always @(posedge CLK or negedge RSTn) 
begin
         if (~RSTn) BA_AD_REG = 8'h0;
    else if (BA_WR) BA_AD_REG = AD_REG[31:24];  
end

//sample the address comparator output while its looking at address
always @(posedge CLK) HIT = (AD_REG[31:24] == BA_AD_REG);
assign   PCI_REQ_D = (HIT & AVALID) | CONFIG_SPACE; 

always @(posedge CLK or negedge RSTn)
begin		//controls OE of PCI_PAR
	       if (~RSTn) PAROE <= 1'b0;
	else if (~ENDEDn) PAROE <= 1'b0;
	else              PAROE <= AD_OE;//follows PCI_AD by one clock
end

endmodule

`include "d:\xilapp\pci\verilog\critpath.v"
`include "d:\xilapp\pci\verilog\parity.v"
`include "d:\xilapp\pci\verilog\arbiter.v"
`include "d:\xilapp\pci\verilog\pipcon.v"
