/****************************************************************************/
/* MAIN.C  - Skeleton Driver for COFF Loader       Version 4.50 5/92        */
/****************************************************************************/
/*                                                                          */
/* General:  This file is a custom module to interface the generic COFF     */
/*           loader to the TMS320C30 EVM and TMS320C30 XDS500 via the       */
/*           scan interface.   This module can be compiled and linked       */
/*           with the Microsoft 5.1 C compiler tool set.  Other compilers   */
/*           and linkers may work but are not formerly supported.           */
/*                                                                          */
/* Usage:    COFF_load <-options> filename                                  */
/* Options:     -b        clear .bss section                                */
/*              -q        quiet mode (no banner)                            */
/*              -v        verbose (debug)                                   */
/*              -h        help, print options                               */
/*              -n        no execution after load                           */
/*              -p        use I/O address supplied                          */
/*--------------------------------------------------------------------------*/
/* Functions defined in this module:                                        */
/*                                                                          */
/*    main()             - Main driver for loader.                          */
/*    printhelp()        - Print the loader options                         */
/*                                                                          */
/****************************************************************************/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

extern void load_msg(char *, long,long,long,long,long,long);

#define RUN_FROM_RESET     1
#define RUN_FROM_ENTRY     0
/*--------------------------------------------------------------------------*/
/* APPLICATION INTERFACE VARIBLES                                           */
/*--------------------------------------------------------------------------*/
unsigned long   reloc = 0;               /* RELOCATION AMOUNT               */
int             quiet = 0;               /* NO BANNER                       */
int             norun = 0;               /* NO RUN AFTER LOAD               */
unsigned short  emuaddr = 0x240;         /* DEFAULT C30EVM I/O ADDR         */
                                         /* CHANGE TO 0X220 FOR XDS500      */
/*--------------------------------------------------------------------------*/
/* LOADER EXTERNAL VARIBLES AND PROTOTYPES                                  */
/*--------------------------------------------------------------------------*/
extern int     TARG_init(void);
extern int     TARG_execute(int);
extern int     TARG_reset(void);
extern int     COFF_load(char *);
extern void    printhelp(void);

extern FILE   *fin;                  /* COFF INPUT FILE                     */
extern int     verbose;              /* PRINT PROGRESS INFO                 */
extern int     need_data;            /* READ IN RAW DATA                    */
extern int     need_symbols;         /* READ IN SYMBOL TABLE                */
extern int     clear_bss;            /* CLEAR BSS SECTION                   */

/****************************************************************************/
/*                                                                          */
/* MAIN() - Main driver for loader.  Read in options and filename, open     */
/*          COFF file, and call 'COFF_load'.                                */
/*                                                                          */
/****************************************************************************/
void main(argc, argv)
   int argc;
   char *argv[];
{
   char fname[256], *ctmp, *env_str;
   int  i, files = 0;
   need_symbols = 0;                  /* TELL LOADER NOT TO READ SYM TABLE  */

   /*--------------------------------------------------------------*/
   /*  Search the environment for XDS/EVM  i/o address             */
   /*--------------------------------------------------------------*/
   if((env_str = (char *)getenv("D_OPTIONS")) != NULL)
   {
       while( *env_str != '\0')
       {
	   if(((*env_str == '-') || (*env_str == '/')) &&
	      ((*(env_str+1) == 'P') || (*(env_str+1) == 'p')))
	   {
	      env_str +=2;
	      while( *env_str == ' ') env_str++;
              sscanf(env_str, "%x" , &emuaddr);
	      break;
	   }
	   env_str++;
       }
   }
   /*-----------------------------------------------------------------------*/
   /* PROCESS COMMAND LINE ARGUMENTS                                        */
   /*-----------------------------------------------------------------------*/
   for (i = 1; i < argc; ++i)
   {
      char *argp = argv[i];
      if (*argp == '-')                      /* OPTIONS                     */
         while (*++argp) switch(*argp)
         {
            case 'H': case 'h':  printhelp();                       break;
            case 'B': case 'b':  clear_bss = 1;                     break;
            case 'Q': case 'q':  quiet     = 1;                     break;
            case 'V': case 'v':  ++verbose;                         break;
            case 'N': case 'n':  norun     = 1;                     break;
            case 'P': case 'p':  sscanf(argv[++i], "%x",&emuaddr);  break;
            default :            continue;
         }
      else                                   /* FILENAMES                   */
      {
         strcpy(fname, argp);
         files = 1;
      }
   }

   /*-----------------------------------------------------------------------*/
   /*  HOW TO LOAD:                                                         */
   /*                                                                       */
   /*  Loading a program is a three step process.                           */
   /*      1. TARG_init();     Intialize the emulator.                      */
   /*      2. COFF_load();     Load the program.                            */
   /*      3. TARG_execute();  Execute the program.                         */
   /*                                                                       */
   /*  Loading a second program is also a three step process.               */
   /*      1. TARG_reset();    Intialize the emulator.                      */
   /*      2. COFF_load();     Load the program.                            */
   /*      3. TARG_execute();  Execute the program.                         */
   /*                                                                       */
   /*  Function Summary:                                                    */
   /*                                                                       */
   /*  TARG_init() Intialize the TBC or XDS500 to a known good state and    */
   /*              do a softreset on the TMS320C30.  The TARG_init()        */
   /*              function should only be called once in a application     */
   /*              program.                                                 */
   /*                                                                       */
   /*  COFF_load() Load a COFF file to the target.  Should only be called   */
   /*              after a TARG_init() or TARG_reset() function.            */
   /*                                                                       */
   /*  TARG_execute() Will start the program executing from a reset         */
   /*                 condition, from the code entry point, or just set     */
   /*                 the entry point but not execute.  Should only be      */
   /*                 be called following a COFF_load() or TARG_reset()     */
   /*                 function.                                             */
   /*                                                                       */
   /*                 You can run from reset (RUN_FROM_RESET).  Which       */
   /*                 requires a reset vector be set up in your code.  This */
   /*                 is the same as doing a "reset" and then a "runf"      */
   /*                 command in the HLL debugger.                          */
   /*                                                                       */
   /*                 Or you can run from the entry point of your code      */
   /*                 (RUN_FROM_ENTRY). This requires a valid code entry    */
   /*                 point be present in the COFF file. This is the same as*/
   /*                 doing a "runf" command in the HLL debugger.           */
   /*                                                                       */
   /*  TARG_reset()   Do a soft reset of the processor to get it to a known */
   /*                 good state.  The processor is left in a halted state  */
   /*                 following the reset. The COFF_load() and TARG_execute */
   /*                 can be executed after a TARG_reset(). The TARG_reset()*/
   /*                 must be used to stop the processor execution prior to */
   /*                 loading a new source file.                            */
   /*                                                                       */
   /*-----------------------------------------------------------------------*/

   /*-----------------------------------------------------------------------*/
   /* SAY HELLO TO USER AND PROMPT FOR FILENAME IF NOT GIVEN.               */
   /*-----------------------------------------------------------------------*/
   if (!quiet) printf("COFF Loader   v4.50\n");

   if (!files)
   {
      char *p;
      printf("   COFF file [.out]: ");
      fgets(fname, 64, stdin);
      if (p = strrchr(fname, '\n')) *p = '\0';
   }
   /*-----------------------------------------------------------------------*/
   /* INITIALIZE THE SYSTEM PRIOR TO LOADING THE COFF FILE.                 */
   /*-----------------------------------------------------------------------*/
   if(TARG_init())
     { printf("\n>> can't intialize the target system\n"); exit(-1); }

   /*-----------------------------------------------------------------------*/
   /* ADD THE .OUT EXTENSION TO THE COFF FILE IF NOT PRESENT.               */
   /*-----------------------------------------------------------------------*/
   ctmp = strrchr(fname,'.');
   if ((!ctmp) || (ctmp < strrchr(fname,'\\'))) strcat(fname,".out");

   /*-----------------------------------------------------------------------*/
   /* LOAD THE FILE.                                                        */
   /*-----------------------------------------------------------------------*/
   if (COFF_load(fname)) { printf("\n>> error loading file\n"); exit(-1); }
   fclose(fin);

   /*-----------------------------------------------------------------------*/
   /* EXECUTE THE PROGRAM.                                                  */
   /*-----------------------------------------------------------------------*/
   if(TARG_execute(RUN_FROM_ENTRY))
     { printf("\n>> can't run the target system\n"); exit(-1); }

   exit(0);
}

/****************************************************************************/
/*                                                                          */
/* PRINT_HELP() - Print out help screen.                                    */
/*                                                                          */
/****************************************************************************/

void printhelp()
{
    printf(" ******TMS320C30 SCAN BASED LOADER OPTIONS*******\n\n");
    printf("  evmload <-options> filename.out                \n");
    printf(" -h        Help, print options                   \n");
    printf(" -b        Clear .bss section with zeros         \n");
    printf(" -q        Quiet (no banner)                     \n");
    printf(" -n        No execution after load               \n");
    printf(" -p xxxx   Use host I/O address supplied         \n");
    printf(" -v        Verbose (print debug information      \n");
    exit(0);
}

