;**************************************************************************
; FILENAME: c2xxprog.asm - Generic name
;           c2xx_bp0.asm  
;
; DESCRIPTION:
;   Flash Programmer control module with flash algorithms -P
;   called by the JTAG loader, PRG2xxw.exe
;   Uses the on-chip SARAM for algorithms and flash data buffer
;
; GLOBALS:
;
;   unsigned * PRG_bufaddr      Address of buffer for flash/program data
;   unsigned   PRG_bufsize      Size of program data buffer
;   unsigned   PRG_devsize      Size of programmable device
;   unsigned * PRG_paddr        First programming address
;   unsigned   PRG_page         Programming page
;   unsigned   PRG_length       Length of programming data
;   unsigned   PRG_status       Status of programming functions
;
; PUBLIC FUNCTIONS:
;   PRG_init                    Initialize system for programming
;   PRG_program                 Program a block
;   PRG_erase                   Erase a programmable device
;   PRG_verify                  Verify a block
;
; PRIVIATE FUNCTIONS:
;   None
;
; USAGE/LIMITATIONS
;   The global and public function symbol values must be available in the
;   COFF file for the loader to work properly. The functions are not
;   intended to be called by another program.
;
; NOTES:
;   The program needs three memory sections to operate:
;   Actual location of these sections in the memory is defined in the
;   linker command file - *.cmd
;
;       PRG_text    Executable section for this program
;       PRG_parms   Variable section for this program
;       PRG_data    Data buffer used to hold programming data
;
;   The size and location of each section  is user defined.  The
;   loader program will read the value of PRG_bufaddr and PRG_bufsize to
;   determine the location and size of the PRG_data section.
;
; JTAG loader command file format
;   PRG2xxw  -[options] c2xxprog.out       Flashcode.out
;  <JTAG loader)      <Flash algorithm>  <COFF file to be flashed>
;
;   Source : TI Tools group, Houston
;   Version : 1.10
;   Modified for F2xx devices: Sam Saba, TI Houston            12/24/96
; H**************************************************************************

;
; ***************************************************************************
; INCLUDE FILES
; ***************************************************************************
    .mmregs
    .include "svar0.h"     ; Variable declarations
    .ref GPGMJ             ; References Flash algorithms GCLR or GERS or GPGMJ
; ***************************************************************************
; PUBLIC DECLARATIONS
; ***************************************************************************
        .global PRG_init, PRG_program, PRG_erase, PRG_verify, PRG_stop
        .global PRG_bufaddr, PRG_bufsize, PRG_devsize, PRG_paddr, PRG_page
        .global PRG_length, PRG_status,PARMS
        .global PROTECT, FL_ST,FL_END
; ***************************************************************************
; PRIVATE DECLARATIONS
; ***************************************************************************
                       
BUFFER_SIZE .set    200         ;Size of program buffer size
                                ;Can be increased based available
                                ;memory

DEVICE_SIZE .set    0ffffh      ;Size of device to be programmed.
                                ;Default is maximum address range for
                                ;the F2xx

; ***************************************************************************
; Define the PRG_parm section
; ***************************************************************************
    .sect   "PRG_parm"          ;Actual location in memory defined is
PARMS:                          ;linker command file
PRG_bufaddr .word   PrgBuffer   ;Address of buffer for program
PRG_bufsize .word   BUFFER_SIZE ;Tells host of buffer size
PRG_devsize .word   DEVICE_SIZE ;Tells host of device size

*                               ;The following parameters will be redefined by
*                               ;PRG2xxw based on the flashcode.out
                                ;Algorithm Array variables
PRG_paddr   .word   4000h       ;First address to program
PRG_page    .word   0           ;Page to program
PRG_length  .word   0           ;Length of block to program
PRG_status  .word   0           ;Status of programming functions
Temp        .word   0           ;Temp location for verify
PROTECT     .word   0FF00h      ;Enable all 8 segments
FL_ST       .word   4000h       ;Define Flash array start addres
                                ;for CLEAR and ERASE
                                ;0x0000 for flash0
                                ;0x4000 for flash1
FL_END      .word   8000h
FL_LEN      .word   4000h       ;Length of array to be CLEARED
                                ;or ERASED
PRG_END     .word   0ABCDh      ;LOAD SEGMENT END PATTERN
                             

; ***************************************************************************
; Define the PRG_data section
; ***************************************************************************
    .sect   "PRG_data"          ;Flash code data buffer 
PrgBuffer   .space  BUFFER_SIZE*16  ;Initializes buffer for program data
    .sect   "ary_var"           ; Initialize buffer to 0x0000
            .space  16*16
;
; ***************************************************************************
; Define the PRG_text section
; ***************************************************************************
    .sect   "PRG_text"          ; Control and algorithm module

; F**************************************************************************
; Name: PRG_init
; 
; Description
;   Initialize the F2xx device for programming
; 
; Inputs
;   None
; Outputs
;   int PRG_status              Pass =0; Fail = 1;
;
; Notes: Can be used to include device specific initialization before executing
;        Flash algorithms in PRG_erase,PRG_program, PRG_verify and PRG_stop
;
; F**************************************************************************
 
PRG_init:   ldp #PARMS          ; Defines data page
	    splk    #0,PRG_status
            splk    #0,ERROR
            b       PRG_stop
;
; F**************************************************************************
; Name: PRG_program
; 
; Description
;   Transfers a block of data from RAM to Flash/programmable memory
;   PRG2xxw will execute this after PRG_init by default. If the command
;   line does not have any options -e or -v.
;   If the option specified is -e  then the execution sequence will be:
;   PRG_init > PRG_erase > PRG_program and then PRG_stop
;   If the option specified is -v, then the execution sequence will be:
;   PRG_init > PRG_program, PRG_verify and then PRG_stop
; Inputs
;   unsigned *PRG_paddr         Pointer to first address to program
;   unsigned  PRG_page          Page to program, Program == 0, Data == 1
;   unsigned  PRG_length        Number of words to program
;   unsigned *PRG_bufaddr       Pointer to program data buffer
;
; Outputs
;   int PRG_status              Pass =0; Fail = 1;
;
; Notes:
;   The PRG_page value is setup by the host but not used in this case.
;   The assumption is that the programmable device is in program space.
; F**************************************************************************
 
PRG_program:    ldp #PARMS              ;Point to data page 
                                        ;no  calls for Flash algorithms
                lacl  ERROR
                sub   #1
                bcnd  err2,eq
                call  GPGMJ
                splk    #0,PRG_status   ;update algorithm status
                b       PRG_stop        ;Branch to stop
                
err2:           splk    #1,PRG_status   ;Error in Program stage
                b       PRG_stop
;
; F**************************************************************************
; Name: PRG_erase
; 
; Description
;   Erase a programmed device. This module will be executed if the PRG2xxw
;   command line option specified is -e.
; Inputs
;   None:
;
; Outputs
;   int PRG_status      Pass =0; Fail = 1;
;
; Notes:
;   The erase is device specific. 
; F**************************************************************************
 
PRG_erase:
                ldp #PARMS
err1:           splk    #1,PRG_status; Error in CLEAR or Erase
                b       PRG_stop

;
; F**************************************************************************
; Name: PRG_verify
; 
; Description
;   Verify a block of programmed data. This module will be excuted if the
;   command line option in PRG2xxw is -v.
; 
; Inputs                        Defined by host
;   unsigned *PRG_paddr         Pointer to first address to verify
;   unsigned  PRG_page          Page to verify, Program == 0, Data == 1
;   unsigned  PRG_length        Number of words to verify
;   unsigned *PRG_bufaddr       Pointer to verify data buffer
;
; Outputs
;   int PRG_status      Pass =0; Fail = 1;
;
; Notes:
;   The PRG_page value is setup by the host but not used in this case.
;   The assumption is that the programmable device is in program space.
; F**************************************************************************
 
PRG_verify:
                ldp     #PARMS
error:          splk    #1,PRG_status   ;
                b       PRG_stop        ;Generate a breakpoint

;
; F**************************************************************************
; Name: PRG_stop
; 
; Description
;       Exit routine for all programming functions.
; 
; Inputs
;       None
;
;
; Outputs
;       None
;
; Notes:
;       Exit point for all programming functions
;       The ESTOP opcode gets executed as a NOP when not in emulation
;       mode.  The "b $" will keep the program from running off if
;       not in emulation mode.
;
; F**************************************************************************
PRG_stop:

        nop
        .word   0BE90h                  ;SWI instruction
        nop
        b $
                .end



