/************************************************************************
*                            ispCODE4.c                                 *
*                   ispCODE for ispLSI 2000LV Devices                   *
*              Lattice Semiconductor Corp. Copyright 1996.              *
*                                                                       *
* ISP(tm) daisy chain turbo programming is supported by ispCODE.        *
*                                                                       *
* The function of ispCODE is to step the IEEE 1149.1 TAP state machine  *
* in the ispLSI devices for programming and verification. The input file*
* for ispCODE is ispSTREAM file.                                        *
*                                                                       *
* The ispSTREAM file is created by the command line program dld2isp.exe *
* and the ISP Daisy Chain Download program version 1.3 or higher.       *
*                                                                       *
* All ispLSI 2000LV family of devices support the IEEE 1149.1 32-bit    *
* USERCODE. The USERCODE data must be appended into the JEDEC file if it*
* is to be programmed into the device. The UES editor on the ISP Daisy  *
* Chain Download should be used to add or modify the USERCODE data.     *
* It is important to note that USERCODE is only 32 bits long. The extra *
* bits will still be written into the JEDEC file but will be discarded  *
* during USERCODE programming if more than 32 bits of data is entered   *
* on the UES editor.                                                    *
*
* Programming and verification always include all Array Cells and the   *
* entire USERCODE. The USERCODE is programmed to be FFFFFFFF in hex by  *
* default.                                                              *
*                                                                       *
* The procedure of reading USERCODE is as follows:                      *
* 1. Raise TMS to VIH and pulse TCK 6 times to step to Test-Logic Reset *
*    state.																*
* 2. TMS to VIL, pulse TCK 1 time.                                      *
* 3. TMS to VIH, pulse TCK 2 times.                                     *
* 4. TMS to VIL, pulse TCK 2 times to step to Shift-IR state.           *
* 5. TMS stay at VIL, shift in the VERIFY_USERCODE instruction 10111    *
*    into the IR via TDI and pulse TCK. The instruction is shifted in   *
*    from right to left, i.e. 1, 1, 1, 0 and 1. Repeat the instruction  *
*    if there is more than one ispLSI device in the chain. Raise TMS to *
*    VIH while shifting in the last bit.                                *
* 6. Keep TMS at VIH, pulse TCK 2 times to step to Select-DR-Scan state.*
* 7. TMS to VIL, pulse TCK 2 times to step to Shift-DR state.           *
* 8. Keep TMS at VIL, pulse TCK 32 times to shift out the 32 bits       *
*    USERCODE on TDO. Pulse TCK 32 times more if more than one device   *
*    in the chain. Raise TMS to VIH while shifting out the last bit.    *
*                                                                       *
* 9. Keep TMS at VIH and pulse TCK 6 times to go back to Test-Logic     *
*    Reset state to end the procedure.                                  *    
*                                                                       *
* Revision history:                                                     *
*   4.0   Howard Tang   05/16/96 Convert from V3.xx to support ISP(tm)  *
*                                programming using the IEEE 1149.1 TAP  *
*                                for ispLSI 2000LV family of devices.   *
*                                ispLSI 3000 family of devices are also *
*                                supported if they are on bscan         *
*                                daisy chain configuration.             *
*                                Simplify the code of ispstream_pump    *
*                                for easier reading and use.            *
*                                                                       *
************************************************************************/

#include <stdio.h>
#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include "lattice.h"

/*Global variables */
static int      inputport,       /*port address for the input port*/
                outputport,      /*port address for the output port*/
                port;            /*LPT number */
unsigned long   checksum;            
static int      isp_pins=NUL;    /*3.02 holds the value of parallel port*/
                                 /*     intialized to drive all pins to LOW*/
static char     state;           /*4.0  Keep track of the state of the TAP
                                        state machine.*/

/*prototypes*/
void            find_port(void);
int             power_ok(void);
int             port_ok(void);
void            pulse_width(unsigned int milliseconds);
void            execute(void);
void            move_to_id_state(void);
int             ispstream_pump(char *filename, int operation, int *end);
void            error_handler(int rcode, char *message);
void            ispInstruction(unsigned int Length,unsigned char *Command);
unsigned char   GetByte(FILE *fp);
void ReadispSTREAMHeader(FILE *fp, int *ChainLength, int *ErasePulse, 
                         int *ProgramPulse, int *RowLength, 
                         unsigned int *DataSize, int *EraseRowEnd);
void ReadispSTREAMData(FILE *fp, unsigned int *DataSize, unsigned char *Data,
                       int Row, int EraseRowEnd);
void ispAddress(unsigned int Length, unsigned char *Address);

/***************************************************************************
   Function: isp_setpin(byte pins, byte value)

   Purpose:
   To apply the specified value to the pins indicated. This routine will
   likely be modified for specific systems. As an example, this code
   is for the PC, as described below.

   This routine uses the IBM-PC standard Parallel port, along with the
   schematic shown in Lattice documentation, to apply the signals to the
   programming loop. 

   PC Parallel port pin    Signal name
   --------------------    -----------
         2                   out_SDI
         3                   out_SCLK
         4                   out_MODE
         5                   out_ISP
         6                   out_RESET
         7                   DO5
         8                   out_SENSE_CABLE_OUT    
         9                   DO7
         10                  in_SDO
         12                  in_CABLE_SENSE_IN
         15                  in_VCC_OK 
         20                  GND

   Parameters:
         - pins, which is actually a set of bit flags (defined in lattice.h) 
           that correspond to the bits of the data port. Each of the I/O port 
           bits that drives an isp programming pin is assigned a flag 
           (through a #define) corresponding to the signal it drives. To 
           change the value of more than one pin at once, the flags are added 
           together, much like file access flags are.

           The bit flags are only set if the pin is to be changed. Bits that 
           do not have their flags set do not have their levels changed. The 
           state of the port is always manintained in the static global 
           variable isp_pins, so that each pin can be addressed individually 
           without disturbing the others.

         - value, which is either HIGH (0x01 ) or LOW (0x00 ). Only these two
           values are valid. Any non-zero number sets the pin(s) high.

   Returns: nothing.


**************************************************************************/
void isp_setpin(unsigned char pins, unsigned char value)
{
  /* isp_pins is a Global value that keeps track of the current state
     of the pins  */

  if( value ) /* set flagged pins HIGH */
      isp_pins = pins | isp_pins;
  else        /* set flagged pins LOW */
      isp_pins = ~pins & isp_pins;

  /* value is put on Parallel port pins */
  outp(outputport, isp_pins);

} /* isp_setpin() */


/***************************************************************************
   Function: isp_SDO()

   Purpose:
   To get the value of the SDO pin from the input port.

   This routine is specific to the PC parallel port setup, but can easily
   be changed to address each user's hardware.

   Parameters: none

   Returns: The value of SDO, as a byte, with a value of either 0 or 1.

   Notes:
   - This routine uses the I/O port addresses contained in the global
     variable isp_active, declared in the isplsi.c file. If that
     variable is not set, these routines will not work.


****************************************************************************/
unsigned char isp_SDO(void)
{
 /* MUST return either 0x00 or 0x01 */
  return( (unsigned char) ((inp(inputport) & in_SDO ) ? HIGH : LOW) );

} /* isp_SDO() */

/*************************************************************
*                                                            *
*                         FIND_PORT                          *
*                                                            *
*  This procedure tries to locate the isp cable on parallel  *
*  ports 1,2, or 3, and sets inputport, and outputport       *
*  appropriately.  It returns upon successfully finding the  *
*  first port which has the ispDOWNLOAD cable attached, or   *
*  upon failing to find the cable attached to any port.      *
*                                                            *
*************************************************************/

void            find_port()
{
   int rcode;
   port = 0;

   do {                                          /* find cable on port */
      ++port;

      if (port == 1) 
         {
          outputport = outport1;
          inputport = inport1;
         }
      else if (port == 2) 
              {
               outputport = outport2;
               inputport = inport2;
              }
      else {
            outputport = outport3;
            inputport = inport3;
           }
   rcode = port_ok();
   } while ((rcode != 1) && (port < 3));
}


/*************************************************************
*                                                            *
*                          POWER OK                          *
*                                                            *
*  This procedure determines if the ispDOWNLOAD cable has    *
*  power.                                                    *
*                                                            *
*************************************************************/

int             power_ok()
{
   int             rcode,
                   oldrcode;

   outp(outputport, NUL);
   oldrcode = (in_VCC_OK & inp(inputport));
   pulse_width(5);
   rcode = (in_VCC_OK & inp(inputport));
   if ((oldrcode != rcode) || (rcode == 0) || (oldrcode == 0))
      rcode = POWER_OFF;
   return (rcode);
}


/*************************************************************
*                                                            *
*                          PORT OK                           *
*                                                            *
*  This procedure determines if the ispDOWNLOAD cable is     *
*  connected to specified parallel port.                     *
*                                                            *
*************************************************************/

int             port_ok()
{
   int             d1,
                   d2;
   int             rcode;
   outp(outputport, isp_pins + out_SENSE_CABLE_OUT);
   d1 = (in_CABLE_SENSE_IN & inp(inputport));
   outp(outputport, isp_pins);
   d2 = (in_CABLE_SENSE_IN & inp(inputport));


   if ((d2 != d1) || (NUL > 0))
      rcode = 1;
   else
      rcode = CABLE_NOT_FOUND;
   return (rcode);
}

/*************************************************************
*                                                            *
*                         PULSE_WIDTH                        *
* This procedure is tailored to providing accurate delay in  *
* milliseconds on PC DOS utilizing the down counter of the   *
* timer chip.                                                *
* Users must devise their own timing procedures to ensure    *
* the specified minimum delay is observed when using         *
* different platform.                                        *
*************************************************************/


void            pulse_width(unsigned int milliseconds)
{
   static unsigned int ticks_per_ms = 0;
   static int      mode2 = false;
   static long     pre_ticks = 0L;
   unsigned int    start,
                   end;
   long            ticks,
                   cur_ticks;
   static int      count = 0;


   cur_ticks = 0L;
   outp(0x43, 0x00);
   start = inp(0x40);
   start += inp(0x40) << 8;
   if (ticks_per_ms == 0) {
      ticks_per_ms = (0xFFFF / 55) * 2;
      if (start & 0x0001) {
         ticks_per_ms = 0xFFFF / 55;
         mode2 = true;
      }
   }
   ticks = (long) ticks_per_ms *milliseconds;
   while (ticks > 0L) {
      if (pre_ticks == 0L)
         pre_ticks = cur_ticks;
      else if (count < 5) {                      /* calibration count *//* keep
                                                  * the most conservative ticks
                                                  * per loop as calibrated
                                                  * value */
         if (pre_ticks > cur_ticks)
            pre_ticks = cur_ticks;
         count++;
      }
      outp(0x43, 0x00);
      end = inp(0x40);
      end += inp(0x40) << 8;
      if ((!mode2) && (end & 0x0001)) {
         ticks_per_ms = 0xFFFF / 55;
         ticks -= (long) ticks_per_ms *milliseconds;
         mode2 = true;
         if (ticks <= 0L)
            break;                               /* done already */
      }
      cur_ticks = (long) (start - end);          /* number of ticks elapsed */
      if (cur_ticks == 0L) {                     /* this can happen only when
                                                  * in Window DOS */
         cur_ticks = pre_ticks;
         ticks -= cur_ticks;
      } else if (cur_ticks > 0L)
         ticks -= cur_ticks;
      else {
         cur_ticks = (long) (0xFFFF - end + start);
         ticks -= cur_ticks;
      }
      /* safe guard against stolen ticks in Window DOS */
      if ((pre_ticks > 0L) && ((cur_ticks - pre_ticks) > (ticks_per_ms / 4))) {
         ticks += (cur_ticks - pre_ticks);
         cur_ticks = pre_ticks;
      }
      start = end;
   }
}



/*************************************************************
*                                                            *
*                      GETBYTE                               *
* This procedure reads a byte from the ispSTREAM and update  *
* the checksum.                                              *
*************************************************************/
unsigned char GetByte(FILE *fp)
{
 unsigned char data;
     data = fgetc(fp);
     checksum += data;
     return (data);
}


/*************************************************************
*                                                            *
*                      SCLOCK                                *
* This procedure apply a clock to TCK.                       *
*************************************************************/
void sclock(void)
{
 isp_setpin(out_SCLK,HIGH);
 isp_setpin(out_SCLK,LOW);
}


/*************************************************************
*                                                            *
*                       SHIFT/EXECUTE                        *
* This procedure walk all devices in the daisy chain from a  *
* given state to the next desirable state.                   * 
*                                                            *
*If at Idle state when enter then move to Shift-IR state.    *
*If at Shift-IR state when enter then move to Shift-DR state.*
*If at Shift-DR state when enter then move to Shift-IR state.*
*If at Shift-IR state and state is set to time_state when    *
*enter then move to Idle state.                              *
*************************************************************/
void execute()
{
 int i,count;

     if (state==time_state) 
        {count = 1;   /*move from shift-IR to Idle state to start timing*/ 
         state = idle_state;
        }
     else if (state==shift_state) 
             {count = 2;  /*move from shift-IR to shift-DR*/
              state = execute_state;
             } 
     else if (state==idle_state)
             {count = 2;  /*move from Idle to shift-IR state*/
              state = shift_state;
             }
     else if (state==execute_state)
             {count = 3;  /*move from shift-DR to shift-IR*/
              state = shift_state;
             }
     else count = 0;
     isp_setpin(out_SCLK,LOW);
     isp_setpin(out_MODE,LOW);
     isp_setpin(out_SDI,LOW);
     for (i=0; i<count; i++)
         {
          isp_setpin(out_MODE,HIGH);
          sclock();
         }
     isp_setpin(out_MODE,LOW);
     sclock();
     if (count > 1) sclock();
       /* at Shift-DR or Shift-IR state*/

}


/*************************************************************
*                                                            *
*                      MOVE TO ID STATE                      *
* This procedure walk all devices in the daisy chain to the  *
* Test-Logic-Reset state then Run-Test/Idle state.           *
*                                                            *
*************************************************************/

void move_to_id_state()
{
int i;
       isp_setpin(out_SCLK,LOW);
       isp_setpin(out_MODE+out_SDI,LOW);
       pulse_width(2);
       isp_setpin(out_MODE,HIGH);            
       pulse_width(2);
       isp_setpin(out_SCLK,HIGH);
       pulse_width(2);
       isp_setpin(out_SCLK,LOW);
       for (i=0; i<6; i++) sclock();
       isp_setpin(out_MODE,LOW);
       pulse_width(2);
       sclock();   /*device is in Idle state*/
       state = idle_state;
}

/*************************************************************
*                                                            *
*                      PROGRAM ENABLE                        *
* perform the 3 read id instruction to get devices into      *
* programming mode.                                          *
*************************************************************/

void program_enable(int chips)
{
int i,j,loops;

execute();           /*devices in Shift-IR state*/
for (loops=0; loops<3; loops++)
    {
     for (i=0; i<chips+1; i++)
         for (j=0; j<5; j++)  /*send the instruction into TDI*/
             {if ((i==chips)&&(j==4)) isp_setpin(out_MODE,HIGH);  /*last bit*/ 
              isp_setpin(out_SDI,(PROGRAM_ENABLE >> j) & 0x01);
              sclock();
             }     
     for (i=0; i<3; i++) sclock();
     isp_setpin(out_MODE,LOW);
     sclock(); sclock(); /*devices at Shift-IR state*/
     
    }     
}

/*************************************************************
*                                                            *
*                      DEVICE ENABLE                         *
* perform the read id instruction to get devices into        *
* functional mode.                                           *
*************************************************************/

void device_enable(int chips)
{
int i,j;

/*devices come in at Shift-IR state*/
     for (i=0; i<chips+1; i++)
         for (j=0; j<5; j++)  /*send the instruction into TDI*/
             {if ((i==chips)&&(j==4)) isp_setpin(out_MODE,HIGH);  /*last bit*/ 
              isp_setpin(out_SDI,(PROGRAM_ENABLE >> j) & 0x01);
              sclock();
             }     
     sclock();                /*devices at Update-IR state*/
     isp_setpin(out_MODE,LOW);
     for (i=0; i<20; i++) sclock();   /*go and loop on Run-Test/Idle for 10uS*/
     isp_setpin(out_MODE,HIGH);
     for (i=0; i<3; i++) sclock();   /*devices at Test-Logic-Reset and active*/
}     

/*************************************************************
*                                                            *
*                     READ_ISPSTREAM_HEADER                  *
* Extract the header from the ispSTREAM file.                *
* Return													 *
*     ChainLength--------number of ispLSI devices in the     *
*                        given daisy  chain.                 *
*     ErasePulse---------pulse width in mS for device        *
*                        bulk erase.                         *
*     ProgramPulse-------pulse width in mS for device        *
*                        row by row programming.             *
*     DataWidth----------largest data size of a row.         *
*     EraseRowEnd--------end of the post bulk erase          *
*                        verification.                       *
*************************************************************/
void ReadispSTREAMHeader(fp, ChainLength, ErasePulse, ProgramPulse,
                           RowLength,DataSize,EraseRowEnd)
   int             *RowLength,
                   *ErasePulse,
                   *ProgramPulse,
                   *ChainLength;
   unsigned int    *DataSize;
   int             *EraseRowEnd;
   FILE           *fp;
                          
{
   *ChainLength = GetByte(fp);              /*Byte 1 of ispSTREAM is number of*/
   *ErasePulse = GetByte(fp) * 0x100;
   *ErasePulse += GetByte(fp);
   *ProgramPulse = GetByte(fp);                 /*Byte 4 contains the width of the*/
   *RowLength = GetByte(fp) * 0x100;
   *RowLength += GetByte(fp);
   *DataSize = GetByte(fp) * 0x100;                        /*ispSTREAM file.*/
   *DataSize += GetByte(fp);   
   *EraseRowEnd = GetByte(fp) * 0x100;
   *EraseRowEnd += GetByte(fp);
   /*End of the header of the ispSTREAM file.*/   
   
}


/*************************************************************
*                                                            *
*                     READ_ISPSTREAM_DATA                    *
* Extract one row of data from the ispSTREAM file.           *
* Note that the ispSTREAM file does not observe byte boundary*
* Return													 *
*    DataSize-----The data size in bits of the current row.  *
*    Data---------The buffer contains the data of the current*
*                 row.                                       *
*    Row----------The ispSTREAM address of the current row.  *
*    EraseRowEnd--The last post bulk erase verify row.       *
*************************************************************/
void ReadispSTREAMData(fp, DataSize, Data, Row, EraseRowEnd)
FILE *fp;
unsigned int *DataSize;
unsigned char *Data;
int Row, EraseRowEnd;
{
 static in_col=0;
 static unsigned char curch;
 unsigned int i;    /*maximum data_lenght is 0xFFFF*/
 int condition;
 *DataSize = 0;
 for (i = 0; i<16; i++)   /* Read the 16 bits length of the current row.*/
     {--in_col;
      if (in_col < 0) 
         {
          in_col = 7;                       
          curch = GetByte(fp);                
         }
      if ((curch >> in_col) & 0x01)
               *DataSize |= 0x0001 << 15-i;
      }
 condition = (Row-9) % 12;
 if ((Row < EraseRowEnd) && ((condition==3)||(condition==8)))
    { for (i=0; i < *DataSize/8+1; i++) Data[i] = 0xFF;
      return;
    }
 for  (i=0; i < *DataSize/8+1; i++) Data[i] = 0x00; /*initialize the data to 0s*/
 
 for  (i=0; i < *DataSize; i++) 
      {--in_col;
       if (in_col < 0) 
         {
          in_col = 7;                       
          curch = GetByte(fp);                
         }
       if ((curch >> in_col) & 0x01)
           Data[i/8] |= 0x01 << 7-(i%8);
      }
      
 
}

/*************************************************************
*                                                            *
*                     ispInstruction                         *
* Send the instruction streams to the devices.               *
*    Length-------The size of the instruction stream.        *
*    Command------The buffer contains the command stream.    *
*************************************************************/
void ispInstruction(Length, Command)
unsigned int Length;
unsigned char *Command;
{
 int in_col,in_pos;
 unsigned int index;
 unsigned char curch;
 in_col=0;
 in_pos=0;  
 
 for (index = 0; index < Length; index++) { 
      --in_col;                                  
      if (in_col < 0) {                          
               curch=Command[in_pos++];
               in_col = 7;  
            }
      if (index==Length-1) isp_setpin(out_MODE,HIGH); /*last bit*/
      isp_setpin(out_SDI, (curch >> in_col)&0x01);
      sclock();
     }
 isp_setpin(out_MODE,LOW);
 execute(); 
}

/*************************************************************
*                                                            *
*                     ispAddress                             *
* Send the instruction streams to the devices.               *
*    Length-------The size of the instruction stream.        *
*    Address------The buffer contains the addres stream.     *
*************************************************************/
void ispAddress(Length, Address)
unsigned int Length;
unsigned char *Address;
{
 int in_col,in_pos;
 unsigned int index;
 unsigned char curch;
         
 in_col=0;
 in_pos=0;
 for (index = 0; index < Length; index++) { 
      --in_col;                                  
      if (in_col < 0) {                          
               curch = Address[in_pos++];
               in_col = 7;  
            }
      if (index==Length-1) isp_setpin(out_MODE,HIGH); /*last bit*/
      isp_setpin(out_SDI, (curch >> in_col)&0x01);
      sclock();
     }
 isp_setpin(out_MODE,LOW);
 execute();   
}

/*************************************************************
*                                                            *
*                     ISPSTREAM_PUMP                         *
*                                                            *
*************************************************************/

int   ispstream_pump(char *filename, int operation, int *end)
{
   unsigned int    n,
                   error,
                   length,
                   index,
                   width;
   int             rcode,
                   i,
                   j,
                   out_pos,
                   out_col,
                   in_col;
   unsigned char  *data,
                  *buf,
                  *buff,
                   curch,
                   sdo,
                   cur_bit,
                   uch;
   char            do_shift_instruction,
                   condition;
   char            do_program,
                   do_erase,
                   do_shift_in_data,
                   do_shift_out_data;
   char            do_verify_delay,
                   done_shift_in_data,
                   xch,
                   ych;
   unsigned int    bit_1_count;
   register int    row;
   int             last_row,
                   erase_pulse,
                   program_pulse,
                   devices_in_chain;
   unsigned int    maxi_data,
                   data_length;
   int             last_be_row;
   FILE           *fp;
   unsigned char   var[2];

   if ((fp = fopen(filename, "rb")) == NULL)   /*Look for ispSTREAM file */ 
      return FILE_NOT_FOUND;
   /*Start reading the header of the ispSTREAM file*/
   checksum = 0;                               /*Initialize the checksum*/
   uch = GetByte(fp);                          /*Read the ispSTREAM file type*/
   if ( uch != 0xF0)                           /*Check for correct file type*/
       {
        fclose(fp);
        return FILE_NOT_JEDEC;
       }
   ReadispSTREAMHeader(fp, &devices_in_chain, &erase_pulse, &program_pulse,
                           &last_row, &maxi_data, &last_be_row);

   /*Check the switches for verify devices only or verify UES only. It is
     assumed that the ispSTREAM file was generated with the operation PV.*/
   
   if ((operation==verify) || (operation==verify_ues))
       {program_pulse = 0;       /*skip programming and erase*/ 
        erase_pulse = 0;}
   
   xch = LOW;                                 /*out_SDI = 0 by default.*/
   rcode = OK;                                /*Success by default.*/
   
   /*Allocate memory to store one row of data from the ispSTREAM file.*/
   if ((buf = (unsigned char *) (malloc(maxi_data / 8 + 1))) == NULL) {
      free(buf);
      return ABORT;
   }

   /*Allocate memory to store data for comparison.*/
   if ((buff = (unsigned char *) (malloc(maxi_data / 8 + 1))) == NULL) {
      free(buf);
      free(buff);
      return ABORT;
   }
   move_to_id_state();                           /*Load the IDs*/
   isp_setpin(out_MODE,HIGH);
   sclock();
   isp_setpin(out_MODE,LOW);
   sclock();
   sclock();
   state = execute_state;                        /* Devices at Shift-DR state*/
   done_shift_in_data = false;                   /* Flag to turn on simultaneous*/
                                                 /* in/out data shifting.*/
   error = 0;                                    /* Verify failure.*/
   length = 0;                                   /* Number of bits of data to shift out*/
   data = buff;                                  /* store output data*/
   bit_1_count=0;                                /* Skip programming for all 1s row*/
   
   /*start processing the rows in the ispSTREAM file*/
   for (row = 0; row <= last_row; row++) {
      *end = row;                                /* The row number where process ended.*/
      condition = 0;                             /* 13 possible cases in the ispSTREAM file.*/
      do_program = false;                        /* Flag to execute the program instruction.*/
      do_erase = false;                          /* Flag to execute the erase instruction.*/
      do_shift_in_data = false;                  /* Flag for the current data row is JEDEC data.*/
      do_shift_out_data = false;                 /* Flag to shift out and verify data from devices.*/ 
      do_verify_delay = false;                   /* Flag to execute the verify instructions.*/
   
   /*read one row of data from the ispSTREAM file*/   
      ReadispSTREAMData(fp,&data_length,data,row,last_be_row); 
      width = data_length;  /*The number of bits fetched from the ispSTREAM file.*/
      if (row < 9) {                             
         if (data_length == 0)                        /* empty row */
            condition = 9;                      
         else
            condition = row;
         switch (condition) {
            case 0: do_shift_out_data = true;         /* Verify 32-bit IDs*/
                    break;
            case 2: ispInstruction(data_length,data); /* 8-bit ID read command*/
                    break;
            case 3: do_shift_out_data = true;         /* Verify 8-bit IDs*/
                    data = buf;                       /* store input data*/
                    break;            
            case 8: move_to_id_state();               /* Reset the devices.*/
                    program_enable(devices_in_chain); /* Set devices to programming mode*/
                    if (erase_pulse > 0)  {          
                       do_erase = true;                            
                       state = time_state;            /* Step to Run-Test/Idle to erase*/
                       ispInstruction(data_length,data); /*Erase commands*/
                       } 
                    break;
            case 9:                                      /*Do nothing*/
            default: break;
         }
      } else {                                   /*Program and Verify JEDEC patterns and UES.*/
         if (data_length == 0)                   /*Current row data is blank or not action row.*/
            condition = 13;                    
         else if (row == last_row)
            condition = 12;                      /*The last row is always for security fuse. */

         /*If the ispSTREAM file was built with "PV" operation and you want to force it to
           perform "V" operation only by setting erase pulse width to 0, then
           all the relevant instructions to perform the post Bulk Erase verify
           must be trashed and voided.*/
         /*If the ispSTREAM file was built with "V" operation then last_be_row would be set
           to 9 to skip this section of the code.*/

         else if (((erase_pulse == 0) && (row < last_be_row)) 
                  ||((operation==verify_ues)&&(row <= last_row-DATAROW)))
         
         {/*skip all post bulk erase verifications or skip all operations
            execept verify ues*/
                                
            condition = 13;     
         } 
         
        /*Branch to here if no "PV" operation override or ready to start the process
          exactly per the ispSTREAM file.*/          
         else  
            condition = (row - 9) % 12;         /*From row 9 till the last_row, the ispSTREAM
                                                  file is constructed in many loops of 12 
                                                  cases per loop.*/
         switch (condition) {                
            case 0:                             /* address shift instructions.*/
            case 2:                             /* data shift instructions.*/ 
            case 7:                             /* data shift instructions.*/
               ispInstruction(data_length,data); 
               if (row < last_be_row) data = buff;  /*save data for verify*/
               else data = buf;
               break; 
            case 1:                             /* address data.*/
               ispAddress(data_length,data);
               break;                       
            case 3:                             /* High Order JEDEC data.*/
            case 8:                             /* Low Order JEDEC data.*/
               if (row > last_be_row)           /* In post Bulk Erase verify?*/
                  do_shift_in_data = true;      /* No! Set the flag to shift data into the 
                                                   data shift registers for program and verify
                                                   or just verify.*/
               if (done_shift_in_data)          /* If the verify instruction has been executed,*/
                  do_shift_out_data = true;     /* then shift the data out from devices for*/
                                                /* verification.*/
               done_shift_in_data = true;       /* Set the flag to show that data registers of
                                                   devices already have data shifted in. So 
                                                   verify can be done when encounter the next
                                                   data shift instructions.*/
               length = width;                  /* Remember the number of JEDEC data bits 
                                                   shifted into the data shift registers of 
                                                   devices. The same number of bits must be
                                                   shifted out for verification.*/ 
               data = buf;                      /* Temp storage for one row of ispSTREAM*/
               bit_1_count = 0;                 /* Reset the all 1 bits count.*/
               break;
            case 4:                              /* Program High Order Data instructions.*/
            case 9:                              /* Program Low Order Data instructions.*/
                                                 /* Skip programming if overrided, no data
                                                    was shifted into devices or the data
                                                    shifted into devices are all 1s.*/
                  if ((program_pulse > 0) && (length > 0) && (bit_1_count != length))
                     {state = time_state;
                      ispInstruction(data_length,data); 
                      do_program = true;}        /* Set the flag to wait for programming delay*/
                  break;
            case 12:                             /* Program security cells*/
                  if (program_pulse > 0)         /* Skip programming if overrided.*/
                     {state = time_state;
                      ispInstruction(data_length,data); 
                      do_program = true;}
                  break;
            case 5:                              /* Instructions to verify the High Order Data.*/
            case 10:                             /* Instructions to verify the Low Order Data.*/
               state = time_state;
               ispInstruction(data_length,data);  
               do_verify_delay = true;           /* Set the flag to wait for the verify delay 
                                                    after the verify instructions are shifted
                                                    into the devices and executed.*/ 
               break;
            case 6:                              /* Data shift instructions for High Order Data.*/
            case 11:                             /* Data shift instructions for Low Order Data.*/
               done_shift_in_data = false;       /* The data shift instructions are for shifting
                                                    data out from devices only.*/
               width = length;                   /* The number of data bits to be shifted out
                                                    from devices is the same as what has 
                                                    already been shifted into the devices 
                                                    previously at case 3 or 8. */
               ispInstruction(data_length,data); 
               do_shift_out_data = true;         /* Set the flag to shift data out from devices.
                                                    after the data shift instructions were
                                                    shifted into the devices and executed.*/
               break;
            case 13:                             /* A null row in the ispSTREAM file.*/
                    if ((condition==3)||(condition==8))   /*No data was shifted into devices*/
                          length=0;                 
               break;
            default:                             
               break;
         }
      }
      

      /* If the data shift instructions were shifted into devices and executed
         above, then perform data shift in, shift out or simultaneous shift in and out.*/
      
      if ((do_shift_in_data) || (do_shift_out_data)) 
         {
          printf("Operating On ispSTREAM Row %04d", row);
          uch = 13;                                 /*Carriage return without linefeed*/
          printf("%c", uch);
         out_pos = 0;                               /* Byte pointer points to the 1st byte
                                                       of the previously stored data.*/
         out_col = 0;                               /* Bit pointer points to the most significant
                                                       bit of the 1st byte of the stored data.*/
         xch = HIGH;
         for (index = 0; index < width; index++) 
             {
              --out_col;
              if (out_col < 0) {                      /* Complete a Byte of stored data?*/
                  uch = buff[out_pos];
                  curch = buf[out_pos];
                 if (do_shift_in_data) buff[out_pos] = curch;
                 out_pos++;                           /* Yes! Move the Byte pointer to the next byte.*/
                 out_col = 7;                         /* Reset the bit pointer to the most significant
                                                       bit of the next byte.*/
                }
              if (do_shift_in_data)                   /* Shift data bit by bit into devices.*/
                 {xch = (curch >> out_col) & 0x01;    /* The current bit a 1 ?*/
                  if (xch) bit_1_count++;
                 }
              if (do_shift_out_data) {                /* Shift data out bit by bit from devices.*/
                  cur_bit = (uch >> out_col) & 0x01;     
               sdo = isp_SDO();                       /* Read a bit of data from the SDO pin.*/
               if (sdo != cur_bit) {                  /* Data read does not match with 
                                                         data expected.*/
                  if (row == 0 )                      /* If fail at ID rows then flag fail
                                                         ID check.*/  
                     rcode = UNKNOWN_CHIP;
                  else
                     rcode = VALIDATION_ERROR;        /* Program and verify fail.*/ 
                  error++;
               }
            }                                    
            /* The rising edge of the clock shifts a bit of data into devices.
               The falling edge of the clock shifts out a bit of data from 
               devices.*/
            if (index==width-1)                   /*last bit*/ 
               isp_setpin(out_MODE,HIGH);         /*set devices to Exit-DR state*/
            isp_setpin(out_SDI, xch);             /* Drive the SDI pin.*/
            sclock();
         }
       execute();                                 /* Devices at Shift-DR state*/
      }  /*End of shifting in and/or out of data from Data Shift Registers
           of devices.*/

      else if ((do_erase) || (do_program)) {          
            sclock();
            if (do_erase) pulse_width(erase_pulse);
            else if (do_program) pulse_width(program_pulse);
            isp_setpin(out_MODE, HIGH);          /*set devices to DR-Scan state*/
            pulse_width(1);
            isp_setpin(out_SCLK, HIGH);          /*end erase timing*/
            pulse_width(1);                      /* 1ms clock width to discharge*/
            isp_setpin(out_SCLK, LOW);
            sclock();
            isp_setpin(out_MODE,LOW);
            sclock(); sclock();
            state = shift_state;                 /*devices at Shift-IR state*/
            }                           
      else if (do_verify_delay) {
            isp_setpin(out_SCLK, HIGH);
            for (i = 0; i < 30; i++)             /* simulate a 30uS min delay */
                 isp_setpin(out_SCLK, LOW);
            
            execute();                           /* Back to Shift-IR state*/
         }
      if (error > 0) {                           /* Terminate operation if 
                                                    failure occured.*/
                     break;
                     }
   } /*End of for (row = 0; row <= last_row; row++) */

   /*Fetch from the ispSTREAM file the 16 bit ispSTREAM file checksum and
     compare it with the calculated checksum while reading the ispSTREAM
     file to ensure that there was no data lost.*/
   maxi_data = fgetc(fp) * 0x100;
   maxi_data += fgetc(fp);   /*checksum in the ispSTREAM file*/
   fclose(fp);
   if ((error == 0) && ((checksum & 0xFFFF)!=maxi_data)) {
      error++;
      rcode = FILE_ERROR;
   }  /* End of ispSTREAM file checksum comparison.*/
   device_enable(devices_in_chain);

   if (buf != NULL)
      free(buf);
   if (buff != NULL)
      free(buff);
   if (error > 0) {
      return (rcode);
   } else 
   return (OK);

}


/*************************************************************
*                                                            *
*                         error_handler                      *
*                                                            *
*  rcode                  - error code                       *
*                                                            *
*                                                            *
*  This procedure return the address of the message string   *
*  for the corresponding error code.                         *
*                                                            *
*************************************************************/

void error_handler(int rcode, char *message)
{

/*added by ht for error handling*/
char *error_message[] = {{"PASS"},{""},{""},{"PC Hardware Problem"},{"Chip and JEDEC File Doesn't Match"},
                   {"Can't Find the Download Cable"},{"The VCC on Board Is Not On"},
                   {"No Device Found/Open Download Cable/Open Daisy Chain"},
                   {"Can't Find the File"},{"The File Is Not Complete"},{"Not Enough PC Memory"},
                   {"Verification Fail"},{"The Device is Secured"},{"Non isp Part Found"},
                   {"The File Contain Operation Not Supported in This Mode"},
                   {"The Chip Count in the Chain and the File Do Not Match"},
                   {"Unknown Device Found"},{"Wrong File Type"},{"File Error"},
                   {"The File Has Invalid UES Characters"}};

 strcpy(message, error_message[-rcode]);
} 




/*************************************************************
*                                                            *
*                            MAIN                            *
*                                                            *
*************************************************************/

int             main(int argc, char *argv[])
{
   int             rcode = OK,
                   end = 0;
   FILE           *fpr;
   char           *ptr,
                   filename[300],
                   str[300];
   
   int operation=prog_verify;  /*3.05 Default to Program and Verify*/
   
   printf("\n                 Lattice Semiconductor Corp.\n");
   printf("\n              ispCODE V4.0 Copyright 1995,1996\n");
   printf("\n    ispSTREAM Driver for ispLSI 2000LV Family of Devices\n");
   printf("\n                   IEEE 1149.1 Compliance\n\n");

   if ((argc < 2) || (argc > 3)) {
      printf("\nUsage: jturbo [drive:][path]isp_filename [operation]\n");
      printf("Example: jturbo my_file.isp pv\n");
      printf("\n");
      printf("isp_filename         The ispSTREAM File Created From A Valid \n");
      printf("                     Daisy Chain Configuration File\n");
      printf("operation            \"pv\" = program and verify\n");
      printf("                     \"v\" = Verify only\n");
      printf("                     \"uv\" = Verify UES only\n");
      printf("                     Default to pv if operation is not entered.\n");							
      exit(1);
   }
   if ((fpr = fopen(argv[1], "r")) == NULL) {
      printf("\n%s File not found!\n", argv[1]);
      exit(1);
   }
   fclose(fpr);
   strcpy(str, argv[1]);
   ptr = strchr(str, '.');
   if (ptr)
      *ptr = '\0';
   sprintf(filename, "%s.isp", str);
   strcpy(str, argv[1]);
   if (stricmp(str, filename)) {
      printf("\nFile %s Is Not An ispSTREAM File\n", str);
      exit(1);
   }
   

   /*3.02 set the initial condition of the parallel port*/
   /*     All the port_pins defined on lattice.h will be controlled by
          ispstream_pump(). The rest can be initialized here either to
          HIGH or LOW and won't be altered by isptream_pump*/
            
   isp_pins = NUL;          /*3.02 intialize to drive all port pins LOW*/
   isp_setpin(out_ISP, LOW);/*4.0  drive ispEN pin low to enable the 
                                   IEEE 1149.1 TAP controlling pins:
                                   TCK,TMS,TDI and TDO.*/ 
   find_port();             /*look for the parallel port with download cable*/
   rcode = port_ok();       /*verify if the parallel port found is valid*/
   if (rcode > 0) {
      rcode = power_ok();   /*check if the VCC on Board is on*/
      if (rcode > 0){        /*start programming and verification*/
                
          if(!strcmp(strlwr(argv[2]),"uv")){                  /*3.05 What to do? uv=ues verify only*/
          	operation=verify_ues;    								  /*for verify ues only =1*/  
          	printf("\nStart Verify UES of ispLSI devices.......\n");}  
          else if(!strcmp(strlwr(argv[2]),"v")){               /*v = verify only*/
          	operation=verify;
          	printf("\nStart Verify.......\n");}
          else { 									          /*pv = program and verify: Default*/
          	    operation=prog_verify;
          	    printf("\nStart Program and Verify.......\n");
               }	

   rcode = ispstream_pump(filename, operation, &end);       /*3.05 add operation switch*/
       }
   }
   if (rcode != OK) {
      error_handler(rcode, str);
      printf("\nFailed At Row %d in Program and Verify Due To %s\n", end, str);
      printf("\n");
      printf("+-------+\n");
      printf("| FAIL! |\n");
      printf("+-------+\n");
      return (-rcode);
   } else {
      printf("\n");
      printf("+=======+\n");
      printf("| PASS! |\n");
      printf("+=======+\n");
   }

   return(0);
} 
