/*  C code to read and write ANSI escape sequences.

Limitations: 
No variable-length non-distributive command sequences.  In plain English:
If a control sequence is variable-length -- i.e. we don't know in advance
how many parameters it should have -- then we handle it by "distributing"
it -- e.g. "ESC [ = 15 ; 5 ; 9 Z" would become "ESC [ = 15 Z ESC [ = 5 Z
ESC [ = 9 Z".  (The analogy is with "x * (15 + 5 + 9) = x * 15 + x * 5 +
x * 9".)  Variable-length sequences that don't distribute are NOT
PERMITTED, because distributing them would change their meaning.  I don't
know of any variable-length non-distributive commands in the ANSI
standard.  The only example I can think of is "tar" in UNIX, where
"tar c x y z" is not the same as "tar c x ; tar c y ; tar c z".

Preston Gardner at Valid, April-May 1983.
*/

#include <stdio.h>
#include <ctype.h>
#include "escdefs.h"


static char sccsid[] = "@(#)escseq.c	1.30 (Valid)  2/9/84";


/*
IO buffers...

When writing, io_buf + io_buflen = io_mark.
When reading, io_buflen is the number of chars read and io_mark is the
place we want to read the next thing.
*/


static char *bufend;	/* end of buffer */
#undef MARKFLUSHES
#ifdef MARKFLUSHES
/* MARKFLUSHES -- put a 377 in monitor file to mark where flushes on
   pipe were done.  */
char flushMark[2];
#endif

#define MONINIT 0
/* Initial value for monitor files: 0 means start with them closed; 1 means
   start with them open.  */

/*
    Do NOT change the contents of this structure without warning the people
    who write Pascal programs that link to it!!  They allocate the space.
 */
struct IOStruct {
    int io_fdesc;		/* file descriptor */
    int io_EOF;			/* hit EOF yet? */
    int io_monfdesc;		/* monitor file descriptor */
    char *io_mark;		/* place mark -- divides buffer */
    int  io_buflen;		/* no. of meaningful chars */
    char io_buf[ESCSEQSTRSIZE+1];  /* +1 so we can null-terminate */
};

/* get file descriptor from Pascal file variable */
#define FDESC(f) ((short) *((f)+15))

static char *nabrtmsg = "null pointer in %s -- abort\n";
#define ABORTIFNULL(p,proc) \
  if(p==NULL){fprintf(stderr,nabrtmsg,proc);abort();}

/* exceptions that can occur when reading escape sequences */
#define XNORMAL		0
#define XOKEOF		1
#define XBADEOF		2
#define XNOTANSI	3
#define XTOOLONG	4
#define XBADNUM		5
#define XBADSKIP	6
#define XRDERR		7
#define XTRASTUFF	8
static int exception = 0;
#define CLREXC (exception=0)
#define TSTEXC if(exception){goto handle;}
#define SETEXC(extype) (exception=(extype))



#ifdef SHOWSPACE
main() {
    fprintf(stderr, "sizeof(struct IOStruct) = %d bytes\n",
        sizeof(struct IOStruct));
}
#endif

/*
DEBUG turns on debugging messages.  DUMMY is used for testing assembly
code linkage wrappers and puts test values into the parameters and return
value.   DEBUG should always be turned on if you turn on DUMMY.  Otherwise
there's no point.  These are turned on and off on a per-function basis.
DUMMY is only relevant for functions callable from Pascal.
*/
#undef DUMMY
#define DEBUG











/***********************************************************************

	Low-level functions -- NOT callable from Pascal.

***********************************************************************/

/* The ANSI standard doesn't allow negative values in the escape sequences.
But it does allow parameters of arbitrary size.  Therefore we encode point
values that are positive or negative by multiplying by 2 and adding 1 if
it is negative. */
#define ENCODNEG(d) (((d)<0)?(((-(d))<<1)+1):((d)<<1))
#define DECODNEG(d) (((d)&0x1)?((-((d)-1))>>1):((d)>>1))

/* sscanint and sscanneg are both in sscanint.h -- they are exactly the
same, except that sscanint decodes the number as an +/- integer. */
#undef DEBUG
#define	MKSSCANINT
#include "sscanint.h"
#undef DEBUG
#undef	MKSSCANINT
#include "sscanint.h"

/*****************************************************************/

#undef DEBUG
int sskip(str,cmpstr)
char *str;
char *cmpstr;
{
    register char *p, *q;  int ret = 0;
#   ifdef DEBUG
    fprintf(stderr, "sskip(%lx,%s)\n", str, cmpstr);
#   endif
    p = str;
    q = cmpstr;
    *bufend = '\0';
    while ((*q != '\0') && (*p == *q)) {
	p++;
	q++;
    }
    if (*q != '\0') {
#	ifdef DEBUG
	fprintf(stderr, "sskip: XBADSKIP\n");
#	endif
	SETEXC(XBADSKIP);
	ret = 0;
    } else {
	ret = q - cmpstr;
    }
#   ifdef DEBUG
    fprintf(stderr, "sskip returns %d\n", ret);
#   endif
    return(ret);
}



#undef DEBUG
#define MKSCANCTL
#include "scanctl.h"
#undef DEBUG
#undef  MKSCANCTL
#include "scanctl.h"

/********************FIXSPLITSEQ*************************************/


/*
    Fundamental problem:  ANSI sequences require lookahead of indefinite
    length before you can see the last char and know what type of
    sequence it is.  A normal one-pass scan would require accumulation
    of all intermediate stuff before you'd see the end.

    Instead I scan it twice -- the first time with "scanctl" or
    "scanesc" to see the char at the end, and the second time to pick
    up the parameters once we know what's supposed to be there.

    This gets me into trouble if the sequence is broken across two IO
    buffers, because once I see the end the beginning is already gone.
    So if this happens I take the current string I'm looking at and
    shift it to the left end of the IO buffer, and then read enough
    characters to fill the rest.   This only causes trouble if there
    is an escape sequence larger than the buffer.

    I know it's kludgy but in some ways it's cleaner than making a
    finite state machine and putting all the parameters in local
    variables.  If it makes enough people vomit I'll fix it someday.

    There is no problem like this with text strings.  If a text string
    is broken across two buffers, we treat it like two text strings.
*/
#undef DUMMY
#undef DEBUG
int FixSplitSeq(iop,nread)
register struct IOStruct *iop;
int *nread;
{
    register char *p, *q;	/* used for copying strings */
    extern char *bufend;	/* end of buffer */
    int unread;			/* no of unread chars in buffer */
    int shift;			/* distance to shift things */

#   ifdef DEBUG
    fprintf(stderr, "FixSplitSeq(%lx,%lx)\n", iop, nread);
#   endif

    shift = iop->io_mark - iop->io_buf;	/* distance to shift */
    unread = bufend - iop->io_mark;	/* no of chars not read yet */
    if (unread >= ESCSEQSTRSIZE) {
#   ifdef DEBUG
	fprintf(stderr, "FixSplitSeq: XTOOLONG\n");
#   endif
	SETEXC(XTOOLONG);
	goto handle;
    }
    if (shift != 0) {
	/* shift the current sequence to the beginning of the buffer */
	p = iop->io_mark; q = iop->io_buf;
	while (p < bufend) {
	    *q++ = *p++;
	}
	bufend -= shift;
	*bufend = '\0';
    }
    *nread = read(iop->io_fdesc,bufend,ESCSEQSTRSIZE - unread);
    if (*nread <= 0) {
	if (*nread == 0) {
	    iop->io_EOF = 1;
#	    ifdef DEBUG
	    fprintf(stderr, "FixSplitSeq: XBADEOF\n");
#	    endif
	    SETEXC(XBADEOF);
	    goto handle;
	} else {
	    iop->io_EOF = 1;
#	    ifdef DEBUG
	    fprintf(stderr, "FixSplitSeq: XRDERR\n");
#	    endif
	    SETEXC(XRDERR);
	    goto handle;
	}
    }
    if (iop->io_monfdesc > 0) {
	/* monitor is on */
	write(iop->io_monfdesc,bufend,*nread);
#	ifdef MARKFLUSHES
	write(iop->io_monfdesc,flushMark,1);/* mark where flush was done */
#	endif
    }

    iop->io_mark = iop->io_buf;
    iop->io_buflen = unread + *nread;
    bufend = iop->io_buf + iop->io_buflen;
    *bufend = '\0';
#   ifdef DEBUG
    fprintf(stderr, "IO buffer contents: \n");
    dumpiop(stderr, iop);
#   endif

#   define MYNAME "FixSplitSeq"
    handle:
    switch (exception) {
	case XNORMAL:
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal return(%d)\n", MYNAME, shift);
#	    endif
	    return(shift);
	    break;
	case XOKEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal EOF -- return(%d)\n", MYNAME, shift);
#	    endif
	    return(shift);
	    break;
	case XBADEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: EOF in mid-sequence -- return(%d)\n",
		MYNAME, shift);
#	    endif
	    return(shift);
	    break;
	case XNOTANSI: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: sequence doesn't follow ANSI pattern -- return(%d)\n",
		MYNAME, shift);
#	    endif
	    return(shift);
	    break;
	case XTOOLONG: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: sequence too long -- return(%d)\n",
		MYNAME, shift);
#	    endif
	    return(shift);
	    break;
	case XBADNUM: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: garbage where number expected\n", MYNAME);
#	    endif
	    return(shift);
	    break;
	case XBADSKIP: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: garbage where explicit string expected\n", MYNAME);
#	    endif
	    return(shift);
	    break;
	case XRDERR: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: read error\n", MYNAME);
#	    endif
	    return(shift);
	    break;
	case XTRASTUFF: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: extra chars among sequence parameters\n", MYNAME);
#	    endif
	    return(shift);
	    break;
	default: 
	    fprintf(stderr,
		"%s: panic: exception of unknown type\n", MYNAME);
	    abort();
	    break;
    }
}

/**********************SHITCAN************************************/

#undef DEBUG
shitcan(br,iop,endptr)
register struct EscSeqBuf *br;
register struct IOStruct *iop;
char *endptr;
{
#   ifdef DEBUG
    fprintf(stderr, "shitcan(%lx,%lx,%lx)\n", br, iop, endptr);
#   endif
    br->es_dataType = ESCERROR;
    iop->io_mark = endptr;
}

/************************CLEARESB*************************************/

clearesb(br)
register struct EscSeqBuf *br;
{
    br->es_dataType = 0;
    br->es_x1 = 0;
    br->es_x2 = 0;
    br->es_y1 = 0;
    br->es_y2 = 0;
    br->es_num = 0;
    br->es_strlen = 0;
}

/*********************DUMPSTR****************************************/

dumpstr(fp,start,finish)
FILE *fp;
char *start;
char *finish;
{
    char *p;

    for(p = start; p < finish; p++) {
	if (iscntrl(*p)) {
	    fprintf(fp, "\\%03o", *p);
	} else {
	    putc(*p, fp);
	}
    }
}

/*************************DUMPIOP****************************************/

dumpiop(fp,iop)
FILE *fp;
register struct IOStruct *iop;
{
    char *be;

    fprintf(fp, "\tiop = %lx\n", iop);
    fprintf(fp, "\tio_fdesc = %d\n", iop->io_fdesc);
    fprintf(fp, "\tio_EOF = %d\n", iop->io_EOF);
    fprintf(fp, "\tio_buf = %lx\n", iop->io_buf);
    fprintf(fp, "\tio_mark = %lx\n", iop->io_mark);
    fprintf(fp, "\tio_buflen = %d\n", iop->io_buflen);
    fprintf(fp, "\tbufend = %lx\n", bufend);

    be = iop->io_buf + iop->io_buflen;
    fprintf(fp, "\tcontents of buffer before mark:\n\t\t<");
    dumpstr(fp, iop->io_buf, iop->io_mark);
    fprintf(fp, ">\n\tcontents of buffer after mark:\n\t\t<");
    dumpstr(fp, iop->io_mark, be);
    fprintf(fp, ">\n");
}

/************************DUMPESB*****************************************/

dumpesb(fp,br)
FILE *fp;
register struct EscSeqBuf *br;
{
    fprintf(fp, "\tes_dataType = %ld\n", br->es_dataType);
    fprintf(fp, "\tes_x1 = %ld\n", br->es_x1);
    fprintf(fp, "\tes_x2 = %ld\n", br->es_x2);
    fprintf(fp, "\tes_y1 = %ld\n", br->es_y1);
    fprintf(fp, "\tes_y2 = %ld\n", br->es_y2);
    fprintf(fp, "\tes_num = %ld\n", br->es_num);
    fprintf(fp, "\tes_strlen = %ld\n", br->es_strlen);
    br->es_str[br->es_strlen] = '\0';
    fprintf(fp, "\tes_str = %s\n", br->es_str);
}













/***********************************************************************

	Top-level functions -- callable from Pascal.

***********************************************************************/

/*******************xinit_es****************************************/

#undef DUMMY
#undef DEBUG

int xinit_es(iop,fd)
struct IOStruct *iop;
int fd;
{

#   ifdef DEBUG
    fprintf(stderr, "initfdesio(%d)\n", fd);
#   endif


#   ifdef DUMMY
    return(1);
#   else
    ABORTIFNULL(iop,"initfdesio");
#ifdef MARKFLUSHES
    flushMark[0] = (char)0xFF; flushMark[1] = '\0';
#endif
    iop->io_fdesc = fd;
    iop->io_buflen = 0;
    iop->io_buf[0] = '\0';
    iop->io_mark = iop->io_buf;
    iop->io_EOF = 0;
    iop->io_monfdesc = -1;
    if (MONINIT) {
        xch_monitor(MONINIT,iop);
    }
    return(1);
#   endif
}


/************************FORCE_A_FLUSH*****************************/


int force_a_flush(iop)
/* all writes go through here */
register struct IOStruct *iop;
{
    if ((iop->io_buflen) <= 0) {
	/* don't write if there's nothing there */
	return(1);
    } else {
	if (write(iop->io_fdesc,iop->io_buf,iop->io_buflen) < iop->io_buflen) {
	    fprintf(stderr, "write error\n");
	    return(0);
	} else { 
	    if (iop->io_monfdesc > 0) {
		/* monitor is on */
		write(iop->io_monfdesc,iop->io_buf,iop->io_buflen);
#ifdef MARKFLUSHES
		/* mark where flush was done */
		write(iop->io_monfdesc,flushMark,1);
#endif
	    }

	    iop->io_buf[0] = '\0';
	    iop->io_mark = iop->io_buf;
	    iop->io_buflen = 0;
	    return(1);
	}
    }
}

/*********************UPDATE_BUFFER********************************/

int update_buffer(iop,buf)
register struct IOStruct *iop;
char *buf;
/* write iop->io_buf if adding buf makes length > 512.  Then concatinate
   buf to iop->io_buf (SUE 5/5)*/
{
    register int n;
    int ret;

    ret = 1;
    n = strlen(buf);
    if (n + iop->io_buflen >= ESCSEQSTRSIZE) {
       ret &= force_a_flush(iop);
       }
    strcat(iop->io_buf,buf);
    iop->io_buflen += n;
    iop->io_mark = iop->io_buf + iop->io_buflen;
    return(ret);
}

#undef DUMMY
#undef DEBUG
#ifdef S32
#define xch_monitor xch_moni
#endif S32
xch_monitor(turnon,iop)
int turnon;
struct IOStruct *iop;
{
    char fname[20];

    ABORTIFNULL(iop,"xch_monitor");
    if (turnon) {
	/* turn on monitor */
	if (iop->io_monfdesc > 0) {
	    /* already on, do nothing */
	} else {
	    sprintf(fname, "mon%d.%d", getpid(), iop->io_fdesc);
	    if (access(fname,0)) {
		iop->io_monfdesc = creat(fname, 0664);
	    } else {
		iop->io_monfdesc = open(fname, 1);
		/* go to end */
		lseek(iop->io_monfdesc,0,2);
	    }
	}
    } else {
	/* turn off monitor */
	if (iop->io_monfdesc > 0) {
	    close(iop->io_monfdesc);
	    iop->io_monfdesc = -1;
	} else {
	    /* already off, do nothing */
	}
    }
}


/*************************xflush_e**********************************/

#undef DUMMY
#undef DEBUG
#ifdef S32
#define xflush_es xflush_e
#endif S32
int xflush_es(iop)
struct IOStruct *iop;
{
    ABORTIFNULL(iop,"xflush_e");
    return(force_a_flush(iop));
}

/************************xwrite_e*************************************/

#undef DUMMY
#undef DEBUG
#ifdef S32
#define xwrite_es xwrite_e
#endif S32
int xwrite_es(b,iop)
struct EscSeqBuf *b;		/* this must match Pascal structure */
struct IOStruct *iop;		/* points to IO buffer */
{
    static char currentSequence[ESCSEQSTRSIZE+1];	/* holds current sequence */
    register struct EscSeqBuf *br;	/* points to seq buffer */
    int ret;

#   ifdef DEBUG
#   define SHOWNAME(escname) fprintf(stderr, "ESC val = %s\n", escname);
#   else
#   define SHOWNAME(escname)
#   endif




    /***************************************************************

	macros for writing escape sequences...

    ***************************************************************/

    /* send a 0-param private-use sequence */
#   define MK0PSEQ(escname,puchr,trmchr)\
	SHOWNAME(escname);\
	sprintf(currentSequence,"%c[%c%c",\
	    ESCAPE, puchr, trmchr);\
        ret &=  update_buffer(iop,currentSequence);

    /* send a standard 1-param ANSI sequence */
#   define MK1ANUMP(escname,trmchr)\
	SHOWNAME(escname);\
	sprintf(currentSequence,"%c[%d%c",\
	    ESCAPE, br->es_num, trmchr);\
        ret &=  update_buffer(iop,currentSequence);

    /* send a 1-param private-use sequence */
#   define MK1PNUMP(escname,puchr,trmchr)\
	SHOWNAME(escname);\
	sprintf(currentSequence,"%c[%c%d%c",\
	    ESCAPE, puchr, br->es_num, trmchr);\
        ret &=  update_buffer(iop,currentSequence);

    /* send a pair of point coords, ANSI standard.  Note no
    negatives allowed. */
#   define MK1APPAIR(escname,trmchr)\
	SHOWNAME(escname);\
	sprintf(currentSequence,"%c[%d;%d%c",\
	    ESCAPE,\
	    br->es_x1,\
	    br->es_y1,\
	    trmchr);\
        ret &=  update_buffer(iop,currentSequence);

    /* send a pair of point coords, private use */
#   define MK1PPPAIR(escname,puchr,trmchr)\
	SHOWNAME(escname);\
	sprintf(currentSequence,"%c[%c%d;%d%c",\
	    ESCAPE,\
	    puchr,\
	    ENCODNEG(br->es_x1),\
	    ENCODNEG(br->es_y1),\
	    trmchr);\
        ret &=  update_buffer(iop,currentSequence);

 










#   ifdef DEBUG
    fprintf(stderr, "xwrite_e(%lx,%lx)\n", b, iop);
    dumpesb(stderr,b);
#   endif

#   ifdef DUMMY
    return(1);
#   else
    ABORTIFNULL(iop,"xwrite_e");
    ABORTIFNULL(  b,"xwrite_e");
    ret = 1;
    CLREXC;
    br = b;
    switch (br->es_dataType) {
	case ESCCMD: 
	    MK1PNUMP("ESCCMD",'=','C');
	    break;
	case ESCCPP: 
	    MK1PPPAIR("ESCCPP",'=','H');
	    break;
	case ESCCPR: 
	    MK1APPAIR("ESCCPR",'R');
	    break;
	case ESCCUB :
	    MK1ANUMP("ESCCUB",'D');
	    break;
	case ESCCUD :
	    MK1ANUMP("ESCCUD",'B');
	    break;
	case ESCCUF :
	    MK1ANUMP("ESCCUF",'C');
	    break;
	case ESCCUP :
	    MK1APPAIR("ESCCUP",'H');
	    break;
	case ESCCUU :
	    MK1ANUMP("ESCCUU",'A');
	    break;
	case ESCDCH :
	    MK1ANUMP("ESCDCH",'P');
	    break;
	case ESCDCS: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s\n", "ESCDCS");
#	    endif
	    sprintf(currentSequence,"%c%c",
		ESCAPE,
		'P');
            ret &=  update_buffer(iop,currentSequence);
	    break;
	case ESCDL :
	    MK1ANUMP("ESCDL",'M');
	    break;
	case ESCED :
	    MK1ANUMP("ESCED",'J');
	    break;
	case ESCEL :
	    MK1ANUMP("ESCEL",'K');
	    break;
	case ESCEND:
	    MK0PSEQ("ESCEND",'=','e');
	    break;
	case ESCEOF: 
	    fprintf(stderr, "panic -- tried to send ESCEOF\n");
	    abort();
	    break;
	case ESCERRMSG: 
	    MK1PNUMP("ESCERRMSG",'=','E');
	    break;
	case ESCERROR: 
	    fprintf(stderr, "panic -- tried to send ESCERROR\n");
	    abort();
	    break;
	case ESCGETFROMMENU:
	    MK1PNUMP("ESCGETFROMMENU",'=','m');
	    break;
	case ESCICH :
	    MK1ANUMP("ESCICH",'@');
	    break;
	case ESCIL :
	    MK1ANUMP("ESCIL",'L');
	    break;
	case ESCIQUIT:
	    MK0PSEQ("ESCIQUIT",'=','I');
            ret &= force_a_flush(iop);
	    break;
	case ESCMC: 
	    MK1ANUMP("ESCMC",'i');
	    break;
	case ESCPL: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s\n", "ESCPL");
#	    endif
	    sprintf(currentSequence,"%c[%c%d:%d;%d:%d;%d%c",
		ESCAPE,
		'=',
		br->es_num,
		ENCODNEG(br->es_x1),
		ENCODNEG(br->es_y1),
		ENCODNEG(br->es_x2),
		ENCODNEG(br->es_y2),
		'L');
            ret &=  update_buffer(iop,currentSequence);
	    break;
	case ESCPPR: 
	    MK1PPPAIR("ESCPPR",'=','R');
	    break;
	case ESCPROMPT:
	    MK1PNUMP("ESCPROMPT",'=','p');
            ret &= force_a_flush(iop);
	    break;
	case ESCPUTINMENU:
	    MK1PNUMP("ESCPUTINMENU",'=','M');
	    break;
	case ESCRM: 
	    MK1ANUMP("ESCRM",'l');
	    break;
	case ESCRSM: 
	    MK1PNUMP("ESCRSM",'=','S');
	    break;
	case ESCRVM: 
	    MK1PNUMP("ESCRVM",'=','l');
	    break;
	case ESCSD: 
	    MK1ANUMP("ESCSD",'T');
	    break;
	case ESCSM: 
	    MK1ANUMP("ESCSM",'h');
	    break;
	case ESCST: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s\n", "ESCST");
#	    endif
	    sprintf(currentSequence,"%c%c",ESCAPE,'\\');
            ret &=  update_buffer(iop,currentSequence);
	    break;
	case ESCSU: 
	    MK1ANUMP("ESCSU",'S');
	    break;
	case ESCSVM: 
	    MK1PNUMP("ESCSVM",'=','h');
	    break;
	case ESCTEXT :
#	    ifdef DEBUG
	    fprintf(stderr, "ESCTEXT\n");
#	    endif
/* using STRNCPY because br->es_strlen doesnt seem to be \0 terminated */
	    strncpy(currentSequence,br->es_str,br->es_strlen);
            currentSequence[br->es_strlen] = '\0';
            ret &=  update_buffer(iop,currentSequence);
	    break;
	case ESCWAKEUP:
	    MK0PSEQ("ESCWAKEUP",'=','w');
	    break;
	default :
	    fprintf(stderr, "panic -- dataType = %d unknown\n",
		br->es_dataType);
	    abort();
	    break;
    }
    return(ret);
#   endif
}
#undef SHOWNAME





/**********************xread_es**************************************/


#undef DUMMY
#undef DEBUG
int xread_es(b,iop)
struct EscSeqBuf *b;		/* this must match Pascal structure */
struct IOStruct *iop;		/* points to IO struct */
{
    register struct EscSeqBuf *br;	/* points to esc struct */
    register char *p;			/* cursor for searches */
    char *l, *nextstart;			/* place markers */
    char endch;				/* last char in the sequence */
    extern char *bufend;		/* end of the buffer */
    int scan;				/* no. chars scanned by various fns */
    char puchr;				/* public-use character */
    int shift;				/* no. chars shifted by FixSplitSeq */
    int nread;				/* no. chars read by FixSplitSeq */





    /* macros for parsing escape sequences *******************************/

#   ifdef DEBUG
#   define SHOWEVAL(escname) fprintf(stderr, "ESC val = %d\n", escname);
#   else
#   define SHOWEVAL(escname)
#   endif

    /* test for exceptions and move l */
#   define TSTNMOVL TSTEXC;l+=scan;

    /* test for extra stuff at end of sequence */
#   define TSTXTRAS if(l!=(nextstart-1)){SETEXC(XTRASTUFF);goto handle;}

    /* handle a control sequence with no parameters */
#   define RD0PSEQ(escname)\
    SHOWEVAL(escname);\
    br->es_dataType = escname;\
    TSTXTRAS;

    /* handle a control sequence with one numerical parameter */
#   define RD1NUMP(escname,def)\
    SHOWEVAL(escname);\
    br->es_dataType = escname;\
    scan = sscannat(l,&(br->es_num),def);\
    TSTNMOVL;\
    TSTXTRAS;

    /* read a pair of points for an ANSI point pair (no negatives). */
#   define RD1APTPAIR(escname,defx,defy)\
    SHOWEVAL(escname);\
    br->es_dataType = escname;\
    scan = sscannat(l,&(br->es_x1),defx);\
    TSTNMOVL;\
    scan = sskip(l,";");\
    TSTNMOVL;\
    scan = sscannat(l,&(br->es_y1),defy);\
    TSTNMOVL;\
    TSTXTRAS;

    /* read a pair of points for private use point pair. */
#   define RD1PPTPAIR(escname,defx,defy)\
    SHOWEVAL(escname);\
    br->es_dataType = escname;\
    scan = sscanint(l,&(br->es_x1),defx);\
    TSTNMOVL;\
    scan = sskip(l,";");\
    TSTNMOVL;\
    scan = sscanint(l,&(br->es_y1),defy);\
    TSTNMOVL;\
    TSTXTRAS;




    /* xread_es begins.... ******************************************/
#   ifdef DEBUG
    fprintf(stderr, "xread_es(%lx,%lx):\n", b, iop);
#   endif

    br = b;

#   ifdef DUMMY
    /* put in some values we can easily check on the other end */
    br->es_dataType = 1;
    br->es_x1 = 4;
    br->es_x2 = 9;
    br->es_y1 = 25;
    br->es_y2 = 36;
    br->es_num = 64;
    strcpy(br->es_str, "Hi, Mom!   ");
    br->es_strlen = strlen(br->es_str);

    return(1);
#   else

    /* initializations */
    ABORTIFNULL(iop,"xread_es");
    ABORTIFNULL(  b,"xread_es");
    CLREXC;
    clearesb(br);

    if (iop->io_EOF) {
	fprintf(stderr, "tried to read past EOF\n");
	return(0);
    }

    /* initialize bufend.  This equation should be true throughout. */
    bufend = iop->io_buf + iop->io_buflen;
    *bufend = '\0';


#   ifdef DEBUG
    fprintf(stderr, "IO buffer contents: \n");
    dumpiop(stderr, iop);
#   endif
    /* is IO buffer empty? */
    if (iop->io_mark >= bufend) {
	/* buffer empty -- fill it */
	iop->io_buflen = read(iop->io_fdesc,iop->io_buf,ESCSEQSTRSIZE);
	if (iop->io_buflen <= 0) {
	    if (iop->io_buflen == 0) {
		iop->io_EOF = 1;
#		ifdef DEBUG
		fprintf(stderr, "xread_es: XOKEOF\n");
#		endif
		SETEXC(XOKEOF);
	    } else {
		iop->io_EOF = 1;
#		ifdef DEBUG
		fprintf(stderr, "xread_es: XRDERRF\n");
#		endif
		SETEXC(XRDERR);
	    }
	} else {
	    if (iop->io_monfdesc > 0) {
		/* monitor is on */
		write(iop->io_monfdesc,iop->io_buf,iop->io_buflen);
#ifdef MARKFLUSHES
		/* mark where flush was done */
		write(iop->io_monfdesc,flushMark,1);
#endif
	    }
	    iop->io_mark = iop->io_buf;
	    bufend = iop->io_buf + iop->io_buflen;
	    *bufend = '\0';
#	    ifdef DEBUG
	    fprintf(stderr, "IO buffer contents: \n");
	    dumpiop(stderr, iop);
#	    endif
	}
    }
    TSTEXC;



    /* dig in and start parsing */
    l = iop->io_mark;
    if (*l == ESCAPE) {
	/* escape or control sequence, or error */
	l++;
	if (l >= bufend) {
	    /* ESC is the last thing in the buffer; get some more */
	    shift = FixSplitSeq(iop, &nread);
	    TSTEXC;
	    l -= shift;
	}
	switch (*l) {
	    case '[' :  /* control sequence */
#		ifdef DEBUG
		fprintf(stderr, "control sequence\n");
#		endif
		l++;	/* skip '[' */
		scan = scanctl(iop,&l);
		TSTEXC;
		nextstart = l + scan;
#		ifdef DEBUG
		fprintf(stderr, "sequence scanned: <");
		dumpstr(stderr, iop->io_mark, nextstart);
		fprintf(stderr, ">\n");
#		endif
		endch = *(nextstart-1);



		if (((0x3c <= *l) && (*l <= 0x3f)) ||
		    ((0x70 <= endch) && (endch <= 0x7e))) {
		    /* private-use sequences */
		    puchr = *l;
		    l++;	/* skip private-use char */
		    switch (endch) {
			case 'C':	/* CMD */
			    RD1NUMP(ESCCMD,0);
			    break;
			case 'H':	/* CPP */
			    RD1PPTPAIR(ESCCPP,0,0);
			    break;
			case 'e':	/* END */
			    RD0PSEQ(ESCEND);
			    break;
			case 'E':	/* ERRMSG */
			    RD1NUMP(ESCERRMSG,0);
			    break;
			case 'm':	/* GETFROMMENU */
			    /* GETFROMMENU */
			    RD1NUMP(ESCGETFROMMENU,1);
			    break;
			case 'I':	/* IQUIT */
			    RD0PSEQ(ESCIQUIT);
			    break;
			case 'L':	/* PL */
			    br->es_dataType = ESCPL;
			    scan = sscannat(l,&(br->es_num),2);
			    TSTNMOVL;
			    scan = sskip(l,":");
			    TSTNMOVL;
			    scan = sscanint(l,&(br->es_x1),0);
			    TSTNMOVL;
			    scan = sskip(l,";");
			    TSTNMOVL;
			    scan = sscanint(l,&(br->es_y1),0);
			    TSTNMOVL;
			    scan = sskip(l,":");
			    TSTNMOVL;
			    scan = sscanint(l,&(br->es_x2),0);
			    TSTNMOVL;
			    scan = sskip(l,";");
			    TSTNMOVL;
			    scan = sscanint(l,&(br->es_y2),0);
			    TSTNMOVL;
			    TSTXTRAS;
			    break;
			case 'M':	/* PUTINMENU */
			    /* PUTINMENU */
			    RD1NUMP(ESCPUTINMENU,1);
			    break;
			case 'p':	/* PROMPT */
			    RD1NUMP(ESCPROMPT,0);
			    break;
			case 'R':	/* PPR */
			    RD1PPTPAIR(ESCPPR,0,0);
			    break;
			case 'S':	/* RSM */
			    /* variable length sequence */
			    br->es_dataType = ESCRSM;
			    scan = sscannat(l,&(br->es_num),0);
			    TSTNMOVL;
			    if (*l == ';') {
				/* don't use sskip or it will cause an
				    exception */
				scan = 1;
				TSTNMOVL;

				/* Load the start of this escape sequence
				OVER the current parameter, so that the
				next time we read one we get the same
				sequence again, but with the first
				parameter thrown away.  This is how we
				"distribute" the sequence into several
				sequences. */
				l -= 3;
				*l = ESCAPE; *(l+1) = '['; *(l+2) = 'S';
				nextstart = l;
			    } else {
				TSTXTRAS;
			    }
			    break;
			case 'l':	/* RVM */
			    RD1NUMP(ESCRVM,0);
			    break;
			case 'h':	/* SVM */
			    RD1NUMP(ESCSVM,0);
			    break;
			case 'w':	/* WAKEUP */
			    RD0PSEQ(ESCWAKEUP);
			    break;
			default:
			    br->es_dataType = ESCERROR;
			    break;
		    }
		} else {
		    /* standard ANSI */
		    switch (endch) {
			case 'R':	/* CPR */
			    RD1APTPAIR(ESCCPR,0,0);
			    break;
			case 'D':	/* CUB */
			    RD1NUMP(ESCCUB,1);
			    break;
			case 'B':	/* CUD */
			    RD1NUMP(ESCCUD,1);
			    break;
			case 'C':	/* CUF */
			    RD1NUMP(ESCCUF,1);
			    break;
			case 'H':	/* CUP */
			    RD1APTPAIR(ESCCUP,1,1);
			    break;
			case 'A':	/* CUU */
			    RD1NUMP(ESCCUU,1);
			    break;
			case 'P':	/* DCH */
			    RD1NUMP(ESCDCH,1);
			    break;
			case 'M':	/* DL */
			    RD1NUMP(ESCDL,1);
			    break;
			case 'J':	/* ED */
			    RD1NUMP(ESCED,0);
			    break;
			case 'K':	/* EL */
			    RD1NUMP(ESCEL,0);
			    break;
			case '@':	/* ICH */
			    RD1NUMP(ESCICH,1);
			    break;
			case 'L':	/* IL */
			    RD1NUMP(ESCIL,1);
			    break;
			case 'i':	/* MC */
			    RD1NUMP(ESCMC,0);
			    break;
			case 'l':	/* RM */
			    RD1NUMP(ESCRM,0);
			    break;
			case 'T':	/* SD */
			    RD1NUMP(ESCSD,1);
			    break;
			case 'h':	/* SM */
			    RD1NUMP(ESCSM,0);
			    break;
			case 'S':	/* SU */
			    RD1NUMP(ESCSU,1);
			    break;
			default:
			    br->es_dataType = ESCERROR;
			    break;
		    }
		}



		break;
	    default :
#		    ifdef DEBUG
		fprintf(stderr, "escape sequence\n");
#		    endif
		scan = scanesc(iop,&l);
		nextstart = l + scan;
		TSTEXC;
#		ifdef DEBUG
		fprintf(stderr, "sequence scanned: <");
		dumpstr(stderr, iop->io_mark, nextstart);
		fprintf(stderr, ">\n");
#		endif
		/* these are all ANSI standard */
		switch (*l) {
		    case 'P':	/* DCS */
			br->es_dataType = ESCDCS;
			break;
		    case '\\':	/* ST */
			br->es_dataType = ESCST;
			break;
		    default:
			br->es_dataType = ESCERROR;
			break;
		}
		break;
	}
    } else {  /* doesn't start with ESCAPE */
#	ifdef DEBUG
	fprintf(stderr, "plain text\n");
#	endif
	br->es_dataType = ESCTEXT;

	/* search for an ESC later in the buffer */
	*bufend = ESCAPE;		/* sentinel */
	for (p = iop->io_mark; *p != ESCAPE; p++);

#	ifdef DEBUG
	if (p == bufend) {
	    fprintf(stderr, "no ESC in buffer\n");
	} else {
	    fprintf(stderr, "there is an ESC in buffer\n");
	}
#	endif

	/* copy into the esc seq structure */
	*p = '\0';			/* so strcpy will work */
	strcpy(br->es_str, iop->io_mark);
	br->es_strlen = p - iop->io_mark;
	iop->io_mark = nextstart = p;

	/* fix it back */
	*p = ESCAPE;

	/* if there was no ESC there, p==bufend; this will fix it. */
	*bufend = '\0';

#	ifdef DEBUG
	fprintf(stderr, "IO buffer contents: \n");
	dumpiop(stderr, iop);
#	endif
    }



    /* exception handler */
    handle:
#   undef MYNAME
#   define MYNAME "xread_es"
    iop->io_mark = nextstart;
    switch (exception) {
	case XNORMAL:
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal return\n", MYNAME);
#	    endif
	    return(1);
	    break;
	case XOKEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: normal EOF return\n", MYNAME);
#	    endif
	    br->es_dataType = ESCEOF;
	    iop->io_EOF = 1;
	    return(1);
	    break;
	case XBADEOF: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: EOF in mid-sequence\n", MYNAME);
#	    endif
	    br->es_dataType = ESCEOF;
	    iop->io_EOF = 1;
	    return(0);
	    break;
	case XNOTANSI: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: sequence doesn't follow ANSI pattern\n",
		MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    return(0);
	    break;
	case XTOOLONG: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: panic: sequence too long\n", MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    abort();
	    break;
	case XBADNUM: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: garbage where number expected\n", MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    return(0);
	    break;
	case XBADSKIP: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: garbage where explicit string expected\n", MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    return(0);
	    break;
	case XRDERR: 
#	    ifdef DEBUG
	    fprintf(stderr, "%s: read error\n", MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    return(0);
	    break;
	case XTRASTUFF: 
#	    ifdef DEBUG
	    fprintf(stderr,
		"%s: extra chars among sequence parameters\n", MYNAME);
#	    endif
	    shitcan(br,iop,nextstart);
	    return(0);
	    break;
	default: 
	    fprintf(stderr,
		"%s: panic: exception of unknown type\n", MYNAME);
	    shitcan(br,iop,nextstart);
	    abort();
	    break;
    }
#   endif
}
#undef SHOWEVAL



