/* Insert/delete balanced binary tree: otherwise known as avl trees.
		      Algorithm from Knuth, Searching & Sorting, pp 455-468 
   Key is string or char *.
									    */

#ifndef SAVL_M_INCLUDED
#define SAVL_M_INCLUDED

#include "avl.m.h"

#define Xscompare(a,b,rslt,brec,type,dirty,vr,rv,unlock)\
{\
char *ca, *cb;\
vr(a,ca,char,RO);\
vr(b,cb,char,RO);\
rslt=strcmp(ca,cb);\
unlock(ca); unlock(cb);\
}

/*                M A N I P U L A T I N G      M A C R O S                    */

/*----------------------------------------------------------------------------*/

/* Access the key in an SAVL or SAVLH structure
   Access the rsp in a SAVL or SAVLH structure
   Access the lsp in a SAVL or SAVLH structure
   Access the root in a SAVLH structure

   rec_avl ==> name within record of AVL or AVLH member.
   rec     ==> -> to record containing the rec_avl structure.

***Not assertion proofed.
									      */
#define SAVLO_KEYD(val,rec_avl,rec)\
\
RO_ADRCK(rec);\
ADRCK(val);\
RO_VO(val,AVL_KEY(rec_avl,rec),char)

#define SAVL_RSP AVL_RSP
#define SAVL_LSP AVL_LSP

#define SAVLH_RSP SAVL_RSP
#define SAVLH_LSP SAVL_LSP

#define SAVLR_NULL AVLR_NULL
#define SAVLRO_NULL AVLRO_NULL

#define SAVLR_TOP AVLR_TOP
#define SAVLRO_TOP AVLRO_TOP

#define SAVL_NULL AVL_NULL
#define SAVLH_NULL AVLH_NULL

#define SAVL_CHK(rec_avl,rec)\
\
( (SAVL_KEY(rec_avl,rec) DNE) &&\
  (AVL_CK(rec_avl,rec))\
)
#define SAVLH_CHK(rec_avl,rec)\
\
( (SAVL_CHK(rec_avl,rec) DNE) &&\
  (B(rec_avl,rec) DNE)\
)

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from any member in the tree.

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  Initially set by you to the record from which
		  you wish the search to proceed from.
		  On completion, SET to record with matching key or SET to 
		  NULL if not found. 
									    */

#define SAVL_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key_sought could point anywhere*/\
AVL_LOCATE(key_sought,rec_avl,found,type,Xscompare,NOP)

#define SAVLO_LOCATE(key_sought,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key_sought could point anywhere*/\
AVLO_LOCATE(key_sought,rec_avl,found,type,Xscompare,dirty)

#define SAVLH_LOCATE SAVL_LOCATE
#define SAVLHO_LOCATE SAVLO_LOCATE

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from the root of the tree

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree from which key_sought is to be sought.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  SET to record with matching key;
		  SET to NULL if not found.
									     */

#define SAVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key_sought may point anywhere*/\
AVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xscompare,NOP)

#define SAVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,dirty)\
\
ADRCK(key_sought);/*key_sought may point anywhere*/\
AVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,Xscompare,dirty)

#define SAVLH_FIND SAVL_FIND
#define SAVLHO_FIND SAVLO_FIND

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVL binary-tree.

   rec_avl    ==> name within rec of AVL member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVL member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */

#define SAVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
ADRCK(insert);\
ADRCK(SAVL_KEY(rec_avl,insert));\
AVL_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xscompare)

#define SAVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there)\
\
RO_ADRCK(insert);\
ADRCK(SAVL_KEY(rec_avl,insert));/*key string could point anywhere*/\
AVLO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,key_already_there,Xscompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVL binary-tree

   rec_avl    ==> name within rec of AVL member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVL member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			                                                   */
#define SAVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
ADRCK(delete);\
ADRCK(SAVL_KEY(rec_avl,delete));\
AVL_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xscompare)

#define SAVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
RO_ADRCK(delete);\
ADRCK(SAVL_KEY(rec_avl,delete));/*key could point anywhere*/\
AVLO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xscompare)


/*----------------------------------------------------------------------------*/

/* Insert a key into a AVLH binary-tree.

   rec_avl    ==> name within rec of AVLH member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVLH member to be inserted
		  into the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			 */
#define SAVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
ADRCK(insert);\
ADRCK(SAVL_KEY(rec_avl,insert));\
AVLH_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xscompare)

#define SAVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there)\
\
RO_ADRCK(insert);\
ADRCK(SAVL_KEY(rec_avl,insert));/*key could point anywhere*/\
AVLHO_INSERT(rec_avl,insert,rec_type,hd_avlr,hd,htype,key_already_there,Xscompare)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVLH binary-tree
   
   rec_avl    ==> name within rec of AVLH member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVLH member to be deleted
		  from the avl tree.
   rec_type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_already_there ==> operations to perform if the key already happens to be
			 in the tree.
			                                                   */
#define SAVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
ADRCK(delete);\
ADRCK(SAVL_KEY(rec_avl,delete));\
AVLH_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xscompare)

#define SAVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there)\
\
RO_ADRCK(delete);\
ADRCK(SAVL_KEY(rec_avl,delete));/*key could point anywhere*/\
AVLHO_PICK(rec_avl,delete,rec_type,hd_avlr,hd,key_not_there,Xscompare)

/*----------------------------------------------------------------------------*/
#endif
