/* Homogeneous Insert/delete balanced binary tree: otherwise known as avl trees.
		      Algorithm from Knuth, Searching & Sorting, pp 455-468 */

#ifndef AVL_M_INCLUDED
#define AVL_M_INCLUDED

#include "avl.s.h"
#include "../misc/constants.h"
#include "vm.h"

/*                M A N I P U L A T I N G      M A C R O S                    */

/*----------------------------------------------------------------------------*/

/* Null a AVLR structure

   hd_avlr ==> name within head of AVLR member to be nulled.
   hd      ==> -> to head containing the AVLR structure to be nulled.
									      */

#define AVLR_NULL(hd_avlr,hd)\
{\
as(hd E);\
(hd)->hd_avlr.root = NULL;\
/*\
(hd)->hd_avlr.depth = (hd)->hd_avlr.entries = 0;\
*/\
}

#define AVLRO_NULL AVLR_NULL

/*----------------------------------------------------------------------------*/

/* Null a AVL structure

   rec_avl ==> name within record of AVL member to be nulled.
   rec    ==> -> to record containing the AVL structure to be nulled.
									      */

#define AVL_NULL(rec_avl,rec)\
{\
as(rec E);\
rec->rec_avl.lsp = rec->rec_avl.fp = NULL;\
rec->rec_avl.key = 0;\
}

/*----------------------------------------------------------------------------*/

/* Null an AVLH structure

   rec_avlh ==> name within record of AVLH member to be nulled.
   rec      ==> -> to record containing the AVLH structure to be nulled.
									      */

#define AVLH_NULL(rec_avlh,rec)\
{\
as(rec E);\
rec->rec_avlh.lsp = rec->rec_avlh.fp = NULL;\
rec->rec_avlh.hp = NULL;\
rec->rec_avlh.key = 0;\
}

/*----------------------------------------------------------------------------*/

/* Access the key in an AVL or AVLH structure
   Access the rsp in a AVL or AVLH structure
   Access the lsp in a AVL or AVLH structure
   Access the root in a AVLH structure

   rec_avl ==> name within record of AVL or AVLH member.
   rec     ==> -> to record containing the rec_avl structure.

***Not assertion proofed.
									      */
#define AVL_RSP(rec_avl,rec) (rec->rec_avl.fp)
#define AVL_LSP(rec_avl,rec,type) Xavllsp(rec->rec_avl.lsp,type)

#define AVLH_RSP AVL_RSP
#define AVLH_LSP AVL_LSP

/*----------------------------------------------------------------------------*/

/* Access the top from a AVLR structure

   hd_avlr ==> name within record of AVLR member containing top to get.
   hd      ==> -> to record containing the AVLR structure whose top is to be 
	     gotten.

***Not assertion proofed.
                                                                              */
#define Xavlr_top(hd_avlr,hd,top,type,dirty,vr,roadrck)\
{\
roadrck(hd);\
top=(hd)->hd_avlr.root;\
if (top E) {\
    vr(top,top,type,dirty);\
    }\
}

#define AVLR_TOP(hd_avlr,hd,top,type,dirty)\
\
Xavlr_top(hd_avlr,hd,top,type,NOP,V_R,ADRCK)

#define AVLRO_TOP(hd_avlr,hd,top,type,dirty)\
\
Xavlr_top(hd_avlr,hd,top,type,dirty,VO_RO,ADRCK)

/*----------------------------------------------------------------------------*/

#define AVLR_TOPCK(hd_avlr,hd) ((hd)->hd_avlr.root DNE)

#define AVL_CK(rec_avl,rec)\
\
((rec->rec_avl.fp DNE) &&\
(Xavllsp(rec->rec_avl.lsp,char) DNE))

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from any member in the tree.

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  Initially set by you to the record from which
		  you wish the search to proceed from.
		  On completion, SET to record with matching key or SET to 
		  NULL if not found. 
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      The 1st argument to compare is the 'key_sought',
		      the second argument is the record_to_test->rec_avl.key.
									    */

#define Xavl_locate(key_sought,rec_avl,found,type,compare,dirty,vr,rv,unlock,roadrck)\
\
{\
register int   Xkey_comparison;\
register type *Xfound;\
Xfound=(found);\
roadrck(found);\
while (Xfound E) {\
     compare((key_sought),Xfound->rec_avl.key,Xkey_comparison,Xfound,type,dirty,vr,rv,unlock);\
     if (Xkey_comparison>0) {\
          if (found=Xfound->rec_avl.fp) {\
              vr(found,found,type,RO);\
	      }\
	  unlock(Xfound);\
	  Xfound=found;\
	  }\
     else {\
	  if (Xkey_comparison==0) break;\
          if (found=Xavllsp(Xfound->rec_avl.lsp,type)) {\
               vr(found,found,type,RO);\
	       }\
	  unlock(Xfound);\
	  Xfound=found;\
	  }\
     }\
if (found) {/*make found dirty as specified*/\
     dirty(found);\
     }\
}

#define AVL_LOCATE(key_sought,rec_avl,found,type,compare,dirty)\
\
Xavl_locate(key_sought,rec_avl,found,type,compare,NOP,V_R,R_V,NOP,ADRCK)


#define AVLO_LOCATE(key_sought,rec_avl,found,type,compare,dirty)\
\
Xavl_locate(key_sought,rec_avl,found,type,compare,dirty,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

#define AVLH_LOCATE  AVL_LOCATE
#define AVLHO_LOCATE AVLO_LOCATE

/*----------------------------------------------------------------------------*/

/* Locate a key in an AVL or AVLH binary-tree from the root of the tree

/* 
   key_sought ==> typedef 'Key' key to locate in avl tree.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree from which key_sought is to be sought.
   rec_avl    ==> name within 'found' record of AVL member which forms the 
		  avl tree.
   found      ==> -> to record which forms the avl tree in which AVL member
		  is located:  SET to record with matching key;
		  SET to NULL if not found.
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      The 1st argument to compare is the 'key_sought',
		      the second argument is the record_to_test->rec_avl.key.
									     */

#define Xavl_find(key_sought,hd_avlr,hd,rec_avl,found,type,compare,dirty,vr,rv,unlock,roadrck)\
\
Xavlr_top(hd_avlr,hd,found,type,RO,vr,roadrck);\
if (found E) {\
     Xavl_locate(key_sought,rec_avl,found,type,compare,dirty,vr,rv,unlock,roadrck);\
     }

#define AVL_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,compare,dirty)\
\
Xavl_find(key_sought,hd_avlr,hd,rec_avl,found,type,compare,NOP,V_R,R_V,NOP,ADRCK)

#define AVLO_FIND(key_sought,hd_avlr,hd,rec_avl,found,type,compare,dirty)\
\
Xavl_find(key_sought,hd_avlr,hd,rec_avl,found,type,compare,dirty,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

#define AVLH_FIND  AVL_FIND
#define AVLHO_FIND AVLO_FIND

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVL binary-tree.

   rec_avl    ==> name within rec of AVL member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVL member to be inserted
		  into the avl tree.
   type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> a set of instructions to be executed when the key
			 already is in the tree.
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      Both arguments are rec->rec_avl.key.
   NOTE: rebalance points to the LOWEST record encountered in the tree which 
	 is NOT balanced.
			 */

#define Xavl_insert(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare,vr,rv,unlock,roadrck)\
\
{\
register type *locate,*rebalance,*rebalances_son,*rebalances_father;\
register type *vrebalance, *vrebalances_son;\
register int   side_which_became_deeper_at_rebalance;\
roadrck(hd); roadrck(insert);\
as(Xavllsp(insert,type) == (insert));\
as(Xavllsp(insert->rec_avl.lsp,type) DNE);\
as(insert->rec_avl.fp DNE);\
rebalances_father=NULL;\
locate=(hd)->hd_avlr.root;\
if (locate DNE) {\
     rv(insert,(hd)->hd_avlr.root,type);\
/*\
     as((hd)->hd_avlr.depth==0); as((hd)->hd_avlr.entries==0);\
     (hd)->hd_avlr.depth++;\
     (hd)->hd_avlr.entries++;\
*/\
     /*insert->rec_avl.bd=Xavlequal; is assured since lsp is NULL */\
     }\
else {\
     register int Xkey_comparison;\
     rebalance=locate;\
     vr(locate,locate,type,RO);\
     for(;;) {\
          register type *son, *vson;\
          compare(insert->rec_avl.key,locate->rec_avl.key,Xkey_comparison,locate,type,NOP,vr,rv,unlock);\
          if(!Xkey_comparison) {\
	       unlock(locate);\
	       key_already_there;\
	       }\
          if (Xkey_comparison>0) {\
	       vson=locate->rec_avl.fp;\
	       if (vson DNE) {\
	            rv(locate,locate,type);/*make locate RW*/\
		    vr(locate,locate,type,RW);\
		    unlock(locate);\
	            rv(insert,locate->rec_avl.fp,type);\
	            break;\
	            }\
	       }\
          else {\
	       vson=Xavllsp(locate->rec_avl.lsp,type);\
	       if (vson DNE) {\
	            register type *vinsert;\
	            rv(locate,locate,type);/*make locate RW*/\
		    vr(locate,locate,type,RW);\
	            rv(insert,vinsert,type);\
		    Xavllsps(locate->rec_avl.lsp,vinsert,type);\
		    unlock(locate);\
	            break;\
	            }\
	       }\
	  vr(vson,son,type,RO);\
	  if (Xavlbd(son->rec_avl.lsp)!=Xavlequal) {\
	       rv(locate,rebalances_father,type);\
	       rebalance=vson;\
	       }\
          unlock(locate);\
          locate=son;\
          }\
     unlock(locate);\
     vrebalance=rebalance;\
     vr(rebalance,rebalance,type,RW);\
\
     Xavlbds(insert->rec_avl.lsp,Xavlequal,type);\
/*\
     (hd)->hd_avlr.entries++;\
*/\
\
     compare(insert->rec_avl.key,rebalance->rec_avl.key,Xkey_comparison,rebalance,type,NOP,vr,rv,unlock);\
     if(Xkey_comparison>0) {\
          vrebalances_son=rebalances_son=locate=rebalance->rec_avl.fp;\
          side_which_became_deeper_at_rebalance=Xavlright;\
          }\
     else {\
          vrebalances_son=rebalances_son=locate=Xavllsp(rebalance->rec_avl.lsp,type);\
          side_which_became_deeper_at_rebalance=Xavlleft;\
          }\
\
     vr(locate,locate,type,RW);\
     while (locate!=insert) {/*adjust balance weights from rebalances_son to insert*/\
          register type *tmp;\
          compare(insert->rec_avl.key,locate->rec_avl.key,Xkey_comparison,locate,type,NOP,vr,rv,unlock);\
          if(Xkey_comparison>0) {\
	       Xavlbds(locate->rec_avl.lsp,Xavlright,type);\
	       vr(locate->rec_avl.fp,tmp,type,RW);\
	       unlock(locate);\
	       locate=tmp;\
	       }\
          else {\
	       Xavlbds(locate->rec_avl.lsp,Xavlleft,type);\
	       vr(Xavllsp(locate->rec_avl.lsp,type),tmp,type,RW);\
	       unlock(locate);\
	       locate=tmp;\
	       }\
          }\
     unlock(locate);\
\
     if (Xavlbd(rebalance->rec_avl.lsp)==Xavlequal) { /*tree grown higher*/\
          Xavlbds(rebalance->rec_avl.lsp,side_which_became_deeper_at_rebalance,type);\
/*\
          (hd)->hd_avlr.depth++;\
*/\
          }\
     else {\
          register type *new_root, *vnew_root;\
          if (Xavlbd(rebalance->rec_avl.lsp)==side_which_became_deeper_at_rebalance) {\
	  /*tree out of balance*/\
               vr(rebalances_son,rebalances_son,type,RW);\
	       if (Xavlbd(rebalances_son->rec_avl.lsp)==side_which_became_deeper_at_rebalance) {\
	       /*single rotation*/\
	            new_root=rebalances_son;\
		    vnew_root=vrebalances_son;\
	            if (side_which_became_deeper_at_rebalance==Xavlleft) {\
		         Xavllsps(rebalance->rec_avl.lsp,rebalances_son->rec_avl.fp,type);\
		         rebalances_son->rec_avl.fp=vrebalance;\
		         }\
	            else {\
		         rebalance->rec_avl.fp=Xavllsp(rebalances_son->rec_avl.lsp,type);\
		         Xavllsps(rebalances_son->rec_avl.lsp,vrebalance,type);\
		         }\
	            Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
		    Xavlbds(rebalances_son->rec_avl.lsp,Xavlequal,type);\
	            }\
	       else { /*double rotation*/\
	            if (side_which_became_deeper_at_rebalance==Xavlleft) {\
		         vnew_root=rebalances_son->rec_avl.fp;\
		         vr(vnew_root,new_root,type,RW);\
		         rebalances_son->rec_avl.fp=Xavllsp(new_root->rec_avl.lsp,type);\
		         Xavllsps(new_root->rec_avl.lsp,vrebalances_son,type);\
		         Xavllsps(rebalance->rec_avl.lsp,new_root->rec_avl.fp,type);\
		         new_root->rec_avl.fp=vrebalance;\
		         }\
	            else {\
		         vnew_root=Xavllsp(rebalances_son->rec_avl.lsp,type);\
		         vr(vnew_root,new_root,type,RW);\
		         Xavllsps(rebalances_son->rec_avl.lsp,new_root->rec_avl.fp,type);\
		         new_root->rec_avl.fp=vrebalances_son;\
		         rebalance->rec_avl.fp=Xavllsp(new_root->rec_avl.lsp,type);\
		         Xavllsps(new_root->rec_avl.lsp,vrebalance,type);\
		         }\
	            if (Xavlbd(new_root->rec_avl.lsp)==Xavlequal) {\
		         Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
			 Xavlbds(rebalances_son->rec_avl.lsp,Xavlequal,type);\
		         }\
	            else {\
		         if (Xavlbd(new_root->rec_avl.lsp)==side_which_became_deeper_at_rebalance) {\
		              if (side_which_became_deeper_at_rebalance==Xavlright) {\
			           Xavlbds(rebalance->rec_avl.lsp,Xavlleft,type);\
			           }\
			      else Xavlbds(rebalance->rec_avl.lsp,Xavlright,type);\
			      Xavlbds(rebalances_son->rec_avl.lsp,Xavlequal,type);\
			      }\
		         else {\
		              Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
			      Xavlbds(rebalances_son->rec_avl.lsp,side_which_became_deeper_at_rebalance,type);\
			      }\
		         }\
	            Xavlbds(new_root->rec_avl.lsp,Xavlequal,type);\
		    unlock(new_root);\
	            }\
	       if (rebalances_father==NULL) {\
	            (hd)->hd_avlr.root=vnew_root;\
	            }\
	       else {\
                    vr(rebalances_father,rebalances_father,type,RW);\
	            if (vrebalance==rebalances_father->rec_avl.fp) {\
	                 rebalances_father->rec_avl.fp=vnew_root;\
	                 }\
	            else Xavllsps(rebalances_father->rec_avl.lsp,vnew_root,type);\
		    unlock(rebalances_father);\
	            }\
               unlock(rebalances_son);\
	       }\
          else {  /*tree more balanced*/\
	       Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
	       }\
          }\
     unlock(rebalance);\
\
     /*This check is very tricky*/\
/*\
     printf("d %d e %d <= %d",(hd)->hd_avlr.depth,(hd)->hd_avlr.entries,( (1<<(hd)->hd_avlr.depth) - 1 ) );\
     printf(" >= %d\n",( (1<<((hd)->hd_avlr.depth-1)) ) );\
     as((hd)->hd_avlr.entries <= ( (1<<(hd)->hd_avlr.depth) - 1 ) );\
     as((hd)->hd_avlr.entries >= ( (1<<((hd)->hd_avlr.depth-1)) ) );\
*/\
     }\
}

#define AVL_INSERT(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare)\
\
Xavl_insert(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare,V_R,R_V,NOP,ADRCK)

#define AVLO_INSERT(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare)\
\
Xavl_insert(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVL binary-tree

   rec_avl    ==> name within rec of AVL member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVL member to be deleted
		  from the avl tree.
   type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree.
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      Both arguments are rec->rec_avl.key.
			                                                   */
#define Xavl_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,vr,rv,unlock,roadrck)\
\
{\
static AVLWORK;  /*stack work area for deletion*/\
register type *locate,*vlocate,*deletes_father,*vnew_root,*new_root;\
register int depth={0};\
roadrck(hd); roadrck(delete);\
deletes_father=NULL;\
vlocate=(hd)->hd_avlr.root;\
vr(vlocate,locate,type,RO);\
for(;;){\
    register int Xkey_comparison;\
    if (locate DNE) {unlock(locate); key_not_there;}\
    compare(delete->rec_avl.key,locate->rec_avl.key,Xkey_comparison,locate,type,NOP,vr,rv,unlock);\
    if (!Xkey_comparison) {\
         if (delete==locate) break;\
	 }\
    as(depth<Xavlwork.entries);\
    Xavlwork.Xavlstack[depth].ptr=(struct Xavlwork *)vlocate;\
    if (Xkey_comparison>0) {\
	 deletes_father=vlocate;\
	 vlocate=locate->rec_avl.fp;\
         Xavlwork.Xavlstack[depth++].which_way=Xavlright;\
	 }\
    else {\
	 deletes_father=vlocate;\
	 vlocate=Xavllsp(locate->rec_avl.lsp,type);\
         Xavlwork.Xavlstack[depth++].which_way=Xavlleft;\
	 }\
    unlock(locate);\
    vr(vlocate,locate,type,RO);\
    }\
unlock(locate);\
\
/*\
(hd)->hd_avlr.entries--;\
*/\
\
/*locate new_root*/\
if (delete->rec_avl.fp E) {\
     if (Xavllsp(delete->rec_avl.lsp,type) E) {/*find left-most key, new_root, in right branch*/\
	  register int replacment_depth;\
	  as(depth<Xavlwork.entries);\
	  replacment_depth=depth; depth++;\
	  vnew_root=delete->rec_avl.fp;\
	  vr(vnew_root,new_root,type,RO);\
          if (Xavllsp(new_root->rec_avl.lsp,type) E) {\
               register type *left_father;\
               while (Xavllsp(new_root->rec_avl.lsp,type) E) {\
	             as(depth<Xavlwork.entries);\
	             Xavlwork.Xavlstack[depth].ptr=(struct Xavlwork *)vnew_root;\
	             Xavlwork.Xavlstack[depth].which_way=Xavlleft;  depth++;\
		     left_father=vnew_root;\
		     vnew_root=Xavllsp(new_root->rec_avl.lsp,type);\
		     unlock(new_root);\
		     vr(vnew_root,new_root,type,RO);\
		     }\
	       unlock(new_root);/*make new_root RW*/\
               vr(vnew_root,new_root,type,RW);\
               Xavllsps(new_root->rec_avl.lsp,Xavllsp(delete->rec_avl.lsp,type),type);\
	       vr(left_father,left_father,type,RW);\
	       Xavllsps(left_father->rec_avl.lsp,new_root->rec_avl.fp,type);\
	       unlock(left_father);\
	       new_root->rec_avl.fp=delete->rec_avl.fp;\
	       }\
          else {\
	       unlock(new_root);/*make new_root RW*/\
               vr(vnew_root,new_root,type,RW);\
	       Xavllsps(new_root->rec_avl.lsp,Xavllsp(delete->rec_avl.lsp,type),type);\
	       }\
	  Xavlbds(new_root->rec_avl.lsp,Xavlbd(delete->rec_avl.lsp),type);\
          Xavlwork.Xavlstack[replacment_depth].ptr=(struct Xavlwork *)vnew_root;\
	  Xavlwork.Xavlstack[replacment_depth].which_way=Xavlright; /*replaced delete*/\
	  unlock(new_root);\
	  }\
     else vnew_root=delete->rec_avl.fp; /*delete is deleted directly*/\
     }\
else vnew_root=Xavllsp(delete->rec_avl.lsp,type);/*delete is deleted directly*/\
\
if (deletes_father E) { /*by-pass delete to new_root*/\
     vr(deletes_father,deletes_father,type,RW);\
     if (deletes_father->rec_avl.fp==delete)\
          deletes_father->rec_avl.fp=vnew_root;\
     else Xavllsps(deletes_father->rec_avl.lsp,vnew_root,type);\
     unlock(deletes_father);\
     }\
else (hd)->hd_avlr.root=vnew_root;\
\
delete->rec_avl.lsp=delete->rec_avl.fp=NULL;\
\
while(depth>=0) { /*adjust balance factors*/\
     register type *rebalance, *vrebalance, *rebalance_son, *vrebalance_son;\
     int became_shallower;\
     if (--depth<0) { /*whole tree decreased in height*/\
/*\
	  (hd)->hd_avlr.depth--;\
*/\
	   break;\
	  }\
     vrebalance=(type *)Xavlwork.Xavlstack[depth].ptr;\
     vr(vrebalance,rebalance,type,RW);\
     became_shallower=Xavlwork.Xavlstack[depth].which_way;\
     if (Xavlbd(rebalance->rec_avl.lsp)==Xavlequal) {   /*tree less balanced*/\
          if (became_shallower==Xavlright)\
	       Xavlbds(rebalance->rec_avl.lsp,Xavlleft,type);\
	  else Xavlbds(rebalance->rec_avl.lsp,Xavlright,type);\
	  unlock(rebalance);\
	  break;\
	  }\
     if(Xavlbd(rebalance->rec_avl.lsp)==became_shallower) { /*tree more balanced*/\
	  Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
	  }\
     else {    /*tree out of balance*/\
	  if (became_shallower==Xavlright)\
	       vrebalance_son=Xavllsp(rebalance->rec_avl.lsp,type);\
	  else vrebalance_son=rebalance->rec_avl.fp;\
	  vr(vrebalance_son,rebalance_son,type,RW);\
	  if (Xavlbd(rebalance_son->rec_avl.lsp)==became_shallower) { /*double rotation*/\
	       if (Xavlbd(rebalance_son->rec_avl.lsp)==Xavlright) {\
		    vnew_root=rebalance_son->rec_avl.fp;\
		    vr(vnew_root,new_root,type,RW);\
		    rebalance_son->rec_avl.fp=Xavllsp(new_root->rec_avl.lsp,type);\
		    Xavllsps(new_root->rec_avl.lsp,vrebalance_son,type);\
		    Xavllsps(rebalance->rec_avl.lsp,new_root->rec_avl.fp,type);\
		    new_root->rec_avl.fp=vrebalance;\
		    }\
	       else {\
		    vnew_root=Xavllsp(rebalance_son->rec_avl.lsp,type);\
		    vr(vnew_root,new_root,type,RW);\
	            Xavllsps(rebalance_son->rec_avl.lsp,new_root->rec_avl.fp,type);\
		    new_root->rec_avl.fp=vrebalance_son;\
		    rebalance->rec_avl.fp=Xavllsp(new_root->rec_avl.lsp,type);\
		    Xavllsps(new_root->rec_avl.lsp,vrebalance,type);\
		    }\
	       if (Xavlbd(new_root->rec_avl.lsp)==Xavlequal) {\
		    Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
		    Xavlbds(rebalance_son->rec_avl.lsp,Xavlequal,type);\
		    }\
	       else {\
		    if (Xavlbd(new_root->rec_avl.lsp)==became_shallower) {\
		         if (became_shallower==Xavlright) {\
			    Xavlbds(rebalance_son->rec_avl.lsp,Xavlleft,type);\
			      }\
		         else  Xavlbds(rebalance_son->rec_avl.lsp,Xavlright,type);\
			 Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
			 }\
		    else {\
		         Xavlbds(rebalance_son->rec_avl.lsp,Xavlequal,type);\
		         Xavlbds(rebalance->rec_avl.lsp,became_shallower,type);\
			 }\
		    }\
	       Xavlbds(new_root->rec_avl.lsp,Xavlequal,type);\
	       unlock(new_root);\
	       }\
	  else { /*single rotation*/\
	       vnew_root=vrebalance_son;\
	       if (became_shallower==Xavlright) {\
		    Xavllsps(rebalance->rec_avl.lsp,rebalance_son->rec_avl.fp,type);\
		    rebalance_son->rec_avl.fp=rebalance;\
		    }\
	       else {\
		    rebalance->rec_avl.fp=Xavllsp(rebalance_son->rec_avl.lsp,type);\
		    Xavllsps(rebalance_son->rec_avl.lsp,rebalance,type);\
		    }\
	       if (Xavlbd(rebalance_son->rec_avl.lsp)==Xavlequal) {\
		    if(became_shallower==Xavlright)\
		         Xavlbds(rebalance_son->rec_avl.lsp,Xavlright,type);\
		    else Xavlbds(rebalance_son->rec_avl.lsp,Xavlleft,type);\
		    if (depth==0) (hd)->hd_avlr.root=vnew_root;\
		    else {\
		         unlock(rebalance);\
                         vr((type *)Xavlwork.Xavlstack[depth-1].ptr,rebalance,type,RW);\
                         if (Xavlwork.Xavlstack[depth-1].which_way==Xavlright)\
			      rebalance->rec_avl.fp=vnew_root;\
			 else Xavllsps(rebalance->rec_avl.lsp,vnew_root,type);\
			 }\
                    unlock(rebalance_son);\
                    unlock(rebalance);\
		    break;\
		    }\
	       else {\
	            Xavlbds(rebalance->rec_avl.lsp,Xavlequal,type);\
	            Xavlbds(rebalance_son->rec_avl.lsp,Xavlequal,type);\
		    }\
	       }\
          unlock(rebalance_son);\
	  if (depth==0) { /*point to new root*/\
	       (hd)->hd_avlr.root=vnew_root;\
	       }\
	  else {\
	       unlock(rebalance);\
               vr((type *)Xavlwork.Xavlstack[depth-1].ptr,rebalance,type,RW);\
               if(Xavlwork.Xavlstack[depth-1].which_way==Xavlright)\
		    rebalance->rec_avl.fp=vnew_root;\
	       else Xavllsps(rebalance->rec_avl.lsp,vnew_root,type);\
	       }\
	  }\
     unlock(rebalance);\
     }\
     /*This check is very tricky*/\
/*\
if ((hd)->hd_avlr.depth>0) {\
     as((hd)->hd_avlr.entries <= ( (1<<(hd)->hd_avlr.depth) - 1 ) );\
     as((hd)->hd_avlr.entries >= ( (1<<((hd)->hd_avlr.depth-1)) ) );\
     }\
else as((hd)->hd_avlr.entries==0);\
*/\
}

#define AVL_PICK(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare)\
\
Xavl_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,V_R,R_V,NOP,ADRCK)

#define AVLO_PICK(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare)\
\
Xavl_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

/*----------------------------------------------------------------------------*/

/* Insert a key into a AVLH binary-tree.

   rec_avl    ==> name within rec of AVLH member which is to be inserted into
		  the avl tree.
   insert     ==> -> to record containing the rec_avl AVLH member to be inserted
		  into the avl tree.
   type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. insert's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVL member which is the top
		  of the tree to which insert will be placed.
   key_already_there ==> a set of instructions to be executed when the key
			 already is in the tree.
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      Both arguments are rec->rec_avl.key.
			 */
#define Xavlh_insert(rec_avl,insert,type,hd_avlr,hd,htype,key_already_there,compare,vr,rv,unlock,roadrck)\
\
Xavl_insert(rec_avl,insert,type,hd_avlr,hd,key_already_there,compare,vr,rv,unlock,roadrck)\
as(insert->rec_avl.hp DNE);\
rv(hd,insert->rec_avl.hp,htype)

#define AVLH_INSERT(rec_avl,insert,type,hd_avlr,hd,htype,key_already_there,compare)\
\
Xavlh_insert(rec_avl,insert,type,hd_avlr,hd,htype,key_already_there,compare,V_R,R_V,NOP,NOP,ADRCK)

#define AVLHO_INSERT(rec_avl,insert,type,hd_avlr,hd,htype,key_already_there,compare)\
\
Xavlh_insert(rec_avl,insert,type,hd_avlr,hd,htype,key_already_there,compare,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

/*----------------------------------------------------------------------------*/

/* Delete a key in an AVLH binary-tree
   
   rec_avl    ==> name within rec of AVLH member which is to be deleted from
		  the avl tree.
   delete     ==> -> to record containing the rec_avl AVLH member to be deleted
		  from the avl tree.
   type   ==> name of the type (structure) of the record which forms the
		  avl tree. i.e. delete's type.
   hd_avlr    ==> name within hd record of AVLR member which is the top (head)
		  of the avl tree.
   hd         ==> -> to a record which contains the AVLR member which is the top
		  of the avl tree which delete is to be removed.
   key_not_there ==> operations to perform if the key is not present
		     in the tree.
   compare    ==> a macro or subroutine which yields an integer value such
		  that 0 means the two keys given to compare are identical,
		      >0 means the 1st key is greater than the second,
		      <0 means the 1st key is less than the second.
		      Both arguments are rec->rec_avl.key.
			                                                   */
#define Xavlh_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,vr,rv,unlock,roadrck)\
\
Xavl_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,vr,rv,unlock,roadrck)\
delete->rec_avl.hp=NULL

#define AVLH_PICK(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare)\
\
Xavlh_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,V_R,R_V,NOP,NOP,ADRCK)

#define AVLHO_PICK(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare)\
\
Xavlh_pick(rec_avl,delete,type,hd_avlr,hd,key_not_there,compare,VO_RO,RO_VO,UNLOCK,RO_ADRCK)

/*----------------------------------------------------------------------------*/
#endif AVL_M_INCLUDED
