
/* 

    This file implements the following utility routines in C for the
    following operating systems.  The routines all return 0 if successful.
    The last parameter of many is an error string.  If it is non-NULL, then
    it is assumed to be MAX_STRING_LENGTH long and an error message will
    be returned in it upon failure. 

        UNIX 4.2 BSD (compile with -DUNIX -DBSD42)        
	UNIX 4.1C BSD (compile with -DUNIX -DBSD41C)
	UNIX SYSTEM 5 (OPUS) (compile with -DUNIX -DOPUS)
        

    const
      MAX_STRING_LENGTH = 255; { as high as we can count with 1 byte }
    type
      Cint: longint;
      char_array = packed array[0..MAX_STRING_LENGTH] of char;
      xtring = ^char_array;  { with count in first byte }
      time_value = record  { an 8 byte value }
        high: longint;     { high order 4 bytes }
	low: longint;      { low order 4 bytes }
      end;
      
    function firm(name,err: xtring): Cint;
      { remove the named file, provided that it is a regular file. }

    function fiexists(name: xtring): Cint;
      { return 0 if the named file exists.  Return -1 otherwise. }
    
    function fitime(name: xtring; var timestamp: time_value; err: xtring): Cint;
      { Return last-modified time for the named file }

    function sytime(var t: time_value): Cint;
      { Return the current system time }

    function syxtime(var t: time_value): Cint;
      { Return some time in the interval from start of process to the current
        time (x stands for "approximate". }

    function ficreate(name, err: xtring): Cint;
      { If the named file does not exists, then create it.  (Creation
        is done with read/write permissions set entirely by umask). 
	Return 0 iff the file exists (or has been created) and the
	calling program has read/write permission for it. }

*/


#include <errno.h>

typedef xtring *char; /* Valid pascal string */

#define MAX_STRING_LENGTH 255 /* Max length of an xtring */

/* Error numbers for errors caught here -- these are negative, while the
   system error numbers are positive. */
#define NOT_REGULAR_FILE (-1)


struct time_value {
	long high;
	long low;
	}


extern int errno;  /* system error number */


static void
string_to_xtring(s,x)
    /* convert C string s to Valid pascal string x */
    register char *s;
    xtring x;
{
    register char *p,*end;
    
    end=x+MAX_STRING_LENGTH;  /* insure no overflow */
    for (p=x+1;(*s!='\0')&&(p<=end);p++) *p=(*s++);
    *x=(char)(p-x)-1;
}


static void
xtring_to_string(x,s)
    /* convert Valid pascal string x to C string s */
    xtring x;
    register char *s;
{
    register char *p,*end;
    
    end=x+(*x);
    for (p=x+1;(p<=end);s++) *s=(*p++);
    *s='\0';
}


/*
 * If err exists, fill it with the appropriate error message.
 */
static void
do_error(num, err)
    int num;
    xtring err;
{
    if (!err) return;

    switch (num) {
	case NOT_REGULAR_FILE:
	    string_to_xtring("Not a regular file", err);
	    break;
	case ENOTDIR:
	    string_to_xtring("A component of the path is not a directory");
	    break;
	case EPERM:
	    string_to_xtring("Pathname contains char with high-order bit set");
	    break;
	case ENOENT:
	    string_to_xtring("File does not exist or pathname too long");
	    break;
	case EACCES:
	    string_to_xtring("Permission denied");
	    break;
	case ELOOP:
	    string_to_xtring("Too many symbolic links encountered"
	    break;
	case EBUSY:
	    string_to_xtring("File is a mount device, and is busy"
	    break;
	case EROFS:
	    string_to_xtring("File is on a read-only file system");
	    break;
        default:
	    string_to_xtring("Unknown reason for failure", err);
	    break;
	}
}
             

int
firm(name, err)
    xtring name, err;
{
    char cname[MAX_STRING_LENGTH];

    xtring_to_string(name, cname);
    if (stat(cname, &st) != 0) {
        do_error(errno, err);
	return(FAILURE);
	}

    if ((st.st_mode & ST_IFMT) != ST_IFREG) {
        do_error(NOT_REGULAR_FILE, err);
	return(FAILURE);
	}

    if (unlink(cname) != 0) {
        do_error(errno, err);
	return(FAILURE);
	}

    return(SUCCESS);
}


int
fitime(name, timestamp, err)
    xtring name;
    struct time_value *timestamp;
    xtring err;
{
    char cname[MAX_STRING_LENGTH];

    xtring_to_string(name, cname);
    if (stat(cname, &st) != 0) {
        do_error(errno, err);
	return(FAILURE);
	}

    if ((st.st_mode & ST_IFMT) != ST_IFREG) {
        do_error(NOT_REGULAR_FILE, err);
	return(FAILURE);
	}

    
    timestamp->low = st.st_mtime;
    timestamp->high = 0;
    return(SUCCESS);
}


static struct time_value saved_time = {0, 0};


int
sytime(t)
    struct time_value *t;
{
    saved_time.low = time(0);
    t->low = saved_time.low;
    t->high = saved_time.high;
    return(SUCCESS);
}


int
syxtime(t)
    struct time_value *t;
{
    if (!saved_time.low) return(sytime(t));
    t->low = saved_time.low;
    t->high = saved_time.high;
    return(SUCCESS);
}


ficreate(name, err)
    xtring name, err;
{
}

    int fd;
    char cname[MAX_STRING_LENGTH];

    xtring_to_string(name,cname);

    if ((fd=open(cname,O_RDWR|O_CREAT,0666))<0) {
        do_error(errno, err);
	return(FAILURE);
	}
    close(fd);
    if (access(cname,R_OK|W_OK)!=0) {
        do_error(errno, err);
	return(FAILURE);
	}
    return(SUCCESS);
}
