(**){--------------- NEW & RELEASE for SCHEMA STRUCTURES --------}


{---------------------------------------------------------------}
{ All new routines for schema objects obtain a new object and   }
{ insert it to the head of the list.  All release routines      }
{ (for these) delete the object from its list.                  }
{ All release_entire... routines and release_all.. routines     }
{ relase an entire list and  return NIL.                        }
{ Examples:                                                     }
{   new_object(head_of_list);                                   }
{   new_object(parent_of_new_one^.next);                        }
{   release_object(head_of_list);                               }
{   release_entire_list(head_of_list);                          }
{   release_object(parent_of_one_to_delete^.next);              }
{---------------------------------------------------------------}


procedure new_text_macro(var head: text_macro_ptr);
  { Gets a new text_macro (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: text_macro_ptr; { new one }
begin
  if free_text_macros <> NIL then
    begin  
      newone := free_text_macros;  
      free_text_macros := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_TEXT_MACRO, 3*POINTER_SIZE+BOOL_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      name := NIL;
      text := NIL;
      reserved := FALSE;
    end;
  head := newone;
end { new_text_macro } ;


procedure release_text_macro(var head: text_macro_ptr);
  { Releases a text_macro for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: text_macro_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_text_macros;
      free_text_macros := old;
    end;
end { release_text_macro } ;


procedure release_all_text_macros(var head: text_macro_ptr);
  { releases a text_macro list by insertion into freelist. Returns NIL. }
  var
    last: text_macro_ptr; { last element of list }
begin
  if head <> NIL then
    begin
      last := head;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_text_macros;
      free_text_macros := head;
      head := NIL;
    end;
end { release_all_text_macros } ;




procedure new_expandable_id(var head: expandable_id_ptr);
  { Gets a new expandable_id (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: expandable_id_ptr; { new one }
begin
  if free_expandable_ids <> NIL then
    begin  
      newone := free_expandable_ids;  
      free_expandable_ids := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_EXPANDABLE_ID, 2*POINTER_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      name := NIL;
    end;
  head := newone;
end { new_expandable_id } ;


procedure release_expandable_id(var head: expandable_id_ptr);
  { Releases a expandable_id for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: expandable_id_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_expandable_ids;
      free_expandable_ids := old;
    end;
end { release_expandable_id } ;


procedure release_all_expandable_ids(var head: expandable_id_ptr);
  { releases a expandable_id list by insertion into freelist. Returns NIL. }
  var
    last: expandable_id_ptr; { last element of list }
begin
  if head <> NIL then
    begin
      last := head;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_expandable_ids;
      free_expandable_ids := head;
      head := NIL;
    end;
end { release_all_expandable_ids } ;


procedure new_dependency_list(var head: dependency_list_ptr);
  { Gets a new text_macro (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: dependency_list_ptr; { new one }
begin
  if free_dependency_lists <> NIL then
    begin  
      newone := free_dependency_lists;  
      free_dependency_lists := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_DEPENDENCY_LIST, 2*POINTER_SIZE+INT_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      file_name := NIL;
      last_modified_time := 0;
    end;
  head := newone;
end { new_dependency_list } ;


procedure release_dependency_list(var head: dependency_list_ptr);
  { Releases a dependency_list for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: dependency_list_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_dependency_lists;
      free_dependency_lists := old;
    end;
end { release_dependency_list } ;


procedure release_all_dependency_lists(var head: dependency_list_ptr);
  { releases a dependency_list list by insertion into freelist. Returns NIL. }
  var
    last: dependency_list_ptr; { last element of list }
begin
  if head <> NIL then
    begin
      last := head;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_dependency_lists;
      free_dependency_lists := head;
      head := NIL;
    end;
end { release_all_dependency_lists } ;


procedure new_compiled_context_list(var head: compiled_context_list_ptr);
  { Gets a new compiled_context_list element,
    initializes it and inserts it at the head of the list. }
  var
    newone: compiled_context_list_ptr; { new one }
begin
  if free_compiled_context_lists <> NIL then
    begin  
      newone := free_compiled_context_lists;  
      free_compiled_context_lists := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_COMPILED_CONTEXT_LIST, 2*POINTER_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      context := NIL;
      dirty := FALSE;
      dirty_for_pass_2 := FALSE;
    end;
  head := newone;
end { new_compiled_context_list } ;


procedure release_compiled_context_list(var head: compiled_context_list_ptr);
  { Releases a compiled_context_list for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: compiled_context_list_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_compiled_context_lists;
      free_compiled_context_lists := old;
    end;
end { release_compiled_context_list } ;


procedure release_all_compiled_context_lists(var head: compiled_context_list_ptr);
  { releases a compiled_context_list list by insertion into freelist. Returns NIL. }
  var
    last: compiled_context_list_ptr; { last element of list }
begin
  if head <> NIL then
    begin
      last := head;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_compiled_context_lists;
      free_compiled_context_lists := head;
      head := NIL;
    end;
end { release_all_compiled_context_lists } ;


procedure new_paged_schema(var head: paged_schema_ptr);
  { Gets a new paged_schema (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: paged_schema_ptr; { new one }
begin
  if free_paged_schemas <> NIL then
    begin  
      newone := free_paged_schemas;  
      free_paged_schemas := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_PAGED_SCHEMA, 
			   7*POINTER_SIZE+3*INT_SIZE+BOOL_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      drawing_type := NIL;
      version := 0;
      page := 0;
      last_modified_time := 0;
      expandable_ids := NIL;
      local_text_macros := NIL;
      dependencies := NIL;
      properties := NIL;
      compiled_contexts := NIL;
      { has_expr := FALSE; } { obsolete }
      make_performed := FALSE;
    end;
  head := newone;
end { new_paged_schema } ;


procedure release_paged_schema(var head: paged_schema_ptr);
  { Releases a paged_schema for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: paged_schema_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      release_all_expandable_ids(head^.expandable_ids);
      release_all_numbered_token_lists(head^.properties);
      release_all_text_macros(head^.local_text_macros);
      release_all_dependency_lists(head^.dependencies);
      release_all_compiled_context_lists(head^.compiled_contexts);
      old := head;
      head := head^.next;
      old^.next := free_paged_schemas;
      free_paged_schemas := old;
    end;
end { release_paged_schema } ;


procedure release_all_paged_schemas(var head: paged_schema_ptr);
  { releases a paged_schema list by insertion into freelist. Returns NIL. }
begin
  while head <> NIL do release_paged_schema(head);
end { release_all_paged_schemas } ;


procedure new_parameter(var head: parameter_ptr);
  { Gets a new parameter (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: parameter_ptr; { new one }
begin
  if free_parameters <> NIL then
    begin  
      newone := free_parameters;  
      free_parameters := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_PARAMETER, 3*POINTER_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      name := NIL;
      text := NIL;
    end;
  head := newone;
end { new_parameter } ;


procedure release_parameter(var head: parameter_ptr);
  { Releases a parameter for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: parameter_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_parameters;
      free_parameters := old;
    end;
end { release_parameter } ;


procedure release_all_parameters(var head: parameter_ptr);
  { releases a parameter list by insertion into freelist. Returns NIL. }
  var
    last: parameter_ptr; { last element of list }
begin
  if head <> NIL then
    begin
      last := head;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_parameters;
      free_parameters := head;
      head := NIL;
    end;
end { release_all_parameters } ;


procedure new_context_definition(var head: context_definition_ptr);
  { Gets a new context_definition (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: context_definition_ptr; { new one }
begin
  if free_context_definitions <> NIL then
    begin  
      newone := free_context_definitions;  
      free_context_definitions := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_CONTEXT_DEFINITION,
                           2*POINTER_SIZE+2*INT_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      number := 0;
      version := 1;
      parameters := NIL;
    end;
  head := newone;
end { new_context_definition } ;


procedure release_context_definition(var head: context_definition_ptr);
  { Releases a context_definition for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: context_definition_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      release_all_parameters(head^.parameters);
      old := head;
      head := head^.next;
      old^.next := free_context_definitions;
      free_context_definitions := old;
    end;
end { release_context_definition } ;


procedure release_all_context_definitions(var head: context_definition_ptr);
  { releases a context_definition list by insertion into freelist. 
    Returns NIL. }
begin
  while head <> NIL do release_context_definition(head);
end { release_all_context_definitions } ;


procedure new_property_attribute(var head: property_attribute_ptr);
  { Gets a new property_attribute (from freelist, if possible),
    initializes it and inserts it at the head of the list. }
  var
    newone: property_attribute_ptr; { new one }
begin
  if free_property_attributes <> NIL then
    begin  
      newone := free_property_attributes;  
      free_property_attributes := newone^.next;  
    end
  else
    begin
      new(newone);  
      increment_heap_count(HEAP_PROPERTY_ATTRIBUTE,
                           2*POINTER_SIZE+INT_SIZE);
    end;
  with newone^ do
    begin
      next := head;
      property := NIL;
      attributes := [];
    end;
  head := newone;
end { new_property_attribute } ;


procedure release_property_attribute(var head: property_attribute_ptr);
  { Releases a property_attribute for re-use. Sets head to
    the former head^.next (so can be used for list deletion). }
  var
    old: property_attribute_ptr; { saves old head for release }
begin
  if head <> NIL then
    begin
      old := head;
      head := head^.next;
      old^.next := free_property_attributes;
      free_property_attributes := old;
    end;
end { release_property_attribute } ;


procedure release_all_property_attributes(var head: property_attribute_ptr);
  { releases a property_attribute list by insertion into freelist. 
    Returns NIL. }
  var
    tail: property_attribute_ptr;
begin
  if head <> NIL then
    begin
      tail := head;
      while tail^.next <> NIL do tail := tail^.next;
      tail^.next := free_property_attributes;
      free_property_attributes := head;
      head := NIL;
    end;
end { release_all_property_attributes } ;


procedure release_all_schema_fields(var schema: schema_definition);
  { releases all fields of the schema. When done, all is as if the
    schema had just been initialized. }
begin
  with schema do
    begin
      release_numbered_dictionary(id_dictionary);
      release_numbered_dictionary(string_dictionary);
      release_all_paged_schemas(paged_schemas);
      release_all_text_macros(used_global_TMs);
      release_all_context_definitions(contexts);
      release_all_property_attributes(used_properties);
      release_entire_property_list(local_TMs);
      local_TMs_defined := FALSE;
      file_name := nullstring;
      directory := NIL;
      file_accessible := FALSE;
      changed := FALSE;
      bubble_check := FALSE;
      enable_cardinal_tap := FALSE;
    end;
end { release_all_schema_fields } ;


(**){----------- SCHEMA HANDLING --------------------------------}


procedure dump_text_macros(*var f: textfile; list: text_macro_ptr*);
  { dump the list of text macros to f for debugging purposes. }
  var
    tm: text_macro_ptr;   { current text macro }
begin
  tm := list;
  if tm = NIL then writeln(f, ' <NIL>')
  else repeat
    write(f, ' ');
    writealpha(f, tm^.name^.identifier_p^.name);
    write(f, '=');
    writestring(f, tm^.text^.string_p);
    tm := tm^.next;
    if tm = NIL then writeln(f, ';')
    else
      begin
        writeln(f, ',');  write(f, '   ');
      end;
  until tm = NIL;
end { dump_text_macros } ;


procedure add_to_dependency_list(fi: plumbing_page_ptr);
  { adds the file name to the dependency list for the page being compiled,
    inserting it into the dictionary for the schema file. }
  var
    entry: dependency_list_ptr;    { an entry already in list }
    found: boolean;                { TRUE iff name is already in list }
    number: numbered_token_ptr;    { schema string dictionary entry for name }
    name: xtring;                  { name of file being added }
begin
  name := fi^.filename;
  if debug_24 then
    begin
      write(outfile, ' depends on ');
      dump_string(outfile, name);
      writeln(outfile);
    end;

  entry := paged_schema_of_this_page.dependencies;  found := FALSE;
  while not found and (entry <> NIL) do
    if entry^.file_name^.string_p = name then found := TRUE
                                         else entry := entry^.next;
  if not found then
    begin
      if fi^.last_modified_time = 0 then
        if not get_time_stamp(name, fi^.last_modified_time) then ;
      number := enter_numbered_string(name, 
	          schema_of_drawing_being_compiled.string_dictionary);
      if number <> NIL then
        begin
          new_dependency_list(paged_schema_of_this_page.dependencies);
          paged_schema_of_this_page.dependencies^.file_name := number;
	  paged_schema_of_this_page.dependencies^.last_modified_time :=
	    fi^.last_modified_time;
	end;
    end;
end { add_to_dependency_list } ;


procedure log_used_global_TMs_in_schema(var schema: schema_definition);
  { This routine updates the global text macro list in the schema file.
    It first removes the old one and then creates a new one by perusing
    the expandable id lists and adding an appropriate entry for each 
    expandable id (whether globally defined or not).  Undefined
    identifiers are output with nullstring as their definition (and not
    reserved).
    
    This routine is to be called immediately before outputting the
    schema file for the drawing being compiled. 
    The expandable id lists are assumed to be sorted in 
    ascending order of id number.  The list is guaranteed to be kosher
    (in order, with no NIL numbered tokens or NIL numbered_token values) }
  var
    current_page: paged_schema_ptr;     { schema of current page }
    current_tm: text_macro_ptr;         { current expandable id }
    current_exp_id: expandable_id_ptr;  { current expandable id }
    nullstring_tok: numbered_token_ptr; { numbered token for nullstring }


  procedure insert_TM(name: numbered_token_ptr);
    { Insert name into used_global_TMs list if not already there.
      Update current_tm to point to the element representing name. }
    var
      temp: text_macro_ptr;    { current element in search }
      done: boolean;           { TRUE when element or insert point is found }
      def: numbered_token_ptr; { definition of text macro (if new) }        
      found: boolean;          { TRUE iff entry already exists }
  begin

    { safety check  - orders TM list even if exp id list is not }

    if current_tm <> NIL then
      if current_tm^.name^.number >= name^.number then
	begin
	  { the exp_id list must be out of order - could put an
	    assertion message here. }
	  current_tm := NIL;  { start search at head of list }
	end;
    
    { find element or insertion point }

    if current_tm = NIL then temp := schema.used_global_TMs
    else temp := current_tm^.next;
    done := FALSE;
    while (temp <> NIL) and not done do
      if temp^.name^.number < name^.number then
	begin
	  current_tm := temp;
	  temp := temp^.next;
	end
      else done := TRUE;

    { insert if not there }

    if current_tm = NIL then found := FALSE
    else if current_tm^.name^.number = name^.number then found := TRUE
    else found := FALSE;

    def := NIL;
    if not found then
      if ([UNRESERVED,RESERVED] * name^.identifier_p^.kind = []) then
        begin
          def := nullstring_tok;
	  if name^.identifier_p^.definition <> nullstring then
	    assert(237 { should be null !! });
	  end
      else
        def := enter_numbered_string(name^.identifier_p^.definition,
                                     schema.string_dictionary);

    if def <> NIL then
      begin
	{ if entry to table was non-kosher, def will be NIL, with
	  assertion violation already reported. }
	new_text_macro(schema.used_global_TMs);
	current_tm := schema.used_global_TMs;
	current_tm^.name := name;
	current_tm^.text :=  def;
	current_tm^.reserved := (RESERVED in name^.identifier_p^.kind);
      end;
  end { insert_TM } ;


begin { log_used_global_TMs_in_schema }
  if debug_24 then writeln(outfile, ' log_used_global_TMs_in_schema');
  nullstring_tok :=
    enter_numbered_string(nullstring, schema.string_dictionary);
  release_all_text_macros(schema.used_global_TMs);
  current_page := schema.paged_schemas;
  while current_page <> NIL do
    begin
      current_exp_id := current_page^.expandable_ids;
      current_tm := NIL;
      while current_exp_id <> NIL do
	begin
	  insert_TM(current_exp_id^.name);
	  current_exp_id := current_exp_id^.next;
	end;
      current_page := current_page^.next;
    end;
end { log_used_global_TMs_in_schema } ;

  
procedure enter_local_TM(var page: paged_schema;
                         name: name_ptr; val: xtring);
  { enter the local text macro def into the given page.  These are to
    be sorted by name token number.  There will be NO duplicates. }
  var
    tm: text_macro_ptr;      { list entry created from name and val }
    parent: text_macro_ptr;  { parent of tm in sorted list }
    done: boolean;           { TRUE when insertion point is found }
begin
  if debug_24 then 
    begin
      write(Outfile, ' defines local TM: ');
      writealpha(Outfile, name^.name);
      write(Outfile, '=');
      writestring(Outfile, val);
      writeln(Outfile);
    end;

  tm := NIL;
  new_text_macro(tm);
  tm^.name := 
    enter_numbered_id(name, schema_of_drawing_being_compiled.id_dictionary);
  tm^.text := 
    enter_numbered_string(val, 
                          schema_of_drawing_being_compiled.string_dictionary);

  if (tm^.text = NIL) or (tm^.name = NIL) then
    release_text_macro(tm) { assertion failure in enter_numbered... }
  else
    begin
      parent := page.local_text_macros;
      if parent <> NIL then
        if parent^.name^.number > tm^.name^.number then
	  parent := NIL
	else if parent^.name^.number = tm^.name^.number then
	  begin
	    assert(0);
	    writeln(CmpLog, ' Duplicate TM name in enter_local_TM !!');
	    parent := NIL;
	  end
	else
	  begin
	    done := FALSE;
	    while (parent^.next <> NIL) and not done do
	      begin
                if parent^.next^.name^.number < tm^.name^.number then
		  parent := parent^.next
	        else if parent^.next^.name^.number = tm^.name^.number then
	          begin
	            assert(0);
	            writeln(CmpLog,
		      ' Duplicate TM name in enter_local_TM !!');
		    done := TRUE;
	          end
	        else done := TRUE;
	      end;
	  end;

      if parent = NIL then
        begin
          tm^.next := page.local_text_macros;
          page.local_text_macros := tm;
	end
      else
        begin
          tm^.next := parent^.next;
          parent^.next := tm;
	end
    end;
end { enter_local_TM } ;


procedure enter_expandable_id(*id: name_ptr*);
  { enter the id into the expandable id list for the page being compiled.  
    (This list is headed by paged_schema_of_this_page.expandable_ids. 
     The ids are sorted by token number (within schema of this page
     dictionary) }
  label
    90; { return }
  var
    tok: numbered_token;        { for passing to insert_numbered_token }
    newtok: numbered_token_ptr; { table entry for this id }
    element: expandable_id_ptr; { new or found element for list }
    parent: expandable_id_ptr;  { parent of element }
    found: boolean;             { TRUE if element found in list }
begin
  if debug_24 then 
    begin
       write(outfile, ' enter_expandable_id ');
       if id = NIL then write(outfile, 'NIL')
       else print_alpha(outfile, id^.name);
       if PERMANENT in id^.kind then writeln(outfile, ' permanent');
       if page_being_compiled = 0 then
         writeln(outfile, '   not compiling a page -- ignored');
     end;
       
  if id = NIL then
    begin
      assert(ASSERT_NIL_NAME_TO_ENTER_EXPANDABLE);
      goto 90 { return } ;
    end;

  if page_being_compiled = 0 then goto 90 { return } ;
  { not compiling a page (just examining versions) -- ignore }

  if PERMANENT in id^.kind then goto 90 { return } ; { don't log these }

  { number the id }

  init_numbered_token(tok, IDENTIFIER_NUMBER);
  tok.identifier_p := id;
  newtok := 
    insert_numbered_token(tok, schema_of_drawing_being_compiled.id_dictionary);
  if newtok = NIL then
    begin
      assert(ASSERT_FAILED_ENTER_EXPANDABLE_ID);
      goto 90 { return } ;
    end;

  { find id or place to add it to the list }

  parent := NIL;
  element := paged_schema_of_this_page.expandable_ids;
  found := FALSE;
  while (element <> NIL) and not found do
    if element^.name = NIL then
      begin
	assert(205 { Nil token value });
        if parent = NIL then
	  begin
	    release_expandable_id(paged_schema_of_this_page.expandable_ids);
	    element := paged_schema_of_this_page.expandable_ids;
	  end
	else
	  begin
	    release_expandable_id(parent^.next);
	    element := parent^.next;
	  end;
      end
    else if element^.name^.number = newtok^.number then found := TRUE
    else if element^.name^.number > newtok^.number then element := NIL
    else
      begin
	parent := element;
	element := element^.next;
      end;


  if not found then
    begin
      if debug_24 then writeln(outfile, '   new entry');
      if parent = NIL then
	begin
	  new_expandable_id(paged_schema_of_this_page.expandable_ids);
	  element := paged_schema_of_this_page.expandable_ids;
	end
      else
	begin
	  new_expandable_id(parent^.next);
	  element := parent^.next;
	end;
      element^.name := newtok;
    end
  else if debug_24 then writeln(outfile, '   found it');
  
  { report the expandable id to data services }
  report_expandable_id_to_ds(id);
90:
end { enter_expandable_id } ;


procedure compute_current_local_TM_context(current_page: paged_schema_ptr;
                                           page_read: boolean);
  { Update schema.local_TMs to reflect the local text macros defined
    on all pages.  Assume that it is not currently valid.  current_page
    is the (old) schema representation of the current page .  If page_read, 
    then use the values just read for the page instead of those stored in
    the schema (in current_page).  This procedure has the side effect of
    checking the currently defined values for errors and dirtying the pages
    where such errors are found. NOTE: error messages are issued only
    for the current page and only if page_read. }
  var
    page: paged_schema_ptr; { current page of schema }
    found: boolean;         { TRUE when first page of this module is found }
    done: boolean;          { TRUE when one with all pages of module }
    prop: property_ptr;     { TM found in list }
    current_TM:
      text_macro_ptr;       { current TM from schema page }
begin
  if debug_23 or debug_24 then
    writeln(Outfile, ' compute_current_local_TM_context:');

  release_entire_property_list(schema_of_drawing_being_compiled.local_TMs);

  page := schema_of_drawing_being_compiled.paged_schemas;
  found := FALSE;
  while (page <> NIL) and not found do
    if (page^.drawing_type = current_page^.drawing_type) and
       (page^.version = current_page^.version) then found := TRUE
    else page := page^.next;

  done := FALSE;
  while (page <> NIL) and not done do
    begin
      if (page = current_page) and page_read then
        current_TM := paged_schema_of_this_page.local_text_macros
      else current_TM := page^.local_text_macros;

      while current_TM <> NIL do
        begin
          if find_property(
            schema_of_drawing_being_compiled.local_TMs,
            current_tm^.name^.identifier_p, prop) then
            begin
              if current_TM^.text^.string_p <> prop^.text then
                if (page = current_page) and page_read then
                  begin
                    error(114 { text macro already exists });
                    error_dump_body_name(DEFINE_string);
                    error_dump_text_macro(
                      current_tm^.name^.identifier_p);
                  end
                else mark_dirty_for_pass_2(page);
            end
          else
            if RESERVED IN current_TM^.name^.identifier_p^.kind then
              begin
                if (page = current_page) and page_read then
                  begin
                    error(105 { reserved TM name });
                    error_dump_body_name(DEFINE_string);
                    error_dump_text_macro(
                      current_TM^.name^.identifier_p);
                  end
                else mark_dirty_for_pass_2(page);
              end
            else
              add_to_prop_list(
                schema_of_drawing_being_compiled.local_TMs,
                current_TM^.name^.identifier_p,
                current_TM^.text^.string_p);

          current_TM := current_TM^.next;
        end;

      page := page^.next;
      if page <> NIL then
        if (page^.drawing_type <> current_page^.drawing_type) or
           (page^.version <> current_page^.version) then 
          done := TRUE;
    end;

  schema_of_drawing_being_compiled.local_TMs_defined := TRUE;

  if debug_23 or debug_24 then
    dump_property_list(Outfile, schema_of_drawing_being_compiled.local_TMs);
end { compute_current_local_TM_context } ;


procedure sort_expandable_ids(var head: expandable_id_ptr);
  { Sorts the expandable ids in ascending order of id number. 
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and not efficient), but are 
    quietly handled. }
  var
    next_element: expandable_id_ptr;  { next element to be checked }
    parent: expandable_id_ptr;        { parent of next_element }
    insert_point: expandable_id_ptr;  { new parent of next_element 
                                        (if it must be moved) }
begin
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      if next_element^.name^.number = parent^.name^.number then
        begin
          assert(ASSERT_DUPLICATE_EXPANDABLE_ID_NUMBERS);
          release_expandable_id(parent^.next);
        end
      else if next_element^.name^.number > parent^.name^.number then
          parent := next_element
      else { out of order - move it }
        begin
          parent^.next := next_element^.next;
          if next_element^.name^.number = head^.name^.number then
            begin
              assert(ASSERT_DUPLICATE_EXPANDABLE_ID_NUMBERS);
              next_element^.next := NIL;
              release_expandable_id(next_element);
            end
          else if next_element^.name^.number < head^.name^.number then
            begin
              next_element^.next := head;
              head := next_element;
            end
          else
            begin
              insert_point := head;
              while insert_point^.next^.name^.number < 
                next_element^.name^.number do
                insert_point := insert_point^.next;
              if insert_point^.next^.name^.number = 
                next_element^.name^.number then
                begin
                  assert(ASSERT_DUPLICATE_EXPANDABLE_ID_NUMBERS);
                  next_element^.next := NIL;
                  release_expandable_id(next_element);
                end
              else
                begin
                  next_element^.next := insert_point^.next;
                  insert_point^.next := next_element^.next;
                end;
            end;
        end { out of order } ;
      next_element := parent^.next;
    end { while } ;
end { sort_expandable_ids } ;


procedure sort_text_macros(var head: text_macro_ptr);
  { Sorts the text macros in ascending order of name token number (like
    expandable ids).
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and not efficient), but are 
    quietly handled. }
  var
    next_element: text_macro_ptr;  { next element to be checked }
    parent: text_macro_ptr;        { parent of next_element }
    insert_point: text_macro_ptr;  { new parent of next_element 
                                        (if it must be moved) }
begin
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      if next_element^.name^.number = parent^.name^.number then
        begin
          assert(0);
	  writeln(CmpLog, ' Duplicate TM logged!');
          release_text_macro(parent^.next);
        end
      else if next_element^.name^.number > parent^.name^.number then
          parent := next_element
      else { out of order - move it }
        begin
          parent^.next := next_element^.next;
          if next_element^.name^.number = head^.name^.number then
            begin
              assert(0);
	      writeln(CmpLog, ' Duplicate TM logged!');
              next_element^.next := NIL;
              release_text_macro(next_element);
            end
          else if next_element^.name^.number < head^.name^.number then
            begin
              next_element^.next := head;
              head := next_element;
            end
          else
            begin
              insert_point := head;
              while insert_point^.next^.name^.number < 
	            next_element^.name^.number do
                insert_point := insert_point^.next;
              if insert_point^.next^.name^.number = 
                 next_element^.name^.number then
                begin
                  assert(0);
	          writeln(CmpLog, ' Duplicate TM logged!');
                  next_element^.next := NIL;
                  release_text_macro(next_element);
                end
              else
                begin
                  next_element^.next := insert_point^.next;
                  insert_point^.next := next_element^.next;
                end;
            end;
        end { out of order } ;
      next_element := parent^.next;
    end { while } ;
end { sort_text_macros } ;


procedure log_property_use(*id: name_ptr*);
  { enter the id into the used property list for the page being compiled.  
    (This list is headed by paged_schema_of_this_page.properties). 
    The ids are sorted by token number (within schema of this page
    dictionary) }
  label
    90; { return }
  var
    tok: numbered_token;              { for passing to insert_numbered_token }
    newtok: numbered_token_ptr;       { table entry for this id }
    element: numbered_token_list_ptr; { new or found element for list }
    parent: numbered_token_list_ptr;  { parent of element }
    found: boolean;                   { TRUE if element found in list }
begin
  if debug_24 then 
    begin
       write(outfile, ' log_property_use ');
       if id = NIL then write(outfile, 'NIL')
       else print_alpha(outfile, id^.name);
       writeln(outfile);
       if page_being_compiled = 0 then
         writeln(outfile, '   not compiling a page -- ignored');
     end;
       
  if id = NIL then
    begin
      assert(0 { not expected });
      writeln(cmplog, ' NIL id passed to log_property_use');
      goto 90 { return } ;
    end;

  if page_being_compiled = 0 then
    { not compiling a page (just examining versions) -- ignore }
    begin
      assert(0 { This should no longer happen });
      Writeln(CmpLog, ' Examining versions in log_property_use');
      goto 90 { return } ;
    end; 

  { number the id }

  init_numbered_token(tok, IDENTIFIER_NUMBER);
  tok.identifier_p := id;
  newtok := 
    insert_numbered_token(tok, 
                          schema_of_drawing_being_compiled.id_dictionary);
  if newtok = NIL then
    begin
      assert(ASSERT_FAILED_ENTER_EXPANDABLE_ID);
      goto 90 { return } ;
    end;

  { find id or place to add it to the list }

  parent := NIL;
  element := paged_schema_of_this_page.properties;
  found := FALSE;
  while (element <> NIL) and not found do
    if element^.token = NIL then
      begin
        assert(205 { Nil token value });
        if parent = NIL then
          begin
            release_numbered_token_list(paged_schema_of_this_page.properties);
            element := paged_schema_of_this_page.properties;
          end
        else
          begin
            release_numbered_token_list(parent^.next);
            element := parent^.next;
          end;
      end
    else if element^.token^.number = newtok^.number then found := TRUE
    else if element^.token^.number > newtok^.number then element := NIL
    else
      begin
        parent := element;
        element := element^.next;
      end;

  if not found then
    begin { insert a new one }
      if parent = NIL then
        begin
          new_numbered_token_list(paged_schema_of_this_page.properties);
          element := paged_schema_of_this_page.properties;
        end
      else
        begin
          new_numbered_token_list(parent^.next);
          element := parent^.next;
        end;
      element^.token := newtok;
      if debug_24 then writeln(outfile, '   new entry');
    end;
90:
end { log_property_use } ;


procedure sort_propertm_attributes(var head: property_attribute_ptr);
  { Sorts the property names in ascending order of id number. 
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and not efficient), but are 
    quietly handled. }
  var
    next_element: property_attribute_ptr;  { next element to be checked }
    parent: property_attribute_ptr;        { parent of next_element }
    insert_point: property_attribute_ptr;  { new parent of next_element 
                                             (if it must be moved) }
    done: boolean;                         { TRUE when head is legal }
begin
  done := FALSE;
  while (head <> NIL) and not done do
    if head^.property = NIL then 
      begin
        assert(213 { Nil token referenced });
        release_property_attribute(head);
      end
    else done := TRUE;
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      if next_element^.property = NIL then
        begin
          assert(213 { Nil token referenced });
          release_property_attribute(parent^.next);
        end
      else if next_element^.property^.number = parent^.property^.number then
        begin
          assert(ASSERT_DUPLICATE_PROPERTY_NUMBERS);
          release_property_attribute(parent^.next);
        end
      else if next_element^.property^.number > parent^.property^.number then
          parent := next_element
      else { out of order - move it }
        begin
          parent^.next := next_element^.next;
          if next_element^.property^.number = head^.property^.number then
            begin
              assert(ASSERT_DUPLICATE_PROPERTY_NUMBERS);
              next_element^.next := NIL;
              release_property_attribute(next_element);
            end
          else if next_element^.property^.number < head^.property^.number then
            begin
              next_element^.next := head;
              head := next_element;
            end
          else
            begin
              insert_point := head;
              while insert_point^.next^.property^.number < 
                next_element^.property^.number do
                insert_point := insert_point^.next;
              if insert_point^.next^.property^.number = 
                next_element^.property^.number then
                begin
                  assert(ASSERT_DUPLICATE_PROPERTY_NUMBERS);
                  next_element^.next := NIL;
                  release_property_attribute(next_element);
                end
              else
                begin
                  next_element^.next := insert_point^.next;
                  insert_point^.next := next_element^.next;
                end;
            end;
        end { out of order } ;
      next_element := parent^.next;
    end { while } ;
end { sort_property_attributes } ;


function funky_compare_parameters(parm1: parameter_ptr;
                                  parm2: property_ptr): compare_type;
  { compares parameters according to the following.
      1. SIZE parameter is less than other parameter names.
      2. Lexicographic order of rest of parameter names. 
      3. Lexicographic order of value.  (redundant)
    Note the different types of the operands }
begin
  case compare_identifiers(parm1^.name^.identifier_p, parm2^.name) of
    LT: 
      if (parm2^.name = SIZE_prop_name) then 
        funky_compare_parameters := GT
      else funky_compare_parameters := LT;
    GT: 
      if (parm1^.name^.identifier_p = SIZE_prop_name) then 
        funky_compare_parameters := LT
      else funky_compare_parameters := GT;
    EQ: funky_compare_parameters := 
      compare_strings(parm1^.text^.string_p, parm2^.text);
  end;
end { funky_compare_parameters } ;


function compare_parameters(parm1, parm2: parameter_ptr): compare_type;
  { compares parameters according to the following.
      1. SIZE parameter is less than other parameter names.
      2. Lexicographic order of rest of parameter names. 
      3. Lexicographic order of value. (redundant) }
begin
  case compare_identifiers(parm1^.name^.identifier_p,
                           parm2^.name^.identifier_p ) of
    LT: 
      if (parm2^.name^.identifier_p = SIZE_prop_name) then 
        compare_parameters := GT
      else compare_parameters := LT;
    GT: 
      if (parm1^.name^.identifier_p = SIZE_prop_name) then 
        compare_parameters := LT
      else compare_parameters := GT;
    EQ: compare_parameters := 
      compare_strings(parm1^.text^.string_p, parm2^.text^.string_p);
  end;
end { compare_parameters } ;


procedure sort_parameters(var head: parameter_ptr);
  { Sorts the parameters in compare_parameter_names order 
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and n-squared), but are quietly handled. 
    Repeated parameters are deleted }
  var
    next_element: parameter_ptr;  { next element to be checked }
    parent: parameter_ptr;        { parent of next_element }
    insert_point: parameter_ptr;  { new parent of next_element }
    comparison: compare_type;     { relation of elements being checked }
begin
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      case compare_parameter_names(next_element^.name^.identifier_p,
                                   parent^.name^.identifier_p) of
        EQ: 
          begin
          { assert(ASSERT_DUPLICATE_PARAMETERS); }{ who cares - just delete it}
            release_parameter(parent^.next);
          end;
        GT: parent := next_element;
        LT: { out of order - move it }
          begin
            parent^.next := next_element^.next;
            case compare_parameter_names(next_element^.name^.identifier_p,
                                         head^.name^.identifier_p) of
              EQ:
                begin
                { assert(ASSERT_DUPLICATE_PARAMETERS); }
                  next_element^.next := NIL;
                  release_parameter(next_element);
                end;
              LT:
                begin
                  next_element^.next := head;
                  head := next_element;
                end;

              GT:
                begin
                  insert_point := head;
                  comparison := compare_parameter_names(
                    next_element^.name^.identifier_p,
                    insert_point^.next^.name^.identifier_p);
                  while comparison = GT do
                    begin
                      insert_point := insert_point^.next;
                      comparison := compare_parameter_names(
                        next_element^.name^.identifier_p,
                        insert_point^.next^.name^.identifier_p);
                    end;
                  if comparison = EQ then
                    begin
                    { assert(ASSERT_DUPLICATE_PARAMETERS); }
                      next_element^.next := NIL;
                      release_parameter(next_element);
                    end
                  else
                    begin 
                      next_element^.next := insert_point^.next;
                      insert_point^.next := next_element;
                    end;
                end { GT } ;
            end { case } ;
          end { LT -> out of order } ;
      end { case } ;
      next_element := parent^.next;
    end { while } ;
end { sort_parameters } ;


function compare_parameter_lists(context1, 
                                 context2: parameter_ptr): compare_type;
  { compares parameter lists }
  var
    parm1: parameter_ptr;   { current parameter in context1 }
    parm2: parameter_ptr;   { current parameter in context2 }
    done: boolean;          { TRUE when comparison has been made }
begin
  parm1 := context1;  parm2 := context2;
  done := FALSE;
  repeat
    if (parm1 = NIL) then
      if (parm2 = NIL) then 
        begin
          compare_parameter_lists := EQ;  done := TRUE;
        end
      else 
        begin
          compare_parameter_lists := LT;  done := TRUE;
        end
    else
      if (parm2 = NIL) then
        begin
          compare_parameter_lists := GT;  done := TRUE;
        end
      else case compare_parameters(parm1, parm2) of
        LT: 
          begin
            compare_parameter_lists := LT;  done := TRUE;
          end;
        GT: 
          begin
            compare_parameter_lists := GT;  done := TRUE;
          end;
        EQ:
          begin
            parm1 := parm1^.next;  parm2 := parm2^.next;
          end;
      end { case }
  until done;
end { compare_parameter_lists } ;


function funky_compare_parameter_lists(
  context1: parameter_ptr; context2: property_ptr): compare_type;
  { compares parameter lists where 1 is a parameter list and 2 is
    a property_list list }
  var
    parm1: parameter_ptr;   { current parameter in context1 }
    parm2: property_ptr;    { current parameter in context2 }
    done: boolean;          { TRUE when comparison has been made }
begin
  parm1 := context1;  parm2 := context2;
  done := FALSE;
  repeat
    if (parm1 = NIL) then
      if (parm2 = NIL) then 
        begin
          funky_compare_parameter_lists := EQ;  done := TRUE;
        end
      else 
        begin
          funky_compare_parameter_lists := LT;  done := TRUE;
        end
    else
      if (parm2 = NIL) then
        begin
          funky_compare_parameter_lists := GT;  done := TRUE;
        end
      else case funky_compare_parameters(parm1, parm2) of
        LT: 
          begin
            funky_compare_parameter_lists := LT;  done := TRUE;
          end;
        GT: 
          begin
            funky_compare_parameter_lists := GT;  done := TRUE;
          end;
        EQ:
          begin
            parm1 := parm1^.next;  parm2 := parm2^.next;
          end;
      end { case }
  until done;
end { funky_compare_parameter_lists } ;


procedure sort_contexts(var head: context_definition_ptr);
  { Sorts the contexts according to their parameter lists 
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and not efficient), but are 
    quietly handled. }
  var
    next_element: context_definition_ptr;  { next element to be checked }
    parent: context_definition_ptr;        { parent of next_element }
    insert_point: context_definition_ptr;  { new parent of next_element }
    done: boolean;                         { TRUE when insert point found }
begin
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      case compare_parameter_lists(next_element^.parameters, 
                                   parent^.parameters) of
        EQ: { redundant entry -- delete it }
          begin
            assert(236 { redundant context entries });
            release_context_definition(parent^.next);
          end;
        GT: parent := next_element;
        LT: { out of order - move it }
          begin
            parent^.next := next_element^.next;
            case compare_parameter_lists(next_element^.parameters, 
                                         head^.parameters) of
              EQ:
                begin
                  assert(236 { redundant context entries });
                  next_element^.next := NIL;
                  release_context_definition(next_element);
                end;
              LT:
                begin
                  next_element^.next := head;
                  head := next_element;
                end;
              GT:
                begin
                  insert_point := head;  done := FALSE;
                  repeat
                    case 
                      compare_parameter_lists(next_element^.parameters, 
                                              insert_point^.next^.parameters)
                                              of
                      EQ: { redundant }
                        begin
                          assert(236);
                          next_element^.next := NIL;
                          release_context_definition(next_element);
                        end;
                      LT: { insert here }
                        begin
                          next_element^.next := insert_point^.next;
                          insert_point^.next := next_element;
                          done := TRUE;
                        end;
                      GT: insert_point := insert_point^.next;
                    end { case } ;
                  until done;
                end { GT } ;
            end { case } ;
          end { LT -> out of order } ;
      end { case } ;
      next_element := parent^.next;
    end { while } ;
end { sort_contexts } ;


function find_context_definition_insertion_point(
  parms: property_ptr; contexts: context_definition_ptr;
  var parent: context_definition_ptr): context_definition_ptr;
  { Finds the context matching the parameter list.  Both must be
    sorted before calling this routine.  If not found, NIL is
    returned.  In either case, returns a ptr to the parent of the
    context (NIL if context is or would be first element of list. }
  var
    current_context: context_definition_ptr;   { current element of contexts }
    current_parm: property_ptr;                { current element of parms }
    result: context_definition_ptr;            { found context }
    done: boolean;                             { TRUE when we find context or
                                                 go too far }
begin
  result := NIL;  parent := NIL;
  current_context := contexts;  current_parm := parms;  done := FALSE;
  while (current_context <> NIL) and not done do
    case funky_compare_parameter_lists(current_context^.parameters, parms) of
      LT:
        begin
          parent := current_context;
          current_context := current_context^.next;
        end;
      GT: done := TRUE;
      EQ:
        begin
          done := TRUE;
          result := current_context;
        end;
    end;
  find_context_definition_insertion_point := result;
end { find_context_definition_insertion_point } ;
  

function find_context_definition(
  parms: property_ptr; 
  contexts: context_definition_ptr): context_definition_ptr;
  { Finds the context matching the parameter list.  Both must be
    sorted before calling this routine.  If not found, NIL is
    returned. }
  var
    dummy: context_definition_ptr;   { dummy parent }
begin
  find_context_definition := 
    find_context_definition_insertion_point(parms, contexts, dummy);
end { find_context_definition } ;


function enter_context_definition(
  parms: property_ptr; 
  var schema: schema_definition): context_definition_ptr;
  { Finds the context matching the parameter list.  Both must be
    sorted before calling this routine.  If not found, one is created. }
  label
    90; { return } 
  var
    cont: context_definition_ptr;    { the context found/created }
    parent: context_definition_ptr;  { parent of cont }
    dic_error: boolean;              { TRUE iff numbered dictionary error }


  function copy_parameters(source: property_ptr; 
                           var dic_error: boolean): parameter_ptr;
    { return a copy of the source list, entering things into the
      dictionaries of the current schema. dic_error indicates an
      error in entering something, in which case the parms are not
      copied }
    var
      head: parameter_ptr;           { head of returned list }
      tail: parameter_ptr;           { tail of returned list }
      current: property_ptr;         { current element of source list }
  begin
    head := NIL;  tail := NIL;  current := source;  dic_error := FALSE;
    while current <> NIL do
      begin
        if head = NIL then 
          begin
            new_parameter(head);  tail := head;
          end
        else
          begin
            new_parameter(tail^.next);  tail := tail^.next;
          end;
        tail^.name := enter_numbered_id(current^.name,
                                        schema.id_dictionary);
        if tail^.name = NIL then dic_error := TRUE;
        tail^.text := enter_numbered_string(current^.text,
                                            schema.string_dictionary);
        if tail^.text = NIL then dic_error := TRUE;
        current := current^.next;
      end { while } ;
    if dic_error then release_all_parameters(head);
    copy_parameters := head;
  end { copy_parameters } ;
  

begin { enter_context_definition }
  cont := find_context_definition_insertion_point(parms, 
                                                  schema.contexts, parent);
  if cont = NIL then
    begin
      new_context_definition(cont);
      if schema.highest_context_number = MAXINT then
        begin
          assert(240 { integer overflow });
          writeln(CmpLog, ' In enter_context_definition');
          cont^.number := 0;
        end
      else
        begin
          schema.highest_context_number := schema.highest_context_number + 1;
          cont^.number := schema.highest_context_number;
        end;
      cont^.parameters := copy_parameters(parms, dic_error);
      if dic_error or (cont^.number = 0) then
        begin
          { This should never actually happen - but if it does, kill
            this instance by returning NIL }
          release_context_definition(cont);
          enter_context_definition := NIL;
          goto 90 { return } ;
        end;
      if parent = NIL then
        begin
          cont^.next := schema.contexts;
          schema.contexts := cont;
        end
      else
        begin
          cont^.next := parent^.next;
          parent^.next := cont;
        end;
    end;
  enter_context_definition := cont;
90:
end { enter_context_definition } ;


function find_context_number(contexts: context_definition_ptr;
                             number: natural_number): context_definition_ptr;
  { find the context having number as its number }
  var
    current: context_definition_ptr; { current context }
    found: boolean;                  { TRUE when found }
begin
  current := contexts;  found := FALSE;
  while (current <> NIL) and not found do
    if current^.number = number then found := TRUE
    else current := current^.next;
  find_context_number := current;
end { find_context_number } ;


function compare_paged_schemas(var page1, 
                                   page2: paged_schema): compare_type;
  { compares paged schemas by type, version, page.  The pages are not
    altered (pass by reference is for efficiency). }
begin
  case compare_identifiers(page1.drawing_type^.identifier_p, 
                           page2.drawing_type^.identifier_p) of
    LT: compare_paged_schemas := LT;
    GT: compare_paged_schemas := LT;
    EQ:
      if page1.version < page2.version then compare_paged_schemas := LT
      else if page1.version > page2.version then compare_paged_schemas := GT
      else if page1.page < page2.page then compare_paged_schemas := LT
      else if page1.page > page2.page then compare_paged_schemas := GT
      else compare_paged_schemas := EQ;
  end;
end { compare_paged_schemas } ;


procedure sort_paged_schemas(var head: paged_schema_ptr);
  { Sorts the paged_schemas.
    The sort is order N for the following 2 cases.
      1. already in order.
      2. in opposite order.
    other cases are not expected (and not efficient), but are 
    quietly handled. }
  var
    next_element: paged_schema_ptr;  { next element to be checked }
    parent: paged_schema_ptr;        { parent of next_element }
    insert_point: paged_schema_ptr;  { new parent of next_element }
    comparison: compare_type;        { saves last compare }
begin
  parent := head;
  if parent = NIL then next_element := NIL
                  else next_element := parent^.next;
  while next_element <> NIL do
    begin
      case compare_paged_schemas(next_element^, 
                                 parent^) of
        EQ: parent := next_element; 
        GT: parent := next_element;
        LT: { out of order - move it }
          begin
            parent^.next := next_element^.next;
            case compare_paged_schemas(next_element^, parent^) of
              EQ:
                begin
                  next_element^.next := head;
                  head := next_element;
                end;
              LT:
                begin
                  next_element^.next := head;
                  head := next_element;
                end;
              GT:
                begin
                  insert_point := head;
                  comparison := 
                    compare_paged_schemas(next_element^, insert_point^.next^);
                  while comparison = GT do
                    begin
                      insert_point := insert_point^.next;
                      comparison := 
                        compare_paged_schemas(next_element^,
                                              insert_point^.next^);
                    end;
                  next_element^.next := insert_point^.next;
                  insert_point^.next := next_element^.next;
                end { GT } ;
            end { case } ;
          end { LT -> out of order } ;
      end { case } ;
      next_element := parent^.next;
    end { while } ;
end { sort_paged_schemas } ;


function insert_schema_page(var page_spec: paged_schema;  { not altered }
                            var element: paged_schema_ptr { altered }):
  paged_schema_ptr;
  { insert a page corresponding to page_spec ahead of
    element. Initialize the drawing_type, version and page of the
    new element.
    Return with element = the inserted element and also return the ptr }
begin
  new_paged_schema(element);
  with element^ do
    begin
      drawing_type := page_spec.drawing_type;
      version := page_spec.version;
      page := page_spec.page;
    end;
  insert_schema_page := element;
end { insert_schema_page } ;
    

function find_schema_page(var schema: schema_definition;
                          var page_spec: paged_schema { unchanged } ): 
			  paged_schema_ptr;
  { Findthe old schema page corresponding to page_spec and return a
    pointer to it (or NIL, if unfound). }
  var
    current: paged_schema_ptr; { current element }
    done: boolean;             { TRUE iff search has ended }
begin
  current := schema.paged_schemas;  done := FALSE;
  while (current <> NIL) and not done do
    if compare_paged_schemas(current^, page_spec) = LT then 
      current := current^.next
    else done := TRUE;
  find_schema_page := current;
end { find_schema_page } ;


function enter_schema_page(var schema: schema_definition;
                           var page_spec: paged_schema; { unchanged }
                           var parent: paged_schema_ptr;
			   var is_new: boolean): paged_schema_ptr;
  { Find/create the old schema page corresponding to page_spec and return a
    pointer to it. Also return a pointer to its parent (NIL if root).
    Schema_spec is unchanged, but passed by reference for efficiency. 
    Return is_new TRUE iff a new page has been added. }
  var
    current: paged_schema_ptr; { current/found element }
    comparison: compare_type;  { remembers last comparison }
begin
  parent := NIL;  current := schema.paged_schemas;
  is_new := TRUE;
  comparison := LT;
  while (current <> NIL) and (comparison = LT) do
    begin
      comparison := 
        compare_paged_schemas(current^, page_spec);
      if comparison = LT then
        begin
          parent := current;
          current := current^.next;
        end;
    end;
  if comparison = EQ then
    begin
      if debug_24 then writeln(outfile, '   found page');
      enter_schema_page := current;
      is_new := FALSE;
    end
  else
    if parent = NIL then
      begin
        if debug_24 then writeln(outfile, '   adding page');
        enter_schema_page := 
          insert_schema_page(page_spec, schema.paged_schemas);
      end
    else
      begin
        if debug_24 then writeln(outfile, '   adding page');
        enter_schema_page := insert_schema_page(page_spec, parent^.next);
      end;
end { enter_schema_page } ;


(**){--------------- SCHEMA FILE I/O ------------------------------}


procedure parse_schema_file(var schema: schema_definition);
  { Parses the schema file into schema (which is presumed to have
    already been initialized).  The schema file is assumed to have
    been freshly opened as the current parse file }
  label
    90;  { return }

  var
    id_prefix_sy: symbols;     { initialized = SHARP and then used as const }
    string_prefix_sy: symbols; { initialized = DOLLAR and then used as const }
    end_symbols: setofsymbols;         { symbols marking the end of a section }
    final_end_symbols: setofsymbols;   { symbols marking end of data }
    file_type_found: file_types;       { type of file we have }

    end_page_symbols: setofsymbols;    { symbols indicating end of a page }
    legal_at_id_list: setofsymbols;    { symbols legal when we could be
                                         starting a schema id list element   }
    legal_at_local_TM: setofsymbols;   { symbols legal when we could be
                                         starting a local text macro element }
    end_of_TM_element: setofsymbols;   { symbols legal at the end of a
                                         local text macro entry (or the
                                         start of the next one) }
    legal_at_dependencies:
      setofsymbols;                    { symbols legal when we could have
                                         a dependency list }
    end_of_dependency_element:
      setofsymbols;                    { symbols legal after an element of a
                                         dependency list }
    legal_at_result_list:
      setofsymbols;                    { symbols legal when we could be
                                         looking at a result file list }
    legal_at_page_properties:
      setofsymbols;                    { symbols legal when we could be 
                                         looking at a page's property list }
    legal_at_time: setofsymbols;       { symbols legal when we could be
                                         looking at a :<time> }
    context_table: avl_ptr;            { table of contexts sorted by number }


  function parse_schema_version: boolean;
    { parse the version number and return TRUE if it exists and matches the
      current number.   This number is incremented whenever a change is
      made to the syntax or semantics of the schema file or the expansion
      file. }
    var
      val: boolean;         { return value }
  begin
    if sy = CONSTANT then
      begin
        val := (const_val = SCHEMA_SYNTAX_VERSION);
        insymbol;
      end
    else val := FALSE;
    parse_schema_version := val;
    if (debug_23 or debug_24) and not val then 
      writeln(Outfile, 'Schema syntax number outdated');
  end { parse_schema_version } ;
  

  procedure parse_directives(var schema: schema_definition);
    { parse the directive number and separate it into its bit fields }
  begin
    if sy <> CONSTANT then assert(ASSERT_EXPECTED_CONSTANT)
    else
      begin
        schema.bubble_check := bit_and(const_val, BUBBLE_CHECK_MASK) <> 0;
        schema.enable_cardinal_tap :=
	  bit_and(const_val, CARDINAL_TAP_MASK) <> 0;
        insymbol;
      end;
  end { parse_directives } ;
  

  function parse_paged_schemas: paged_schema_ptr;
    { parses the paged schema section of a schema file and returns a
      sorted list }
    label
      10; { cycle }
    var
      ps: paged_schema_ptr;              { head of list of paged schemas }


    function parse_time(var stamp: time_stamp): boolean;
      { parses a time stamp - return TRUE iff found, FALSE and 0 if not }
      var
        sign: -1..1;      { "sign" of partial time-stamp in schema file }
    begin
      if sy = MINUS then
        begin
          sign := -1;
          insymbol;
        end
      else sign := 1;

      if (sy <> CONSTANT) then 
        begin
	  assert(ASSERT_EXPECTED_CONSTANT);
	  parse_time := FALSE;
	end
      else
        begin
	  parse_time := TRUE;
          stamp := const_val * sign;
          insymbol; { eat the high byte }
	end;
    end { parse_time } ;


    function parse_schema_id_list(var head: expandable_id_ptr): boolean;
      { parses and sorts the expandable id list of a schema file.
        Sort is as done by sort_expandable_ids.  Return FALSE if
        error is discovered. }
      label
        90; { return }
      var
        exp_id: expandable_id_ptr;           { current expandable id }
        nid: numbered_token_ptr;             { just parsed numbered id }
    begin
      parse_schema_id_list := TRUE;
      head := NIL;

      if not (sy in legal_at_id_list) then
        begin
          assert(ASSERT_EXPECTED_ID_LIST);
          parse_schema_id_list := FALSE;
          goto 90 { return } ;
        end;

      if sy <> MINUS then goto 90 { return } ;

      insymbol; { eat the - }
      while not (sy in legal_at_local_TM) do
        begin
          if sy <> id_prefix_sy then
            begin
              assert(ASSERT_EXPECTED_ID_NUMBER);
              skip(legal_at_local_TM);
	      parse_schema_id_list := FALSE;
            end
          else
            begin
              nid := parse_numbered_token(schema.id_dictionary);
              if nid = NIL then 
                begin
                  parse_schema_id_list := FALSE;
                  skip(legal_at_local_TM);
                end
              else 
                begin
                  new_expandable_id(exp_id);
                  exp_id^.name := nid;
                  exp_id^.next := head;
                  head := exp_id;
                end;
            end;
        end { while } ;
      sort_expandable_ids(head);
    90: { return }
    end { parse_schema_id_list } ;


    function parse_local_TMs(var head: text_macro_ptr): boolean;
      { parses the local text macros section of a paged schema.
        Return FALSE if error is detected. }
      label
        90; { return }
      var
        nid: numbered_token_ptr;            { just parsed id number }
        str: numbered_token_ptr;            { just parsed string number }
    begin
      parse_local_TMs := TRUE;  head := NIL;

      if not (sy in legal_at_local_TM) then
        begin
          assert(ASSERT_EXPECTED_LOCAL_TM);
          parse_local_TMs := FALSE;
          goto 90 { return } ;
        end;

      if sy <> AMPERSAND then goto 90 { return } ;

      insymbol; { eat the & }
      while sy = id_prefix_sy do
        begin 
          nid := parse_numbered_token(schema.id_dictionary);
          if nid = NIL then skip(end_of_TM_element)
          else if sy <> string_prefix_sy then
            begin
              assert(ASSERT_EXPECTED_NUMBERED_STRING);
              skip(end_of_TM_element);  parse_local_TMs := FALSE;
            end
          else
            begin { nid ok }
              str := parse_numbered_token(schema.string_dictionary);
              if str = NIL then 
                begin
                  skip(end_of_TM_element);  parse_local_TMs := FALSE;
                end
              else
                begin
                  new_text_macro(head);
                  with head^ do 
                    begin
                      name := nid;
                      text := str;
                      reserved := FALSE;
                    end;
                end;
            end { nid ok } ;
        end { while } ;
	
      sort_text_macros(head);

    90: { return }
    end { parse_local_TMs } ;


    function parse_dependencies(var head: dependency_list_ptr): boolean;
      { parses the file dependencies section of a schema file.  Return
        FALSE if error discovered. }
      label
        90; { return }
      var
        str: numbered_token_ptr;           { just parsed string number }
    begin
      parse_dependencies := FALSE;  head := NIL;

      if not (sy in legal_at_dependencies) then
        begin
          assert(ASSERT_EXPECTED_DEPENDENCIES);
          goto 90 { return } ;
        end;

      if sy = LPAREN then
        begin
          insymbol; { eat the ( }
          while sy = string_prefix_sy do
            begin
              str := parse_numbered_token(schema.string_dictionary);
              if str = NIL then goto 90 { return };
	      new_dependency_list(head);
              head^.file_name := str;
	      if not parse_time(head^.last_modified_time) then
	        goto 90 { return };
            end;
	end;
      parse_dependencies := TRUE;
    90:
    end { parse_dependencies } ;


    function parse_properties(var head: numbered_token_list_ptr): boolean;
      { parses the properties used section of a page.  Return
        FALSE if error discovered. }
      label
        90; { return }
      var
        id_token: numbered_token_ptr;    { parsed property name }
    begin
      parse_properties := TRUE;  head := NIL;
      if not (sy in legal_at_page_properties) then
        begin
          assert(158 { expected ) });
          skip(end_page_symbols);
          goto 90 { return } ;
       end;

      if sy <> PLUS then goto 90 { return } ;

      insymbol; { eat the + }
      while sy = id_prefix_sy do
        begin
          id_token := parse_numbered_token(schema.id_dictionary);
          if id_token = NIL then
            begin
              skip(end_page_symbols + [id_prefix_sy]);
              parse_properties := FALSE;
            end
          else
            begin
              new_numbered_token_list(head);
              head^.token := id_token;
            end;
        end;
    90:
    end { parse_properties } ;


    function parse_compiled_contexts(var head: compiled_context_list_ptr):
      boolean;
      { parse the compiled context list specification.  Use context_table
        (declared in parse_schema_file) to locate the context_definition
        associated with each context number.  Return FALSE if error
        has occurred. The list is not sorted. }
      label 
        90; { return }
      var
        dummy_context: context_definition_ptr; { for supplying key to
                                                 avl_find }
        object: avl_object_ptr;                { type required for avl_find }
        found: avl_ptr;                        { context found in table }
    begin
      head := NIL;
      parse_compiled_contexts := FALSE;
      if not (sy in legal_at_result_list) then
        begin
          assert(158 { expected ) });
          goto 90 { return } ;
        end;

      new_context_definition(dummy_context);
      object.context_number := dummy_context;
{     object.tag := AVL_CONTEXT_NUMBER;                       }(*AVL*)
      while sy = RPAREN do
        begin
          insymbol; { eat the ) }
          if sy <> CONSTANT then
            begin
              assert(ASSERT_EXPECTED_CONSTANT);
              release_context_definition(dummy_context);
              goto 90 { return } ;
            end;

          dummy_context^.number := const_val;
          found := avl_find(object, context_table, AVL_CONTEXT_NUMBER);
          if found = NIL then
            begin
              assert(0 { should never happen });
              writeln(cmplog, ' Unable to find context number in table!!');
              release_context_definition(dummy_context);
              goto 90 { return } ;
            end;

          insymbol; { eat context_number }

          new_compiled_context_list(head);
          head^.context := found^.object.context_number;
          if sy = IDENT then
	    if id.name = D_identifier then
	      begin
	        head^.dirty := TRUE;
		insymbol;
	      end;
        end;
      parse_compiled_contexts := TRUE;
      release_context_definition(dummy_context);
    90: { for returns }
    end { parse_compiled_contexts } ;


  begin { parse_paged_schemas }
    ps := NIL;
    parse_paged_schemas := NIL;

    while not (sy in end_symbols) do
      begin
        if sy = EQUAL then insymbol
        else
          begin
            assert(178 { expected = });
            skip(end_page_symbols);
            goto 10 { cycle } ;
          end;
        if sy <> id_prefix_sy then
          begin
            assert(ASSERT_EXPECTED_NUMBERED_ID);
            skip(end_page_symbols);
            goto 10 { cycle } ;
          end;
        new_paged_schema(ps);
        with ps^ do 
          begin
            drawing_type := parse_numbered_token(schema.id_dictionary);
            if drawing_type = NIL then
              begin
                release_paged_schema(ps);
                skip(end_page_symbols);
                goto 10 { cycle } ;
              end;
            if sy <> PERIOD then
              begin
                assert(ASSERT_EXPECTED_PERIOD);
                release_paged_schema(ps);
                skip(end_page_symbols);
                goto 10 { cycle } ;
              end;
            insymbol; { eat the . }
            if sy <> CONSTANT then
              begin
                assert(ASSERT_EXPECTED_CONSTANT);
                release_paged_schema(ps);
                skip(end_page_symbols);
                goto 10 { cycle } ;
              end;
            version := const_val;  insymbol; { eat the version number }
            if sy <> PERIOD then
              begin
                assert(ASSERT_EXPECTED_PERIOD);
                release_paged_schema(ps);
                skip(end_page_symbols);
                goto 10 { cycle } ;
              end;
            insymbol; { eat the . }
            if sy <> CONSTANT then
              begin
                assert(ASSERT_EXPECTED_CONSTANT);
                release_paged_schema(ps);
                skip(end_page_symbols);
                goto 10 { cycle } ;
              end;
            page := const_val;  insymbol; { eat the page number }

            if not parse_time(last_modified_time) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;

	    if sy = IDENT then
	      begin
		if id.name <> E_identifier then
		  begin
		    assert(190 { misc. parse error });
		    writeln(CmpLog, ' Expected E');
		    skip(end_page_symbols);
		    release_paged_schema(ps);
		    goto 10 { cycle } ;
		  end;
		{ has_expr := TRUE; } { obsolete }
		insymbol;  { eat the E }
	      end
	    { else has_expr := FALSE } { obsolete } ;

            if not parse_compiled_contexts(compiled_contexts) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;

            if not parse_schema_id_list(expandable_ids) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;

            if not parse_local_TMs(local_text_macros) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;

            if not parse_dependencies(dependencies) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;

            if not parse_properties(properties) then
              begin
                skip(end_page_symbols);
                release_paged_schema(ps);
                goto 10 { cycle } ;
              end;
          end { with ps^ } ;
      10:
      end { while } ;
    if sy = SEMI then insymbol
    else
      begin
        assert(116 { unexpected end of data });
      end;
    sort_paged_schemas(ps);
    parse_paged_schemas := ps;
  end { parse_paged_schemas } ;


  function parse_used_global_TMs: text_macro_ptr;
    { parses the used global text macros section of a schema file
      (these are not sorted and do not need to be). }
    var
      head: text_macro_ptr;              { head of list for return }
      end_of_TM_element: setofsymbols; { symbols indicating 
                                           start of next item }
      nid: numbered_token_ptr;           { just parsed id number }
      str: numbered_token_ptr;           { just parsed string number }


    function parse_reserved: boolean;
      { parses the reserved spec within a used_global_tm spec and
        returns value for the "reserved" field of that record. }
    begin
      parse_reserved := FALSE;
      if not (sy in ([COLON,id_prefix_sy] + end_symbols)) then
        begin
          assert(ASSERT_EXPECTED_RESERVED);
          skip(end_of_TM_element);
        end
      else
        begin
          if sy = COLON then
            begin
              insymbol; { eat the : }
              if sy <> IDENT then 
                begin
                  assert(ASSERT_EXPECTED_R);
                  skip(end_of_TM_element);
                end
              else if id.name <> R_identifier then
                begin
                  assert(ASSERT_EXPECTED_R);
                  skip(end_of_TM_element);
                end
              else
                begin
                  insymbol; { eat the R }
                  parse_reserved := TRUE;
                end;
            end;
        end;
    end { parse_reserved } ;


  begin { parse_used_global_TMs }
    head := NIL;
    end_of_TM_element := end_symbols + [id_prefix_sy];

    while not (sy in end_symbols) do
      if sy <> id_prefix_sy then
        begin
          assert(ASSERT_EXPECTED_NUMBERED_ID);
          skip(end_of_TM_element);
        end
      else
        begin { parse a TM spec }
          nid := parse_numbered_token(schema.id_dictionary);
          if nid = NIL then skip(end_of_TM_element)
          else if sy <> string_prefix_sy then
            begin
              assert(ASSERT_EXPECTED_NUMBERED_STRING);
              skip(end_of_TM_element);
            end
          else
            begin { nid ok }
              str := parse_numbered_token(schema.string_dictionary);
              if str = NIL then skip(end_of_TM_element)
              else
                begin
                  new_text_macro(head);
                  with head^ do 
                    begin
                      name := nid;
                      text := str;
                      reserved := parse_reserved;
                    end;
                end;
            end { nid ok } ;
        end { parse a TM spec } ;
    if sy = SEMI then insymbol
    else
      begin
        assert(116 { unexpected end of data });
      end;

    parse_used_global_TMs := head;
  end { parse_used_global_TMs } ;


  function parse_used_properties: property_attribute_ptr;
    { parses the used properties section of a schema file }
    label
      10;  { cycle }
    var
      head: property_attribute_ptr;      { head of list for return }
      nid: numbered_token_ptr;           { just parsed id number }


    procedure unpack_attributes(int: longint; var attributes: name_type_set);
      { unpack the integer attribute representation into the set }
    begin
      attributes := [];
      if bit_and(int, ET_PERMIT_BODY) <> 0 then 
        attributes := attributes + [PERMIT_BODY];
      if bit_and(int, ET_PERMIT_PIN) <> 0 then 
        attributes := attributes + [PERMIT_PIN];
      if bit_and(int, ET_PERMIT_SIGNAL) <> 0 then 
        attributes := attributes + [PERMIT_SIGNAL];
      if bit_and(int, ET_INHERIT_BODY) <> 0 then 
        attributes := attributes + [INHERIT_BODY];
      if bit_and(int, ET_INHERIT_PIN) <> 0 then 
        attributes := attributes + [INHERIT_PIN];
      if bit_and(int, ET_INHERIT_SIGNAL) <> 0 then 
        attributes := attributes + [INHERIT_SIGNAL];
      if bit_and(int, ET_FILTER) <> 0 then 
        attributes := attributes + [DONT_OUTPUT];
      if bit_and(int, ET_CONTROL) <> 0 then 
        attributes := attributes + [IS_ET_CONTROL];
      if bit_and(int, ET_PARAMETER) <> 0 then 
        attributes := attributes + [IS_PARAMETER];
      if bit_and(int, ET_INT_PARAMETER) <> 0 then 
        attributes := attributes + [IS_INT_PARAMETER];
    end { unpack_attributes } ;


  begin { parse_used_properties }
    head := NIL;

    while not (sy in end_symbols) do
      begin
        if sy <> id_prefix_sy then
          begin
            assert(ASSERT_EXPECTED_NUMBERED_ID);
            skip(end_symbols + [id_prefix_sy]);
            goto 10 { cycle } ;
          end;

        nid := parse_numbered_token(schema.id_dictionary);
        if nid = NIL then
          begin
            skip(end_symbols + [id_prefix_sy]);
            goto 10 { cycle } ;
          end;

        if sy <> EQUAL then
          begin
            assert(178 { expected = });
            skip(end_symbols + [id_prefix_sy]);
            goto 10 { cycle } ;
          end;
        insymbol; { eat = }

        if sy <> CONSTANT then
          begin
            assert(ASSERT_EXPECTED_CONSTANT);
            skip(end_symbols + [id_prefix_sy]);
            goto 10 { cycle } ;
          end;
        
        new_property_attribute(head);
        head^.property := nid;
        unpack_attributes(const_val, head^.attributes);
      10: { for cycle }
      end;
    if sy = SEMI then insymbol
    else
      begin
        assert(116 { unexpected end of data });
      end;
    parse_used_properties := head;
  end { parse_used_property_attributes } ;


  function parse_contexts: context_definition_ptr;
    { parses CONTEXTS section of schema file  The contexts are then
      sorted. }
    var
      head: context_definition_ptr;          { head of list for return }
      end_of_context_symbols: setofsymbols;  { symbols to find for recovery }
      legal_at_parameters: setofsymbols;     { symbols legal when expecting
                                               parameter specs or the 
                                               following item. }
      context_number: natural_number;        { the context number }


    function parse_parameters: parameter_ptr;
      { parses  and sorts the parameter list associtated with a context }
      var
        head: parameter_ptr;        { head of list for return }
        nid: numbered_token_ptr;    { just parsed id number }
    begin
      head := NIL;
      if (sy = CONSTANT) then
        begin
          new_parameter(head);
          head^.name := enter_numbered_id(SIZE_prop_name, 
                                          schema.id_dictionary);
          head^.text := enter_numbered_string(number_to_string(const_val),
                                              schema.string_dictionary);
          if (head^.name = NIL) or (head^.text = NIL) then
            release_parameter(head);
          insymbol;
        end;
      while not (sy in end_of_context_symbols) do
        begin
          if not (sy = id_prefix_sy) then
            begin
              assert(126 { expected parameters or next thing });
              skip(legal_at_parameters);
            end
          else
            begin
              nid := parse_numbered_token(schema.id_dictionary);
              if nid = NIL then skip(legal_at_parameters)
              else if sy = string_prefix_sy then
                begin
                  new_parameter(head);
                  head^.name := nid;
                  head^.text := 
                    parse_numbered_token(schema.string_dictionary);
                  if head^.text = NIL then 
                    begin
                      release_parameter(head);
                    end;
                end
              else
                begin
                  assert(123 { expected a numbered string });
                  skip(legal_at_parameters);
                end;
            end { ok to start } ;
        end { while } ;
      sort_parameters(head);
      parse_parameters := head;
    end { parse_parameters } ;


  begin { parse_contexts }
    head := NIL;
    end_of_context_symbols := end_symbols + [PERCENT];
    legal_at_parameters := end_of_context_symbols + [id_prefix_sy];

    while not (sy in end_symbols) do
      if sy <> PERCENT then
        begin
          assert(ASSERT_EXPECTED_PERCENT);
          skip(end_of_context_symbols);
        end
      else
        begin
          insymbol; { eat the % }
          if sy <> CONSTANT then
            begin
              assert(ASSERT_EXPECTED_CONSTANT);
              skip(end_of_context_symbols);
            end
          else
            begin { context number there }
              context_number := const_val;
              insymbol;
                 
              if sy <> CONSTANT then
                begin
                  assert(ASSERT_EXPECTED_CONSTANT);
                  skip(end_of_context_symbols);
                end
              else
                begin
                  new_context_definition(head);
                  head^.number := context_number;
                  head^.version := const_val;
                  insymbol; { eat version number }
                  head^.parameters := parse_parameters;
                end;
            end { context name there }
        end;

    if sy = SEMI then insymbol
    else
      begin
        assert(116 { unexpected end of data });
      end;

    sort_contexts(head);    { sorts list by parameters and deletes illegal
                              entries if found }
    parse_contexts := head;
  end { parse_contexts } ;


  function sort_contexts_by_number(head: context_definition_ptr): avl_ptr;
    { return the root of an avl tree that orders the contexts by number }
    var
      tree: avl_ptr;            { root of tree for return }
      current: avl_object_ptr;  { current context }
      dummy: avl_ptr;           { current table entry }
  begin
    tree := NIL;
    current.context_number := head;
    while current.context_number <> NIL do
      begin
        dummy := avl_insert(current, tree, AVL_CONTEXT_NUMBER);
        current.context_number := current.context_number^.next;
      end;
    sort_contexts_by_number := tree;
  end { sort_contexts_by_number } ;


begin { parse_schema_file }

  { tolerate an empty file }
  
  if sy <> ENDOFDATASY then
    begin

      { parse the schema file }

      file_type_found := get_file_type;
      if file_type_found <> SCHEMA_FILE then
        begin
          { !!! horrible fatal error condition }
          error(FATAL_SCHEMA_FILE_WRONG_TYPE);
          goto 90 { return }
        end;

      if parse_schema_version then
        begin
          { initialize local vars }

          context_table := NIL;
          id_prefix_sy := SHARP;
          string_prefix_sy := DOLLAR;
          final_end_symbols := [ENDSY,ENDOFDATASY];
          end_symbols := [SEMI,ENDSY,ENDOFDATASY];
          end_page_symbols := end_symbols + [EQUAL];
          legal_at_page_properties := end_page_symbols + [PLUS];
          legal_at_dependencies := legal_at_page_properties + [LPAREN];
          end_of_dependency_element := end_page_symbols + [string_prefix_sy];
          legal_at_local_TM := legal_at_dependencies + [AMPERSAND];
          end_of_TM_element := legal_at_dependencies + [id_prefix_sy]; 
          legal_at_id_list := legal_at_local_TM + [MINUS];
          legal_at_result_list := legal_at_id_list + [RPAREN];
          legal_at_time := legal_at_result_list + [COLON,IDENT];
          
          parse_directives(schema);

          parse_numbered_dictionary(schema.string_dictionary);
          if not (sy in final_end_symbols) then
            parse_numbered_dictionary(schema.id_dictionary);

          if not (sy in final_end_symbols) then
            schema.contexts := parse_contexts;
          context_table := sort_contexts_by_number(schema.contexts);
          if not (sy in final_end_symbols) then
            schema.paged_schemas := parse_paged_schemas;
          if not (sy in final_end_symbols) then
            schema.used_global_TMs := parse_used_global_TMs;
          if not (sy in final_end_symbols) then
            schema.used_properties := parse_used_properties;

          release_entire_avl_tree(context_table);
        end;
    end;
90:
end { parse_schema_file } ;


(**)


procedure output_parameters(var f: pipe;
                            parms: parameter_ptr);
  { outputs the parameter body properties for schema and expansion files }
  const
    PARMS_PER_LINE = 5;
  var
    current: parameter_ptr;     { current element }
    count: 0..PARMS_PER_LINE;   { number of parms since CRLF }
begin
  current := parms;  count := 0;
  if current <> NIL then
    if (current^.name^.identifier_p = SIZE_prop_name) then
      begin
        pipe_dump_char(f, ' ');
        pipe_dump_integer(f, 
          string_to_natural_number(current^.text^.string_p));
        current := current^.next;  count := 1;
      end;
  while current <> NIL do with current^ do
    begin
      if count = PARMS_PER_LINE then
        begin
          pipe_dump_CRLF(f);
          count := 1;
        end
      else count := count + 1;
      output_numbered_token(f, name^, IDENTIFIER_NUMBER);
      output_numbered_token(f, text^, STRING_NUMBER);
      current := next;
    end;
  if parms <> NIL then pipe_dump_CRLF(f);
end { output_parameters } ;


procedure output_schema_file(var f: pipe; var schema: schema_definition);
  { output a schema file.  File f must be open. Schema is output for
    final output (which causes output_numbers to be set for numbered
    tokens, effectively renumbering them).  The _continue output routines
    are used to stay within the proper line width. }


  procedure output_current_schema_directives(var f: pipe);
    { output current pertinent directives to the schema file }
    var
      directive_code: natural_number;
  begin
    directive_code := 0;
    if bubble_check then
      directive_code := directive_code + BUBBLE_CHECK_MASK;
    if enable_cardinal_tap then 
      directive_code := directive_code + CARDINAL_TAP_MASK;
    pipe_dump_integer(f, directive_code);
  end { output_current_schema_directives } ;


  procedure output_paged_schemas(var f: pipe; ps: paged_schema_ptr);
    { output the paged schema list to a schema file }
    var
      current: paged_schema_ptr;  { current element }


    procedure output_compiled_contexts(var f: pipe;
                                       contexts: compiled_context_list_ptr);
      { outputs the local text macros defined in this page }
      const
        CONTEXTS_PER_LINE = 15;
      var
        current: compiled_context_list_ptr;  { current element }
        count: 0..CONTEXTS_PER_LINE;         { number of contexts since CRLF }
    begin
      if contexts <> NIL then 
        begin
          current := contexts;  count := 0;
          while current <> NIL do with current^ do
            begin
              if count = CONTEXTS_PER_LINE then
                begin
                  pipe_dump_CRLF(f);
                  count := 1;
                end
              else count := count + 1;
              pipe_dump_char(f, ')');
              pipe_dump_integer(f, context^.number);
	      if (dirty or dirty_for_pass_2) then pipe_dump_char(f, 'D');
              current := next;
	    end;
          pipe_dump_CRLF(f);
        end;
    end { output_compiled_contexts } ;


    procedure output_expandable_ids(var f: pipe; exids: expandable_id_ptr);
      { outputs the expandable ids  }
      const
        EXP_IDS_PER_LINE = 10;
      var
        current: expandable_id_ptr;  { current element }
        count: 0..EXP_IDS_PER_LINE;  { number of expanded ids since CRLF }
    begin
      if exids <> NIL then
        begin
          pipe_dump_char(f, '-');
          current := exids;  count := 0;
          while current <> NIL do with current^ do
            begin
              if count = EXP_IDS_PER_LINE then
                begin
                  pipe_dump_CRLF(f);
                  count := 1;
                end
              else count := count + 1;
              output_numbered_token(f, name^, IDENTIFIER_NUMBER);
              current := next;
            end;
          pipe_dump_CRLF(f);
        end;
    end { output_expandable_ids } ;


    procedure output_local_text_macros(var f: pipe;
                                       tms: text_macro_ptr);
      { outputs the local text macros defined in this page }
      const
        LOCAL_TMS_PER_LINE = 5;
      var
        current: text_macro_ptr;        { current element }
        count: 0..LOCAL_TMS_PER_LINE;   { number of TMs since CRLF }
    begin
      if tms <> NIL then 
        begin
          pipe_dump_char(f, '&');
          current := tms;  count := 0;
          while current <> NIL do with current^ do
            begin
              if count = LOCAL_TMS_PER_LINE then
                begin
                  pipe_dump_CRLF(f);
                  count := 1;
                end
              else count := count + 1;
              output_numbered_token(f, name^, IDENTIFIER_NUMBER);
              output_numbered_token(f, text^, STRING_NUMBER);
            { if reserve then pipe_dump_alpha(f, ':R              '); }
              current := next;
            end;
          pipe_dump_CRLF(f);
        end;
    end { output_local_text_macros } ;


    procedure output_dependencies(var f: pipe;
                                  deps: dependency_list_ptr);
      { outputs the local text macros defined in this page }
      const
        DEPENDENCIES_PER_LINE = 12;
      var
        current: dependency_list_ptr;      { current element }
        count: 0..DEPENDENCIES_PER_LINE;   { number printed since CRLF }
    begin
      if deps <> NIL then 
        begin
          pipe_dump_char(f, '(');
          current := deps;  count := 0;
          while current <> NIL do with current^ do
            begin
              if count = DEPENDENCIES_PER_LINE then
                begin
                  pipe_dump_CRLF(f);
                  count := 1;
                end
              else count := count + 1;
              output_numbered_token(f, file_name^, STRING_NUMBER);
              pipe_dump_char(f, ' ');
	      pipe_dump_integer(f, last_modified_time);
              current := next;
            end;
          pipe_dump_CRLF(f);
        end;
    end { output_dependencies } ;


    procedure output_properties(var f: pipe;
                                props: numbered_token_list_ptr);
      { outputs the local text macros defined in this page }
      const
        PROPERTIES_PER_LINE = 12;
      var
        current: numbered_token_list_ptr; { current element }
        count: 0..PROPERTIES_PER_LINE;    { number of properties since CRLF }
    begin
      if props <> NIL then 
        begin
          pipe_dump_char(f, '+');
          current := props;  count := 0;
          while current <> NIL do with current^ do
            begin
              if count = PROPERTIES_PER_LINE then
                begin
                  pipe_dump_CRLF(f);
                  count := 1;
                end
              else count := count + 1;
              output_numbered_token(f, token^, IDENTIFIER_NUMBER);
              current := next;
            end;
          pipe_dump_CRLF(f);
        end;
    end { output_properties } ;


  begin { output_paged_schemas }
    current := ps;
    while current <> NIL do with current^ do
      begin
        pipe_dump_char(f, '=');
        output_numbered_token(f, drawing_type^, IDENTIFIER_NUMBER);
        pipe_dump_char(f, '.');
        pipe_dump_integer(f, version);
        pipe_dump_char(f, '.');
        pipe_dump_integer(f, page);
        pipe_dump_char(f, ' ');
        pipe_dump_integer(f, last_modified_time);
	(* obsolete
	if has_expr then 
	  begin
	    pipe_dump_char(f, ' ');
	    pipe_dump_char(f, 'E');
	  end;
	*)
        pipe_dump_CRLF(f);
        output_compiled_contexts(f, compiled_contexts);
        output_expandable_ids(f, expandable_ids);
        output_local_text_macros(f, local_text_macros);
        output_dependencies(f, dependencies);
        output_properties(f, properties);
        current := next;
      end;
    pipe_dump_char(f, ';');
    pipe_dump_CRLF(f);
  end { output_paged_schemas } ;


  procedure output_used_global_TMs(var f: pipe;
                                     tms: text_macro_ptr);
    { outputs the used global text macro definitions  }
    const
      GLOBAL_TMS_PER_LINE = 5;
    var
      current: text_macro_ptr;         { current element }
      count: 0..GLOBAL_TMS_PER_LINE;   { number of TMs since CRLF }
  begin
    current := tms;  count := 0;
    while current <> NIL do with current^ do
      begin
        if count = GLOBAL_TMS_PER_LINE then
          begin
            pipe_dump_CRLF(f);
            count := 1;
          end
        else count := count + 1;
        output_numbered_token(f, name^, IDENTIFIER_NUMBER);
        output_numbered_token(f, text^, STRING_NUMBER);
        if reserved then pipe_dump_alpha(f, ':R              ');
        current := next;
      end;
    pipe_dump_char(f, ';');
    pipe_dump_CRLF(f);
  end { output_used_global_TMs } ;


  procedure output_contexts(var f: pipe; cont: context_definition_ptr);
    { Renumber and output the context list -- note that any other
      references to the context by its number are now obsolete.  The
      result file lists refer to the context by pointer, so they remain ok. }
    var 
      current: context_definition_ptr; { current element }
      new_number: natural_number;
  begin
    current := cont;  new_number := 0;
    while current <> NIL do with current^ do 
      begin
        new_number := new_number + 1;
        number := new_number;
        pipe_dump_char(f, '%');
        pipe_dump_integer(f, new_number);
        pipe_dump_char(f, ' ');
        pipe_dump_integer(f, version);
        pipe_dump_char(f, ' ');
        output_parameters(f, parameters);
        current := next;
      end;
      pipe_dump_char(f, ';');
      pipe_dump_CRLF(f);
  end { output_contexts } ;
    

  procedure output_used_properties(var f: pipe; 
                                   props: property_attribute_ptr);
    { output the attributes of properties used in the drawing pages }
    const
      PROPERTIES_PER_LINE = 5;
    var
      current: property_attribute_ptr; { current element }
      count: 0..PROPERTIES_PER_LINE;   { number of properties since CRLF }
  begin
    current := props;  count := 0;
    while current <> NIL do with current^ do
      begin
        if count = PROPERTIES_PER_LINE then
          begin
            pipe_dump_CRLF(f);
            count := 1;
          end
        else count := count + 1;
        output_numbered_token(f, current^.property^, IDENTIFIER_NUMBER);
        pipe_dump_char(f, '=');
        pipe_dump_integer(f,
          et_property_attributes(current^.property^.identifier_p^.kind));
        current := next;
      end;
    pipe_dump_char(f, ';');
    pipe_dump_CRLF(f);
  end { output_used_properties } ;


begin { output_schema_file }
  init_output_continue;
  pipe_dump_alpha(f, 'FILE_TYPE=SCHEMA');
  pipe_dump_char(f, ';');  
  pipe_dump_integer(f, SCHEMA_SYNTAX_VERSION);
  pipe_dump_char(f, ' ');
  output_current_schema_directives(f);
  pipe_dump_CRLF(f);
  output_numbered_dictionary(f, schema.string_dictionary, TRUE);
  output_numbered_dictionary(f, schema.id_dictionary, TRUE);
  output_contexts(f, schema.contexts);
  output_paged_schemas(f, schema.paged_schemas);
  output_used_global_TMs(f, schema.used_global_TMs);
  output_used_properties(f, schema.used_properties);
  pipe_dump_alpha(f, 'END.            ');
  pipe_dump_CRLF(f);
end { output_schema_file } ;


(**){--------------- SCHEMA file access ---------------------------}


function read_schema_file(macro_name: xtring; macro: macro_module_ptr;
                          var schema: schema_definition): boolean;
  { Reads the schema file associated with the macro.  schema fields
    are released before reading the file. Current file is set to
  UNKNOWN_FILE when done, but CmpSchema is left open to preserve
  its advisory lock. }
  var
    extension: xtring;
begin { read_schema_file }
  if debug_20 or debug_23 or debug_24 then 
    begin
      write(outfile, ' --- read_schema_file for ');
      writestring(outfile, macro_name);
      writeln(outfile, ' ---');
    end;

  release_all_schema_fields(schema);
  schema.file_name := nullstring;
  extension := er_extension(macro);
  if extension <> NIL then
    copy_string(er_filename(macro, ord(SCHEMA_FILE), 0, NIL), 
                schema.file_name);

  current_file_name := schema.file_name;

  allowed_key_words := schema_keysys;  parse_SCALDconstants := FALSE;
  upper_case_strings := FALSE;

  schema.file_accessible := FALSE;

  if schema.file_name <> nullstring then
    if open_a_file(current_file_name, CMPSCHEM_FILE) then
      begin
        schema.file_accessible := TRUE;
        parse_schema_file(schema);
        current_file := UNKNOWN_FILE;  current_file_name := nullstring;

        { don't close it -- it is locked to this program and will later
          be written without breaking the lock }
      end
    else 
      begin
        error(232 { unable to access schema });
        error_dump_macro_name(macro_name, 0);
        error_dump_file_name(current_file_name);
        current_file := UNKNOWN_FILE;  current_file_name := nullstring;
      end;

  allowed_key_words := [];  parse_SCALDconstants := TRUE;
  upper_case_strings := TRUE;
  schema.changed := FALSE;
  read_schema_file := schema.file_accessible or (extension = NIL);
  schema_file_open := schema.file_accessible;

  if debug_20 or debug_23 or debug_24 then 
    writeln(outfile, ' --- end read_schema_file ', 
                     schema.file_accessible or (extension = NIL), 
		     ' ---');
end { read_schema_file } ;


procedure write_schema_file(var schema: schema_definition);
 { update used global TMs and writes the schema file for the given schema
   if necessary. }
begin
  enter_critical_section;
  if schema.file_accessible then
    begin
      if debug_31 then kill_self;
      if not schema.changed then 
        begin
	  if close_parse_file(CMPSCHEM_FILE) then ;
	end
      else
        begin
          log_used_global_TMs_in_schema(schema);
          if rewrite_locked_file(CMPSCHEM_FILE, schema.file_name, 
                                 CMPSCHEM_FILE_NAME) then
            begin
              pipe_from_open_file(CmpSchem_pipe, SCHEMA_PIPE, 
                                  schema.file_name);
              output_schema_file(CmpSchem_pipe, schema);
              if debug_24 then
                begin
                  write(outfile, ' Dictionaries for ');
                  dump_string(outfile, schema.file_name);
                  writeln(outfile);  writeln(outfile);
                  dump_numbered_dictionary(outfile, schema.string_dictionary);
                  dump_numbered_dictionary(outfile, schema.id_dictionary);
                end;
              pipe_close(Cmpschem_pipe);
            end
          else error(232 { unable to write a schema file });
        end;

#if UNIX
{
      //no explicit unlocking of files, to be done by data services as a part
      //of transaction management.
      if efs_unlock(0, nullstring, schema.file_name, NIL) then ;
}
#endif
    end;

  schema.file_accessible := FALSE;
  schema_file_open := FALSE;
  release_string(schema.file_name);
  exit_critical_section;
end { write_schema_file } ;
