(**){--------------- NEW & RELEASE for NUMBERED_TOKENs --------}

{---------------------------------------------------------------}
{ All new routines for numbered tokens obtain a new object and  }
{ insert it to the head of the list.  All release routines      }
{ (for these) delete the object from its list.                  }
{ All release_all_<object>s... routines relase an entire list   }
{ and return NIL.                                               }
{ Examples:                                                     }
{   new_object(head_of_list);                                   }
{   new_object(parent_of_new_one^.next);                        }
{   release_object(head_of_list);                               }
{   release_all_lists(head_of_list);                        }
{   release_object(parent_of_one_to_delete^.next);              }
{---------------------------------------------------------------}

procedure init_numbered_token(var tok: numbered_token;
			     token_kind: numbered_token_type);
  { Initializes a numbered_token record to UNKNOWN_TOKEN_NUMBER, 
    token_kind, NOT used and NIL. This is automatically done by 
    new_numbered_token, but this routine should be used when needing 
    to initialize a local (non-dynamic) record of this type. 
    NOTE: sets tok.next = NIL }
begin
  with tok do 
    begin
      number := UNKNOWN_TOKEN_NUMBER;
      output_number := UNKNOWN_TOKEN_NUMBER;
      next := NIL;
      case token_kind of
	STRING_NUMBER: string_p := NIL;
	IDENTIFIER_NUMBER: identifier_p := NIL;
	OTHERWISE 
	  begin
	    assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	    string_p := NIL;
	  end;
      end;
    end;
end { init_numbered_token } ;


procedure new_numbered_token(var tok: numbered_token_ptr;
			     token_kind: numbered_token_type);
  { Gets a new numbered_token (from freelist, if possible) and
    initializes (per init_numbered_token) and inserts it at the
    head of the list. }
  var
    nt: numbered_token_ptr; { new one }
begin
  if free_numbered_tokens <> NIL then
    begin  
      nt := free_numbered_tokens;  
      free_numbered_tokens := nt^.next;  
    end
  else
    begin
      new(nt);  
      increment_heap_count(HEAP_NUMBERED_TOKEN, 
        2*POINTER_SIZE+2*INT_SIZE+BOOL_SIZE);
    end;
  init_numbered_token(nt^, token_kind);
  nt^.next := tok;  tok := nt;
end { new_numbered_token } ;


procedure release_numbered_token(var tok: numbered_token_ptr;
				 token_kind: numbered_token_type);
  { Releases a numbered_token for re-use. Sets tok to
    the former tok^.next (so can be used for list deletion). }
  var
    nt: numbered_token_ptr; { saves old tok for release }
begin
  if tok <> NIL then
    begin
      nt := tok;
      tok := tok^.next;
      nt^.next := free_numbered_tokens;
      free_numbered_tokens := nt;
    end;
end { release_numbered_token } ;


procedure new_numbered_token_list(
  var toklist: numbered_token_list_ptr);
  { Gets a new numbered_token (from freelist, if possible) and
    initializes it to NIL values and inserts it to head of list. }
  var
    ntl: numbered_token_list_ptr; { new one }
begin
  if free_numbered_token_lists <> NIL then
    begin  
      ntl := free_numbered_token_lists;  
      free_numbered_token_lists := ntl^.next;  
    end
  else
    begin
      new(ntl);  
      increment_heap_count(HEAP_NUMBERED_TOKEN_LIST, 2*POINTER_SIZE);
    end;
  ntl^.token := NIL;
  ntl^.next := toklist;  toklist := ntl;
end { new_numbered_token_list } ;


procedure release_all_numbered_token_lists(
  var toklist: numbered_token_list_ptr);
  { releases a numbered_token_list list by insertion into freelist. 
    Returns NIL. Has no effect on numbered tokens referenced. }
  var
    last: numbered_token_list_ptr; { last element of list }
begin
  if toklist <> NIL then
    begin
      last := toklist;
      while last^.next <> NIL do last := last^.next;
      last^.next := free_numbered_token_lists;
      free_numbered_token_lists := toklist;
      toklist := NIL;
    end;
end { release_all_numbered_token_lists } ;


procedure release_numbered_token_list(
  var toklist: numbered_token_list_ptr);
  { deletes and releases a numbered_token_list object. 
    Has no effect on numbered tokens referenced. }
  var
    ntl: numbered_token_list_ptr; { saves old toklist for release }
begin
  if toklist <> NIL then
    begin
      ntl := toklist;
      toklist := toklist^.next;
      ntl^.next := free_numbered_token_lists;
      free_numbered_token_lists := ntl;
    end;
end { release_numbered_token_list } ;


(**){--------------- DICTIONARY ACCESS for NUMBERED_TOKENs --------}


{--------------------------------------------------------------------}
{ In the access routines, lookups can be based on number or token    }
{ value.  Each procedure takes a parameter TOK which is passed by    }
{ reference.  The lookup rule is:                                    }
{                                                                    }
{   if tok.number = UNKNOWN_TOKEN_NUMBER then perform lookup based   }
{     on token value                                                 }
{   else perform lookup based on token number.                       }
{                                                                    }
{ It is an assertion violation to attempt to enter Nil values or     }
{ illegal identifiers.                                               }
{--------------------------------------------------------------------}


function find_numbered_token_by_number(
  var {const} tok: numbered_token; 
  var {const} dic: numbered_dictionary;
  var index: numbered_token_table_range; 
  var parent: numbered_token_ptr;
  var found_item: numbered_token_ptr): boolean;
  { Finds parent of token in dic.table (if parent is NIL, then
    token goes at head of INDEXth bucket).  If token is found,
    it is returned as found_item (which will be NIL if not found).
    Function returns FALSE iff brain damage is detected. }
  label
   90; { return }
  var
    found: boolean;  { TRUE when and if found }
begin
  find_numbered_token_by_number := TRUE;
  if tok.number = UNKNOWN_TOKEN_NUMBER then
    begin
      index := 0;
      found_item := NIL;
      parent := NIL;
      assert(ASSERT_LOOKING_FOR_UNKOWN_TOKEN);
      find_numbered_token_by_number := FALSE;
      goto 90 { return } ;
    end;

  index := tok.number mod (LAST_NUMBERED_TOKEN_INDEX + 1);

  found_item :=  dic.table[index];  parent := NIL;  found := FALSE;
  while (found_item <> NIL) and not found do
    if found_item^.number = tok.number then found := TRUE
    else if found_item^.number < tok.number then
      begin
        parent := found_item;  found_item := found_item^.next;
      end
    else found_item := NIL;
90:
end { find_numbered_token_by_number } ;


function find_numbered_token_by_value(
  var {const} tok: numbered_token; 
  var {const} dic: numbered_dictionary;
  var index: numbered_token_table_range; 
  var parent: numbered_token_list_ptr;
  var found_item: numbered_token_list_ptr
): boolean;
  { Finds parent of token in dic.inverted_table (if parent is NIL, then
    token goes at head of INDEXth bucket).  If token is found,
    it is returned as found_item (which will be NIL if not found).
    Function returns FALSE iff brain damage is detected (in which case
    the returned values are undefined). }
  var
    found: boolean;               { TRUE if and when we find the value }
begin
  find_numbered_token_by_value := TRUE;

  case dic.kind of

    STRING_NUMBER:
      begin
        index := (ord(tok.string_p) div 4) mod
	         (LAST_NUMBERED_TOKEN_INDEX + 1);
        found_item :=  dic.inverted_table[index];
	parent := NIL;  found := FALSE;
        while (found_item <> NIL) and not found do
          begin
            if found_item^.token = NIL then
	      begin
	        assert(206 { Nil token field in token list });
                found_item := NIL;
	        find_numbered_token_by_value := FALSE;
	      end;
            if found_item^.token^.string_p = tok.string_p then found := TRUE
            else if ord(found_item^.token^.string_p) < ord(tok.string_p) then
	      begin
	        parent := found_item;  found_item := found_item^.next;
	      end
	    else found_item := NIL; { not there and parent is correct parent }
          end;
      end;

    IDENTIFIER_NUMBER:
      begin
        index := (ord(tok.identifier_p) div 4) mod 
	         (LAST_NUMBERED_TOKEN_INDEX + 1);
        found_item :=  dic.inverted_table[index];
	parent := NIL;  found := FALSE;
        while (found_item <> NIL) and not found do
          begin
            if found_item^.token = NIL then
	      begin
	        assert(206 { Nil token field in token list });
		found_item := NIL;
	        find_numbered_token_by_value := FALSE;
	      end;
            if found_item^.token^.identifier_p = tok.identifier_p then
	      found := TRUE
            else if ord(found_item^.token^.identifier_p) < 
	            ord(tok.identifier_p) then
	      begin
	        parent := found_item;  found_item := found_item^.next;
	      end
	    else found_item := NIL; { not there and parent is correct parent }
          end;
      end;

    OTHERWISE 
      begin
	assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
        find_numbered_token_by_value := FALSE;
      end;
  end;

end { find_numbered_token_by_value } ;


function insert_numbered_token(
  var {const} tok: numbered_token; 
  var dic: numbered_dictionary): numbered_token_ptr;
  { Searches table for numbered token in specified dictionary. 
    Returns pointer to found (or newly inserted) numbered token.
    The token is marked USED.
    Tok is NOT altered.  NIL is returned if assertion violation has
    occurred. Note -- this routine is GUARANTEED not to enter into the
    table a token with a non-kosher value (Non kosher values are NIL for
    all both types of tokens and an identifier entry for NULL_ALPHA. }
  label
    90; { return }
  var
    result: numbered_token_ptr;            { returned value }
    table_index: numbered_token_table_range;     { bucket in table }
    inv_table_index: numbered_token_table_range; { bucket in inverted table }
    parent: numbered_token_ptr;            { parent in table bucket }
    inv_parent: numbered_token_list_ptr;   { parent in inverted table bucket }
    the_item: numbered_token_ptr;          { found or created table entry }
    the_inv_item: numbered_token_list_ptr; { found or created inv table entry }
    temp: numbered_token;                  { copy of tok with new number }


  function alpha_is_id(id: alpha):  boolean;
    { return TRUE iff the id is a legal id }
    var
      ok: boolean;           { TRUE as long as id is ok }
      in_blanks: boolean;    { TRUE when in trailing blanks }
      i: id_range;           { index into id }
  begin
    ok := isupper[id[1]];  in_blanks := FALSE;  i := 1;
    while ok and not in_blanks and (i < ID_LENGTH) do
      begin
        i := i + 1;
        if id[i] = ' ' then in_blanks := TRUE
        else ok := isidentchar[id[i]];
      end;
    while ok and (i < ID_LENGTH) do
      begin
        i := i + 1;
        ok := (id[i] = ' ');
      end;
    alpha_is_id := ok;
  end { alpha_is_id } ;


  function value_is_kosher(var {const} tok: numbered_token;
                           kind: numbered_token_type): boolean;
    { test whether value is kosher -- non-NIL and if an identifer, then
      a legal one. }
  begin
    case kind of
      STRING_NUMBER:
        if tok.string_p = NIL then
	  begin
	    assert(221 { null string });
	    value_is_kosher := FALSE;
	  end
	else value_is_kosher := TRUE;
      IDENTIFIER_NUMBER:
        if tok.identifier_p = NIL then
	  begin
	    assert(221 { null string });
	    value_is_kosher := FALSE;
	  end
	else if not alpha_is_id(tok.identifier_p^.name) then
	  begin
	    assert(233 { bad id });
	    value_is_kosher := FALSE;
	  end
	else value_is_kosher := TRUE;
      OTHERWISE 
	begin
	  assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	  value_is_kosher := FALSE;
	end;
    end;
  end { value_is_kosher } ;
      
	
  function insert_number_only({const}var tok: numbered_token): 
    numbered_token_ptr;
    { inserts tok into table and returns pointer to new object. }
    var
      newtok: numbered_token_ptr;
  begin
    if dic.highest < tok.number then dic.highest := tok.number;
    if parent = NIL then
      begin
        new_numbered_token(dic.table[table_index], dic.kind);
        newtok := dic.table[table_index];
      end
    else
      begin
        new_numbered_token(parent^.next, dic.kind);
        newtok := parent^.next;
      end;
    with newtok^ do
      begin
	number := tok.number;
	case dic.kind of
	  IDENTIFIER_NUMBER: identifier_p := tok.identifier_p;
	  STRING_NUMBER: string_p := tok.string_p;
	  OTHERWISE
	    begin
	      assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	      string_p := NIL;
	    end;
	end;
      end;
    insert_number_only := newtok;
  end { insert_number_only } ;


  function insert_new_token({const}var tok: numbered_token): numbered_token_ptr;
    { inserts tok into both lists. Returns pointer to the new element }
    var
      newtok: numbered_token_ptr;  { new token }
  begin
    newtok := insert_number_only(tok);
    if inv_parent = NIL then
      begin
	new_numbered_token_list(dic.inverted_table[inv_table_index]);
	dic.inverted_table[inv_table_index]^.token := newtok;
      end
    else
      begin
	new_numbered_token_list(inv_parent^.next);
	inv_parent^.next^.token := newtok;
      end;
    insert_new_token := newtok;
  end { insert_new_token } ;


begin { insert_numbered_token }
  insert_numbered_token := NIL;  result := NIL;

  if not dic.active then
    begin
      assert(249 { insertion to non-active dictionary });
      goto 90 { return } ;
    end;

  if tok.number = UNKNOWN_TOKEN_NUMBER then
    begin
      if not find_numbered_token_by_value(tok, dic, inv_table_index, 
				          inv_parent, the_inv_item) then
	goto 90 { return } ;  { assertion message already issued }

      if the_inv_item <> NIL then
	begin { found it }
	  result := the_inv_item^.token;
	  if the_inv_item^.token = NIL then
	    begin
	      assert(206 { Nil token field in token list });
	    end;
	end
      else 
	begin { create new entry }
          if not value_is_kosher(tok, dic.kind) then goto 90 { return } ;
	  if dic.highest = MAXINT then
	    begin
	      assert(208 { numbered token table overflow });
	      goto 90 { return } ;
	    end;
	  dic.highest := dic.highest + 1;
	  temp := tok;
	  temp.number := dic.highest;

	  if not find_numbered_token_by_number(temp, dic, table_index, 
					       parent, the_item) then
	    goto 90 { return } ;  { assertion message already issued }

	  if the_item = NIL then
	    begin
	      result := insert_new_token(temp);
	    end
	  else
	    begin
	      assert(ASSERT_NEW_NUMBER_IS_OLD); { serious abberation }
	      goto 90 { return } ;
	    end;
	end;
    end
  else
    begin
      if not find_numbered_token_by_number(tok, dic, table_index, 
				           parent, the_item) then
	goto 90 { return } ;  { assertion message already issued }

      if the_item <> NIL then 
	begin { found it }
	  result := the_item; 
	end
      else
	begin { create new entry }
          if not value_is_kosher(tok, dic.kind) then goto 90 { return } ;

	  if not find_numbered_token_by_value(tok, dic, inv_table_index, 
					      inv_parent, the_inv_item) then
	    goto 90 { return } ;  { assertion message already issued }

	  if the_inv_item = NIL then
	    begin
	      result := insert_new_token(tok);
	    end
	  else
	    begin
	      assert(ASSERT_DUPLICATE_TOKEN_VALUE); { warning }
	      result := insert_number_only(tok);
	    end;
	  if dic.highest < tok.number then dic.highest := tok.number;
	end;

    end;

    if result <> NIL then result^.used := TRUE;

    insert_numbered_token := result;
90:
end { insert_numbered_token } ;


function find_numbered_token(
  var {const} tok: numbered_token; 
  var {const} dic: numbered_dictionary): numbered_token_ptr;
  { Searches table for numbered token in specified dictionary. 
    If found, returns pointer to found numbered token and the
    found token is marked USED.
    Tok is NOT altered.  NIL is returned if assertion violation has
    occurred or token was not found. This function does not assume
    that not finding the item is an assertion violation.  }
  label
    90; { return }
  var
    table_index: numbered_token_table_range;     { bucket in table }
    inv_table_index: numbered_token_table_range; { bucket in inverted table }
    parent: numbered_token_ptr;            { parent in table bucket }
    inv_parent: numbered_token_list_ptr;   { parent in inverted table bucket }
    the_item: numbered_token_ptr;          { found or created table entry }
    the_inv_item: numbered_token_list_ptr; { found or created inv table entry }
begin
  find_numbered_token := NIL;

  if tok.number = UNKNOWN_TOKEN_NUMBER then
    begin
      if not find_numbered_token_by_value(tok, dic, inv_table_index, 
				          inv_parent, the_inv_item) then
	goto 90 { return } ;  { assertion message already issued }
      if the_inv_item <> NIL then
	begin { found it }
	  find_numbered_token := the_inv_item^.token;
	  if the_inv_item^.token = NIL then
	    begin
	      assert(206 { Nil token field in token list });
	    end
	  else the_inv_item^.token^.used := TRUE;
	end;
    end
  else
    begin
      if not find_numbered_token_by_number(tok, dic, table_index, 
				           parent, the_item) then
	goto 90 { return } ;  { assertion message already issued }

      find_numbered_token := the_item;
      if the_item <> NIL then the_item^.used := TRUE;
    end;
90:
end { find_numbered_token } ;


procedure delete_numbered_token(var {const} tok: numbered_token; 
			       var dic: numbered_dictionary);
  { Searches table for numbered token in specified dictionary. 
    If found, that token is deleted from the dictionary.
    (Deletion is implemented by marking the token USED=FALSE)
    Tok is NOT altered.                                   }
  var temp: numbered_token_ptr;
begin
  temp := find_numbered_token(tok, dic);
  if temp <> NIL then temp^.used := FALSE;
end { delete_numbered_token } ;


function enter_numbered_string(str: xtring;  var dic: numbered_dictionary):
  numbered_token_ptr;
  { enter the string into the dictionary and returns a pointer to
    the entered element (may return NIL if SEVERE brain damage is
    detected). NEVER return non-NIL token with a NIL value. }
  var
    temptok: numbered_token;
begin
  if dic.kind <> STRING_NUMBER then
    begin
      assert(224 { illegal dict insertion });
      writeln(cmplog, ' enter_numbered_string');
      enter_numbered_string := NIL;
    end
  else
    begin
      init_numbered_token(temptok, STRING_NUMBER);
      temptok.string_p := str;
      enter_numbered_string := insert_numbered_token(temptok, dic);
    end;
end { enter_numbered_string } ;


function enter_numbered_id(id: name_ptr;  var dic: numbered_dictionary):
  numbered_token_ptr;
  { enter the id into the dictionary and returns a pointer to
    the entered element (may return NIL if severe brain damage is
    detected).  }
  var
    temptok: numbered_token;
begin
  if dic.kind <> IDENTIFIER_NUMBER then
    begin
      assert(224 { illegal dict insertion });
      writeln(cmplog, ' enter_numbered_id');
      enter_numbered_id := NIL;
    end
  else
    begin
      init_numbered_token(temptok, IDENTIFIER_NUMBER);
      temptok.identifier_p := id;
      enter_numbered_id := insert_numbered_token(temptok, dic);
    end;
end { enter_numbered_id } ;


(**){--------------- NUMBERED_TOKEN  OUTPUT  ----------------------}


procedure output_numbered_token(var f: pipe; var tok: numbered_token; 
			       kind: numbered_token_type);
  { Prints a numbered token to f using the _continue routines.
    tok.output_number is the number that is used (as it is assumed that
    the dictionary has already been printed, thus the output number
    will match the number printed for the dictionary entry). }
begin
  case kind of
    IDENTIFIER_NUMBER: 
      begin
	pipe_dump_char(f, '#');
        if tok.output_number = UNKNOWN_TOKEN_NUMBER then
          begin
            assert(225 { should be defined });
            writeln(cmplog, ' output_numbered_token !');
	    pipe_dump_string_quoted(f, tok.string_p);
          end
	else pipe_dump_integer(f, tok.output_number);
      end;
    STRING_NUMBER: 
      begin
	pipe_dump_char(f, '$');
        if tok.output_number = UNKNOWN_TOKEN_NUMBER then
          begin
            assert(225 { should be defined });
            writeln(cmplog, ' output_numbered_token !');
	    pipe_dump_alpha(f, tok.identifier_p^.name);
          end
	else pipe_dump_integer(f, tok.output_number);
      end;
    OTHERWISE assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
  end;
end { output_numbered_token } ;


procedure lookup_and_print_string_continue(var f: pipe;
					   str: xtring;
					   var dic: numbered_dictionary);
  { looks up the string, if found prints it as number, if not prints
    it as literal  NOTE this procedure will not print anything if str
    is NIL, so calling procedure better be sure that it isn't }
  var
    temptok: numbered_token;    { for passing to find_numbered_token }
    result: numbered_token_ptr; { found entry } 
begin
  if str = NIL then
    begin
      assert(220 { NIL string});
      writeln(cmplog, ' lookup_and_print_string_continue ! ');
    end
  else if dic.kind <> STRING_NUMBER then
    begin
      assert(224 { illegal dict insertion });
      writeln(cmplog, ' lookup_and_print_string_continue ! ');
    end
  else
    begin
      pipe_dump_char(f, '$');
      init_numbered_token(temptok, STRING_NUMBER);
      temptok.string_p := str;
      result := find_numbered_token(temptok, dic);
      if result = NIL then pipe_dump_string_quoted(f, str)
      else if result^.output_number = UNKNOWN_TOKEN_NUMBER then
        begin
	  assert(225 {should not be there});
          writeln(cmplog, ' lookup_and_print_string_continue ! ');
	  pipe_dump_string_quoted(f, str);
	end
      else pipe_dump_integer(f, result^.output_number);
    end;
end { lookup_and_print_string_continue } ;


procedure lookup_and_print_id_continue(var f: pipe;
                                       id: name_ptr;
				       var dic: numbered_dictionary);
  { looks up the id, if found prints it as number, if not prints
    it as literal. }
  var
    temptok: numbered_token;    { for passing to find_numbered_token }
    result: numbered_token_ptr; { found entry } 
begin
  if id = NIL then
    begin
      assert(221 { NIL name_ptr });
      writeln(cmplog, ' lookup_and_print_id_continue ! ');
    end
  else if dic.kind <> IDENTIFIER_NUMBER then
    begin
      assert(224 { illegal dict insertion });
      writeln(cmplog, ' lookup_and_print_id_continue ! ');
    end
  else
    begin
      pipe_dump_char(f, '#');
      init_numbered_token(temptok, IDENTIFIER_NUMBER);
      temptok.identifier_p := id;
      result := find_numbered_token(temptok, dic);
      if result = NIL then pipe_dump_alpha(f, id^.name)
      else if result^.output_number = UNKNOWN_TOKEN_NUMBER then
        begin
	  assert(225 {should not be there});
          writeln(cmplog, ' lookup_and_print_id_continue ! ');
	  pipe_dump_alpha(f, id^.name);
        end
      else pipe_dump_integer(f, result^.output_number);
    end;
end { lookup_and_print_id_continue } ;


(**){------------- NUMBERED_DICTIONARY OPERATIONS --------------}


procedure output_numbered_dictionary(var f: pipe;
			    var dic: numbered_dictionary;
			    final_output: boolean);
  { Writes the dictionary to the (already opened for writing) file.
    If FINAL_OUTPUT, then the OUTPUT_NUMBER fields of the entries
    are set to "take up the slack" (so that numbers 1..n are used
    to represent n entries) and these OUTPUT_NUMBERs are written
    to correspond with the tokens.  
    Note that these OUTPUT_NUMBERs MUST then be used when 
    writing references to these tokens later in the file.
    If not FINAL_OUTPUT, then the usual NUMBERs are used and
    the OUTPUT_NUMBERs are set to match them.
    It is highly illegal to attempt to insert objects into a
    dictionary after it has been written for final output. }
  var
    i: numbered_token_table_range;  { current bucket index }
    nt: numbered_token_ptr;         { current element of bucket }


  procedure number_dictionary;
    { Assigns output_number := number for all entries in dictionary }
    var
      i: numbered_token_table_range; { current bucket }
      nt: numbered_token_ptr;        { entry in bucket }
  begin
    for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
      begin
	nt := dic.table[i];
	while nt <> NIL do 
	  begin
	    nt^.output_number := nt^.number;  nt := nt^.next;
	  end;
      end;
  end {number_dictionary } ;


  procedure renumber_dictionary;
    { Assigns numbers to output_number fields so as to compress
      the range of numbers used. }
    label
      90; { return }
    type
      ptr_table = array[numbered_token_table_range] of numbered_token_ptr;
    var
      i: numbered_token_range;           { current number }
      index: numbered_token_table_range; { current bucket }
      missing: numbered_token_range;     { number missing }
      ntptr: ptr_table;                  { current element of each bucket }
  begin
    dic.active := FALSE;  { No further insertions allowed }

    with dic do
      for index := 0 to LAST_NUMBERED_TOKEN_INDEX do
        ntptr[index] := table[index];

    missing := 0;
    index := 1;    { we never use 0 as a token number }

    for i := 1 to dic.highest do
      begin
	if ntptr[index] = NIL then missing := missing + 1
	else with ntptr[index]^ do 
	  begin
	    if number > i then missing := missing + 1
	    else if number = i then
	      begin
	        if used then output_number := i - missing
	        else 
		  begin
		    missing := missing + 1;
		    output_number := UNKNOWN_TOKEN_NUMBER;
		  end;
	        ntptr[index] := next;
	      end
	    else { < is a no-no }
	      begin
		assert(ASSERT_BUCKET_UNORDERED);
		number_dictionary;
		goto 90 { return } ;
	      end;
	  end;
	if index = LAST_NUMBERED_TOKEN_INDEX then index := 0
	else index := index + 1;
      end;
    dic.highest := dic.highest - missing;
  90:
  end { renumber_dictionary } ;


begin { output_numbered_dictionary }
  if final_output then renumber_dictionary
		  else number_dictionary;

  case dic.kind of

    IDENTIFIER_NUMBER:

      for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
	begin
          nt := dic.table[i];
	  while nt <> NIL do with nt^ do
	    begin
              if used then
                if output_number = UNKNOWN_TOKEN_NUMBER then
	          begin
	            assert(225 { UNKNOWN found });
		    writeln(cmplog, ' output_numbered_dictionary (id)');
		  end
		else
                  begin
		    pipe_dump_char(f, '!');
                    pipe_dump_integer(f, output_number);
		    pipe_dump_char(f, ' ');
                    pipe_dump_alpha(f, identifier_p^.name);
                    pipe_dump_CRLF(f);
                  end;
              nt := next;
	    end;
        end;

   STRING_NUMBER:

      for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
        begin
          nt := dic.table[i];
          while nt <> NIL do with nt^ do
	    begin
              if used then
                if output_number = UNKNOWN_TOKEN_NUMBER then
	          begin
	            assert(225 { UNKNOWN found });
		    writeln(cmplog, ' output_numbered_dictionary (string)');
		  end
		else
                  begin
		    pipe_dump_char(f, '!');
                    pipe_dump_integer(f, output_number);
                    pipe_dump_string_quoted(f, string_p);
                    pipe_dump_CRLF(f);
                  end;
              nt := next;
            end;
	end;

    OTHERWISE

      begin
	assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
      end;

  end { case } ;

  pipe_dump_char(f,';');   pipe_dump_CRLF(f);

end { output_numbered_dictionary } ;


procedure dump_numbered_dictionary(var f: textfile;
			    var dic: numbered_dictionary);
  { dump the numbered dictionary to file f (for debugging purposes) }
  var
    i: numbered_token_table_range;  { current bucket index }
    nt: numbered_token_ptr;         { current element of bucket }
begin
  write(f, ' Number Out-number Used');

  case dic.kind of

    IDENTIFIER_NUMBER:
      begin
        writeln(f, ' Identifier');  writeln(f);

        for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
	  begin
            nt := dic.table[i];
	    while nt <> NIL do with nt^ do
              begin
                write(f, ' ', number:6, output_number:11);
		if used then write(f, '    T ')
		        else write(f, '    F ');
                if identifier_p = NIL then write(f, '<NIL> (bad!)')
                else writealpha(f, identifier_p^.name);
		writeln(f);
                nt := next;
              end;
          end;
      end;

   STRING_NUMBER:
     begin
       writeln(f, ' String');  writeln(f);

        for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
	  begin
            nt := dic.table[i];
	    while nt <> NIL do with nt^ do
              begin
                write(f, ' ', number:6, output_number:11);
		if used then write(f, '    T ')
		        else write(f, '    F ');
                if string_p = NIL then write(f, '<NIL> (bad!)')
                else writestring(f, string_p);
		writeln(f);
                nt := next;
              end;
          end;
      end;

    OTHERWISE

      begin
        writeln(f, ' ???? (bad)');
	assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
      end;

  end { case } ;

  writeln(f);
end { dump_numbered_dictionary } ;


procedure release_numbered_dictionary(var dic: numbered_dictionary);
  { releases of all entries into the dictionary by insertion into
    freelists.  This disposal affects ONLY the numbered_token
    and numbered_token_list records in the dictionary -- it does
    not affect the XTRINGs and NAME_ELEMENTs to which they refer. 
    When done, the dictionary is empty, has same type, and is ready to go. }
  var
    i: numbered_token_table_range;  { current bucket index }
begin
  with dic do
    begin
      for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
	begin
	  while table[i] <> NIL do release_numbered_token(table[i], kind);
	  release_all_numbered_token_lists(inverted_table[i]);
	end;
      highest := 0;
      active := TRUE;
    end;
end { release_numbered_dictionary } ;


procedure init_numbered_dictionary(*var dic: numbered_dictionary; 
			  token_kind: numbered_token_type*);
  { Initializes the dictionary to empty -- it is assumed to contain
    garbage values. }
  var
    i: numbered_token_table_range;  { current bucket index }
begin
  with dic do
    begin
      highest := 0;
      kind := token_kind;
      active := TRUE;
      for i := 0 to LAST_NUMBERED_TOKEN_INDEX do
	begin
	  table[i] := NIL;
	  inverted_table[i] := NIL;
	end;
    end;
end { init_numbered_dictionary } ;


(**){------------- NUMBERED_TOKEN PARSING --------------}


procedure parse_numbered_dictionary(var dic: numbered_dictionary);
  { Parses a numbered dictionary section of the parse file.
    The dic.kind field is expected to indicate the type of token 
    expected, and dic is expected to have been initialized. 
    The routine ends with sy being the token following the SEMI
    which ends the dictionary (or (an error condition) the ENDSY
    or ENDOFDATASY that was prematurely encountered)   }
  label
    10, { cycle }
    90; { return }
  var
    end_symbols: setofsymbols;  { symbols indicating the end of the section }
    skip_symbols: setofsymbols; { symbols for recovery (heaven forbid) }
    kind: numbered_token_type;  { value of dic.kind }
    tok: numbered_token;        { the token that has just been parsed }
    assert_expected_token: 
      assert_range;             { assertion message for expected token }
    tokptr: numbered_token_ptr; { returned ptr to table entry }
    token_sy: symbols;          { expected token symbol (STRINGS or IDENT)}
begin
  end_symbols := [SEMI,ENDOFDATASY,ENDSY];
  skip_symbols := end_symbols + [EXCLAMATION,SEMI];
  kind := dic.kind;
  init_numbered_token(tok, dic.kind);
  case kind of
    IDENTIFIER_NUMBER:
      begin
	tok.identifier_p := NIL;
	token_sy := IDENT;
	assert_expected_token := ASSERT_EXPECTED_IDENT;
      end;
    STRING_NUMBER:
      begin
	tok.string_p := NIL;
	token_sy := STRINGS;
	assert_expected_token := ASSERT_EXPECTED_STRINGS;
      end;
    OTHERWISE 
      begin
	assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	skip(end_symbols);
	goto 90 { return } ;
      end;
  end;

  while not (sy in end_symbols) do
    begin
      if sy = EXCLAMATION then insymbol
      else
	begin
	  assert(ASSERT_EXPECTED_EXCLAMATION);
	  skip(skip_symbols);
	  goto 10 { cycle } ;
	end;
      if sy <> CONSTANT then
	begin
	  assert(ASSERT_EXPECTED_CONSTANT);
	  skip(skip_symbols);
	  goto 10 { cycle } ;
	end;
      if const_val <= 0 then
	begin
	  assert(124 { entry number out of range });
	  skip(skip_symbols);
	  goto 10 { cycle } ;
	end;
      tok.number := const_val;  insymbol;
      if sy <> token_sy then
	begin
	  assert(assert_expected_token);
	  skip(skip_symbols);
	  goto 10 { cycle } ;
	end;
      case kind of 
	IDENTIFIER_NUMBER: tok.identifier_p := id.name;
	STRING_NUMBER: tok.string_p := lex_string;
	OTHERWISE 
	  begin
	    assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	    skip(end_symbols);
	    goto 90 { return } ;
	  end;
      end;
      tokptr := insert_numbered_token(tok, dic);  
      tokptr^.used := FALSE;
      insymbol;
    10:
    end;
  if sy <> SEMI then
    begin
      assert(116 { unexpected end of data });
    end
  else insymbol;
90:
end { parse_numbered_dictionary } ;


function parse_numbered_token(var dic: numbered_dictionary): 
  numbered_token_ptr;
  { Parses a numbered token of the given type.  If syntax errors
    prevent this, or it is not found in the table, returns NIL.
    Otherwise, returns a pointer to the dictionary entry of the token. 
    Expects the token to be of the same type as the dictionary, and 
    expects that it can be found in the dictionary. }
  label
    90; { return }
  var
    tok: numbered_token;       { for parsed value of number }
    found: numbered_token_ptr; { found dictionary entry } 
begin
  parse_numbered_token := NIL;
  case dic.kind of
    IDENTIFIER_NUMBER: if sy <> SHARP then
      begin
	assert(114 { expected # });
	insymbol;
	goto 90 { return } ;
      end;
    STRING_NUMBER: if sy <> DOLLAR then
      begin
	assert(115 { expected $ });
	insymbol;
	goto 90 { return } ;
      end;
    OTHERWISE
      begin
	assert(ASSERT_UNRECOGNIZED_TOKEN_TYPE);
	insymbol;
	goto 90 { return } ;
      end;
  end;
  insymbol; { eat the prefix }
  init_numbered_token(tok, dic.kind);
  if sy = CONSTANT then
    begin
      if (const_val <= 0) then
        begin
	  assert(124 { out of range });
	  goto 90 { return } ;
	end;
      tok.number := const_val;
      insymbol;
      found := find_numbered_token(tok, dic);
      if found = NIL then
        begin
          assert(207 { numbered token not found });
          goto 90 { return } ;
        end;
    end
  else if sy = STRINGS then
    begin
      if dic.kind <> STRING_NUMBER then
        begin
          assert(123 { expected numb string });  goto 90 { return } ;
        end;
      tok.string_p := lex_string;  insymbol;
      found := insert_numbered_token(tok, dic);
    end
  else if sy = IDENT then
    begin
      if dic.kind <> IDENTIFIER_NUMBER then
        begin
          assert(122 { expected numb id });  goto 90 { return } ;
        end;
      tok.identifier_p := id.name;  insymbol;
      found := insert_numbered_token(tok, dic);
    end
  else
    begin
      if dic.kind = STRING_NUMBER then assert(123)
                                  else assert(122);
      writeln(cmplog, 'sy=', ord(sy):1);
      goto 90 { return } ;
    end;
  parse_numbered_token := found;
90:
end { parse_numbered_token } ;
  
