{ --------------- Drawing initialization for separate comp ------ }


#if VAX
function hack_around_vaxc_bug_fcreat(BYVALUE name: xtring; var err: xtring):
boolean;  EXTERNAL; 
#endif


function is_context_free_primitive(vers: macro_module_ptr): boolean;
  { Return TRUE iff the current drawing is a simple primitive and contains
    no known expandable ids (as determined by last compilation).
    (If not yet compiled, the latter condition is assumed TRUE.) }
  var
    prim_page: page_range;
    page_spec: paged_schema;           { specification of primitive page }
    schema_page: paged_schema_ptr;     { schema page for primitive }
begin
  prim_page := er_issimple(vers);
  if prim_page = 0 then is_context_free_primitive := FALSE
  else
    begin
      is_context_free_primitive := TRUE;
      if er_version(vers) <> 0 then
	begin
	  with page_spec do
	    begin
	      drawing_type := enter_numbered_id(
		name_from_string(er_extension(vers)),
		schema_of_drawing_being_compiled.id_dictionary);
	      version := er_version(vers);
	      page := prim_page;
	    end;
    
	  schema_page := 
	    find_schema_page(schema_of_drawing_being_compiled, page_spec);
	  if schema_page <> NIL then
	    if schema_page^.expandable_ids <> NIL then 
	      is_context_free_primitive := FALSE;
	end;
    end;
end { is_context_free_primitive } ;


procedure init_context(context: property_ptr);
  { Set up the root symbol table to define the context and enters
    the names and definitions into the expansion dictionaries.
    This has the side effect of creating the mtree_root node.
    Current_mtree_node is set to mtree_root. }
  var
    current_parm: property_ptr;      { current parm of context }


  procedure insert_identifier(parm: property_ptr; var table: identifier_ptr);
    { insert the parameter into the identifier list and the expansion
      dictionaries. }
    var
      temp: identifier_ptr;      { new id }
      dummy: numbered_token_ptr; { don't care what dict entries are }
  begin
    { Note implicit assumption that the context list does NOT contain
      duplicate ids }
    temp := NIL;  new_identifier(temp);
    temp^.next := table;  table := temp;

    temp^.name := parm^.name;  temp^.definition := parm^.text;

    dummy := enter_numbered_id(parm^.name, expansion_id_dictionary);
    dummy := enter_numbered_string(parm^.text, expansion_string_dictionary);
  end { insert_identifier } ;


begin { init_context }
  new_mtree_node(mtree_root);  mtree_root^.macro_name := root_macro_name;
  current_parm := context;
  while current_parm <> NIL do
    begin
      insert_identifier(current_parm, mtree_root^.symbol_table);
      current_parm := current_parm^.next;
    end;
  current_mtree_node := mtree_root;
end { init_context } ;


function init_drawing(command: compile_command;
                      drawing: xtring;
		      var skip_drawing: boolean): boolean;
  { Set up for separate_compilation of drawing pages for context specified
    by the global list "specified_context";
    For commands other than CHECK_FOR_PRIM, return TRUE iff successful
    (no errors).  For CHECK_FOR_PRIM, return TRUE iff successful and 
    drawing is a primitive. 
    Return skip_drawing=TRUE iff drawing is primitive and is not 
    a new drawing.
    Guarantee the following state when TRUE is returned (and not 
    skip_drawing):
      1. Global variables set up to indicate the current drawing.
      2. Schema file has been read.
      3. Global text macros have been checked and any changes have
         "dirtied" the appropriate pages in the schema. }
  var
    new_macro: boolean;  { TRUE if current drawing is a different drawing 
                           from last drawing or is a version that comes from
			   a different drawing directory than last version }
    drawing_ok: boolean; { TRUE while we can access a drawing }
    old_module: macro_module_ptr; { version last compiled }


  procedure init_basic_globals;
    { initialize the basic globals for identifying a new drawing }
  begin
    old_module := module_being_compiled;
    module_being_compiled := NIL;

    root_macro_name := drawing;
    context_being_compiled := NIL;
    extension_being_compiled := null_name;
    version_being_compiled := 0;
    page_being_compiled := 0;
  end { init_basic_globals } ;


  procedure set_extension_and_version(modl: macro_module_ptr);
    { set extension_being_compiled and version_being_compiled, if appropriate }
  begin
    extension_being_compiled := name_from_string(er_extension(modl));
    version_being_compiled := er_version(modl);
  end { set_extension_and_version } ;


begin { init_drawing }
  if debug_20 then 
    begin
      write(outfile, ' --- init_drawing ');
      writestring(outfile, drawing);
      writeln(outfile, ' ---');
      write(outfile, ' Context:');
      dump_property_list(outfile, specified_context);
    end;
  drawing_ok := TRUE;  skip_drawing := FALSE;
  re_init_expansion_structures;
  init_basic_globals;

  { NOTE: for sepcomp command, last_sepcomp = NIL, so new_macro = TRUE;
    schema_of_drawing_being_compiled.file_accessible = FALSE, so the
    call to write_schema_file does nothing.  So the following code for 
    SEPLINK works for SEPCOMP as well. }
  if (last_sepcomp = NIL) then new_macro := TRUE
  else new_macro := (last_sepcomp^.drawing <> drawing);
  if new_macro then
    begin
      { clean up schema from the last drawing }
      write_schema_file(schema_of_drawing_being_compiled);
    end;

  init_context(specified_context);

  { report to data services the new drawing being used}
  if report_current_dwg_to_ds(drawing) <> 0 then
    begin
    end;

  if (command = CHECK_FOR_PRIM) then
    module_being_compiled := select_module(drawing, 
				           context_string(specified_context), 1)
  else
    module_being_compiled := select_module(drawing,
				           context_string(specified_context), 0);

  drawing_ok := (module_being_compiled <> NIL);

  if drawing_ok then
    begin
      set_extension_and_version(module_being_compiled);
      if not new_macro then
	if not er_samedwg(module_being_compiled, old_module) then
	  begin
	    new_macro := TRUE;
	    write_schema_file(schema_of_drawing_being_compiled);
	    if debug_20 then 
	      writeln(outfile, 'Different drawing_dir -- change schema file');
	  end;
    end;

  er_release(old_module);

  if (command = CHECK_FOR_PRIM) and drawing_ok then
    drawing_ok := er_isprim(module_being_compiled);

  if new_macro and drawing_ok then
    begin
      drawing_ok := read_schema_file(drawing, module_being_compiled,
                                     schema_of_drawing_being_compiled);
    end;

  if is_context_free_primitive(module_being_compiled) then
    begin
      if debug_20 or debug_23 or debug_24 then
	writeln(Outfile, '  that is context free');

     { change context to the NULL context and prepare to skip this
       one if the primitive has already been done. Note that the
       symbol table is re-initialized for each page, so we don't
       need to worry about that here }

     specified_context := NIL;
     skip_drawing := not new_macro;
   end;

  if drawing_ok and not skip_drawing then 
    begin
      if new_macro then
        check_global_changes(schema_of_drawing_being_compiled);
      context_being_compiled := 
        enter_context_definition(specified_context, 
	                         schema_of_drawing_being_compiled);
      drawing_ok := context_being_compiled <> NIL;
    end;
    
  drawing_ok := drawing_ok and (errors_encountered * fatal_errors = []);

  init_drawing := drawing_ok;
  if debug_20 then writeln(outfile, ' --- exit init_drawing ',
                           drawing_ok, ' skip=', skip_drawing, ' ---');
end { init_drawing } ;
  

{------------ LINKER COMMUNICATION ROUTINES ----------------}


procedure new_sepcomp_list(var ptr: sepcomp_list_ptr);
  { insert a new element }
  var
    newone: sepcomp_list_ptr;  { new element }
begin
  if free_sepcomp_lists = NIL then
    begin
      new(newone);
      increment_heap_count(HEAP_SEPCOMP_LIST, 
                           4*POINTER_SIZE+ENUM_SIZE+BOOL_SIZE);
    end
  else
    begin
      newone := free_sepcomp_lists;
      free_sepcomp_lists := free_sepcomp_lists^.next;
    end;
  with newone^ do
    begin
      command := COMPILE_SUB;  { the usual case }
      drawing := nullstring;
      compile_type := null_name;
      context := NIL;
      forcecompile := FALSE;
      next := ptr;
    end;
  ptr := newone;
end { new_sepcomp_list } ;


procedure release_sepcomp_list(var ptr: sepcomp_list_ptr);
  { delete and release the element }
 var
    oldnext: sepcomp_list_ptr;  { save ptr^.next for return }
begin
  if ptr <> NIL then
    begin
      release_entire_property_list(ptr^.context);
      oldnext := ptr^.next;
      ptr^.next := free_sepcomp_lists;
      free_sepcomp_lists := ptr;
      ptr := oldnext;
    end;
end { release_sepcomp_list } ;


function open_linker_comm: boolean;
  { open cmpdraw and design, if not specified in command line (and
    thus opened by linker). Return TRUE if successful.  If unsuccessful,
    make sure that both descriptors are CLOSED (essential for uVAX version). }
  var
    ok: boolean;         { TRUE while successful }
begin
  ok := TRUE;
  if CMPDRAW_specified_in_command_line then
    begin
      if not pipe_reset_descriptor(CmpDraw_pipe, 
                                   FROM_ET_PIPE, CmpDraw_fd) then
        begin
          error(166 { no cmpdraw });
	  error_dump_indent(indent);
	  error_dump_alpha('descriptor=     ');
	  error_dump_integer(CmpDraw_fd);
	  error_dump_CRLF;
          ok := FALSE;
	end;
    end
  else
    if not pipe_reset_file(CmpDraw_pipe, FROM_ET_PIPE, nullstring) then
      begin
        error(166 { no cmpdraw });  ok := FALSE;
      end;

  current_file := UNKNOWN_FILE;

  if DESIGN_specified_in_command_line then
    begin
      if not pipe_rewrite_descriptor(Design_pipe, 
	                             TO_ET_PIPE, Design_fd) then
	begin
          error(234 { can't access DESIGN file});
          error_dump_indent(indent);
          error_dump_alpha('descriptor=     ');
          error_dump_integer(Design_fd);
          error_dump_CRLF;
	  if ok then pipe_close(CmpDraw_pipe);
          ok := FALSE;
        end
      else if not ok then pipe_close(Design_pipe);
    end
  else
    if not pipe_rewrite_file(Design_pipe, TO_ET_PIPE, nullstring) then
      begin
        error(234 { hosed });
	if ok then pipe_close(CmpDraw_pipe);
        ok := FALSE;
      end
    else if not ok then pipe_close(Design_pipe);

  open_linker_comm := ok;
end { open_linker_comm } ;
  

function init_cmpdraw_file: boolean;
  { Sets up for parse of the cmpdraw file (PIPE_FROM_ET) (already open). 
    Returns TRUE if successful }
begin
  if current_file <> UNKNOWN_FILE then
    begin
      init_cmpdraw_file := FALSE;
      assert(169 { tried to open two files at once });

      writeln(CmpLog, ' Current file open = ', ord(current_file));
      writeln(CmpLog, ' New file to open = ', ord(CMPDRAW_FILE));
    end
  else
    begin
      init_cmpdraw_file := TRUE;

      allowed_key_words := [ENDSY];
      parse_SCALDconstants := FALSE;

      current_file := CMPDRAW_FILE;
      read_state := FINIT;
      insymbol;
    end;
end { init_cmpdraw_file } ;


procedure pseudo_close_parse_file;
  { "close" reading of parse file without actually closing it.
     Set special parse flags to their default values. }
begin
  allowed_key_words := [];  
  parse_SCALDconstants := TRUE;
  upper_case_strings := TRUE;
  current_file := UNKNOWN_FILE;
end { pseudo_close_parse_file } ;


function read_linker_commands: boolean;
  { Reads the cmpdraw_file into the list headed by sepcomp_list_root.
    The list is appended to the end of the current list.  
    The value of last_sepcomp and current_sepcomp are not changed.
    The file must already be open.  It is closed if ENDOFDATASY
    is encountered right away. Returns FALSE when file has been closed. }
  label
    90; { return }
  var
    current_command: compile_command;     { latest imperative }
    final_end_symbols: setofsymbols;      { indicating end of data }
    end_symbols: setofsymbols;            { indicating end of an element }
    recovery_set: setofsymbols;           { for parse error recovery }
    result: boolean;                      { value for return }
    save_debug: boolean;                  { value of "debug" }
        

  procedure reverse_list(var head: sepcomp_list_ptr);
    { reverse the list }
    var
      previous: sepcomp_list_ptr;  { former parent of head }
      next: sepcomp_list_ptr;      { former child of head }
  begin
    previous := NIL;
    while head <> NIL do 
      begin
        next := head^.next;
        head^.next := previous;
        previous := head;
        head := next;
      end;
    head := previous;
  end { reverse_list } ;


  function parse_cmpdraw_context(var context: property_ptr): boolean;
    { parse context and return TRUE if no syntax errors.
      NOTE: this routine does not call skip on errors -- calling proc must 
      recover }
    label
      90; { return }
    var
      temp: property_ptr; { new element }
      tail: property_ptr; { last element of list }
  begin
    context := NIL;  tail := NIL;  parse_cmpdraw_context := TRUE;
    if not (sy in end_symbols + [IDENT,CONSTANT]) then 
      begin
        assert(234 { expect params });
        parse_cmpdraw_context := FALSE;
        goto 90 { return } ;
      end;
    if sy = CONSTANT then
      begin
        new_property(temp);
        temp^.name := SIZE_prop_name;
        temp^.text := number_to_string(const_val);
        temp^.next := NIL;
        context := temp;  tail := temp;
        insymbol;
      end;
    while sy = IDENT do
      begin
        new_property(temp);
        temp^.name := id.name;
        insymbol;
        if not (sy = STRINGS) then
          begin
            assert(49 { expect prop value });  parse_cmpdraw_context := FALSE;
            release_property(temp);
            goto 90 { return } ;
          end
        else
          begin
            temp^.text := lex_string;
            if tail = NIL then
              begin
                context := temp;  tail := temp;
              end
            else 
              begin
                tail^.next := temp;  tail := temp;
              end;
            insymbol;
          end;
      end;
  90:
  end { parse_cmpdraw_context } ;


  function find_compile_command(name: name_ptr): compile_command;
    { return the command indicated by the name or LAST_COMPILE_COMMAND
      (if not found) }
    var
      command: compile_command;
  begin
    command := succ(FIRST_COMPILE_COMMAND);
    while (command <> LAST_COMPILE_COMMAND) and 
          (compile_command_table[command] <> name) do
      command := succ(command);
    find_compile_command := command;
  end { find_compile_command } ;


  function process_compile_command: boolean;
    { Read the command and enter it in the list of modules to compile }


    procedure handle_gross_failure(num: assert_range);
      { print assertion failure message and clean up }
    begin
      if num <> 0 then assert(num);
      error(217 { fatal });
      pseudo_close_parse_file;
      process_compile_command := FALSE;
      release_sepcomp_list(sepcomp_list_root);
    end { handle_gross_failure } ;


  begin { process_compile_command }
    process_compile_command := TRUE;
    new_sepcomp_list(sepcomp_list_root);
    sepcomp_list_root^.command := current_command;

    if sy <> STRINGS then handle_gross_failure(ASSERT_EXPECTED_STRINGS)
    else
      begin
        sepcomp_list_root^.drawing := lex_string;
        insymbol;
        if sy <> IDENT then handle_gross_failure(ASSERT_EXPECTED_IDENT)
	else
	  begin
            sepcomp_list_root^.compile_type := id.name;
            insymbol;
            if not parse_cmpdraw_context(sepcomp_list_root^.context) then
                handle_gross_failure(0);
          end;
      end;
  end { process_compile_command } ;


begin { read_linker_commands }
  if debug_20 then writeln(outfile, 'enter read_linker_commands');
  if debug_26 then
    begin
      save_debug := debug;  debug := TRUE;
    end;
  final_end_symbols := [ENDSY,ENDOFDATASY];
  end_symbols := [SEMI,ENDSY,ENDOFDATASY];
  recovery_set := [IDENT]+final_end_symbols;

  if not init_cmpdraw_file then 
    begin
      error(166 { can't access CMPDRAW});
      pseudo_close_parse_file;
      read_linker_commands := FALSE; { die }
      goto 90 { return } ;
    end;

  { reverse the list so that insertions can be made at head, re-reversing
    when done. }

  reverse_list(sepcomp_list_root); 

  if sy = ENDOFDATASY then
    begin
      read_linker_commands := FALSE;  
      pseudo_close_parse_file;
      goto 90 { return } ;
    end;

  result := TRUE;

  if not (sy in recovery_set) then 
    begin
      assert(226 { expected sepcomp command });  error(217 { fatal });
      pseudo_close_parse_file;
      read_linker_commands := FALSE;
      goto 90 { return } ;
    end;

  while (sy = IDENT) and result do
    begin
      current_command := find_compile_command(id.name);
      insymbol; { eat the command }
      if current_command = LAST_COMPILE_COMMAND then
        begin
          assert(226 { expected sepcomp command });  error(217 { fatal });
          result := FALSE;
          release_sepcomp_list(sepcomp_list_root);
        end
      else if current_command = STOP_AT_SUBCKT then
	analog_designer_compile := TRUE
      else result := process_compile_command;

      if (sy = SEMI) then insymbol
      else 
        begin
          assert(35 { expected ; });  error(217 { fatal });
          result := FALSE;
        end;
    end;


  if (sy <> ENDSY) then assert(34 { expected END });

  reverse_list(sepcomp_list_root);

  { NOTE: do not insymbol the END as this causes insymbol to get the '.'
    which causes it to look ahead to see if it might be '..'.  The 
    lookahead is not smart enough to realize that it is at an eoln, so
    it tries to get another line.  This causes deadlock between the
    compiler and linker. }

  pseudo_close_parse_file;
  read_linker_commands := result;

90:
  if debug_26 then debug := save_debug;
end { read_linker_commands } ;


procedure write_bogus_expansion_file(var f: pipe; pg: natural_number);
  { writes expansion file containing only the current drawing
    specification and error summary to pipe d, calling it version 1 and
    page pg. (Use page 0 for "drawing not found" errors -- this is why
    pg is not of type page_range.) }
begin
  init_output_continue;
  output_numbered_dictionary(f, expansion_string_dictionary, 
                             FINISHED_DICTIONARY);
  output_numbered_dictionary(f, expansion_id_dictionary, 
                             FINISHED_DICTIONARY);
  output_heading(f, pg);
  output_context(f);
  pipe_dump_char(f, ';');  { body properties section }
  pipe_dump_char(f, ';');  { dependencies section }
  pipe_dump_char(f, ';');  { modules list section }
  pipe_dump_char(f, ';');  { syonyms section }
  pipe_dump_char(f, ';');  { invokes section }
  pipe_dump_char(f, ';');  { signal properties section }
  pipe_dump_CRLF(f);
  pipe_dump_alpha(f, 'END.            ');
  pipe_dump_CRLF(f);
end { write_bogus_expansion_file } ;


procedure write_design_file_page_spec;
  { write design file information for this page }
begin
  if DESIGN_specified_in_command_line then
    begin
      pipe_dump_alpha(CmpTmp_pipe, 'D#READ$         ');
      pipe_dump_string_quoted(CmpTmp_pipe, page_expansion_file_name);
      pipe_dump_char(CmpTmp_pipe, ';');
      pipe_dump_CRLF(CmpTmp_pipe);
    end
  else
    begin
      pipe_dump_alpha(Design_pipe, 'D#READ$         ');
      pipe_dump_string_quoted(Design_pipe, page_expansion_file_name);
      pipe_dump_char(Design_pipe, ';');
      pipe_dump_CRLF(Design_pipe);
    end
end { write_design_file_page_spec } ;


function init_next_sepcomp: boolean;
  { Set up for next separate_compilation requested by the linker.
    Return TRUE if one is needed, FALSE if completely done.
    If the current command is SEPLINK then
    guarantee the following state when TRUE is returned:
      1. Global variables set up to indicate the current drawing.
      2. Schema file has been read.
      3. If PrintCmpLst then the CmpLst file has been opened for writing. }
  var
    drawing_ok: boolean;   { TRUE while we can access a drawing }
    skip_drawing: boolean; { TRUE iff drawing is to be skipped (because it is
                             a repeated request for same primitive) }


  procedure output_et_directives(var f: pipe);
    { write out the bit order, property attributes, max error limit, and
      suppressed messages as et directives. }
    const
      MAX_PROPERTIES_PER_LINE = 4;
      MAX_ERRORS_PER_LINE = 20;
    var
      default_inheritance:
        name_type_set;         { default inheritance attributes }
      count: natural_number;   { number of listed things written to the line }
      i: name_table_range;     { index into name_table }
      name: name_ptr;          { current name }
      temp: xtring;            { temporary string for linker list file name }
      err: error_range;        { current error number }
  begin
    default_inheritance := default_attributes * inheritance_attributes;
    count := 0;
    
    if left_to_right then pipe_dump_alpha(f, 'D#BITORDER#LR;  ')
                     else pipe_dump_alpha(f, 'D#BITORDER#RL;  ');
    pipe_dump_CRLF(f);

    pipe_dump_alpha(f, 'D#ATTRIBUTES    ');  pipe_dump_CRLF(f);
    for i := 0 to NAME_TABLE_SIZE do
      begin
        name := name_table[i];
        while name <> NIL do
          begin
            if (name^.kind <> default_attributes) then
              begin
                if count >= MAX_PROPERTIES_PER_LINE then
                  begin
                    pipe_dump_CRLF(f);  count := 0;
                  end;
                pipe_dump_char(f, '#');
                pipe_dump_alpha(f, name^.name);
                pipe_dump_char(f, '=');
                pipe_dump_integer(f, et_property_attributes(name^.kind));
                count := count + 1;
              end;
            name := name^.next;
          end;
      end;
    pipe_dump_char(f, ';');  pipe_dump_CRLF(f);

    pipe_dump_alpha(f, 'D#MAX_ERROR     ');
    pipe_dump_char(f, ' ');
    pipe_dump_integer(f, max_errors);
    pipe_dump_char(f, ';');  pipe_dump_CRLF(f);
    
    if suppress_errors <> [] then
      begin
        pipe_dump_alpha(f, 'D#SUPPRESS      ');
        count := 0;
        for err := 0 to MAX_ERROR_NUM do
          begin
            if (err in suppress_errors) then
              begin
                if count >= MAX_ERRORS_PER_LINE then
                  begin
                    pipe_dump_CRLF(f);
                    count := 0;
                  end;
                pipe_dump_char(f, ' ');
                pipe_dump_integer(f, err);
                count := count + 1;
              end;
          end;
        pipe_dump_char(f, ';');  pipe_dump_CRLF(f);
      end;
  end { output_et_property_attributes } ;
                  

  procedure output_et_root_directives(var f: pipe);
    { output any et directives needed for a root drawing }
    const
      LNKLST_NAME = 'LNKLST          ';
    var
      full: xtring;       { full page linker list file name (in temp string) }
      err: xtring; 
  begin
    full := nullstring;
    copy_string(er_filename(module_being_compiled, 
                            ord(LINKER_LIST_FILE), 0,
			    context_string(specified_context)), full);

    if (full <> nullstring) then
      begin
        { Compiler creates the file so that it is created with the correct
	  umask }

#if UNIX
        if not file_exists(full) then
	  if rewrite_file(CmpLst, full, CMPLST_FILE_NAME) then
	    close_file(CmpLst, CMPLST_FILE_NAME, full);
#endif UNIX
#if VAX
        if not hack_around_vaxc_bug_fcreat(full, err)  then
          begin
            error(169);  error_dump_file_name(full);
            if (err <> NIL) then
              begin
                error_dump_indent(indent);
                error_dump_string(err);  error_dump_CRLF;
              end;
          end;
#endif VAX
        pipe_dump_alpha(f, 'D#LIST$         ');
        pipe_dump_string_quoted(f, full);
        pipe_dump_char(f, ';');  pipe_dump_CRLF(f);
        release_string(full);
      end;

    if (HIERARCHY in reports_to_generate) then
      begin
        pipe_dump_alpha(f, 'D#REPORT_HIER;  ');
	pipe_dump_CRLF(f);
      end;
  end { output_et_root_directives } ;


begin { init_next_sepcomp }
  if debug_20 then writeln(outfile, 'init_next_sepcomp');
  if (command <> SEPLINK_COMMAND) then drawing_ok := FALSE
  else repeat
    clear_errors;
    drawing_ok := TRUE;  skip_drawing := FALSE;

    { Set up the list for the next drawing }
    while (current_sepcomp <> sepcomp_list_root) and
          (sepcomp_list_root <> NIL { just to be safe }) do
      release_sepcomp_list(sepcomp_list_root);
    last_sepcomp := sepcomp_list_root;
    if last_sepcomp <> NIL then current_sepcomp := last_sepcomp^.next;
    if current_sepcomp = NIL then 
      begin
        { close out last batch }
        
        if last_sepcomp <> NIL then
          begin
            write_schema_file(schema_of_drawing_being_compiled);
            if design_specified_in_command_line then
              begin
                pipe_dump_alpha(CmpTmp_pipe, 'END.            ');
                pipe_dump_CRLF(CmpTmp_pipe);
                pipe_close(CmpTmp_pipe);
                pipe_dump_alpha(Design_pipe, 'D#READ$         ');
                pipe_dump_string_quoted(Design_pipe, temp_file_name);
                pipe_dump_char(Design_pipe,';');
                pipe_dump_CRLF(Design_pipe);
              end;
            pipe_dump_alpha(Design_pipe, 'END.            ');
            pipe_dump_CRLF(Design_pipe);
          end;

        { start the list over }

        release_sepcomp_list(sepcomp_list_root);
        repeat 
          drawing_ok := read_linker_commands;
        until (sepcomp_list_root <> NIL) or not drawing_ok;
        
        if drawing_ok then
          if design_specified_in_command_line then
            drawing_ok := pipe_rewrite_file(CmpTmp_pipe, 
                                            TEMP_PIPE, temp_file_name);

        { last_sepcomp = NIL implies that these are the first commands
          obtained from the linker, in which case we now pass it the
          property attributes.  }

        if drawing_ok and (last_sepcomp = NIL) then
          if design_specified_in_command_line then
            output_et_directives(CmpTmp_pipe)
          else output_et_directives(Design_pipe);

        if not drawing_ok then 
          while (sepcomp_list_root <> NIL) do
            release_sepcomp_list(sepcomp_list_root);

        last_sepcomp := NIL;
        current_sepcomp := sepcomp_list_root;
      end;

    if (current_sepcomp = NIL) then { we are done }
      begin
        drawing_ok := FALSE;
        { pipe is closed elsewhere }
      end
    else
      begin
        specified_context := current_sepcomp^.context;
        drawing_ok := init_drawing(current_sepcomp^.command,
                                   current_sepcomp^.drawing,
                                   skip_drawing);
        if drawing_ok then
          begin
            if (current_sepcomp^.command = COMPILE_ROOT) and
               not skip_drawing then
              if design_specified_in_command_line then
                output_et_root_directives(CmpTmp_pipe)
              else output_et_root_directives(Design_pipe);

            if primitive_directive_found(root_macro_name) then
              begin
                { Implement (un-pathnamed) PRIMITIVE directive }
                if not er_isprim(module_being_compiled) then
                  begin
                    if design_specified_in_command_line then                
                      write_bogus_expansion_file(CmpTmp_pipe, 1)
                    else
		      write_bogus_expansion_file(Design_pipe, 1);
                    skip_drawing := TRUE;
                  end;
              end;
          end
        else
          { inform linker of error(s) }
          if design_specified_in_command_line then
            write_bogus_expansion_file(CmpTmp_pipe, UNDEFINED_PAGE_NUMBER)
          else write_bogus_expansion_file(Design_pipe, UNDEFINED_PAGE_NUMBER);
      end;

  until (drawing_ok and not skip_drawing) or (current_sepcomp = NIL);
    
  init_next_sepcomp := drawing_ok;

  if debug_20 then writeln(outfile, 'exit init_next_sepcomp ', drawing_ok);
end { init_next_sepcomp } ;

