procedure output_error_documentation(var f: textfile);
  { output documentation for the error messages encountered during the
    compile.  Read the documentation from the error messages manual. }
  var
    error_num: error_range;            { current error number being processed }
    last_error_found: longint;         { last error encountered in file }
    end_of_file: boolean;              { TRUE when end of file has been found }
    max_err: -1..MAX_ERROR_NUM;        { largest error number in set }


  procedure read_line;
    { read lines from the input file until current error number is found,
      then copy all lines down to next error. }
    const
      END_O_FILE = 2;
    var
      found: boolean;          { TRUE when we find the start of error_num }
      done: boolean;           { TRUE when done writing error_num }


    function find_error(var num: longint): boolean;
      { check the line for an error number and return it.  Return TRUE
        if there was a number. }
      const
        LAST_SHARP_COLUMN = 16;  { if # not in first 16, it's not there }
      var
        found: boolean;           { TRUE when we find non-white space }
	done: boolean;            { TRUE when done processing number }
	i: string_range;          { index into instring }
	instring_length: string_range;  { length of instring }
    begin
      find_error := FALSE;  found := FALSE;

      { go to the '#' preceding the error number }
      
      i := 0;
      while (i < min(ord(instring^[0]), LAST_SHARP_COLUMN)) and not found do
	begin
	  i := i + 1;
          found := (instring^[i] = '#');
	end;

      if found then
        begin
          { read and evaluate the number }
          instring_length := ord(instring^[0]);
          find_error := TRUE;  
          num := 0;  
	  if i = instring_length then done := TRUE
	  else
	    begin
	      i := i + 1;
	      done := not isdigit[instring^[i]];
	    end;
          while not done do
            begin
	      num := num*10 + ord(instring^[i])-ord('0');
	      if (i = instring_length) then done := TRUE
	      else
	        begin
		  i := i + 1;
		  done := not isdigit[instring^[i]];
		end;
	    end;
	end;

      if debug_24 then
        begin
	  if found then writeln(outfile, 'found ', num:1);
        end;

    end { find_error } ;


    function read_a_line(var f: textfile; line: xtring): boolean;
      { Read a line into the string.  Return TRUE if EOF }
      var
        i: string_range;      { index into the line }
	ch: char;             { current char }
    begin
      i := 0;
      read_a_line := eof(f);
      while not eoln(f) and not eof(f) do
        begin
          read(f, ch);  
          if i < MAX_STRING_LENGTH then
            begin  i := i + 1;  line^[i] := ch;  end
          else
            begin
              while not eof(f) and not eoln(f) do get(f);
            end;
        end;
      if not eof(f) then readln(f);
      line^[0] := chr(i);  
    end { read_a_line } ;


  begin { read_line }
    if debug_24 then writeln(outfile, 'read_line(',error_num:1,')');
    found := (not end_of_file and (last_error_found = error_num));
    while not end_of_file and not found do
      begin
#if !UNIX
        end_of_file := read_a_line(CmpStan, instring);           
#else
        end_of_file := (creadln(CmpStan, instring) = END_O_FILE); 
#endif
	if debug_24 and end_of_file then writeln(outfile, 'end of file');
        if not end_of_file then if find_error(last_error_found) then 
	  found := (last_error_found = error_num);
      end;
      
    if found then
      repeat
        write(f, ' ');  dump_string(f, instring);  writeln(f);
#if !UNIX
        end_of_file := read_a_line(CmpStan, instring);
#else
        end_of_file := (creadln(CmpStan, instring) = END_O_FILE); 
#endif
	if not end_of_file then done := find_error(last_error_found);
      until end_of_file or done;
    if debug_24 then writeln(outfile, 'exit read_line');
  end { read_line } ;


  procedure welcome;
    { display message to indicate start of the output of error documentation }
  begin
    writeln(monitor);
    writeln(monitor, ' Starting output of error documentation.');
    writeln(CmpLog);
    writeln(CmpLog, ' Starting output of error documentation.');
  end { welcome } ;


begin { output_error_documentation }
  if debug_20 then writeln(outfile, ' enter output_error_documentation');
  if errors_encountered <> [] then
    begin
      welcome;

      page(f);
      writeln(f);
      writeln(f,' Descriptions of the errors encountered during compilation');
      writeln(f,' ---------------------------------------------------------');
      writeln(f);

      if not reset_file(error_documentation_file, STANDARD_FILE) then
        begin
          error(138 { cannot open this file });
          error_dump_file_name(error_documentation_file);
        end

      else
        begin
          last_error_found := -1;  end_of_file := FALSE;
	  max_err := -1;
          for error_num := 0 to MAX_ERROR_NUM do
            if error_num IN errors_encountered then max_err := error_num;

          for error_num := 0 to max_err do
            if error_num IN errors_encountered then read_line;


          if not close_parse_file(STANDARD_FILE) then
            begin
              error(168 { cannot close file });
              error_dump_file_name(error_documentation_file);
            end;
        end;

      write(monitor, '   End of error help message output ');
      write(CmpLog, '   End of error help message output ');
      exec_time(last_elapsed_time, last_CPU_time, TRUE);
    end;
  if debug_20 then writeln(outfile, ' exit output_error_documentation');
end { output_error_documentation } ;


