procedure comperr;
  { perform comperr command.  Basic theory: There are 2 structures, a 
    drawing table and a module queue.  The drawing table contains all
    drawings which have so far been seen.  Each entry in the drawing table
    points to a table of modules (one per context so far seen for the
    drawing).  The queue links all of the modules which have been seen
    but not yet processed for children and list files.  When a modules is
    invoked within some expansion file, the table is checked for its
    presence.  If it is not there, it is entered in both structures.
    Processing of the queue continues until it is empty. }
  type
    how_to_process = (PROCESS_PRIMITIVES_ONLY, PROCESS_ANY_EXTENSION, PROCESS_ROOT);

  var
    queue: comperr_module_ptr;              { modules to be processed (ring) }
    drawing_table: avl_ptr;                 { drawings seen }
    free_drawing: comperr_drawing_ptr;      { a free record of this type }
    free_module: comperr_module_ptr;        { a free record of this type }
    id_dictionary: numbered_dictionary;     { id dict for expansion files }
    string_dictionary: numbered_dictionary; { str dict for expansion files }
    current_modl: comperr_module_ptr;       { current module }
    extension_to_process: how_to_process;   { whether or not to ignore
                                              non-primitives in non-root
					      modules }
    last_version_with_schema_read: macro_module_ptr;
  

  procedure error_dump_missing_page(modl: comperr_module_ptr; 
                                    version: macro_module_ptr;
				    page: integer; which: file_types);
    { dump appropriate error for a missing page }
  begin
    error(250 { No results for page });
    error_dump_indent(indent);
    error_dump_alpha('Page name=      ');
    error_dump_string(modl^.drawing^.macro_name);
    error_dump_char('.');
    error_dump_string(er_extension(version));
    error_dump_char('.');
    error_dump_integer(page);
    error_dump_CRLF;
    error_dump_context(modl^.context, 10 { size of 'Page name='});
    error_dump_indent(indent);
    error_dump_alpha('Missing page    ');
    if (which = EXPANSION_FILE) then error_dump_alpha(' expansion file.')
    else error_dump_alpha(' list file.     ');
    error_dump_CRLF;
  end { error_dump_missing_page } ;


  procedure dump_context(var f: textfile;  context: property_ptr);
    { dump the context on one line (no indentation nor CRLF) }
    var
      param: property_ptr;          { a parameter in context }
  begin
    if context = NIL then write(f, 'No parameters');
    param := context;
    while param <> NIL do
      begin
        if param^.name = SIZE_prop_name then
          dump_string(f, param^.text)
        else
          begin
            writealpha(f, param^.name^.name);
            writestring(f, param^.text);
          end;
        param := param^.next;
      end;
  end { dump_context } ;


  procedure dump_module(var f: textfile;  modl: comperr_module_ptr;
                        indentation: natural_number);
    { dump the module }
  begin
    if indentation > 0 then write(f, ' ':indentation);
    if modl = NIL then write(f, '<NIL>')
    else dump_context(f, modl^.context);
    writeln(f);
  end { dump_module } ;


  procedure dump_drawing(var f: textfile; drawing: comperr_drawing_ptr;
                         indentation: natural_number);
    { dump the drawing and its modules }
    var
      avl_module: avl_ptr;          { current module in drawing }
      module_stack: avl_ptr;        { stack for module tree traversal }
  begin
    if indentation > 0 then write(f, ' ':indentation);
    write(f, 'Drawing: ');
    if drawing = NIL then writeln(f, '<NIL>')
    else
      begin
        writestring(f, drawing^.macro_name);
	writeln(f);
        avl_module := avl_inorder_init(drawing^.modules, module_stack);
        if avl_module = NIL then
          writeln(f, ' ':(indentation + 4), 'No modules');
        while avl_module <> NIL do
          begin
            dump_module(f, avl_module^.object.cerr_module, indentation + 4);
            avl_module := avl_inorder(module_stack);
          end;
      end;
  end { dump_drawing } ;


  procedure dump_drawing_table(var f: textfile; indentation: natural_number);
    { dump the drawing table }
    var
      avl_drawing: avl_ptr;         { current drawing in table }
      drawing_stack: avl_ptr;       { stack for drawing tree traversal }
  begin
    if indentation > 0 then write(f, ' ':indentation);
    writeln(f, '------- Dump of the comperr drawing table ------------');
    avl_drawing := avl_inorder_init(drawing_table, drawing_stack);
    if avl_drawing = NIL then
      writeln(f, ' ':(indentation + 4), 'No drawings');
    while avl_drawing <> NIL do
      begin
        dump_drawing(f, avl_drawing^.object.cerr_drawing, indentation);
        avl_drawing := avl_inorder(drawing_stack);
      end;
  end { dump_drawing_table } ;


#include "comperrout.p"


  procedure new_drawing(var dwg: comperr_drawing_ptr);
    { return an initialized new drawing }
  begin
    new(dwg);
    with dwg^ do
      begin
        queue_entry := NIL;
        macro_name := nullstring;
        modules := NIL;
      end;
  end { new_drawing } ;


  procedure new_module(var modl: comperr_module_ptr);
    { return an initialized new module }
  begin
    new(modl);
    with modl^ do
      begin
        queue_next := NIL;
        queue_back := NIL;
        drawing := NIL;
        context := NIL;
        context_str := nullstring;
      end;
  end { new_module } ;


  procedure queue_module(modl: comperr_module_ptr);
    { If drawing is already represented in queue, then insert this module
      after its last queued module, otherwise insert at the rear of the
      queue. }
    var
      parent: comperr_module_ptr;  { last module of same drawing in queue }
  begin
    parent := modl^.drawing^.queue_entry;
    if queue = NIL then
      begin  { empty queue }
        queue := modl;
        modl^.queue_next := modl;
        modl^.queue_back := modl;
      end
    else if parent = NIL then
      begin  { queue to rear }
        modl^.queue_next := queue;
        modl^.queue_back := queue^.queue_back;
        queue^.queue_back := modl;
        modl^.queue_back^.queue_next := modl;
      end
    else
      begin  { insert after parent }
        modl^.queue_back := parent;
        modl^.queue_next := parent^.queue_next;
        modl^.queue_next^.queue_back := modl;
        parent^.queue_next := modl;
      end;
  end { queue_module } ;


  procedure dequeue_module(modl: comperr_module_ptr);
    { delete the module from the queue }
  begin
    if modl^.queue_next = modl then queue := NIL
    else
      begin
        if queue = modl then queue := modl^.queue_next;
        modl^.queue_next^.queue_back := modl^.queue_back;
        modl^.queue_back^.queue_next := modl^.queue_next;
      end;
    modl^.queue_next := NIL;
    modl^.queue_back := NIL;
    if modl^.drawing^.queue_entry = modl then
      modl^.drawing^.queue_entry := NIL;
  end { dequeue_module } ;


  function enter_drawing(macro_name: xtring): comperr_drawing_ptr;
    { Return pointer to entry for drawing in comperr drawing table }
    var
      drawing_object: avl_object_ptr;  { avl xface }
      entry: avl_ptr;                  { entry for drawing in avl tree }
      drawing: comperr_drawing_ptr;    { return value }
  begin
{   drawing_object.tag := AVL_COMPERR_DRAWING;              }(*AVL*)

    if debug_8 then
      begin
        write(Outfile, 'Enter_drawing(');
	writestring(Outfile, macro_name);
	writeln(Outfile, ')');
      end;

    free_drawing^.macro_name := macro_name;
    drawing_object.cerr_drawing := free_drawing;
    entry := avl_insert(drawing_object, drawing_table, AVL_COMPERR_DRAWING);
    drawing := entry^.object.cerr_drawing;
    if drawing = free_drawing then
      begin
        if debug_8 then writeln(Outfile, '  -- a new entry');
        new_drawing(free_drawing);  { for next time }
      end
    else
      begin
        if debug_8 then writeln(Outfile, '  -- already in table');
      end;
    enter_drawing := drawing;
  end { enter_drawing } ;


  procedure insert_module(drawing: comperr_drawing_ptr; 
                          var context: property_ptr);
    { If module is not already in the table, enter it and enqueue it.

      The table records all modules already seen and the queue lists those
      yet to be processed.

      NOTE: the context list is either freed or made part of the module
      table by this routine, so it is returned NIL.  }
    var
      module_object: avl_object_ptr;   { avl xface }
      entry: avl_ptr;                  { entry for drawing in avl tree }
      modl: comperr_module_ptr;        { ditto }
  begin
  { module_object.tag := AVL_COMPERR_MODULE;                }(*AVL*)

    if debug_8 then
      begin
        write(Outfile, 'insert_module(');
	if drawing = NIL then write(Outfile, '<NIL>')
	else writestring(Outfile, drawing^.macro_name);
	write(Outfile, ', ');
	dump_context(Outfile, context);
	writeln(Outfile, ')');
      end;

    free_module^.context := context;
    module_object.cerr_module := free_module;
    entry :=
      avl_insert(module_object, drawing^.modules, AVL_COMPERR_MODULE);
    if entry^.object.cerr_module = free_module then
      begin
        { free_module is now part of the table }
	modl := entry^.object.cerr_module;

        if debug_8 then writeln(Outfile, '  -- a new module');
        free_module^.drawing := drawing;
        free_module^.context_str := context_string(context);
        new_module(free_module);  { for next time }
	context := NIL;
      end
    else
      begin
        { context was already in the table }
	modl := NIL;

        if debug_8 then writeln(Outfile, '  -- already in table');
        free_module^.context := NIL;
        release_entire_property_list(context);
      end;

    if modl <> NIL then queue_module(modl);
  end { insert_module } ;


  function parse_for_modules: boolean;
    { parse the expansion file up to and including its invoke list, entering
      the invokes into the module table.  Insert into the queue those that
      have not yet been seen in an expansion file and marked then as seen. 
      Return TRUE iff the page contains anomalies that meet or exceed the
      specified severity level. }
    label
      90; { for return }
    var
      string_prefix: symbols;           { numbered string prefix ($) }
      id_prefix: symbols;               { numbered id prefix (#) }
      drawing_name: numbered_token_ptr; { macro name of module }
      context: property_ptr;            { context of module just read }
      tail: property_ptr;               { tail of context list }
      prop_name: numbered_token_ptr;    { current parameter name }
      prop_val: numbered_token_ptr;     { current parameter value }
      dump_list: boolean;               { TRUE iff we want to dump list file }


    procedure emit_syntax_error;
      { issue error 248 with appropriate data and clean up for return }
    begin
      release_numbered_dictionary(id_dictionary);
      release_numbered_dictionary(string_dictionary);
      release_entire_property_list(context);

      error(248 { incorrect expansion file syntax });
      error_dump_file_name(current_file_name);
    end { emit_syntax_error } ;


    procedure skip_to_semi;
      { skip to SEMI or ENDOFDATASY.  This is faster than skip([SEMI]) as
        it does not invole set construction. }
    begin
      while (sy <> SEMI) and (sy <> ENDOFDATASY) do insymbol;
    end { skip_to_semi } ;


    function parse_heading: boolean;
      { parse the module heading and mark the page for output if appropriate.
        Return TRUE iff no syntax errors found. }
      label
        90; { for return }
      var
        name: numbered_token_ptr;  { current heading property name }
        val: numbered_token_ptr;   { current heading property value }


      function not_0(val: xtring): boolean;
        { Return TRUE iff val is anything but "0" }
      begin
        not_0 := TRUE;
        if ord(val^[0]) = 1 then if val^[1] = '0' then not_0 := FALSE;
      end { not_0 } ;


    begin { parse_heading }
      if debug_8 then write(Outfile, 'Parsing expansion file heading ');
      parse_heading := TRUE;

      while sy = id_prefix do
        begin
          name := parse_numbered_token(id_dictionary);
          if name = NIL then 
            begin
              emit_syntax_error;  parse_heading := FALSE;
              goto 90 { return } ;
            end;

          if sy <> string_prefix then 
            begin
              emit_syntax_error;  parse_heading := FALSE;
              goto 90 { return } ;
            end;
          val := parse_numbered_token(string_dictionary);
          if val = NIL then 
            begin
              emit_syntax_error;  parse_heading := FALSE;
              goto 90 { return } ;
            end;

          if name^.identifier_p = ERROR_prop_name then
            dump_list := dump_list or not_0(val^.string_p)
          else if (name^.identifier_p = OVERSIGHT_prop_name) and 
                  (ord(specified_severity) <= ord(OVERSIGHT_SEVERITY)) then
            dump_list := dump_list or not_0(val^.string_p)
          else if (name^.identifier_p = WARNING_prop_name) and
                  (ord(specified_severity) <= ord(WARNING_SEVERITY)) then
            dump_list := dump_list or not_0(val^.string_p);

          if debug_8 then
	    begin
	      write(Outfile, '  ');
	      writealpha(Outfile, name^.identifier_p^.name);
	      write(Outfile, '=');
	      writestring(Outfile, val^.string_p);
	      if dump_list then writeln(Outfile, '; (DUMP)')
	                    else writeln(Outfile, '; (no dump)');
	    end;

          if dump_list then skip_to_semi;
        end;

      if sy <> SEMI then 
        begin
          emit_syntax_error;  parse_heading := FALSE;
        end
      else insymbol;

      if debug_8 then writeln(Outfile, 'Done with heading');
    90:
    end { parse_heading } ;


    procedure append_property(name: name_ptr;  val: xtring);
      { append the property to context }
    begin
      if context = NIL then
        begin
          new_property(context);  tail := context;
        end
      else
        begin
          new_property(tail^.next);  tail := tail^.next;
        end;
      tail^.name := name;  tail^.text := val;
    end { append_property } ;


  begin { parse_for_modules }
    dump_list := specified_severity = NO_SEVERITY;

    if get_file_type <> EXPANSION_FILE then
      begin
        error(248 { incorrect expansion file syntax });
        error_dump_file_name(current_file_name);
        goto 90 { return }
      end;

    id_prefix := SHARP;
    string_prefix := DOLLAR;
    context := NIL;

    parse_numbered_dictionary(string_dictionary);
    parse_numbered_dictionary(id_dictionary);
    if not parse_heading then goto 90 { return } ;
    skip_to_semi;  if sy = SEMI then insymbol; { skip context }
    skip_to_semi;  if sy = SEMI then insymbol; { skip drawing_properties }
    skip_to_semi;  if sy = SEMI then insymbol; { skip dependencies }

    if (sy <> SEMI) and (sy <> AMPERSAND) then
      begin
        emit_syntax_error;
        goto 90 { return }
      end;

    { parse the modules }

    while sy = AMPERSAND do
      begin
        insymbol;

        if sy <> CONSTANT { module number } then
          begin
            emit_syntax_error;  goto 90 { return } ;
          end;
	insymbol;
        
        if sy <> string_prefix then
          begin
            emit_syntax_error;  goto 90 { return } ;
          end;
        drawing_name := parse_numbered_token(string_dictionary);
        if drawing_name = NIL then
          begin
            emit_syntax_error;  goto 90 { return } ;
          end;

        if sy = CONSTANT { size value } then
          begin
            append_property(SIZE_prop_name, number_to_string(const_val));
            insymbol;
          end;

        while (sy <> AMPERSAND) and (sy <> SEMI) do
          begin
            if sy <> id_prefix then
              begin
                emit_syntax_error;  goto 90 { return } ;
              end;
            prop_name := parse_numbered_token(id_dictionary);
            if prop_name = NIL then
              begin
                emit_syntax_error;  goto 90 { return } ;
              end;

            if sy <> string_prefix then
              begin
                emit_syntax_error;  goto 90 { return } ;
              end;
            prop_val := parse_numbered_token(string_dictionary);
            if prop_val = NIL then
              begin
                emit_syntax_error;  goto 90 { return } ;
              end;
            append_property(prop_name^.identifier_p, prop_val^.string_p);
          end;

        insert_module(enter_drawing(drawing_name^.string_p), context);
        { context is now NIL }
      end;
    release_numbered_dictionary(id_dictionary);
    release_numbered_dictionary(string_dictionary);

  90:
    parse_for_modules := dump_list;
  end { parse_for_modules } ;


  procedure process_pages(modl: comperr_module_ptr;  
                          version: macro_module_ptr;
			  how: how_to_process);
    { For each page, read the expansion file for sons and barf up the
      list file if necessary. }
    var
      page: page_range;


    function read_sons: boolean;
      { Read first part of expansion file for current page and enter all
        modules invoked into the drawing table and queue as needed.  
	Return TRUE iff list file needs dumping. }
    begin
      read_sons := FALSE;
      parse_SCALDconstants := FALSE;
      allowed_key_words := expansion_keysys;
      current_file_name := nullstring;
      copy_string(er_filename(version, ord(EXPANSION_FILE), page, 
		              modl^.context_str),
		  current_file_name);
						  
      if current_file_name <> nullstring then
	if not open_a_file(current_file_name, STANDARD_FILE) then
	  error_dump_missing_page(modl, version, page, EXPANSION_FILE)
	else
	  begin
	    read_sons := parse_for_modules;
	    if close_parse_file(STANDARD_FILE) then ;
	  end;
      release_string(current_file_name);
      allowed_key_words := [];
    end { read_sons } ;
  
  
    procedure cat_list_file(which: file_types);
      { Cat the current page list file to CMPLST }
      var
	name: xtring;
    begin
       name := nullstring;
       copy_string(er_filename(version, ord(which), page, modl^.context_str),
		   name);
       if (name <> nullstring) then cat_to_cmplst(modl, version, page, name);
       release_string(name);
    end { cat_list_file } ;


  begin { process_pages }
    { NOTE: primitives need to be read to determine whether
      or not their list files should be dumped }

    if (how <> PROCESS_PRIMITIVES_ONLY) or (er_isprim(version)) then
      begin
	if (how = PROCESS_ROOT) then
	  begin
	    page := 0;
	    linker_list_heading;
	    cat_list_file(LINKER_LIST_FILE);
	    page_list_heading;
	  end;

	page := er_page(version);
	while (page <> 0) do
	  begin
	    if read_sons then cat_list_file(LISTING_FILE);
	    page := er_page(version);
	  end;
      end;
    cleanup_root_macro_def;
  end { process_pages } ;


  procedure process_module(modl: comperr_module_ptr;  how: how_to_process);
    { Check module for simple primitive -- if it qualifies, chuck it and
      insert the parameterless version for processing.  Also chuck it if
      a real extension has not been chosen.  Otherwise process all of its
      pages }
    var
      version: macro_module_ptr;
  begin
    root_macro_name := modl^.drawing^.macro_name;
    specified_context := modl^.context;
    init_context(modl^.context);
    if (how = PROCESS_PRIMITIVES_ONLY) then
      version := select_module(modl^.drawing^.macro_name, modl^.context_str, 1)
    else
      version := select_module(modl^.drawing^.macro_name, modl^.context_str, 0);

    if version = NIL then
      begin
        modl := NIL; { error already reported }
        if (how = PROCESS_ROOT) then error(217 { fatal });
      end
    else if er_version(version) = 0 then modl := NIL  { non-graphical model }
    else if er_force_to_prim(version) then modl := NIL
    else
      begin
	if er_issimple(version) <> 0 then
	  begin
	    if (not er_samedwg(last_version_with_schema_read, 
			       version)) then
	      begin
		er_release(last_version_with_schema_read);
		last_version_with_schema_read := version;
		write_schema_file(schema_of_drawing_being_compiled);
		    { if it's open, close it }
		if read_schema_file(
		  modl^.drawing^.macro_name, version, 
		  schema_of_drawing_being_compiled) then ;
	      end;

	    if is_context_free_primitive(version) then
	      begin
		if debug_8 then writeln('  -- forced to NULL context');
		specified_context := NIL;
		insert_module(modl^.drawing, specified_context);
		modl := NIL;  
	      end;
	  end;
      end;

    if modl <> NIL then process_pages(modl, version, how);

    if version <> last_version_with_schema_read then er_release(version);
    specified_context := NIL;
    cleanup_root_macro_def;
  end { process_module } ;


  procedure list_processing_welcome;
  begin
    writeln(Monitor);
    writeln(Monitor, ' Retrieving listing files.');
    writeln(CmpLog);
    writeln(CmpLog, ' Retrieving listing files.');  { for consistency }
  end { list_processing_welcome } ;


  procedure list_processing_done;
  begin
    write(Monitor, '   Listing files retrieved ');
    write(CmpLog, '   Listing files retrieved ');  { for consistency }
    exec_time(last_elapsed_time, last_CPU_time, TRUE);
  end { list_processing_done } ;


  procedure init;
    { initialize comperr variables }
  begin
    queue := NIL;
    drawing_table := NIL;
    new_drawing(free_drawing);   { keep 1 free one until needed }
    new_module(free_module);     { keep 1 free one until needed }
    init_numbered_dictionary(string_dictionary, STRING_NUMBER);
    init_numbered_dictionary(id_dictionary, IDENTIFIER_NUMBER);

    write(CmpLst, COMPERR_WELCOME);
    dump_string(CmpLst, vversion);
    writeln(CmpLst);
    writeln(CmpLst, COPYRIGHT_NOTICE);
    writeln(CmpLst);
  end { init } ;


begin { comperr }
  last_version_with_schema_read := NIL;
  if fatal_errors * errors_encountered <> [] then
    error(217 { we can't go on! })
  else if not open_comperr_list then
    begin
      error(217 { no way });
    end
  else
    begin
      PrintCmpLst := TRUE;
      init;

      if debugging then report_compiler_directives(Outfile);

      setup_global_textmacros_from_ds_module;
      {init_global_text_macros;}           { initialize reserved text macros }
      setup_property_attributes_from_ds_module;
      {read_property_attributes;}

      if fatal_errors * errors_encountered <> [] then
        error(217 { we can't go on! })
      else 
        if (root_macro_name = nullstring) then
            error(148 { no ROOT_DRAWING was specified })
        else 
          begin
            list_processing_welcome;
	    if single_level_compile then 
	      extension_to_process := PROCESS_PRIMITIVES_ONLY
	    else
	      extension_to_process := PROCESS_ANY_EXTENSION;

            { Do root module as a special case }

            specified_context := NIL;  { for now }
            insert_module(enter_drawing(root_macro_name), specified_context);

            current_modl := queue;
            if (current_modl = NIL) then error(217 { fatal -- no root })
	    else
	      begin
		process_module(current_modl, PROCESS_ROOT);
		dequeue_module(current_modl);
	      end;

            { Do other modules, as encountered }

            current_modl := queue;
            while current_modl <> NIL do
              begin
                process_module(current_modl, extension_to_process);
                dequeue_module(current_modl);
                current_modl := queue;
              end;

	    write_schema_file(schema_of_drawing_being_compiled);
		{ if it's open, close it }

            list_processing_done;
          end;

      if printdirectory_ok then dump_drawing_table(outfile, 0);

      if display_error_doc then
        output_error_documentation(CmpLst);

      comperr_goodbye;
      close_comperr_list;  PrintCmpLst := FALSE;
    end;
end { comperr } ;


