{ ------- signal syntax configuration routines ------- }


procedure read_and_check_configuration_file;
  { if configuration directives found, generate configuration file
    otherwise, read configuration from it.  }
  var
    error_occurred: boolean;         { TRUE if error occurred during parse }


  function determine_symbol(var sym: symbols; var ch: char;
                             var name: name_ptr): boolean;
    { determine the symbol, character, or ID from the input file and return
      them.  If an error occurs, return FALSE. }
    var
      specified_character: xtring;  { character specified for symbol }
      ok: boolean;                  { TRUE if all is ok }
  begin
    ok := FALSE;

    sym := NULLSY;  ch := chr(255);  name := null_name;

    create_a_string(specified_character, 1);

    if sy <> STRINGS then
      begin  error(33 { expected a string });  skip([SEMI]);  end

    else if ord(lex_string^[0]) > 1 then
      begin  error(102 { only 1 char });  skip([SEMI]);  end

    else
      begin
        if ord(lex_string^[0]) = 0 then
          begin  sym := NULLSY;  ch := chr(255);  ok := TRUE;  end
        else
          begin
            specified_character^[1] := lex_string^[1];
            ch := specified_character^[1];

            parse_string(specified_character, PARSE_SEPARATELY);
            sym := sy;
            if sym = IDENT then name := id.name;
            pop_parsed_string(specified_character);

            if sym in forbidden_symbols then error(99 { illegal symbol })
                                        else ok := TRUE;
          end;

        insymbol;    { eat the specification }
      end;

    if not ok then error_occurred := TRUE;

    release_string(specified_character);

    determine_symbol := ok;
  end { determine_symbol } ;


  procedure process_syntax;
    { process the signal syntax specification.  The form is:

            <specifier> <specifier> ...      

      where <specifier> is one of the following:

            <negation>
            <assertion>
            <name>
            <subscript>

      <name> and <subscript> MUST always appear. }
    var
      syntax_index: 0..SYNTAX_TABLE_SIZE;   { index into syntax descriptor }
      new_table: signal_syntax_table_type;  { new syntax table }
      encountered: syntax_specifier_set;    { specifiers found }
      specifier: syntax_specifier_type;     { current specifier }
      no_errors: boolean;                   { TRUE if no errors found }


    function get_specifier: syntax_specifier_type;
      { get the syntax specifier and return }
      var
        i: syntax_specifier_type;    { index into specifier table }
        found: boolean;              { TRUE when specifier found }
    begin
      if sy = LESSTHAN then insymbol else error(10 { expected < });

      i := succ(FIRST_SYNTAX_SPECIFIER);  found := FALSE;
      while (i <= pred(LAST_SYNTAX_SPECIFIER)) and not found do
        if syntax_specifier_names[i] = id.name then found := TRUE
        else i := succ(i);

      if not found then
        begin
          error(67 );
	  error_dump_line('Unknown signal syntax specification         ');
          no_errors := FALSE;
          i := NULL_SPECIFIER;
          error_occurred := TRUE;
        end;

      get_specifier := i;
      insymbol;

      if sy = GREATERTHAN then insymbol else error(11 { expected > });
    end { get_specifier } ;


  begin { process_syntax }
    for syntax_index := 1 to SYNTAX_TABLE_SIZE do
      new_table[syntax_index] := null_specifier;

    { read in all the syntax elements }

    syntax_index := 0;  encountered := [];  no_errors := TRUE;
    repeat
      if syntax_index < SYNTAX_TABLE_SIZE then
        begin
          specifier := get_specifier;

          if specifier IN encountered then
            begin  
	       error(67 );
	       error_dump_line('Signal syntax element found twice           ');
	       no_errors := FALSE;  
	    end
          else
            begin
              encountered := encountered + [specifier];
              syntax_index := succ(syntax_index);
              new_table[syntax_index] := specifier;
            end;
        end;
    until (sy <> LESSTHAN) or (syntax_index >= SYNTAX_TABLE_SIZE);

    { general properties must be at the end of the signal }

    new_table[syntax_index] := PROPERTY_SPECIFIER;
    
    if not (NAME_SPECIFIER IN encountered) then
      begin  
        error(67 );
	error_dump_line('Every syntax MUST have a name portion       ');
	no_errors := FALSE;
      end;

    if not (SUBSCRIPT_SPECIFIER IN encountered) then
      begin
        error(67 );
	error_dump_line('Every syntax MUST have a subscript          ');
	no_errors := FALSE;
      end;

    { check the form for temporary restrictions }

    syntax_index := 1;
    if not (new_table[syntax_index] IN [ASSERTION_SPECIFIER,
                                        NEGATION_SPECIFIER,
                                        NAME_SPECIFIER]) then
      begin  
	error(67 );
	error_dump_line('Illegal form for signal syntax              ');
	no_errors := FALSE;  
      end
    else
      begin
        if new_table[syntax_index] <> NAME_SPECIFIER then
          syntax_index := syntax_index + 1;

        if new_table[syntax_index] <> NAME_SPECIFIER then
          begin  
	    error(67 );
	    error_dump_line('Illegal form for signal syntax              ');
	    no_errors := FALSE;  
	  end;
      end;

    if no_errors then signal_syntax_table := new_table
                 else error_occurred := TRUE;
  end { process_SYNTAX_directive } ;


  procedure process_low_assertion;
    { process the low assertion specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := LOW_ASSERTED_SY;
            sym := LOW_ASSERTED_SY;
          end;

        signal_is_asserted_low_symbol := sym;
        signal_is_asserted_low_char := ch;
      end;
  end { process_low_assertion } ;


  procedure process_high_assertion;
    { process the high assertion specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := HIGH_ASSERTED_SY;
            sym := HIGH_ASSERTED_SY;
          end;

        signal_is_asserted_high_symbol := sym;
        signal_is_asserted_high_char := ch;
      end;
  end { process_high_assertion } ;


  procedure process_negation;
    { process the negation specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym = IDENT then
          begin
            name^.kind := name^.kind + [KEY_WORD];
            name^.sy := NEGATION_SY;
            sym := NEGATION_SY;
          end;

        signal_negation_symbol := sym;
        signal_negation_char := ch;
      end;
  end { process_negation } ;


  procedure process_name_prefix;
    { process the name prefix specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym IN expression_symbols + [IDENT] then
          begin  error(99 { illegal symbol });  error_occurred := TRUE;  end
        else
          begin
            name_property_prefix_symbol := sym;
            name_property_prefix_char := ch;
          end;
      end;
  end { process_name_prefix } ;


  procedure process_general_prefix;
    { process the general prefix specification }
    var
      sym: symbols;       { symbol specified }
      ch: char;           { character specified }
      name: name_ptr;     { identifier specified }
  begin
    if determine_symbol(sym, ch, name) then
      begin
        if sym IN expression_symbols + [IDENT] then
          begin  error(99 { illegal symbol });  error_occurred := TRUE;  end
        else
          begin
            general_property_prefix_symbol := sym;
            general_property_prefix_char := ch;
          end;
      end;
  end { process_general_prefix } ;


  procedure process_bit_ordering;
    { process the bit ordering configuration specification }
  begin
    if sy <> IDENT then
      begin
        error(1 { expected IDENT });
        skip([SEMI]);
        error_occurred := TRUE;
        end
    else
      begin    
        if id.name = L_to_R_specifier then left_to_right := TRUE
        else if id.name = R_to_L_specifier then left_to_right := FALSE
        else
          begin  error(52 { invalid });  error_occurred := TRUE;  end;

        insymbol;    { eat the specification }
      end;
  end { process_bit_ordering } ;


  procedure process_subrange;
    { process the subrange configuration specification }
    var
      specifier: xtring;          { character(s) specified }
      sym: symbols;               { symbol specified }
  begin
    specifier := nullstring;
    if sy <> STRINGS then
      begin
        error(33 { expected a string });
        skip([SEMI]);
        error_occurred := TRUE;
      end
    else
      begin
        specifier := lex_string;

        parse_string(specifier, PARSE_SEPARATELY);
        sym := sy;
        pop_parsed_string(specifier);

        if sym = DOTDOTSY then
          begin
            subrangesy := DOTDOTSY;
            fieldsy := COLON;
            insymbol;    { eat the .. }
          end
        else if sym = COLON then
          begin
            subrangesy := COLON;
            fieldsy := COLONCOLONSY;
            insymbol;    { eat the : }
          end
        else
          begin
            error(67 { not permitted });
 	    error_dump_line('Subrange symbol must be .. or :             ');
            skip([SEMI]);
            error_occurred := TRUE;
          end;
      end;
  end { process_subrange } ;

    
  procedure parse_configuration_specification;
    { parse the signal syntax configuration file }
    var
      config_spec: configure_types;    { index into table of configure types }
      found: boolean;                  { TRUE if configuration found }
      specifier: name_ptr;             { configuration specifier from file }
  begin
    while sy = IDENT do
      begin
        specifier := id.name;

        config_spec := succ(FIRST_CONFIGURE_SPECIFIER);  found := FALSE;
        while (config_spec < LAST_CONFIGURE_SPECIFIER) and not found do
          if configure_specifiers[config_spec] = specifier then found := TRUE
          else config_spec := succ(config_spec);

        if not found then
          if (specifier^.name = 'SIGNAL_SYNTAX   ') or
             (specifier^.name = 'SYNTAX          ') then
            begin
              insymbol;     { eat the configuration name }
              if sy = EQUAL then insymbol;
              process_syntax;
            end
          else
            begin
              error(69 { unknown signal syntax specification });
              skip([SEMI]);
              error_occurred := TRUE;
            end
        else
          begin
            insymbol;    { eat the configuration name }
            if sy = EQUAL then insymbol;

            case config_spec of
              CONFIGURE_SUBRANGE:
                  process_subrange;

              CONFIGURE_BIT_ORDERING:
                  process_bit_ordering;

              CONFIGURE_LOW_ASSERTED:
                  process_low_assertion;

              CONFIGURE_HIGH_ASSERTED:
                  process_high_assertion;

              CONFIGURE_NEGATION:
                  process_negation;

              CONFIGURE_NAME_PREFIX:
                  process_name_prefix;

              CONFIGURE_GENERAL_PREFIX:
                  process_general_prefix;

              CONFIGURE_CONCATENATION:
                  skip([SEMI]);
            end { case } ;
          end;

        if sy = SEMI then insymbol else error(12 { expected ; });
      end;
  end { parse_configuration_specification } ;


begin { read_and_check_configuration_file }
  allowed_key_words := directory_keysys;

  if not open_a_file(configuration_file, STANDARD_FILE) then
    begin
      error(207 { cannot open the configuration file });
      error_dump_file_name(configuration_file);
    end

  else
    begin
      if get_file_type <> CONFIGURATION_SPEC then
        begin
          error(67 { problem with syntax spec });
	  error_dump_line('Illegal file type for configuration file    ');
          error_dump_file_name(configuration_file);
          error_occurred := TRUE;
        end
      else
        begin
          error_occurred := FALSE;

          parse_configuration_specification;

          if sy = ENDSY then insymbol else error(40 { expected END });
          if sy <> PERIOD then error(37 { expected . });
        end;


      if error_occurred then
        begin
          error(157 { error occurred in this file });
          error_dump_file_name(configuration_file);
        end;

      if not close_parse_file(STANDARD_FILE) then
        begin
          error(168 { cannot close the file });
          error_dump_file_name(configuration_file);
        end;
    end;
  allowed_key_words := [];
end { read_and_check_configuration_file } ;


(**)     { ------- read in the linker directives ------- }


procedure read_linker_directives_file;
  { read in the file containing linker directives and root macro
    name specification. }
  var
    directive: directive_type;          { directive being parsed }
    directives_encountered: 
                        directive_set;  { all directives encountered }


  function find_directive(name: name_ptr; var directive: directive_type):
                                                                    boolean;
    { search the directive table for the specified name and return its
      directive type if found. }
    var
      search: directive_type;       { directive search index }
      found: boolean;               { TRUE when directive found }
  begin
    search := succ(FIRST_DIRECTIVE);  found := FALSE;
    while (search < LAST_DIRECTIVE) and not found do
      if name = compiler_directive[search] then found := TRUE
      else search := succ(search);

    if ((search IN debug_directives) and not found_debug_password) or
       ((search = CONTEXT_directive) and not found_root_context_password) then
      begin  found := FALSE;  search := LAST_DIRECTIVE;  end;

    find_directive := found;  directive := search;
  end { find_directive } ;


  function ON_or_OFF(specifier: name_ptr; default: boolean): boolean;
    { if specifier = 'ON' then return TRUE else if specifier = 'OFF' then
      return FALSE, else return the default. }
  begin
    if specifier = ON_specifier then ON_or_OFF := TRUE
    else if specifier = OFF_specifier then ON_or_OFF := FALSE
    else
      begin  error(52 { invalid specifier });  ON_or_OFF := default;  end;
  end { ON_or_OFF } ;




  procedure process_OUTPUT_file_list;
    { process the output file list and set the control flags }
    var
      file_name: output_file_names;      { name of output file to create }
      done: boolean;                     { if all output controls read }
      found: boolean;                    { TRUE if file name found }
  begin 
    if not (OUTPUT_DIRECTIVE in directives_encountered) then
      files_to_generate := [];

    done := FALSE;
    while not done and (sy <> SEMI) do
      begin
        if sy <> IDENT then
          begin  error(1 { expected ident });  skip([SEMI,COMMA,IDENT]);  end
        else
          begin
            file_name := succ(first_file_name);  found := FALSE;
	    if id.name <> null_name then
	      while (file_name < last_file_name) and not found do
		if output_file[file_name] = id.name then found := TRUE
		else file_name := succ(file_name);

            if not found then
              begin
                error(97 { unknown output file name });
                file_name := first_file_name;
              end;

            files_to_generate := files_to_generate + [file_name];
            insymbol;
          end;
        if sy = COMMA then insymbol else done := TRUE;
      end;
  end { process_OUTPUT_file_list } ;


  procedure get_file_name_list(var list: file_list_ptr);
    { read a list of files from the input.  Check to see if the file has
      already been specified.  If not, add it to the list.  Otherwise,
      generate an error. }
    var
      done: boolean;                { TRUE when done with directive }
      file_name: xtring;            { name of file being specified }
      current_file: file_list_ptr;  { current file in the list }
      found: boolean;               { TRUE if entry found in library }
  begin    
    done := FALSE;
    repeat
      if sy <> STRINGS then
        begin
          error(33 { expected a string });  skip([SEMI,COMMA,STRINGS]);
        end
      else
        begin
          file_name := lex_string;
          current_file := list;  found := FALSE;
          while (current_file <> NIL) and not found do
            if current_file^.file_name = file_name then found := TRUE
            else current_file := current_file^.next;

          if found then
            error(38 { file was already specified })
          else
            begin
              new_file_list(list);
              list^.file_name := file_name;
            end;

          insymbol;     { eat the file name }
        end;

      if sy = COMMA then insymbol else done := TRUE;
    until done;
  end { get_file_name_list } ;


  procedure process_PASSWORD_directive;
    { process the SECRET SCALD PASSWORDS!!!!! }
  begin
    if sy = EQUAL then insymbol;     { allow space or equal }
    if sy <> IDENT then error(1 { expected ident })
    else
      if id.name^.name = CONFIGURE_PASSWORD then
        { do nothing }
      else if id.name^.name = DEBUG_PASSWORD then
        begin
          found_debug_password := TRUE;
          found_root_context_password := TRUE;
	end
      else if id.name^.name = CONTEXT_PASSWORD then
        found_root_context_password := TRUE;

    insymbol;    { eat the symbol }
  end { process_PASSWORD_directive } ;


  procedure process_COMPILE_directive(compile_type: name_ptr);
    { process the COMPILE directive and set up the extension for compile }
    var
      extension,                { standard file name extension suffix }
      temp: alpha;              { temp for extension creation }
      temp_length: id_range;    { length of the temp identifier }
      new_extension: name_ptr;  { extension for allowed macros }
      id_size,                  { length of the identifier }
      i: id_range;              { index into the temp }
  begin
    if not TYPE_specified_in_command_line then
      begin
        if (compile_type^.name = PART_prefix) then
          begin
            error(216 { PART not allowed; LOGIC assumed });
            compile_type := enter_name(LOGIC_prefix);
          end;

        extension := compile_type^.name;
        id_size := alpha_length(extension);

        temp := EXTENSION_SUFFIX;
        temp_length := alpha_length(temp);

        if id_size > (ID_LENGTH - temp_length) then
          error(65 { identifier too long })
        else
          begin
            for i := 1 to temp_length do extension[id_size+i] := temp[i];
            new_extension := enter_name(extension);

            if new_extension = PRIM_connectivity then
              error(66 { not allowed })
            else
              begin
                temp := DIRECTORY_SUFFIX;
                temp_length := alpha_length(temp);

                if id_size > (ID_LENGTH - temp_length) then
                  error(65 { too long })
                else
                  begin
                    { init to end for append }
                    extension := compile_type^.name;
                    expansion_connectivity := new_extension;

                    for i := 1 to temp_length do
                      extension[id_size+i] := temp[i];

                    file_type_list[ALLOWED_DIRECTORY] :=
                                                        enter_name(extension);
                  end;
              end;
          end;
	  specified_compile_type := compile_type;
      end;
  end { process_COMPILE_directive } ;


  procedure process_filters(filter_flag: longint);
    { process filters for the properties specified.  Add to a list for
      processing after the attributes files.  PASS means to pass and FILTER
      means to filter. }
    var
      temp_name: xtring;     { property name in a string }
      done: boolean;         { TRUE when done processing input list }
  begin
    temp_name := nullstring;
    done := FALSE;
    while (sy = IDENT) and not done do
      begin
        copy_to_string(id.name^.name, temp_name);
        init_property(temp_name, property_enum, filter_flag);
	property_enum := property_enum + 1;

        insymbol;     { eat the identifier }

        if sy = COMMA then insymbol else done := TRUE;
      end;
    release_string(temp_name);
  end { process_filters } ;



  procedure process_command_line_arguments;
    { get parameters from a command line.  The line can contain:
      
      [root_drawing] [compile_type] [context_name] }
    var
      arg: xtring;                 { current argument from command line }
      temp_string: xtring;         { string for conversion to internal }
      len: string_range;           { length of the string parameter }
      ch: char;                    { character from input parameter }
      temp_alpha: alpha;           { temp storage for the compile type }


    procedure process_context_name_arg;
      { process the current arg as context name }
      var
        i: string_range;             { index into the strings }
    begin
      len := ord(arg^[0]);
      if len > 0 then
        begin
          create_a_string(temp_string, len);
          for i := 1 to len do
            begin
              ch := arg^[i];
              if ch IN lower_case then
                temp_string^[i] := chr(ord(ch) - ord('a') + ord('A'))
              else
                temp_string^[i] := ch;
            end;
          context_name := enter_and_release_string(temp_string);

          CONTEXT_specified_in_command_line := TRUE;
        end;
    end { process_context_name_arg } ;


    procedure process_compile_arg;
      { process the current arg as compile type }
      var
        i: string_range;             { index into the strings }
    begin
      len := ord(arg^[0]);
      if len > 0 then
        begin
          temp_alpha := NULL_ALPHA;
          if len > ID_LENGTH then len := ID_LENGTH;
          for i := 1 to len do
            begin
              ch := arg^[i];
              if ch IN lower_case then
                temp_alpha[i] := chr(ord(ch) - ord('a') + ord('A'))
              else
                temp_alpha[i] := ch;
            end;

          process_COMPILE_directive(enter_name(temp_alpha));
          TYPE_specified_in_command_line := TRUE;
        end;
    end { process_compile_arg } ;


    procedure process_root_arg;
      { process the current arg as root drawing name }
      var
        i: string_range;             { index into the strings }
    begin
      len := ord(arg^[0]);
      if len > 0 then
        begin
          create_a_string(temp_string, len);
          for i := 1 to len do
            begin
              ch := arg^[i];
              if ch IN lower_case then
                temp_string^[i] := chr(ord(ch) - ord('a') + ord('A'))
              else
                temp_string^[i] := ch;
            end;

          root_macro_name := enter_and_release_string(temp_string);
          ROOT_specified_in_command_line := TRUE;
        end;
    end { process_root_arg } ;


  begin { process_command_line_arguments }
    init_cli_arg_structures;

    { number of arguments includes the program name }


    if sargc > 1 then     { there is a ROOT drawing parameter }
      begin
        sargv(1, arg);
        process_root_arg;
      end;

    if sargc > 2 then   { there is a COMPILE type parameter }
      begin
        sargv(2, arg);
        process_compile_arg;
      end;

    if sargc > 3 then   { there is a CONTEXT_NAME arg }
      begin
        sargv(3, arg);
        process_context_name_arg;
      end;

  end { process_command_line_arguments } ;


begin { read_linker_directives_file }

  writeln(monitor);  writeln(monitor, ' Reading the Compiler directives.');

  process_command_line_arguments;


  allowed_key_words := directives_keysys;
  if not open_a_file(nullstring, DIRECTIVES_FILE) then
    error(135 { file does not exist! })
  else
    begin
      directives_encountered := [];


      while sy = IDENT do
        begin
          if not find_directive(id.name, directive) then
            begin
              error(51 { unknown directive });
              skip([SEMI] + directives_keysys);
            end

          else if (directive IN directives_encountered) and
                  (directive IN one_time_directives) then
            begin
              error(45 { directive specified more than once });
              skip([SEMI]);
            end

          else
            begin
              if directive IN file_name_directives then
                upper_case_strings := FALSE;

              insymbol;     { eat the directive symbol }

              if sy = EQUAL then insymbol;    { allow an equal }
              case directive of
                ALLOW_PART_NAME_DIRECTIVE:
                       skip([SEMI]);

                AMUSING_MESSAGES_DIRECTIVE:
                       skip([SEMI]);

                BUBBLECHECK_DIRECTIVE:
                       skip([SEMI]);

                CARDINAL_TAP_DIRECTIVE:
                       skip([SEMI]);

                COMMAND_DIRECTIVE:
                       skip([SEMI]);

                COMPILE_DIRECTIVE:
                    begin
                      if sy <> ident then error(1 { expected ident })
                      else process_COMPILE_directive(id.name);
                      insymbol;
                    end;

                CONFIG_FILE_DIRECTIVE:
                    begin
                      if sy <> STRINGS then error(33 { expected name })
                      else configuration_file := lex_string;
                      insymbol;
                    end;

                CONST_BUBBLE_CHK_DIRECTIVE:
                       skip([SEMI]);

                CONTEXT_DIRECTIVE:
                    begin
                      if sy = STRINGS then
                        if CONTEXT_specified_in_command_line then
                          begin   end
                        else
                          context_name := lex_string
                      else
                        error(33);
                      insymbol;
                    end;

                DEBUG_DIRECTIVE:
                       skip([SEMI]);

                DEBUG_AT_PATH_DIRECTIVE:
                       skip([SEMI]);

                DEBUG_CONTROL_DIRECTIVE:
                       skip([SEMI]);

                DECLARE_BODIES_DIRECTIVE:
                       skip([SEMI]);

                DEFAULT_FILTER_DIRECTIVE:
		    begin
                      if sy = IDENT then
			default_filter_value := 
			    ON_or_OFF(id.name,default_filter_value)
                      else
			error(1 { expected ident });
                      insymbol;
		    end;

                DEFAULT_L_OR_G_DIRECTIVE:
                       skip([SEMI]);

                DIRECTORY_DIRECTIVE:
                       skip([SEMI]);

                ERROR_HELP_DIRECTIVE:
                       skip([SEMI]);

                EVACUATE_DIRECTIVE:
                       skip([SEMI]);

		EXPANSION_RULES_DIRECTIVE:
                       skip([SEMI]);

                FILTER_PROPERTY_DIRECTIVE:
                    process_filters(FILTER_THE_PROPERTY);

                HIERARCHICAL_NWC_DIRECTIVE:
                       skip([SEMI]);

                LIBRARY_DIRECTIVE:
                       skip([SEMI]);

                LOCALLY_GLOBAL_DIRECTIVE:
                       skip([SEMI]);

                MASTER_LIBRARY_DIRECTIVE:
                       skip([SEMI]);

                MAX_ERROR_DIRECTIVE:
                       skip([SEMI]);

                MERGE_DRAWINGS_DIRECTIVE:
                       skip([SEMI]);

                NESTED_TM_PARAMS_DIRECTIVE:
                       skip([SEMI]);

                NET_PROCESSING_DIRECTIVE:
                       skip([SEMI]);

                OUTPUT_DIRECTIVE:
                    process_OUTPUT_file_list;

                OVERSIGHT_DIRECTIVE:
                       skip([SEMI]);

                PAGE_SYNONYM_DIRECTIVE:
		    begin
                      if sy = IDENT then
			page_synonym := 
			    ON_or_OFF(id.name,page_synonym)
                      else
			error(1 { expected ident });
                      insymbol;
		    end;

                PASS_PROPERTY_DIRECTIVE:
                    process_filters(PASS_THE_PROPERTY);

                PASSWORD_DIRECTIVE:
                       process_PASSWORD_directive;

                PERMIT_NO_ASSERTION_DIRECTIVE:
                       skip([SEMI]);

                PICK_DIRECTIVE:
                       skip([SEMI]);

                PRIMITIVE_DIRECTIVE:
                       skip([SEMI]);

                PROPERTY_DIRECTIVE:
                    get_file_name_list(property_file);

                PRINT_WIDTH_DIRECTIVE:
                       skip([SEMI]);

                READ_ALL_UDIRS_DIRECTIVE:
                       skip([SEMI]);

                REPORT_DIRECTIVE:
                       skip([SEMI]);

		REPORT_UNKASSERT_DIRECTIVE:
                       skip([SEMI]);

                ROOT_DIRECTIVE:
                    begin
                      if sy = STRINGS then
                        if ROOT_specified_in_command_line then
                          { ignore this one }
                        else
                          root_macro_name := lex_string
                      else
                        error(33 { expected name string });
                      insymbol;
                    end;

                SEPARATE_COMPILE_DIRECTIVE:
                       skip([SEMI]);

                SHADOW_ROOT_DIRECTIVE:
                       skip([SEMI]);

                SHAREABLE_DIRECTIVE:
		        skip([SEMI]);

                SINGLE_LEVEL_DIRECTIVE:
		    begin
                      if sy = IDENT then
			single_level_compile := 
			    ON_or_OFF(id.name,single_level_compile)
                      else
			error(1 { expected ident });
                      insymbol;
		    end;

                SUPPRESS_DIRECTIVE:
                       skip([SEMI]);

                TEXT_MACRO_DIRECTIVE:
                       skip([SEMI]);

                TOKENIZE_PARAMS_DIRECTIVE:
                       skip([SEMI]);

                WARN_DIRECTIVE:
                       skip([SEMI]);

                OTHERWISE
                    begin
                      error(51 { unknown directive });
                      skip([SEMI]+directives_keysys);
                    end;
              end { case } ;

              directives_encountered := directives_encountered + [directive];

              upper_case_strings := TRUE;
            end { else } ;

          if sy = SEMI then insymbol else error(12 { expected ; });
        end { while } ;

      if sy = ENDSY then insymbol else error(40 { expected END });
      if sy <> PERIOD then error(37 { expected . });

      if not close_parse_file(DIRECTIVES_FILE) then
        begin
          error(168 { can't close it });
          error_dump_alpha_file_name('INFILE          ');
	end;

      read_and_check_configuration_file;

    end { else not file open } ;

  if not found_root_context_password then context_name := nullstring;

  allowed_key_words := [];
  if not (OUTPUT_DIRECTIVE in directives_encountered) then
  begin
    PrintCmpExp := TRUE;
    PrintCmpSyn := TRUE;
  end;
    
  write(monitor, '   Compiler directives read ');
  exec_time(last_elapsed_time, last_CPU_time, TRUE);

end { read_linker_directives_file } ;    
    
