#ifndef STRINGTH
#define STRINGTH
// A string is a string table entry -- there is only one string table


#include <string.h>
#include <stream.h>
#include "basictypes.h"
#include "avlbase.h"
#include "gavl.h"
#include "avl.h"


extern char *non_ZERO_int_assigned_to_string; // an error message


gavl_define(str,string_entry,charP); 
class string_entry {
    friend class string;
    friend class string_union_member;
#if COUNT_STRING_REFS
    int references;
#endif
    gavl_inst(str,string_entry,charP);
    string_entry(const char*);
    ~string_entry();
};
gavl_inline(str,string_entry,charP);


class string {
    string_entry *s;
    friend class string_union_member;
public:
    string() { s = 0; }
    string(const char*);
#if COUNT_STRING_REFS
    string(const string &st) { if (s = st.s) s->references++; }

    ~string() { if (s && !(--s->references)) delete s; }

    const string& operator=(const string&);
#endif

    // Careful -- type casting to char * does NOT increment the reference
    // count.  'char *'s obtained from strings that later vanish may end
    // up pointing to garbage.  This is meant to be used for passing
    // strings to system calls that take a char *, use it, and return 
    // without copying it.  The << (output) operator is provided via 
    // type conversion to char *.  The >> (input) operator is not defined
    // here -- its optimum semantics can vary from program to program.
    operator const_charP() { return s ? (char*)(s->str) : 0; } 

    int cmp(const string& x)
        { return s == x.s ? 0 : s ? x.s ? strcmp(s->str, x.s->str) : 1 : -1; }
    int cmp(const char* c)
        { return s ? c ? strcmp(s->str, c) : 1 : c ? -1 : 0; }
    int ptr_cmp(const string& x)
	{ return ::cmp(s,x.s); }  // For building arbitrarily ordered tables

    char operator[](unsigned i) { return ((char*)(s->str))[i]; }

    operator!() { return !s; }
    operator==(const string& x) { return s == x.s; }
    operator!=(const string& x) { return s != x.s; }

    operator!=(const char* c) { return cmp(c) != 0; }
    operator==(const char* c) { return cmp(c) == 0; }

    friend operator!=(const char* c, const string& s) { return s != c; }
    friend operator==(const char* c, const string& s) { return s == c; }

    operator<(const char* c) { return cmp(c) < 0; }
    operator<=(const char* c) { return cmp(c) <= 0; }
    operator>(const char* c) { return cmp(c) > 0; }
    operator>=(const char* c) { return cmp(c) >= 0; }

    friend operator<(const char* c, const string& s) { return s > c; }
    friend operator<=(const char* c, const string& s) { return s >= c; }
    friend operator>(const char* c, const string& s) { return s < c; }
    friend operator>=(const char* c, const string& s) { return s <= c; }

    operator<(const string& x) { return cmp(x) < 0; }
    operator<=(const string& x) { return cmp(x) <= 0; }
    operator>(const string& x) { return cmp(x) > 0; }
    operator>=(const string& x) { return cmp(x) >= 0; }

    // Dump entire string table to given ostream.  Value of "this" irrelevant.
    void dump(ostream& strm = cerr);
    
};


class string_union_member {
    // Permits string membership in a union (normally disallowed as strings have
    // constructors).  Note that this must be handled carefully if reference
    // counting is being done -- be sure to explictly destroy these when clobbering
    // them.
    string_entry* s;
public:
    void construct(const string& str)
    {
	s = str.s;
#ifdef COUNT_STRING_REFS
	if (s) s->references++
#endif
    }
    void destroy()
    {
#ifdef COUNT_STRING_REFS
	if (s && !(--s->references)) delete s;
#endif
	0; // Gets around AT+T 1.2 problem with the empty inline function
    }
    string operator string() 
    { 
	string foo; foo.s = s;
	return foo;
    }
};


inline ostream& operator<<(ostream& o, string s) { return o.operator<<(s); }

gavlkey_define(string);
#endif STRINGTH
