#ifndef SMALLSET_H
#define SMALLSET_H
#ifndef BASICTYPES_H
#include "basictypes.h"
#endif
/*
    smallset.h -- generic small set.  The domain is some enumerated type
    with less than sizeof(unsigned long)*8 elements whose values are in
    the range 0..sizeof(unsigned long)*8-1 (the default for such a type).  
    Class smallset is useful by itself for most applications.

    Operators for class smallset:

	Construction -- smallset foo(member); creates a set with element
	    member.  smallset foo; creates an empty set.
	Union -- | and |=  (or + and +=)
	Intersetion -- & and &= (or * and *=)
	Subset < and <=
	Superset > and >=
	Difference - and -=
	Membership [] ( foo[member] returns boolean value)

    Class gsmallset_define(largest_member) is a text macro that defines
    a class named gsmallset(largest_member) whose domain is constrained
    to be 0..largest_member (which must be a constant name or simple
    constant).  It provides range-checking on the constructor and the
    unary operator ~ (which yields the domain elements not in the set).

    Note: There is no code here to check explicitly that your domain
    size falls within legal limits -- it is assumed that the value of
    largest_member used to define a gsmallset is <= 
    (sizeof(unsigned long)*8-1).
*/


const SMALLSET_DOMAIN_SIZE = sizeof(unsigned long) * 8;

class smallset {
protected:
    unsigned long members;
public:
    smallset() { members = 0; }
    smallset(int member) 
        { members = 1 << (member % SMALLSET_DOMAIN_SIZE); }
    boolean operator[](int member)
	{ smallset s(member);  return (s.members & members) ? TRUE : FALSE; }

    smallset operator|(smallset s)
        { s.members |= members; return s; }
    smallset operator&(smallset s)
        { s.members &= members; return s; }

    smallset operator|=(smallset s)
        { members |= s.members; return *this; }
    smallset operator&=(smallset s)
        { members &= s.members; return *this; }

    smallset operator-(smallset s)
	{ s.members = members & (members ^ s.members); return s; }
    smallset operator-=(smallset s)
	{ members &= members ^ s.members;  return *this; }

    smallset operator+(smallset s) { return operator|(s); }
    smallset operator+=(smallset s) { return operator|=(s); }
    smallset operator*(smallset s) { return operator&(s); }
    smallset operator*=(smallset s) { return operator&=(s); }
};


#define gsmallset(largest_member) name2(smallset_,largest_member)
#define gsmallset_define(large) \
const name2(DOMAIN_,large) =\
    (unsigned long)(~0) >> (SMALLSET_DOMAIN_SIZE - (large+1));\
class gsmallset(large) : public smallset {\
public:\
    gsmallset(large)(int i) : (i)\
    { \
	if i > large fault("smallset range check failure");\
    }\
    operator~() \
    {\
	smallset s;  s.members = name2(DOMAIN_,large);\
        return (gsmallset(large))(s - *this);\
    }\
\
    /* The rest are just wrappers to type-cast returns from the base\
       class -- don't strain your eyes! */\
    gsmallset(large) operator|(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator|(s); }\
    gsmallset(large) operator&(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator&(s); }\
    gsmallset(large) operator|=(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator|=(s); }\
    gsmallset(large) operator&=(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator&=(s); }\
    gsmallset(large) operator-(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator-(s); }\
    gsmallset(large) operator-=(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator-=(s); }\
    gsmallset(large) operator+(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator+(s); }\
    gsmallset(large) operator+=(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator+=(s); }\
    gsmallset(large) operator*(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator*(s); }\
    gsmallset(large) operator*=(gsmallset(large) s)\
	{ return (gsmallset(large))smallset::operator*=(s); }\
};
#endif
