#ifndef gdring
#include <generic.h>
#include "basictypes.h"
#include "fault.h"
#include "giterator.h"
/*
    gdring.h -- generic doubly-linked rings.

    Use -- analagous to the use of the singly linked rings in gring.h,
    but with member function appropriate to doubly-linked rings.
    See gring.h for a full explanation of how this pattern of declarations
    works.

    Given these definitions:

    gdring_define(next,bar);

    class bar {
        // ...
	gdring_inst(next,bar);
        // ...
    };	

    class foo {
        //...
    public:
        // ...
	gdringroot(next,bar) bars;
        // ...
    };	
        
    You get the following member functions for foo->bars:

    // Insertion
        void insert_front(bar*);
        void insert_rear(bar*);
        void insert_before(bar* to_insert, bar* successor);
        void insert_after(bar* to_insert, bar* predecessor);

	void insert(bar*); // same as insert_front
	void append(bar*); // same as insert_rear
	void insert(bar*, bar* successor); // same as insert_before
	void append(bar*, bar* predecessor); // same as insert_after

    // Traversal
        bar* first();
        bar* last();
        bar* next(bar*); // returns successor
        bar* previous(bar*); // returns successor

    // Deletion
	void remove(bar*);
	bar* remove_first();  // returns the object removed
	bar* remove_last();   // returns the object removed

    // Rotation -- second element becomes first, first becomes last
    // (or in reverse)
	void rotate_forward();
	void rotate_backward();

*/


#define gdring(member,type) name3(ring_,type,member)
#define gdringroot(member,type) name3(ringroot_,type,member)
#define gdringiterator(member,type) GITERATOR(member,type)


#define gdring_inst(member,type)\
    friend gdringroot(member,type);  gdring(member,type) member


#define gdring_define_classes(member,type,specifier)\
\
class type;\
\
class gdring(member,type) {\
    friend class gdringroot(member,type);\
    type* forward;\
    type* back;\
    void clear() { forward = back = 0; }\
public:\
    gdring(member,type)() { clear(); }\
};\
\
class gdringroot(member,type) : public collection_base {\
    type* tailp;\
public:\
    /* note that clear() does not clean up members -- it just 0s the root */\
    void clear() { tailp = 0; }\
    gdringroot(member,type)() { clear(); }\
\
    specifier void insert_front(type* rec);\
    specifier void insert_rear(type* rec);\
    specifier void insert_after(type* rec, type* prev);\
    specifier void insert_before(type* rec, type* succ);\
    specifier void insert(type* rec); /* == insert_front */\
    specifier void insert(type* rec, type* prev); /* == insert_before */\
    specifier void append(type* rec); /* == insert_rear */\
    specifier void append(type* rec, type* succ); /* == insert_after */\
\
    specifier type* first();\
    specifier type* last();\
    specifier type* next(type* rec);\
    specifier type* previous(type* rec);\
\
    specifier void remove(type* toremove);\
    specifier type* remove_first();\
    specifier type* remove_last();\
\
    specifier void rotate_forward();\
    specifier void rotate_backward();\
};\
\
/* definition of generic iterator class */\
class gdringiterator(member,type) : public iterator_base {\
protected:\
    type* pos;\
public:\
    specifier gdringiterator(member,type)(const gdringroot(member,type)& r);\
    specifier void init();\
    const gdringroot(member,type)& collection() \
	{ return *(gdringroot(member,type)*)&iterator_base::collection(); }\
    specifier type* operator()(); \
}


#define gdring_define(member,type) gdring_define_classes(member,type,)


/*
    The following must come after the definition of class "type".
*/
#define gdring_functions(member,type,specifier)\
\
/* Insertion */\
specifier void gdringroot(member,type)::insert(type* rec)\
{\
    AS(!(rec->member.forward));\
    AS(!(rec->member.back));\
    if (!tailp) tailp = rec->member.forward = rec->member.back = rec;\
    else {\
	rec->member.forward = tailp->member.forward;\
	rec->member.back = tailp;\
	tailp->member.forward = rec;\
	rec->member.forward->member.back = rec;\
	}\
}\
specifier void gdringroot(member,type)::insert_front(type* rec)\
    { insert(rec); }\
specifier void gdringroot(member,type)::append(type* rec)\
    { insert(rec); tailp = tailp->member.forward; /* rotate */ }\
specifier void gdringroot(member,type)::append(type* rec, type* prev)\
{\
    AS(prev);  AS(rec);\
    AS(prev->member.forward);  AS(prev->member.back);\
    AS(!(rec->member.forward)); AS(!(rec->member.back));\
    rec->member.forward = prev->member.forward;\
    rec->member.back = prev;\
    prev->member.forward = rec;\
    rec->member.forward->member.back = rec;\
    if (tailp == prev) tailp = rec;\
}\
specifier void gdringroot(member,type)::insert_after(type* rec, type* prev)\
    { append(rec, prev); }\
specifier void gdringroot(member,type)::insert(type* rec, type* succ)\
{\
    AS(rec);  AS(succ);\
    AS(succ->member.back);  AS(succ->member.forward);\
    AS(!(rec->member.back)); AS(!(rec->member.forward));\
    rec->member.back = succ->member.back;\
    rec->member.forward = succ;\
    succ->member.back = rec;\
    rec->member.back->member.forward = rec;\
}\
specifier void gdringroot(member,type)::insert_before(type* rec, type* succ)\
    { insert(rec, succ); }\
\
/* Traversal */\
specifier type* gdringroot(member,type)::first()\
    { return tailp ? tailp->member.forward : 0; }\
specifier type* gdringroot(member,type)::last()\
    { return tailp; }\
specifier type* gdringroot(member,type)::next(type* rec)\
    { return rec == tailp ? 0 : rec->member.forward; }\
specifier type* gdringroot(member,type)::previous(type* rec)\
    { rec = rec->member.back;  return rec == tailp ? 0 : rec; }\
\
/* Removal */\
specifier void gdringroot(member,type)::remove(type* toremove)\
{\
    AS(tailp);  AS(toremove);\
    if (toremove == toremove->member.forward) clear();\
    else {\
	toremove->member.forward->member.back = toremove->member.back;\
	toremove->member.back->member.forward = toremove->member.forward;\
	if (tailp == toremove) tailp = toremove->member.back;\
    }\
    toremove->member.clear();\
}\
specifier type* gdringroot(member,type)::remove_first() \
    { type *r = first();  if (r) remove(r); return r; }\
specifier type* gdringroot(member,type)::remove_last() \
    { type *r = last();  if (r) remove(r); return r; }\
\
/* Rotation */\
specifier void gdringroot(member,type)::rotate_forward()\
    { if (tailp) tailp = tailp->member.forward; }\
specifier void gdringroot(member,type)::rotate_backward()\
    { if (tailp) tailp = tailp->member.back; }\
\
/* generic iterator */\
specifier gdringiterator(member,type)::\
gdringiterator(member,type)(const gdringroot(member,type)& r) : (r)\
    { pos = 0; }\
specifier void gdringiterator(member,type)::init() \
    { pos = 0; }\
specifier type* gdringiterator(member,type)::operator()() \
{\
    type* val = pos ? collection().next(pos) : collection().first();\
    if (val) pos = val;\
    return val;\
}


#define gdring_inline(member,type) gdring_functions(member,type,inline)



/* ring traversal -- same as gring_walk in gring.h -- can use either */
#define gdring_walk(root,lvalue)\
    for ((lvalue) = (root).first(); (lvalue); (lvalue)=(root).next(lvalue))
#endif
