// Generic AVL (balanced binary) tree insertion and deletion.  Given
// a (changeble) tree root, and element to insert (delete) and the
// path to the point of insertion (location of element to be deleted),
// the operation can be completed without any regard for the key field.
// Therefore, insertions and deletions of trees of any type can be 
// performed by this one pair of procedures.  See avl.h for the
// declarations of the standard kinds of trees.
//
// Basic algorithms adapted from the ValidLINKER data structure macro
// library originally by Bob Floyd.  It attributes the algorithms to
// Knuth, Searching and Sorting, pp 455-469.
//
// Bill Hunsicker.  June 1987.


#include "avlbase.h"


// Error message used by gavlkey(keytype)::operator=(...) (gavl.h)
char *already_in_tree = "Assignment to key of avl class already in a tree";


class avlpath_itr {
    friend avlroot_base;

    avlpath *path;
    int *current;
    int *marked;

    avlpath_itr()
        { fault("avlpath_itr -- need path to iterate"); }
    avlpath_itr(avlpath& s)
        { path = &s; current = (marked = path->path) - 1; }
    int first()
	{ return (current = path->path) <= path->top ? *current : 0; }
    int next()
        { return ++current < path->top ? *current : 0; }
    int mark()
	{ current = marked; return current < path->top ? *current : 0; }
    void set_mark()
	{ marked = current; }
};


void avlroot_base::insert(avlbase *item, avlpath& path)
{
    if (item->in_tree()) fault("avlbase::insert: Element already in a tree");
    AS(!item->left());
    AS(!item->right());
    ASSERT(((int)item & ~AVL_BALANCE_BITS) == (int)item);

    register avlbase *locate = root;
    if (!locate) {
        root = item;
        item->set_balance(AVL_EVEN);
        return;
    }

    avlbase *rebalances_father = 0;
    avlpath_itr itr(path);
    int comparison = itr.first();
    avlbase* rebalance=locate; // to become LOWEST element not balanced
    itr.set_mark();
    while (1) {
        avlbase *son;
        if (comparison > 0) {
	    son = locate->right();
	    if (!son) {
		locate->set_right(item);
		break;
	    }
	}
        else {
            if (!comparison) fault("avlbase::insert: already in tree");
	    son = locate->left();
	    if (!son) {
		locate->set_left(item);
		break;
	    }
	}
	comparison = itr.next();  // Must be here for itr.set_mark below
	if (son->balance() != AVL_EVEN) {
	    rebalances_father = locate;
	    rebalance=son;
	    itr.set_mark();
	}
	locate=son;
    }

    item->set_balance(AVL_EVEN);

    avlbase *rebalances_son;
    int side_which_became_deeper_at_rebalance;
    if ((comparison = itr.mark() /* compare locate with rebalance) */) > 0) {
        rebalances_son = locate = rebalance->right();
        side_which_became_deeper_at_rebalance = AVL_RIGHT;
    }
    else {
        AS(comparison < 0);
        rebalances_son = locate = rebalance->left();
        side_which_became_deeper_at_rebalance = AVL_LEFT;
    }

    while (locate != item) {
        // adjust balance weights from rebalances_son to item.
        if ((comparison = itr.next()) > 0) {
	    locate->set_balance(AVL_RIGHT);
	    locate = locate->right();
	}
        else {
	    AS(comparison < 0);
	    locate->set_balance(AVL_LEFT);
	    locate=locate->left();
	}
    }

    if (rebalance->balance() == AVL_EVEN) // tree grown higher
        rebalance->set_balance(side_which_became_deeper_at_rebalance);
    else {
        avlbase *new_root;
        if (rebalance->balance() == side_which_became_deeper_at_rebalance) {
	    // tree out of balance
	    if (rebalances_son->balance() == side_which_became_deeper_at_rebalance) {
	        // single rotation
	        new_root = rebalances_son;
	        if (side_which_became_deeper_at_rebalance == AVL_LEFT) {
		    rebalance->set_left(rebalances_son->right());
		    rebalances_son->set_right(rebalance);
		}
	        else {
		    rebalance->set_right(rebalances_son->left());
		    rebalances_son->set_left(rebalance);
		}
	        rebalance->set_balance(AVL_EVEN);
		rebalances_son->set_balance(AVL_EVEN);
	    }
	    else { 
	        // double rotation
	        if (side_which_became_deeper_at_rebalance == AVL_LEFT) {
		    new_root = rebalances_son->right();
		    rebalances_son->set_right(new_root->left());
		    new_root->set_left(rebalances_son);
		    rebalance->set_left(new_root->right());
		    new_root->set_right(rebalance);
		}
	        else {
		    new_root = rebalances_son->left();
		    rebalances_son->set_left(new_root->right());
		    new_root->set_right(rebalances_son);
		    rebalance->set_right(new_root->left());
		    new_root->set_left(rebalance);
		}

	        if (new_root->balance() == AVL_EVEN) {
		    rebalance->set_balance(AVL_EVEN);
		    rebalances_son->set_balance(AVL_EVEN);
		}
	        else {
		    if (new_root->balance() == side_which_became_deeper_at_rebalance) {
		        if (side_which_became_deeper_at_rebalance == AVL_RIGHT)
		            rebalance->set_balance(AVL_LEFT);
			else rebalance->set_balance(AVL_RIGHT);
			rebalances_son->set_balance(AVL_EVEN);
		    }
		    else {
		        rebalance->set_balance(AVL_EVEN);
		        rebalances_son->set_balance(side_which_became_deeper_at_rebalance);
		    }
		}
		new_root->set_balance(AVL_EVEN);
	    }
	    if (!rebalances_father) root = new_root;
	    else
		if (rebalance == rebalances_father->right())
		    rebalances_father->set_right(new_root);
	        else rebalances_father->set_left(new_root);
	}
        else rebalance->set_balance(AVL_EVEN); // tree more balanced
    }
}


static struct pickstack {
    avlbase *ptr;
    avl_balance which_way;
};


void
avlroot_base::pick(avlbase *item, avlpath& path)
{
    avlbase *locate = root;
    int depth = 0;
    pickstack pickstack[sizeof(char *)*8];
    avlbase *deletes_father = 0;
    avlpath_itr itr(path);
    for(int comparison = itr.first(); comparison; comparison = itr.next()) {
        if (!locate) fault("avlbase::pick: element not found");
        pickstack[depth].ptr = locate;
        deletes_father=locate;
	if (comparison > 0) {
	    locate = locate->right();
	    pickstack[depth++].which_way = AVL_RIGHT;
	 }
	else {
	    locate = locate->left();
	    pickstack[depth++].which_way = AVL_LEFT;
	}
    }
    if (locate != item) fault("avlbase::pick: element not found");

    // locate new_root
    avlbase *new_root;

    if (item->right()) {
	if (item->left()) { 
	    // find left-most key, new_root, in right branch
	    int replacement_depth = depth++;
	    new_root = item->right();
	    if (new_root->left()) {
		avlbase *left_father;
		while (new_root->left()) {
		    pickstack[depth].ptr = new_root;
		    pickstack[depth].which_way = AVL_LEFT;  depth++;
		    left_father = new_root;
		    new_root = new_root->left();
		}
		new_root->set_left(item->left());
		left_father->set_left(new_root->right());
		new_root->set_right(item->right());
	    }
	    else new_root->set_left(item->left());

	    new_root->set_balance(item->balance());
	    pickstack[replacement_depth].ptr = new_root;
	    pickstack[replacement_depth].which_way=AVL_RIGHT; // replaced item
	}
	else new_root=item->right(); // item is deleted directly
    }
    else new_root = item->left(); // item is deleted directly*/

    if (deletes_father) // by-pass item to new_root
	if (deletes_father->right() == item)
	    deletes_father->set_right(new_root);
	else deletes_father->set_left(new_root);
    else root = new_root;

    item->init();

    while(depth >= 0) {
	// adjust balance factors
	if (--depth < 0) break;
	avlbase *rebalance = pickstack[depth].ptr;
	int became_shallower = pickstack[depth].which_way;
	if (rebalance->balance() == AVL_EVEN) {   
	    // tree less balanced
	    if (became_shallower == AVL_RIGHT) 
	        rebalance->set_balance(AVL_LEFT);
	    else rebalance->set_balance(AVL_RIGHT);
	    break;
	}

	if(rebalance->balance() == became_shallower)
	    rebalance->set_balance(AVL_EVEN); // tree more balanced
	else {
	    // tree out of balance
	    avlbase *rebalance_son = (became_shallower == AVL_RIGHT) ? 
	        rebalance->left() : rebalance->right();

	    if (rebalance_son->balance() == became_shallower) {
		// double rotation
		if (rebalance_son->balance() == AVL_RIGHT) {
		    new_root = rebalance_son->right();
		    rebalance_son->set_right(new_root->left());
		    new_root->set_left(rebalance_son);
		    rebalance->set_left(new_root->right());
		    new_root->set_right(rebalance);
		}
		else {
		    new_root = rebalance_son->left();
		    rebalance_son->set_left(new_root->right());
		    new_root->set_right(rebalance_son);
		    rebalance->set_right(new_root->left());
		    new_root->set_left(rebalance);
		}
		if (new_root->balance() == AVL_EVEN) {
		    rebalance->set_balance(AVL_EVEN);
		    rebalance_son->set_balance(AVL_EVEN);
		}
		else {
		    if (new_root->balance() == became_shallower) {
			if (became_shallower == AVL_RIGHT)
			    rebalance_son->set_balance(AVL_LEFT);
			else  rebalance_son->set_balance(AVL_RIGHT);
			    rebalance->set_balance(AVL_EVEN);
		    }
		    else {
			rebalance_son->set_balance(AVL_EVEN);
			rebalance->set_balance(became_shallower);
		    }
		}
		new_root->set_balance(AVL_EVEN);
	    }
	    else {
		// single rotation
		new_root = rebalance_son;
		if (became_shallower == AVL_RIGHT) {
		    rebalance->set_left(rebalance_son->right());
		    rebalance_son->set_right(rebalance);
		}
		else {
		    rebalance->set_right(rebalance_son->left());
		    rebalance_son->set_left(rebalance);
		}
		if (rebalance_son->balance() == AVL_EVEN) {
		    if(became_shallower == AVL_RIGHT)
			rebalance_son->set_balance(AVL_RIGHT);
		    else rebalance_son->set_balance(AVL_LEFT);
		    if (depth==0) root = new_root;
		    else {
			rebalance = pickstack[depth-1].ptr;
			if (pickstack[depth-1].which_way == AVL_RIGHT)
			    rebalance->set_right(new_root);
			else rebalance->set_left(new_root);
		    }
		    break;
		}
		else {
		    rebalance->set_balance(AVL_EVEN);
		    rebalance_son->set_balance(AVL_EVEN);
		}
	    }
    
	    if (depth==0) root = new_root;
	    else {
		rebalance = pickstack[depth-1].ptr;
		if(pickstack[depth-1].which_way == AVL_RIGHT)
		    rebalance->set_right(new_root);
		else rebalance->set_left(new_root);
	    }
	}
    }
}

void
avl_inorder::init()
{
    clear();
    avlbase* root = tree_root();
    while (root) {
        push(root);
	root = root->left();
    }
}


avlbase *
avl_inorder::operator()()
{
    avlbase *val = pop();
    avlbase *node = val ? val->right() : 0;
    while (node) {
        push(node);
	node = node->left();
    }
    return val;
}


void
avl_preorder::init()
{
    clear();
    push(tree_root());
}


avlbase *
avl_preorder::operator()()
{
    avlbase *val = pop();
    if (val) {
        avlbase *child;
        if (child = val->right()) push(child);
        if (child = val->left()) push(child);
    }
    return val;
}


unsigned long avlroot_base::num_entries()
{
    avl_inorder next(*this);
    unsigned long count = 0;
    while (next()) count++;
    return count;
}
