/* OrderedCltn.c -- implementation of abstract ordered collections

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2017
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-5363
	uucp: {decvax!}seismo!elsie!cecil!keith
	September, 1985

Function:
	
OrderedCltns are ordered by the sequence in which objects are added and removed
from them.  Object elements are accessible by index.

Modification History:

12-8-89		P. Hackett

1.  Fixed? bug in removeAtIndex. If endIndex == capacity(), size does not
    decrease.

5-1-89		R. Edwards

1.  Fixed bug in removeAtIndex which left element not set to nil

21-Feb-87	K. E. Gorlen

1.  Fixed bug in removeAtIndex that caused attempt to store nil pointer beyond
bounds of contents array.

07-Oct-86	S. M. Orlow

1.  Added typedef int (*pf_qsort_compare)(void*,void*);

2.  Changed declaration  compare_ob (obid&,obid&)  to compare_ob(obid,obid)

3.  Changed invocation of qsort to be compatible with declaration.
     
06-Oct-86	S. M. Orlow

1.  Added binary I/O constructor, storer, and reader functions

1-Oct-86	K. E. Gorlen

1. C++ Release 1.1: restore use of default argument in constructor.

11-Feb-86	K. E. Gorlen

1.  Modify Collection::asOrderedCltn to return OrderedCltn with minimum
capacity of CLTN_DEFAULT_CAPACITY.

Modification History:

5-Feb-86	K. E. Gorlen

1.  Modify remove to return a pointer to the removed object.

9-Jan-86	K. E. Gorlen

1.  Modify OrderedCltn(istream&,OrderedCltn&) to read collection capacity.

7-Jan-86	K. E. Gorlen

1.  Add OrderedCltn::species().

2.  Modify OrderedCltn::isEqual to use species().
*/

#include <libc.hxx>
#include "OrderedCltn.hxx"
#include "oopsIO.hxx"

#define	THIS	OrderedCltn
#define	BASE	SeqCltn
DEFINE_CLASS(OrderedCltn,SeqCltn,1,NULL,NULL);

extern const int OOPS_CLTNEMPTY,OOPS_OBNOTFOUND;

OrderedCltn::OrderedCltn(UNSIGNED size) : contents(size)
{
	endIndex = 0;
}

OrderedCltn::OrderedCltn(const OrderedCltn& c) : contents(c.contents)
{
	endIndex = c.endIndex;
}

void OrderedCltn::operator=(const OrderedCltn& c)
{
	endIndex = c.endIndex;
	contents = c.contents;
}

bool OrderedCltn::operator==(const OrderedCltn& a)
{
	if (endIndex != a.endIndex) return NO;
	else {
		register int i = endIndex;
		register obid* vv = &contents.elem(0);
		register obid* av = &a.at(0);
		while (i--) if (!((*vv++)->isEqual(**av++))) return NO;
	}
	return YES;
}

OrderedCltn OrderedCltn::operator&(const SeqCltn& cltn)
{
	OrderedCltn c(size()+cltn.size());
	addContentsTo(c);
	cltn.addContentsTo(c);
	return c;
}

void OrderedCltn::operator&=(const SeqCltn& cltn)
{
	cltn.addContentsTo(*this);
}

obid OrderedCltn::addAtIndex(int i, const Object& ob)
{
	if (endIndex == capacity()) contents.reSize(capacity()+CLTN_EXPANSION_INCREMENT);
	for (register int j=endIndex; j>i; j--) contents[j] = contents[j-1];
	contents[i] = &ob;
	endIndex++;
	return &ob;
}

obid OrderedCltn::removeAtIndex(int i)
{
	register obid obrem = contents[i];
	for (register int j=i+1; j<endIndex; j++) contents[j-1] = contents[j];
//	if (endIndex < capacity()) contents[--endIndex] = nil;
	if (endIndex <= capacity()) contents[--endIndex] = nil;
	return obrem;
}

obid OrderedCltn::add(const Object& ob)
{
	addAtIndex(endIndex,ob);
	return &ob;
}

obid OrderedCltn::addAfter(const Object& ob, const Object& newob)
{
	register int i = indexOf(ob);
	if (i < 0) errNotFound("addAfter",ob);
	return addAtIndex(i+1,newob);
}

obid OrderedCltn::addAllLast(const OrderedCltn& cltn)
{
	if (endIndex+cltn.size() >= capacity())
		contents.reSize(endIndex+cltn.size()+CLTN_EXPANSION_INCREMENT);
	for (register int i=0; i<cltn.size(); i++) contents[endIndex++] = cltn.contents[i];
	return &cltn;
}

obid OrderedCltn::addBefore(const Object& ob, const Object& newob)
{
	register int i = indexOf(ob);
	if (i < 0) errNotFound("addBefore",ob);
	return addAtIndex(i,newob);
}

Collection& OrderedCltn::addContentsTo(Collection& cltn)
{
	for (register int i=0; i<size(); i++) cltn.add(*contents[i]);
	return cltn;
}

obid OrderedCltn::addLast(const Object& ob) { return add(ob); }

obid OrderedCltn::after(const Object& ob)
{
	register int i=indexOf(ob);
	if (i<0) errNotFound("after",ob);
	if (++i == endIndex) return nil;
	return contents[i];
}

obid& OrderedCltn::at(int i)	{ return contents[i]; }

void OrderedCltn::atAllPut(const Object& ob)
{
	for (register int i=0; i<endIndex; i++) contents[i] = &ob;
}

obid OrderedCltn::before(const Object& ob)
{
	register int i = indexOf(ob);
	if (i < 0) errNotFound("before",ob);
	if (--i < 0) return nil;
	return contents[i];
}

UNSIGNED OrderedCltn::capacity()	{ return contents.capacity(); }

void OrderedCltn::deepenShallowCopy()
{
	BASE::deepenShallowCopy();
	contents.deepenShallowCopy();
}

obid OrderedCltn::first()
{
	if (endIndex==0) errEmpty("first");
	else return contents.elem(0);
}

UNSIGNED OrderedCltn::hash()
{
	register UNSIGNED h = endIndex;
	register int i = endIndex;
	register obid* vv = &contents.elem(0);
	while (i--) h^=(*vv++)->hash();
	return h;
}

int OrderedCltn::indexOf(const Object& ob)
{
	for (register int i=0; i<endIndex; i++)
		if (contents[i]->isEqual(ob)) return i;
	return -1;
}

int OrderedCltn::indexOfSubCollection(const SeqCltn& cltn, int start)
{
	int subsize = cltn.size();
	for (register int i=start; i<(endIndex-subsize); i++) {
		for (register int j=0; j<subsize; j++)
			if (!(contents[i+j]->isEqual(*cltn.at(j)))) goto next;
		return i;
next:;	}
	return -1;
}

bool OrderedCltn::isEmpty()	{ return endIndex==0; }
	
bool OrderedCltn::isEqual(const Object& a)
{
	return a.isSpecies(class_OrderedCltn) && *this==*(OrderedCltn*)&a;
}

const Class* OrderedCltn::species()	{ return &class_OrderedCltn; }

obid OrderedCltn::last()
{
	if (endIndex==0) errEmpty("last");
	else return contents.elem(endIndex-1);
}

UNSIGNED OrderedCltn::occurrencesOf(const Object& ob)
{
	register UNSIGNED n=0;
	for (register int i=0; i<endIndex; i++)
		if (contents[i]->isEqual(ob)) n++;
	return n;
}

void OrderedCltn::printOn(ostream& strm)
{
	strm << className() << "[\n";
	for (register int i=0; i<endIndex; i++) {
		if (i>0) strm << "\n";
		contents[i]->printOn(strm);
	}
	strm << "]\n";
}

obid OrderedCltn::remove(const Object& ob)
{
	for (register int i=0; i<endIndex; i++) {
		if (contents[i]->isEqual(ob)) {
			return removeAtIndex(i);
		}
	}
	return nil;
}

obid OrderedCltn::removeId(const Object& ob)
{
	for (register int i=0; i<endIndex; i++) {
		if (contents[i] == &ob) {
			removeAtIndex(i);
			return &ob;
		}
	}
	return nil;
}

obid OrderedCltn::removeLast()
{
	if (endIndex==0) errEmpty("removeLast");
	else return removeAtIndex(endIndex-1);
}

void OrderedCltn::reSize(UNSIGNED newSize)
{
	if (newSize > size()) contents.reSize(newSize);
}

void OrderedCltn::replaceFrom(int start, int stop, const SeqCltn& replacement, int startAt)
{
	register int j=startAt;
	for (register int i=start; i<=stop; i++,j++)
		contents[i] = replacement.at(j);
}

static int compare_ob(obid* a, obid* b) { return (*a)->compare(**b); }

UNSIGNED OrderedCltn::size()		{ return endIndex; }

void OrderedCltn::sort()
{
     qsort(&contents.elem(0),size(),sizeof(obid),compare_ob);
}

void OrderedCltn::errEmpty(const char* fn)
{
	setOOPSerror(OOPS_CLTNEMPTY,DEFAULT,this,className(),fn);
}

void OrderedCltn::errNotFound(const char* fn, const Object& ob)
{
	setOOPSerror(OOPS_OBNOTFOUND,DEFAULT,this,className(),fn,ob.className(),&ob);
}

OrderedCltn Collection::asOrderedCltn()
{
	OrderedCltn cltn(MAX(size(),CLTN_DEFAULT_CAPACITY));
	addContentsTo(cltn);
	return cltn;
}

static UNSIGNED orderedcltn_capacity;

OrderedCltn::OrderedCltn(istream& strm, OrderedCltn& where) :
	contents((strm >> orderedcltn_capacity, orderedcltn_capacity))
{
	this = &where;
	endIndex = 0;
	UNSIGNED n;
	strm >> n;		// read collection capacity 
	while (n--) add(*readFrom(strm));
}

OrderedCltn::OrderedCltn(fileDescTy& fd, OrderedCltn& where) :
	contents((readBin(fd,orderedcltn_capacity),orderedcltn_capacity))
{
	this = &where;
	endIndex = 0;
	UNSIGNED n;
	readBin(fd,n);
	while (n--) add(*readFrom(fd));
}
