/*
	AVL v1.2 - AVL-tree Functions

	File creation date:	December 22, 1981 @ 17:13 by BDR
	Last revision date:	July 22, 1982 @ 08:58 by BDR


Copyright (C) 1982 by Bruce Robertson


	This file contains several C functions for maintaining AVL-trees.  An
AVL-tree is a special kind of binary tree that eliminates the problem that
normal binary trees have when the items being placed in the tree appear in
alphabetical order.  A normal binary tree becomes effectively a linked list
in this case.  An AVL-tree however, stays "balanced" regardless of the order
the items appear in.

	These comments do not describe the basic algorithm; refer to the text
"Fundamentals of Data Structures" by Horowitz and Sahni, Computer Science
Press, Inc., page 442.

	The items to be placed in the AVL-trees implemented by these functions
may be any nul-terminated string.  The most common application is a symbol
table for a compiler or an assembler; other applications include sorting lines
of text files, or detecting duplicate lines or symbols.

	The following functions are defined in this file:

		avl_insert	Inserts a new symbol into an AVL-tree.
		avl_lookup	Searches an AVL-tree for a symbol.
		avl_inorder	Performs an in-order traversal of a tree.
		avl_preorder	Performs a pre-order traversal.
		avl_postorder	Performs a post-order traversal.


If you find any bugs in this code, please contact me at (714) 545-7339 during
normal business hours.  PLEASE do not modify and redistribute this code your-
self!  This code may be reused and distributed IN ITS ORIGINAL FORM and NOT FOR
PROFIT by anyone, free of charge or license.

				Bruce Robertson

*/

#include "a:bdscio.h"

/*
	Modification History
	====================

v1.2		July 22, 1982 @ 08:58 by BDR
	Fixed the bug in the balancing code of 'avl_insert()' that I put there
after changing all the -1s to 0xFFs, when doing v1.1.    ~sigh~  When will I
learn that '=' and '==' are different?????


v1.1		April 21, 1982 @ 09:27 by BDR
	Comments were rewritten, and the 'balance' field of the node structure
was changed from type 'int' to type 'short' to save one byte per node.  This
required that all balance factors of -1 be changed to 0xFF.  Balance factors
0 and 1 stayed the same.


v1.0		December 22, 1981 @ 17:13 by BDR
	Original file was created.

*/



/*
	Structure of AVL-tree nodes.

	The first four structure members of each AVL-tree node must be declared
in the manner described below.  All subsequent members may be anything you want
them to be, as the functions deal with pointers to structures, rather than the
structures themselves.  The first four fields of every node must be as follows:

*/

#define	BAL	short		/* type of balance factor */

struct	avlnode
    {
	struct	avlnode	*left;		/* pointer to left sub-tree */
	struct	avlnode	*right;		/* pointer to right sub-tree */
	BAL	balance;		/* node balance factor */
	char	*name;			/* pointer to the symbol string */
    };


#define	AVLNODE	struct	avlnode		/* I wish we had 'typedef'... */



/*
	Example.

	Below is an example of the use of these functions.  This sample program
accepts lines from the standard input and places them in an AVL-tree.  The
lines are then written to the standard output in alphabetical order (ASCII
collating sequence.)  In addition to the text of the line, the original
sequence number of the line is displayed.


struct	treenode
    {
	struct	treenode *left, *right;
	BAL	balance;
	char	*name;
	char	linenum;
    };

struct	treenode	tree;		/* root node of tree */
unsigned sequence;			/* line sequence number */

main()
{
	struct	treenode *node;		/* pointer to current node */
	char	line[1000];
	int	printnode();		/* function to print a node */

	_allocp = NULL;		/* we're going to use alloc() */
	sequence = 1;		/* initialize the line sequence count */

/*	The following initialization MUST be done under BDS C	*/

	tree.left = tree.right = tree.balance = 0;
	tree.name = "";
	while (getline(line))
	    {
		node = alloc(sizeof tree);	/* allocate a new node */
		node->name = alloc(strlen(line) + 1); /* get room for string */
		strcpy(node->name, line);
		node->linenum = sequence++;
		avl_insert(node, &tree);
	    }

/*	When we traverse the tree, we must use the left sub-tree.	*/

	avl_inorder(tree.left, printnode);
}


printnode(node)
struct	treenode	*node;
{
	printf("%5d:  %s\n", node->linenum, node->name);
}


getline(line)
char	*line;
{
	int	c;

	while ((c = getchar()) != EOF  &&  c != CPMEOF)
	    {
		if (c == '\n')
		    {
			*line = '\0';
			return 1;	/* end of line seen */
		    }
		*line++ = c;
	    }
	return 0;		/* end of file seen */
}

	>>>  End Of Example  <<<

*/



/*
	Function:	avl_insert()

	This function inserts a new node into an AVL-tree.  The AVL-tree must
contain at least one node; this may be the root node.  See the example above to
find out how to set up an empty AVL-tree.

	The node to insert must be created before calling this function.  The
node may be a static node declared at compile time, or may be allocated
dynamically from the free memory pool using alloc().  The 'left', 'right' and
'balance' fields of the node to insert need not be initialized.  The 'name'
field, however, should point to the string that is being inserted.

	If the string is found to be already in the AVL-tree, the pointer to
the node to insert is thrown away and a pointer to the existing node is
returned.  If the node is actually inserted, the original node pointer is
returned.


	Calling sequence:

		avl_insert(node, tree);

	where:

		node	points to the node to insert.  The 'name' field of this
			node must point to the string to insert.  All other
			AVL fields are not significant.

		tree	points to the first node of the AVL-tree.  This pointer
			must NEVER be nil.

*/

AVLNODE	*avl_insert(node, tree)
AVLNODE	*node,			/* pointer to the node to insert */
	*tree;			/* pointer to the tree to insert into */
{
	AVLNODE	*tptr,		/* pointer to current node in tree */
		*a,		/* ptr to last node with non-zero balance */
		*parenta,	/* pointer to parent node of 'a' */
		*q,		/* helps keep track of 'parenta' */
		*b, *c,		/* ptrs for performing unbalance rotation */
		*c_left,	/* pointer to left sub-tree of 'c' */
		*c_right;	/* pointer to right sub-tree of 'c' */
	int	n;		/* string comparison result (scratch) */
	BAL	d;		/* scratch variable for balance factor */


	if (node == NULL)
		return NULL;	/* return nil if nil node pointer */
	node -> left  =  node -> right  =  NULL;
	node -> balance = 0;
	if ((tptr = tree -> left) == NULL)
	    {
		/* it's a new tree if left branch is NULL */
		tree -> left = node;
		return node;
	    }

/*
	The first thing to do is find the point of insertion for the new node.
The tree is traversed as though we were doing a simple binary tree.  If the
node is already found in the tree, we return with a pointer to the existing
node without inserting the new node.

	During the traversal, 'a' points to the most recent node with a non-
zero balance factor.  'parenta' points to the parent node of 'a'.  'q' follows
the node pointer 'tptr' through the tree, and will be used as the point of
insertion once 'tptr' reaches nil.

*/

	parenta = q = NULL;	/* set the traversal followers to NULL */
	a = tptr;		/* start 'a' off at the top of the tree */
	while (tptr != NULL)	/* loop until at the end of a branch */
	    {
		if (tptr -> balance != 0)
		    {
			/* non-zero balance found; reset 'a' and 'parenta' */
			a = tptr;	/* 'a' points to this node */
			parenta = q;	/* 'parenta' points to its parent */
		    }
		n = strcmp(node -> name, tptr -> name);
		if (n < 0)
		    {
			/* less, so take left branch */
			q = tptr;	/* 'q' will point to the parent */
			tptr = tptr -> left;
		    }
		else if (n > 0)
		    {
			/* greater, so take right branch */
			q = tptr;	/* 'q' will point to the parent */
			tptr = tptr -> right;
		    }
		else
			return tptr;	/* the node is already in the tree */
	    }

/*
	At this point,we have reached the end of a leaf within the tree.  The
variable 'tptr' is nil, and 'q' points to the last node that was looked at. The
node pointed to by 'q' will become the parent node of the new node (before
balance rotations).  A second symbol comparison is performed to determine
whether the new node should be the left or the right child of the parent node.

*/

	if (strcmp(node -> name, q -> name) < 0)
		q -> left = node;		/* left child */
	else
		q -> right = node;		/* right child */

/*
	Balancing of the tree now must be done.  The first stage of balancing
the tree consists of adjusting the balance factors of all tree nodes on the
path between the pointer 'a' (the last node with a non-zero balance factor),
and pointer 'q' (the parent node of the new node).  Note that by definition of
'a', all nodes on this path will have a balance factor of zero.  All of these
balance factors will change to either 1 or 0xFF.

*/

	if (strcmp(node -> name, a -> name) < 0)
	    {
		b = tptr = a -> left;	/* follow the left branch */
		d = 1;
	    }
	else
	    {
		b = tptr = a -> right;	/* follow the right branch */
		d = 0xFF;
	    }

/*
	Now we traverse the tree until we reach the new node, adjusting balance
factors as we go.

*/

	while (tptr != node)
		if (strcmp(node -> name, tptr -> name) < 0)
		    {
			tptr -> balance = 1;
			tptr = tptr -> left;
		    }
		else
		    {
			tptr -> balance = 0xFF;
			tptr = tptr -> right;
		    }

/*
	Next, we check to see if the tree is still balanced after adding the
new node.  If the tree is okay, then we return with a pointer to the new node.
If the tree is unbalanced, we balance it before returning with the new node.

*/

	if (a -> balance == 0)
	    {
		a -> balance = d;
		return node;
	    }
	if (a -> balance + d == (0xFF + 1))  /* is one 0xFF and the other 1? */
	    {
		a -> balance = 0;
		return node;
	    }

/*
	If we get here, the tree needs balancing.  The type of rotation needed
to rebalance the tree is determined and performed.

*/

	if (d == 1)
	    {
		/* left rotation required */
		if (b -> balance == 1)
		    {
			/* LL rotation */
			a -> left = b -> right;
			b -> right = a;
			a -> balance = b -> balance = 0;
		    }
		else
		    {
			/* LR rotation */
			c = b -> right;
			if (c == NULL)
				c_left = c_right = NULL;
			else
			    {
				c_left = c -> left;
				c_right = c -> right;
			    }
			b -> right = c_left;
			a -> left = c_right;
			c -> left = b;
			c -> right = a;
			switch (c -> balance)
			    {

			case 1:		/* LR(b) */
				a -> balance = 0xFF;
				b -> balance = 0;
				break;

			case 0xFF:	/* LR(c) */
				b -> balance = 1;
				a -> balance = 0;
				break;

			default:	/* LR(a) */
				a -> balance = b -> balance = 0;
			    }
			c -> balance = 0;
			b = c;
		    }
	    }
	else
	    {
		/* right rotation */
		if (b -> balance == 0xFF)
		    {
			/* RR rotation */
			a -> right = b -> left;
			b -> left = a;
			a -> balance = b -> balance = 0;
		    }
		else
		    {
			/* RL rotation */
			c = b -> left;
			if (c == NULL)
				c_left = c_right = NULL;
			else
			    {
				c_left = c -> left;
				c_right = c -> right;
			    }
			b -> left = c_right;
			a -> right = c_left;
			c -> right = b;
			c -> left = a;
			switch (c -> balance)
			    {

			case 0xFF:	/* RL(b) */
				a -> balance = 1;
				b -> balance = 0;
				break;

			case 1:		/* RL(c) */
				b -> balance = 0xFF;
				a -> balance = 0;
				break;

			default:	/* RL(a) */
				b -> balance = a -> balance = 0;
			    }
			c -> balance = 0;
			b = c;
		    }
	    }

/*
	The branch whose root is 'b' has been rebalanced and is the new branch
of 'parenta'.  The original branch of 'parenta' had root 'a'.

*/

	if (parenta == NULL)	/* if nil, we rotated around the tree root */
		tree -> left = b;	/* so set the new tree into the root */
	else
		/* decide if 'b' will be the left or right branch */
		if (a == parenta -> left)
			parenta -> left = b;
		else if (a == parenta -> right)
			parenta -> right = b;
	return node;		/* return a pointer to the new node */
}



/*
	Routine:	avl_lookup()

	This function searches an AVL-tree for a symbol (or string).  A pointer
to the node in the tree holding the symbol is returned if the symbol is found
in the tree.  If the symbol isn't found, a nil pointer is returned.

	Calling sequence:

		ptr = avl_lookup(symbol, tree);

	where:

		ptr	is the returned pointer to the node holding the
			symbol.  This pointer will be nil if the symbol isn't
			in the AVL-tree.

		symbol	is a pointer to the symbol to search for.  The symbol
			may be any string of characters terminated by a zero
			byte.

		tree	is a pointer to the root node of the AVL-tree to
			search.

*/


AVLNODE	*avl_lookup(sym, tree)
char	*sym;		/* pointer to the symbol to search for */
AVLNODE	*tree;		/* pointer to root node of AVL-tree to search */
{
	int	n;		/* result of string comparison (scratch) */


	tree = tree -> left;
	while (tree)		/* loop until we reach the end of a branch */
	    {
		if ((n = strcmp(sym, tree -> name)) == 0)
			return tree;	/* symbol found */
		if (n < 0)
			tree = tree -> left;
		else
			tree = tree -> right;
	    }
	return NULL;		/* symbol not found in tree */
}



/*
	Routines:	avl_inorder(), avl_preorder(), avl_postorder()

	These functions perform traversals of an AVL-tree.  For every node in
the tree, a user-specified function is called.  A pointer to the node being
processed is passed to the user's function as the function's only argument.
The order of processing of the nodes in the tree depends on the function
called.

	Calling sequence:

		avl_inorder(tree, func);
		avl_preorder(tree, func);
		avl_postorder(tree, func);

	where:

		tree	is a pointer to the root node of the AVL-tree to
			traverse.  The pointer may be nil.  Due to the way
			sub-trees are linked to the root of a tree, you must
			pass a pointer to the LEFT SUB-TREE of the root node
			for this routine.  See the example.

		func	is a pointer to the function to call for every tree
			node.  The function is supplied with a pointer to the
			current node as its only argument.

*/

avl_inorder(tree, func)
AVLNODE	*tree;			/* pointer to AVL-tree to traverse */
int	(*func)();	/* pointer to function to call */
{
	if (tree != NULL)
	    {
		avl_inorder(tree -> left, func);
		(*func)(tree);		/* call the function for this node */
		avl_inorder(tree -> right, func);
	    }
}


avl_preorder(tree, func)
AVLNODE	*tree;			/* pointer to the AVL-tree to traverse */
int	(*func)();	/* pointer to the function to call */
{
	if (tree != NULL)
	    {
		(*func)(tree);		/* call the function for this node */
		avl_preorder(tree -> left, func);
		avl_preorder(tree -> right, func);
	    }
}


avl_postorder(tree, func)
AVLNODE	*tree;			/* pointer to the AVL-tree to traverse */
int	(*func)();	/* pointer to the function to call */
{
	if (tree != NULL)
	    {
		avl_postorder(tree -> left, func);
		avl_postorder(tree -> right, func);
		(*func)(tree);		/* call the function for this node */
	    }
}
s supplied with a pointer to the
			current node as its only argument.

*/

avl_inorder(tre