	TITLE	'North Star Disk Copy Program V3.1 of 04/28/81'
;************************************************************
;
;  The purpose of this program is to allow the operator
; to copy, format, or validate North Star double density
; disks. Copying can be done between any two North Star
; drives, and can include all tracks, or just the system
; (0 and 1) tracks or data tracks (2 - 34) for special
; system handling. The allowed functions are:
;
;  1)	Copy all tracks from source disk to destination
;	disk. The density of sector zero on track zero
;	of the source determines the density of the
;	destination.
;
;  2)	Copy only the system tracks from source disk
;	to destination disk. If source disk is double
;	density, tracks 0 and 1 only are copied. If
;	source disk is single density, tracks 0, 1,
;	and 2 only are copied.
;
;  3)	Copy only the data tracks from source disk to
;	destination disk. If source disk is double
;	density, tracks 2-34 inclusive are copied.
;	If source disk is single density, tracks 3-34
;	inclusive are copied.
;
;  4)	Validate the source disk. This is done by
;	reading all tracks.
;
;  5)	Format the destination disk in double
;	density. This is done by writing data
;	blocks with each byte containing the
;	hex value "E5" to all sectors of all
;	tracks. To be Lifeboat-compatible, we
;	write a double-density single-sided
;	disk ID block into track zero, sector
;	zero.
;
;  6)	Format the destination disk in single
;	density. Similar to 5) above.
;
;  In addition, the user is given the options of
; warm booting CP/M, or cold booting by jumping
; to the standard North Star ROM boot address. For
; functions 1 - 3, the user is asked for the source
; and destination drive letters. For function 4,
; the user is asked for just the destination drive
; letter, since the source is unused. For function
; 5, the user is asked for just the source letter,
; since the destination is unused.
;
;  To drive the disks at top speed, this program
; contains a complete set of North Star disk
; driver routines. CP/M is still used to write
; messages to the console and get user replies.
;
;  This program is built on the wreckage of the
; CP/M User's group COPY program, after most of
; it was discarded.
;
;************************************************************

	ORG	100H
;
;EQUATES
;
WBOOT	EQU	0	;CP/M WARM BOOT ADDRESS
CBOOT	EQU	0E800H	;NORTH STAR ROM BOOT ADDRESS
NDRIVE	EQU	2	;NUMBER OF NORTH STAR DRIVES ON SYSTEM
LASTSEC	EQU	9
LASTTRK	EQU	34
BDOS	EQU	5
IDOFF	EQU	050H	;Offset into BUF0 where Lifeboat
			; ID should be placed
CR	EQU	0DH
LF	EQU	0AH
;
;************************************************************
;
;  Start the program here. Print out the list of choices
; and ask for the user's reply.
;
START:	LXI	SP,STKTOP	;Set stack ptr
	LXI	H,SIGNON
	CALL	MSGOT	;Tell who we are
;
LIST:	LXI	SP,STKTOP	;Reload stack ptr
			; in case here after error
	CALL	MOFF	;Turn off North Star motors
			; to allow disk switching
	LXI	H,FLIST	;Print out the menu
	CALL	MSGASK	; and get choice in acc
	STA	FUNC	;Save for later
;
;  Dispatch to routine based on choice, unless it was bad.
;
	CPI	'1'
	JZ	ALLSET	;COPY ALL
	CPI	'2'
	JZ	SYSSET	;COPY SYSTEM
	CPI	'3'
	JZ	DATSET	;COPY DATA
	CPI	'4'
	JZ	VALSET	;VALIDATE
	CPI	'5'
	JZ	FORSTD	;FORMAT DOUBLE DENSITY
	CPI	'6'
	JZ	FORSTS	;FORMAT SINGLE DENSITY
	CPI	'7'
	JZ	EXIT	;WARM BOOT
	CPI	3	;ALLOW CTRL-C FOR WARM BOOT TOO
	JZ	EXIT
	CPI	'8'
	JZ	CEXIT	;COLD BOOT
	LXI	H,SELER
	CALL	MSGOT	;BAD NUMBER GIVEN
	JMP	LIST	;GO ASK AGAIN
;************************************************************
;
SIGNON:	DB	CR,LF,'North Star Copy V3.1$'
FLIST:	DB	CR,LF
	DB	CR,LF,'Function List:'
	DB	CR,LF
	DB	CR,LF,'  1. Copy all tracks, using density'
	DB	' of source disk'
	DB	CR,LF,'  2. Copy system tracks only,'
	DB	' using density of source disk'
	DB	CR,LF,'  3. Copy data tracks only, using'
	DB	' density of source disk'
	DB	CR,LF,'  4. Validate'
	DB	CR,LF,'  5. Format DOUBLE density'
	DB	CR,LF,'  6. Format SINGLE density'
	DB	CR,LF,'  7. Warm boot'
	DB	CR,LF,'  8. Cold boot'
	DB	CR,LF
	DB	CR,LF,'Enter function number: $'
SELER:	DB	CR,LF,'Not a valid number',CR,LF,'$'
;
SRCMSG:	DB	'Enter source$'
DSTMSG:	DB	'Enter destination$'
SDMSG:	DB	' drive (A-'
	DB	NDRIVE-1+'A',') or <CR> to reselect: $'
;
MESGA:	DB	CR,LF,'Compare error on track $'
MESGB:	DB	' (hex) sector $'
MESGC:	DB	' -- permanent $'
MESGD:	DB	CR,LF,'Source read error exit $'
MESGE:	DB	CR,LF,'Destination write error exit $'
MESGF:	DB	CR,LF,'Destination read error exit $'
S1:	DB	CR,LF,'Ready to copy $'
S2:	DB	' from disk '
SSRC:	DB	0,' to disk '
SDST:	DB	0,'.$'
SSYS:	DB	'system tracks only$'
SDAT:	DB	'data tracks only$'
SALL:	DB	'all tracks$'
DFOR:	DB	CR,LF,'Ready to format and validate disk '
DFORD:	DB	0,' in DOUBLE density,',CR,LF
	DB	'overwriting all previous data there.$'
SFOR:	DB	CR,LF,'Ready to format and validate disk '
SFORD:	DB	0,' in SINGLE density,',CR,LF
	DB	'overwriting all previous data there.$'
SVAL:	DB	CR,LF,'Ready to validate disk '
SVALS:	DB	0,'.$'
GOMSG:	DB	CR,LF,CR,LF
	DB	'If that is what you want, switch disks if'
	DB	CR,LF
	DB	'appropriate and type <CR> when ready: $'
INSERT:	DB	CR,LF,'Insert system disk in drive A'
	DB	' and type <CR> to $'
BOOT:	DB	' boot: $'
WMSG:	DB	'warm$'
CMSG:	DB	'cold$'
ABMSG:	DB	CR,LF,'** Console abort **$'
DONMSG:	DB	CR,LF,'Function complete.$'
CRLF:	DB	CR,LF,'$'
;
;  This is the Lifeboat-compatible label we write to a
; double density disk when we format it. Note that the
; contents of the second part of the label are actually
; a CP/M disk parameter block. We assume this must match
; the disk we are formatting, so this DPB is for a
; double-density single-sided disk. The name seems to
; be ignored, so we use NSCOPY (Lifeboat uses FORMAT).
; The rest of the data between the name and the DPB
; has no meaning to us, so we are just blindly using
; the stuff Lifeboat puts there, which doesn't seem
; to hurt.
;
IDD:	DB	'NSCOPY  '	;Format program name
	DB	0,0,0,0,90H,0,0,0,90H	; ???
	DW	40	;DPB - Sectors per track
	DB	3	;Block shift factor
	DB	07H	;Block mask
	DB	0	;Null mask
	DW	165-1	;Disk size in groups - 1
	DW	63	;Directory max entries
	DB	11000000B ;Alloc 0
	DB	0	;Alloc 1
	DW	16	;Check size
	DW	2	;Track offset to data
;
IDLEN	EQU	$-IDD	;# bytes in label
;
;  SINGLE DENSITY LIFEBOAT LABEL (SEE IDD ABOVE)
;
IDS:	DB	'NSCOPY  '	;Format program name
	DB	0,0,0,0,10H,0,0,0,10H	; ???
	DW	20	;DPB - Sectors per track
	DB	3	;Block shift factor
	DB	07H	;Block mask
	DB	0	;Null mask
	DW	80-1	;Disk size in groups - 1
	DW	63	;Directory max entries
	DB	11000000B ;Alloc 0
	DB	0	;Alloc 1
	DW	16	;Check size
	DW	3	;Track offset to data
;
;
;  ASKSRC IS USED TO GET THE SOURCE DRIVE NAME FROM
; THE USER. SRCDSK IS SET ON RETURN WITH A VALUE FROM
; 0 TO NDRIVE-1, AND ACC CONTAINS THE ASCII DRIVE
; LETTER. IF USER TYPED <RET>, WE EXIT BACK TO LIST.
;
ASKSRC:	LXI	H,SRCMSG	;GET SOURCE
	CALL	ASKC
	JC	ASKSRC	;GO BACK IF BAD LETTER
	CPI	NDRIVE
	JNC	ASKSRC
	STA	SRCDSK	; ELSE SET DRIVE NUMBER
	ADI	'A'	;RESTORE LETTER FOR CALLER
	STA	SSRC	;SET FOR COPY FUNCTIONS
	RET
;
;  ASKDST IS USED TO GET THE DESTINATION DRIVE
; NAME. DSTDSK IS SET ON RETURN WITH A VALUE FROM
; 0 TO NDRIVE-1, AND ACC CONTAINS THE ASCII DRIVE
; LETTER. IF USER TYPED <RET>, WE EXIT BACK TO LIST.
;
ASKDST:	LXI	H,DSTMSG	;GET DESTINATION
	CALL	ASKC
	JC	ASKDST	;GO BACK IF BAD LETTER
	CPI	NDRIVE
	JNC	ASKDST
	STA	DSTDSK	; ELSE SET DRIVE NUMBER
	ADI	'A'	;RESTORE LETTER FOR CALLER
	STA	SDST	;SET FOR COPY FUNCTIONS
	RET
;
ASKC:	CALL	MSGOT
	LXI	H,SDMSG
	CALL	MSGASK
	CPI	CR	;ABORT?
	JZ	LIST	;GET OUT IF SO
	SUI	'A'
	RET
;
;  MAIN COPY ROUTINE HERE. PERFORM FUNCTION BETWEEN
; TRACKS SPECIFIED BY THE TWO BYTES AT TRKSRT.
;
COPY:	LDA	FUNC	;CHECK FUNCTION
	CPI	'5'	;FORMAT?
	JNC	FRSTRK	;DENSITY ALL SET IF SO
	CALL	SETDD	;ASSUME DOUBLE D IF NOT
	XRA	A	;SET UP SOURCE READ IF SO
	STA	HSTSEC	; ON SECTOR ZERO
	STA	HSTTRK	; OF TRACK ZERO
	LDA	SRCDSK
	STA	HSTDSK
	LXI	H,MESGD	;IN CASE OF ERROR
	SHLD	IOEMSG
	LXI	H,BUF1	;PT TO SCRATCH BUFFER
	SHLD	HSTADR
	CALL	SETUP	;SELECT DISK AND SEEK
	CALL	READ	;READ THE SECTOR
	LDA	ERFLAG	;CHECK FOR ERRORS
	ORA	A
	CNZ	FAILR	;ERROR IF CAN'T READ SECTOR
	LDA	NSDENS	;CHECK DENSITY
	ORA	A
	JZ	FRSTRK	;ALL SET IF DOUBLE D
	CALL	SETSD	;SET SINGLE DENS VARIABLES
	LDA	FUNC	;IF SINGLE, MAY NEED ADJUSTMENTS
	CPI	'1'	;COPY ALL TRACKS?
	JZ	FRSTRK	;ALL SET IF SO
	CPI	'4'	;VALIDATE?
	JZ	FRSTRK	;ALSO ALL SET
	CPI	'2'	;COPY SYSTEM TRACKS ONLY?
	LXI	H,3*256+0 ;ASSUME SO
	JZ	SDSYS	;GO STUFF IF SO
	LXI	H,(LASTTRK+1)*256+3 ;DATA COPY IF NOT
SDSYS:	SHLD	TRKSRT	;NEW START OR END
	MOV	A,L
FRSTRK:	LDA	TRKSRT	;SET 1ST TRK TO DO
	STA	HSTTRK
;
RDLOOP:	MVI	C,11	;CHECK FOR CONSOLE ABORT
	CALL	BDOS
	ORA	A
	JNZ	ABORT	;GET OUT IF SO
	MVI	A,'*'	;IF NOT,
	CALL	CNSOT	; SHOW DOING NEXT TRACK
	XRA	A
	STA	CMPERR
	LDA	FUNC	;SEE IF FORMATTING ONLY
	CPI	'5'
	JNC	RETRYW	;WRITE ONLY IF SO
	LDA	SRCDSK	;IF NOT,
	STA	HSTDSK	; SELECT SOURCE DRIVE
	CALL	READT	;READ ENTIRE TRACK
	LDA	FUNC	;CHECK FUNCTION
	CPI	'4'	;VALIDATE?
	JZ	SKPW	;SKIP WRITE IF SO
RETRYW:
	LDA	DSTDSK	;SELECT DESTINATION DRIVE
	STA	HSTDSK
	CALL	WRITET	;WRITE TRACK
	CALL	COMPT	;REREAD AND COMPARE
	JNZ	RETRYW	;RETRY IF ERR
;
;  Check to see if we just wrote track zero. If so,
; make sure the Lifeboat label area is set to 0E5H's,
; just in case we are formatting. If we are in fact
; copying, it doesn't matter, since we're done with
; this track anyway.
;
	LDA	HSTTRK	;See if just wrote track zero
	ORA	A
	JNZ	SKPW	;Pass if not
	LXI	H,BUF0+IDOFF	;If so, pt to label area
	MVI	B,IDLEN	;This many bytes to fill
;
FFIX:	MVI	M,0E5H	;Stuff an E5
	INX	H
	DCR	B	;Count it off
	JNZ	FFIX	;Do the whole label
;
;  HERE TO SKIP TRACK WRITE ON VALIDATE ONLY
;
SKPW:	LDA	HSTTRK
	INR	A
	STA	HSTTRK
	LXI	H,TRKSRT+1 ;POINT TO LAST TRACK+1 STORAGE
	CMP	M	;ARE WE DONE?
	JNZ	RDLOOP	;LOOP IF NOT
	RET
;
;
COMPT:	LXI	H,MESGF	;Load address of msg for
	SHLD	IOEMSG	; destination read error
	LXI	H,BUF1
	CALL	RT2	;REREAD INTO BUF1
	LHLD	TRKSIZ	;GET TRACK SIZE IN BYTES
	MOV	B,H	; INTO BC
	MOV	C,L
	LXI	H,BUF0
	LXI	D,BUF1
CMPLP:	LDAX	D
	CMP	M
	JNZ	CERR
	INX	H
	INX	D
	DCX	B
	MOV	A,C	;End of buffer yet?
	ORA	B
	JNZ	CMPLP	;Keep checking if not
	RET
CERR:	PUSH	H
	LXI	H,MESGA
	CALL	MSGOT	;REPORT COMPARE ERROR
	LDA	HSTTRK
	CALL	HXBOT
	LXI	H,MESGB
	CALL	MSGOT
	POP	H
	LXI	D,-BUF0
	DAD	D	;Calc offset into buffer
	LDA	NSCNT	;GET DENSITY FLAG
	ORA	A	;SET ZERO COND CODE IF DOUBLE DENS
	MOV	A,H
	JNZ	CERRSD	;NUMBER RIGHT IF SINGLE DENS
	RAR		;A DOUBLE DENS SECTOR IS TWO PAGES
CERRSD:	CALL	HXBOT	;PRINT SECTOR
	LDA	CMPERR
	INR	A
	STA	CMPERR	;INCREMENT ERROR COUNT
	CPI	10
	RNZ
	LXI	H,MESGC	;REPORT GIVING UP
	CALL	MSGOT
	XRA	A
	RET
;
;
READT:	LXI	H,MESGD	;Load address of msg for
	SHLD	IOEMSG	; source read error
	LXI	H,BUF0
RT2:	SHLD	HSTADR
	CALL	SETUP		;SELECT DRIVE AND
				; SEEK TO TRACK
	LDA	ERFLAG		;CHECK FOR ERRORS
	ORA	A
	CNZ	FAILR	;Index pulse not found is fatal
	MVI	C,0
RT3:	PUSH	B
	MOV	A,C
	STA	HSTSEC
	CALL	READ
	LDA	NSDENS	;CHECK FOR DENSITY MISMATCH
	MOV	C,A
	LDA	NSCNT
	CMP	C
	JZ	RT4	;GO ON IF OK
	MVI	A,5	;SET ERROR IF NOT
	STA	ERFLAG
	LXI	H,SDEMSG
	CALL	DSKERR	;RAT ON BAD SECTOR
RT4:	LDA	ERFLAG
	ORA	A	;Any errors?
	CNZ	FAILR
	LHLD	HSTADR	;Advance read address
	XCHG
	LHLD	BLKSIZ
	DAD	D
	SHLD	HSTADR
	POP	B
	MVI	A,LASTSEC
	CMP	C
	RZ
	INR	C
	JMP	RT3
;
;
WRITET:	LXI	H,BUF0
WT2:	SHLD	HSTADR
	CALL	SETUP		;SELECT DRIVE AND
				; SEEK TO TRACK
	LDA	ERFLAG		;CHECK FOR ERRORS
	ORA	A
	CNZ	FAILW	;Index pulse not found is fatal
	MVI	C,0
WT3:	PUSH	B
	MOV	A,C
	STA	HSTSEC
	CALL	WRITE
	LDA	ERFLAG
	ORA	A	;Any errors?
	CNZ	FAILW
	LHLD	HSTADR	;Advance write address
	XCHG
	LHLD	BLKSIZ
	DAD	D
	SHLD	HSTADR
	POP	B
	MVI	A,LASTSEC
	CMP	C
	RZ
	INR	C
	JMP	WT3
;************************************************************
SYSSET:
	CALL	ASKSRC
	CALL	ASKDST
	CALL	PRNTS1
	LXI	H,SSYS
	CALL	MSGOT
	CALL	PRNTS2
	MVI	L,0		;FIRST TRACK TO TRANSFER
	MVI	H,2		;LAST TRACK PLUS ONE
	JMP	PUTPAR
;************************************************************
DATSET:
	CALL	ASKSRC
	CALL	ASKDST
	CALL	PRNTS1
	LXI	H,SDAT
	CALL	MSGOT
	CALL	PRNTS2
	MVI	L,2		;FIRST TRACK TO TRANSFER
	MVI	H,LASTTRK+1	;LAST TRACK PLUS ONE
	JMP	PUTPAR
;************************************************************
ALLSET:
	CALL	ASKSRC
	CALL	ASKDST
	CALL	PRNTS1
	LXI	H,SALL
	CALL	MSGOT
	CALL	PRNTS2
	JMP	DOALL		;DO ALL TRACKS
;************************************************************
;
;  Format disk. Fill BUF0 with 0E5H's, except for a Lifeboat-
; compatible label, so that Lifeboat CP/M can read and write
; our disks. After track zero is formatted, COPY will make
; the entire buffer 0E5H again by overwriting the label.
;
FORSTD:	CALL	ASKDST
	STA	DFORD
	LXI	H,DFOR
	CALL	MSGOT
	CALL	SETDD		;SET UP DOUBLE DENS VARS
	LXI	D,IDD		;PT TO DOUBLE DENS ID
;
FLBOTH:				;MERGE HERE FROM FORSTS
	MOV	B,H		;MOVE # BYTES/TRK TO BC
	MOV	C,L
	LXI	H,BUF0		;Fill track buffer with E5's
FL:	MVI	M,0E5H		;Set a byte to E5
	DCX	B		; and count it off
	INX	H		;Bop buffer ptr
	MOV	A,B
	ORA	C
	JNZ	FL		;Go back if more to set
	LXI	H,BUF0+IDOFF	;Now time to put in label
	MVI	B,IDLEN		;Move in this many bytes
;
FL5:	LDAX	D		;Get label byte
	MOV	M,A		; and store it
	INX	H		; then bop ptrs
	INX	D
	DCR	B		;Count off a byte
	JNZ	FL5		;Go back if more
	JMP	DOALL		;DO ALL TRACKS
;
;  SINGLE DENSITY FORMAT DISK
;
FORSTS:	CALL	ASKDST
	STA	SFORD
	LXI	H,SFOR
	CALL	MSGOT
	CALL	SETSD		;SET UP SINGLE DENS VARS
	LXI	D,IDS		;PT TO SINGLE DENS ID
	JMP	FLBOTH		;GO MERGE WITH FORSTD
;
;  DOUBLE DENSITY VARIABLES SETUP, HL = BYTES/TRK ON RETURN
;
SETDD:	LXI	H,512
	SHLD	BLKSIZ		;# BYTES/SECTOR
	MVI	A,00H
	STA	NSCNT		;DENSITY FLAG = DOUBLE
	LXI	H,512*(LASTSEC+1) ;# BYTES/TRACK
	SHLD	TRKSIZ
	RET
;
;  SINGLE DENSITY VARIABLES SETUP, HL = BYTES/TRK ON RETURN
;
SETSD:	LXI	H,256
	SHLD	BLKSIZ
	MVI	A,80H
	STA	NSCNT
	LXI	H,256*(LASTSEC+1)
	SHLD	TRKSIZ
	RET
;************************************************************
VALSET:
	CALL	ASKSRC
	STA	SVALS
	LXI	H,SVAL
	CALL	MSGOT
;
DOALL:	MVI	L,0		;1st trk to write
	MVI	H,LASTTRK+1	;Last trk + 1
;
PUTPAR:	SHLD	TRKSRT		;PUT PARAMETERS IN
				; TRKSRT AND TRKSRT+1
;
AGIN:	LXI	H,GOMSG		;Wait until user ready
	CALL	MSGASK		;Print message and get reply
	CPI	3		;Check for CTRL-C
	JZ	LIST		;Go back to menu on that
	CPI	CR
	JNZ	AGIN
;************************************************************
	CALL	COPY		;MAIN ROUTINE
	LXI	H,DONMSG	;SAY WE'RE DONE
	CALL	MSGOT
	JMP	LIST		;DONE. GO ASK FOR MORE TO DO.
;
;
PRNTS1:	LXI	H,S1
	JMP	MSGOT
;
PRNTS2:	LXI	H,S2
	JMP	MSGOT
;************************************************************
FAILR:	LHLD	IOEMSG	;Get addr of appro error msg
	JMP	DIE
FAILW:	LXI	H,MESGE
DIE:	CALL	MSGOT
	JMP	LIST		;GO TRY SOMETHING ELSE
;
;  HERE ON CONSOLE ABORT
;
ABORT:	CALL	CNSIN	;FLUSH ABORTING CHAR
	LXI	H,ABMSG
	CALL	MSGOT	;REPORT STOPPING
	JMP	LIST	;GO TRY SOMETHING ELSE
;************************************************************
EXIT:	LXI	H,INSERT
	CALL	MSGOT
	LXI	H,WMSG
	CALL	MSGOT
	LXI	H,BOOT
	CALL	MSGASK		;WAIT BEFORE REBOOTING
	CPI	CR
	JNZ	EXIT
	JMP	WBOOT		;GO WARM BOOT CP/M
;
CEXIT:	LXI	H,INSERT
	CALL	MSGOT
	LXI	H,CMSG
	CALL	MSGOT
	LXI	H,BOOT
	CALL	MSGASK
	CPI	CR
	JNZ	CEXIT
	JMP	CBOOT		;GO COLD BOOT NORTH STAR
	PAGE
;
;  NORTH STAR MEMORY MAPPED I/O ADDRESSES
;
WDATA	EQU	0E900H		;WRITE DATA. DATA IS
				; LOW 8 ADRESS BITS.
CORDER	EQU	0EA00H		;CONTROLLER ORDER
CCMND	EQU	0EB00H		;CONTROLLER COMMAND
;
;  MOFF SHUTS OFF THE NORTH STAR DRIVE MOTORS
; AND RESETS THE CONTROLLER.
;
MOFF:	LDA	CCMND+17H	;RESET CONTROLLER
	RET
;
;  WRITEHST PERFORMS THE PHYSICAL WRITE TO THE
; NORTH STAR DISK. ON ENTRY, SECTOR IS IN HSTSEC. ON
; EXIT, ERROR FLAG IS IN ERFLAG (ZERO IF NONE).
;
;  ***  INTERRUPTS MUST BE DISABLED HERE  **
;
WRITE:	DI			;WRITE NORTH STAR
	XRA	A
	STA	ERFLAG		;NO ERROR YET
	LDA	CCMND+20H	;GET B-STATUS
	ANI	02H		;IS DISK WRITE-PROTECTED?
	MVI	A,6		;ASSUME SO, ERROR CODE 6
	STA	ERFLAG
	JNZ	WPERR		;WRITE ALWAYS FAILS IF SO
	CALL	SEKSEC		;WAIT FOR OUR SECTOR
	LHLD	HSTADR		;GET ADDRESS TO WRITE FROM
	MVI	B,31		;ASSUME DOUBLE DENSITY WRITE
	MVI	E,2
	LDA	NSCNT		; THEN CHECK
	MOV	C,A		;PUT DATA COUNT IN C
	ORA	A
	JZ	WRIT5		;GO ON IF DOUBLE D
	MVI	B,15		; ELSE CORRECT
	DCR	E
WRIT5:	LDA	CCMND+16H	;INITIATE SECTOR WRITE
;
WRIT10:	LDA	CCMND+15H	;KEEP MOTORS RUNNING, GET A-STAT
	ANI	08H		;MUST LOOP UNTIL 96 USEC WINDOW
	JNZ	WRIT10		; PAST, SO LOOP WHILE WI TRUE
;
;  NOW WRITE THE 15 OR 31 BYTES OF LEADING ZEROES ON THE SECTOR
;
WRIT15:	LDA	WDATA+00H	;WRITE A BYTE OF ZEROES
	NOP			;KILL TIME
	MVI	D,WDATA SHR 8	;KILL TIME AND LOAD D FOR LATER
	DCR	B		;COUNT OFF A ZERO BYTE
	JNZ	WRIT15		;DO THEM ALL
;
;  FOLLOW WITH ONE OR TWO SYNCH BYTES
;
WRIT17:	LDA	WDATA+0FBH	;WRITE A SYNCH BYTE
	DCR	E
	JNZ	WRIT17		;GO BACK IF DOUBLE D
;
;  NOW WRITE OUT 256 OR 512 BYTES. DO THIS TWO AT A TIME SO
; WE CAN USE SINGLE PRECISION COUNTER. B IS ZERO FROM ABOVE
; TO INIT THE CRC BYTE.
;
WRIT20:	MOV	A,M		;GET NEXT BYTE TO OUTPUT
	MOV	E,A		; IN OUTPUT REG
	XRA	B		;ADD INTO CHECKSUM
	RLC
	MOV	B,A		;LEAVE CHECKSUM IN B
	LDAX	D		;WRITE DATA BYTE TO DISK
	INX	H		;BOP BUF PTR
	MOV	A,M		;REPEAT FOR NEXT BYTE
	MOV	E,A
	XRA	B
	RLC
	MOV	B,A
	LDAX	D
	INX	H
	DCR	C		;COUNT OFF LAST PAIR OF BYTES
	JNZ	WRIT20		;GO BACK IF MORE TO WRITE
	MOV	E,B		;IF NOT, TIME TO WRITE CHECKSUM
	INX	B		;KILL TIME
	LDAX	D		; THEN WRITE THE BYTE
	XRA	A		;DONE, INDICATE NO ERROR
	STA	ERFLAG
;
WHSTRT:	RET			; AND RETURN TO CALLER
;
WPERR:	LXI	H,WPEMSG	;WRITE PROTECT ERROR
	CALL	DSKERR		;TELL OPERATOR
	JMP	WHSTRT		; THEN LEAVE
;
;  READHST PERFORMS THE PHYSICAL READ FROM THE
; NORTH STAR DISK. ON ENTRY, SECTOR IS IN HSTSEC. ON
; EXIT, ERROR FLAG IS IN ERFLAG (ZERO IF NONE).
;
;  ***  INTERRUPTS MUST BE DISABLED HERE  ***
;
READ:	DI			;READ NORTH STAR
	MVI	A,10		;RETRY COUNT ON ERROR
	STA	RTCNT
;
READRT:	CALL	SEKSEC		;WAIT FOR OUR SECTOR
	MVI	B,08CH		;COUNT FOR SYNCH CHAR LOOP
	LXI	D,CCMND+40H	;SET UP READ DATA REGS
;
;  WAIT FOR RE SO WE CAN DO SECTOR READ, AND TEST
; THE DOUBLE DENSITY BIT.
;
WAITRE:	LDA	CCMND+10H	;GET A-STATUS
	ANI	04H		;CHECK RE
	JZ	WAITRE		;LOOP UNTIL RE TRUE
	XTHL			;KILL TIME TO GET INTO ZEROES
	XTHL
	XTHL
	XTHL
	XTHL
	XTHL
	XTHL
	XTHL
	LDA	CCMND+10H	;GET A-STATUS AGAIN
	ANI	20H		;ARE WE READING DOUBLE DENS?
	RAL			;GET ZERO IN NSDENS IF SO
	RAL			; ELSE GET 80H
	XRI	80H
	STA	NSDENS
;
;  NOW WAIT FOR SYNCH CHAR DETECTED. ERROR IF WE HAVE TO
; WAIT TOO LONG.
;
READ5:	LDA	CCMND+10H	;WAIT FOR SYNCH CHAR DETECTED
	RRC			;CHECK BIT
	JC	READ15		;OUT IF GOT IT
	DCR	B		;COUNT DOWN IF NOT
	JNZ	READ5		;GO BACK IF STILL OK
	MVI	A,1		;IF WAITED TOO LONG, SYNCH ERROR
	LXI	H,SYEMSG	;SYNCH ERROR MESSAGE
;
RERR:	STA	ERFLAG		;STORE ERROR CODE
	LDA	RTCNT		;COUNT OFF A RETRY
	DCR	A
	STA	RTCNT
	JNZ	READRT		;IF COUNT LEFT, GO RETRY
;
;  READ ERROR RETRIES FAILED. FLAG ERROR.
;
FRERR:	CALL	DSKERR		;TELL OPERATOR OF ERROR THEN OUT
;
RHSTRT:	RET
;
;  READ THE DATA INTO HSTBUF
;
READ15:	LHLD	HSTADR		;READ INTO HERE
	MVI	B,0		;INIT CHECKSUM BYTE
	LDA	NSDENS		;LOAD COUNT
	MOV	C,A		; INTO C-REG
;
READ20:	LDAX	D		;READ NEXT DATA BYTE
	MOV	M,A		; AND PUT IT IN BUFFER
	XRA	B		;ADD TO CHECKSUM
	RLC
	MOV	B,A
	INX	H		;BOP BUFFER PTR
	NOP			;KILL TIME
	LDAX	D		;GET NEXT BYTE
	MOV	M,A
	XRA	B
	RLC
	MOV	B,A
	INX	H
	DCR	C		;COUNT OFF LAST PAIR
	JNZ	READ20		;GO BACK IF MORE TO DO
	LDAX	D		; ELSE READ CRC BYTE
	XRA	B		; AND CHECK IT AGAINST OURS
	STA	ERFLAG		;IF OK, ZERO EFLAG
	JZ	RHSTRT		; AND RETURN TO CALLER
	MVI	A,2		; ELSE FLAG CHECKSUM ERROR
	LXI	H,RERMSG	;MESSAGE IF NEEDED
	JMP	RERR		;GO MAYBE RETRY
;
;  SETUP SELECTS THE UNIT CORRESPONDING TO HSTDSK,
; SEEKS TO THE TRACK IN HSTTRK. WRITE PRECOMPENSATION
; IS SET IF REQUIRED AS WELL. ON ERROR, ERFLAG CONTAINS
; A NON-ZERO VALUE ON RETURN.
;
SETUP:	LDA	NSCNT		;GET DOUBLE D BIT
	XRI	80H
	MOV	C,A		; INTO C-REG
	LDA	HSTDSK		;WANT THIS DISK
	INR	A		;REMAP TO CONTROLLER MASK
	CPI	03H		;IF UNIT 1 OR 2,
	JC	SET0		; GOT CORRECT MASK
	RAL			;IF 3 OR 4, MUST MAP
	ANI	0CH		; TO 4 OR 8 RESPECTIVELY
SET0:	ORA	C		;OR IN DENSITY BIT
	MOV	C,A		;SAVE IN REG C
	CALL	WAIT1S		;WAIT A SECTOR TIME, GET B-STAT
	ANI	10H		;ARE MOTORS ALREADY ON?
	LDA	CCMND+15H	;GIVE THEM EXTRA KICK ANYWAY
	JNZ	SET5		;BRANCH IF SO
	MVI	D,17H		;WAIT FOR MOTORS TO
	CALL	SCWAIT		; COME UP TO SPEED IF NOT
	JMP	SET10		; THEN GO SELECT DRIVE
;
SET5:	LDA	CURDSK		;IS THIS THE CURRENT DISK?
	CMP	C
	JZ	SET20		;GO SEEK TO TRACK IF SO
;
;  MUST SELECT NEW DISK
;
SET10:	MOV	A,C
	STA	CURDSK		;NEW CURRENT DISK
	MVI	B,CORDER SHR 8	;SET UP FOR SELECT
	LDAX	B		; AND DO IT
	MVI	D,2		;WAIT TWO SECTOR TIMES
	CALL	SCWAIT		; BEFORE LOOKING FOR INDEX
	MVI	B,12		;DON'T LOOK TOO LONG
;
SET15:	CALL	WAIT1S		;WAIT ONE SECTOR TIME
	LDA	CCMND+10H	;GET A-STATUS
	ANI	40H		;INDEX HOLE SEEN?
	JNZ	SET20		;CAN GO SEEK NOW IF SO
	DCR	B		;COUNT DOWN IF NOT
	JNZ	SET15		; AND MAYBE GO LOOK AGAIN
;
;  INDEX HOLE NOT FOUND. THAT'S A FATAL ERROR.
;
	MVI	A,4		;NO INDEX PULSE
	STA	ERFLAG
	LXI	H,NIPMSG
	JMP	DSKERR		;GO TELL OPERATOR ON WAY OUT
;
;  SEEK TO TRACK SPECIFIED BY HSTTRK NOW, AND SET
; WRITE PRECOMPENSATION IF APPROPRIATE.
;
SET20:	CALL	SEEK		;SEEK TO HSTTRK
	LDA	HSTTRK		;WHERE ARE WE?
	CPI	14H+1		;ARE WE BEYOND TRACK 14H?
	JC	SET25		;GO ON IF NOT
	LDA	NSCNT		;WRITING DOUBLE DENSITY?
	ORA	A
	JNZ	SET25		;NO PRECOMP IF NOT
	LDA	CURDSK		;IF SO, GET CURRENT MASK
	ORI	20H		; AND SET PRECOMP BIT
	MVI	H,CORDER SHR 8
	MOV	L,A
	MOV	A,M		;SET BIT IN CONTROLLER
SET25:	XRA	A		;NO ERRORS
	STA	ERFLAG
	RET			;SETUP IS DONE
;
;  HERE TO POSITION TO HSTSEC WITHIN THE CURRENT TRACK.
;
SEKSEC:	LDA	HSTSEC		;GET TARGET SECTOR
	MOV	C,A		;IN C
	STA	ERSEC		;SET UP FOR POSSIBLE ERROR
;
SET30:	CALL	WAIT1S		;WAIT FOR NEXT SECTOR
	LDA	CCMND+35H	;KICK MOTORS AND GET SECNUM
	ANI	0FH		;STRIP NON-SECTOR BITS
	CMP	C		;IS THIS THE TARGET SECTOR?
	JNZ	SET30		;WAIT UNTIL IT IS
	RET			;POSITIONED TO SECTOR
;
;  SEEK TO TRACK SPECIFIED BY HSTTRK. CURRENT OFFSET INTO
; NSTRK TABLE IS GIVEN BY HSTDSK.
;
SEEK:	LDA	HSTTRK		;GET TARGET TRACK
	MOV	D,A		; INTO D
	LDA	HSTDSK		;GET NSTRK INDEX
	MOV	C,A
	MVI	B,0
	LXI	H,NSTRK		;PT AT TABLE
	DAD	B		; THEN AT CORRECT ENTRY
	MOV	A,M		;GET CURRENT TRACK
	INR	A		;IS THIS DISK NEW?
	JNZ	SEEK5		;PASS IF NOT
	PUSH	D		;MUST HOME HEAD IF SO
	STA	HSTTRK		; TO KNOW WHERE WE ARE
	MVI	A,60		;MAKE SURE STEP TO ZERO
	MOV	M,A
	PUSH	H
	CALL	SEEK		;GO TO TRACK ZERO
	POP	H
	POP	D
	MOV	A,D
	STA	HSTTRK		;RESTORE HSTTRK
;
SEEK5:	MOV	A,D
	SUB	M		;SEE HOW FAR AWAY WE ARE
				; FROM TARGET TRACK
	MOV	M,D		; BUT ALWAYS SET NEW TRACK
	RZ			;IF THERE, DONE
	LXI	H,CORDER+30H	;ASSUME STEPPING IN
	MOV	C,A		;SAVE STEP COUNT
	JP	STEPIN		;BRANCH IF RIGHT
	CMA			;IF WRONG, NEGATE COUNT
	INR	A
	MOV	C,A		; THEN SAVE THAT
	LDA	CCMND+20H	;GET B-STATUS
	ANI	01H		;ARE WE ON TRACK ZERO?
	RNZ			;MUST BE DONE IF SO
	MVI	L,10H		;IF NOT, STEPPING OUT
;
STEPIN:	LDA	CURDSK		;GET CURRENT UNIT MASK
	ORA	L		;FORM FINAL CORDER VALUE
	MOV	L,A
	MOV	D,M		;SET THE STEP FLIP-FLOP
	ORI	10H
	MOV	L,A
	MOV	D,M		;SET IT AGAIN FOR SOME REASON
	XRI	10H
	MOV	L,A
	MOV	D,M		;NOW RESET THE STEP FLIP-FLOP
;
;  WAIT WHILE HEAD STARTS MOVING
;
	MVI	A,14H
STEPW1:	MVI	D,38H
STEPW2:	DCR	D		;KILL TIME
	JNZ	STEPW2
	DCR	A
	JNZ	STEPW1
	MVI	D,2		;NOW WAIT TWO SECTOR TIMES
	CALL	SCWAIT
	LDA	CCMND+25H	;GET B-STATUS AND KICK MOTORS
	ANI	01H		;ARE WE AT TRACK ZERO?
	JNZ	WAIT1S		;DONE STEPPING IF SO
	DCR	C		; ELSE COUNT OFF LAST TRACK
	JNZ	STEPIN		;GO BACK IF MORE TO DO
				; ELSE WAIT ONE MORE SECTOR TIME
				; TO MAKE SURE SAFE FOR I/O
;
;  SECTOR WAIT. ON ENTRY, # SECTORS TO WAIT IS IN D.
; ON EXIT, D=0 AND ACC=A-STATUS. CALL WAIT1S TO WAIT
; ONE SECTOR TIME.
;
WAIT1S:	MVI	D,1		;WAIT ONE SECTOR TIME
;
SCWAIT:	LDA	CCMND+11H	;RESET SECTOR FLAG
SCW5:	LDA	CCMND+10H	;GET A-STATUS
	ORA	A		;CHECK SECTOR FLAG
	JP	SCW5		;WAIT FOR IT IF NOT UP
	LDA	CCMND+11H	; ELSE RESET SECTOR FLAG
	DCR	D		;COUNT DOWN WAIT COUNTER
	JNZ	SCW5		;GO BACK IF MORE TO DO
	RET			; ELSE RETURN A-STATUS IN ACC
;
;  DISK ERRORS REPORTED HERE
;
DSKERR:	PUSH	H		;SAVE BODY ADDRESS
	MVI	A,CR		;EJECT LINE
	CALL	CNSOT
	MVI	A,LF
	CALL	CNSOT
	POP	H
	CALL	MSGOT		;REPORT BODY
	LDA	HSTDSK		;THIS DISK
	ADI	'A'
	STA	ASCDSK
	LXI	H,ERRMSG
	CALL	MSGOT
	LDA	HSTTRK		;THIS TRACK
	CALL	HXBOT
	LXI	H,SECMSG
	CALL	MSGOT
	LDA	ERSEC		;THIS SECTOR
				;FALL INTO HXBOT TO FINISH
;
;  DISPLAY HEXIDECIMAL EQUIVALENT OF ACC CONTENTS
;
HXBOT:	PUSH	PSW		;SAVE CHAR
	RRC			;SWAP NIBBLES
	RRC
	RRC
	RRC
	CALL	HXNOT		;OUTPUT HIGH NIBBLE
	POP	PSW		; THEN LOW NIBBLE
HXNOT:	ANI	0FH		;STRIP HIGH NIBBLE
	CPI	10		;LESS THAN 10
	JC	HXNNM		;PASS IF SO
	ADI	'A'-'9'-1	;ADD LETTER OFFSET
HXNNM:	ADI	'0'		;MAKE ASCII
;
CNSOT:	PUSH	H		;Save all regs
	PUSH	D
	PUSH	B		; but PSW
	MOV	E,A
	MVI	C,2
	CALL	BDOS		;Output char to console
	POP	B		;Restore regs
	POP	D
	POP	H
	RET			;Return to caller
;
CNSIN:	PUSH	H		;SAVE ALL REGS
	PUSH	D
	PUSH	B
	MVI	C,1
	CALL	BDOS		;READ CONSOLE CHAR
	POP	B		;RESTORE ALL REGS
	POP	D
	POP	H
	CPI	'A'+20H		;FOLD LOWER CASE TO UPPER
	RC
	CPI	'Z'+20H+1
	RNC
	SUI	20H
	RET
;
MSGASK:	CALL	MSGOT	;OUTPUT MSG @HL
	CALL	CNSIN	;GET OPERATOR REPLY
	LXI	H,CRLF	; THEN RETURN CARRIAGE
	CALL	MSGOT
	RET		;RETURN REPLY IN ACC
;
MSGOT:	PUSH	PSW		;SAVE CALLER FLAGS
MSGL:	MOV	A,M		;LOAD CHAR
	CPI	'$'		;END OF STRING?
	JZ	MSGR		;OUT IF SO
	CALL	CNSOT		;IF NOT, OUTPUT CHAR
	INX	H		;ADVANCE PTR
	JMP	MSGL		;GO BACK FOR NEXT CHAR
;
MSGR:	POP	PSW		;RESTORE CALLER PSW
	RET
;
;  DISK ERROR MESSAGES
;
WPEMSG:	DB	'Protect$'
SDEMSG:	DB	'Density mismatch$'
RERMSG:	DB	'CRC$'
SYEMSG:	DB	'No synch$'
NIPMSG:	DB	'No index$'
;
ERRMSG:	DB	' err on '
ASCDSK:	DB	' :, trk $'
SECMSG:	DB	', sec $'
;
;  NORTH STAR DRIVER CURRENT TRACK TABLE
;
NSTRK:	DB	0FFH	;NO CURRENT TRACK YET
	DB	0FFH	;EACH ENTRY CONTAINS
	DB	0FFH	; THE LAST TRACK POSITION
	DB	0FFH	; FOR THE UNIT 1-4
;
;  NORTH STAR DRIVER VARIABLES
;
HSTDSK:	DB	0	;HOST DISK NUMBER
ERFLAG:	DB	0	;ERROR REPORTING
CURDSK:	DB	0FFH	;CURRENT ACTIVE DISK,
			; GENNED-IN FORCE SELECT
NSDENS:	DS	1	;DENSITY FLAG FOR LAST
			; BLOCK READ. DD=00H, SD=80H
NSCNT:	DS	1	;DENSITY FLAG AND COUNT FOR
			; NEXT BLOCK TO WRITE. DD=00H,
			; SD=80H
TRKSIZ:	DS	2	;# BYTES/TRACK
BLKSIZ:	DS	2	;# BYTES/SECTOR
HSTADR:	DS	2	;XFER MEM ADDRESS
HSTTRK:	DS	1	;HOST TRACK NUMBER
HSTSEC:	DS	1	;HOST SECTOR NUMBER
RTCNT:	DS	1	;ERROR RETRY COUNTER
ERSEC:	DS	1	;SECTOR IN ERROR
;
;  MAIN COPY PROGRAM DATA AREA
;
TRKSRT:	DS	2	;STORAGE FOR FIRST AND
			; LAST+1 TRACK NUMBERS
CMPERR:	DS	1	;NUMBER OF COMPARE ERRORS
FUNC:	DS	1	;SELECTED FUNCTION NUMBER
SRCDSK:	DS	1	;SOURCE DRIVE NUMBER
DSTDSK:	DS	1	;DESTINATION DRIVE NUMBER
IOEMSG:	DS	2	;FAILR (read error) msg addr
STK:	DS	128
STKTOP:	DS	1
;
BUF0:	DS	512*(LASTSEC+1)
BUF1:	DS	512*(LASTSEC+1)
;
	END
