/*	keyboard.h	2.3	82/10/27	*/

/* (C) Copyright 1982 by Sun Microsystems, Inc. */

/*
 *	Header file for Sun Microsystems parallel keyboard routines
 *
 *	The keyboard is a standard Micro Switch, or otherwise, keyboard,
 *	with an 8048/8748 on the board.  This has been modified to produce
 *	up/down keycodes on 8 parallel output lines.  Each keycode is held
 *	stable on these lines for a minimum of 2.5 ms in order that the 
 *	main processor can read it during its refresh routine, which executes
 *	every 2 ms or so.  
 *
 *	When no keys are depressed, the keyboard transmits a keycode of
 *	7F hex, to indicate that.  Thus, when the last key is released,
 *	a keycode for its release is sent, then a 7F.
 *
 *	Eventually the 8048 will, when idle, alternate its 7F's with 
 *	keyboard id codes, identifying the key layout or other factors
 *	which host programs might want to automatically determine.
 *	The support for this feature is commented out with "#ifdef KBDID"
 *	and is currently untested.
 *
 *	In the meantime, the configuration file for this version of the
 *	ROM monitor must specify what keyboard it will support, by using:
 *		options KEYBKL or KEYBVT
 *
 *				John Gilmore, Sun Microsystems   3August82
 *
 *	Well, Keyboard ID support is here, but since there isn't room for
 *	both sets of keytables, you still have to specify which keyboard
 * 	you are making proms for.  The only good the keyboard id does is
 *	to let us know the state of the keyboard (in particular, the CAPS 
 *	LOCK led) when we power up.
 *
 *	Note that the ID byte has been slightly redefined.  The low 4 bits
 *	now contain the keyboard ID (currently 1 for VT100 keyboard); the
 *	high 4 bits contain state information about the keyboard.  The high
 *	order bits are allocated from the high end in case we need to shift
 *	the boundary.  The only bit currently defined is 0x80, which is
 *	the current state of the CAPS LOCK LED.
 *
 *	The IDLE code is sent when no physical keys are down.  It is always
 *	immediately followed by the ID byte.  Note that virtual keys (CAPS
 *	LOCK) can be down when the IDLE code is received.
 *
 *				John Gilmore, 26 October 82
 *
 */

#ifndef GETKEYPORT

/*
 *	How to get the current keycode being presented by the 8048
 *
 *	GETKEYPORT reads the raw port; GETKEY deglitches it.
 */

#define GETKEYPORT	( *(unsigned char*) 0xE00000 )
#define	GETKEY(key) \
	while (GETKEYPORT != ( (key) = GETKEYPORT ) );

/*
 *	Various special characters that might show up on the port
 */

#define	NOTPRESENT	0xFF	/* Keyboard is not plugged in */
#define IDLEKEY		0x7F	/* Keyboard is idle; i.e. no keys down */
/* See comments above re keyboard id's which might follow IDLEKEYs. */
#define PRESSED		0x00	/* 0x80 bit off means key was pressed */
#define RELEASED	0x80	/* 0x80 bit on  means key was released */

#ifdef KEYBKL
/* Klunker style keyboard wired-in information */
#define KB_UNKNOWN	KB_MS_103SD32	/* Unknown keyboard assumed klunker */
#define	ABORTKEY1	1	/* First key of abort seq - ERASE EOF */
#define ABORTKEY2	77	/* 2nd key of abort seq - "A" */
#endif KEYBKL

#ifdef KEYBVT
/* VT100 style keyboard wired-in information */
#ifdef KBDID
/* With KBDID, the value of KB_UNKNOWN must be different from a real kbd id */
#define KB_UNKNOWN	0xEE		/* Random unlikely to be seen */
#else KBDID
#define KB_UNKNOWN	KB_VT100	/* Unknown keyboard assumed VT100 */
#endif KBDID
#define	ABORTKEY1	1	/* First key of abort seq - Setup */
#define ABORTKEY2	59	/* 2nd key of abort seq - "A" */
#endif KEYBVT


/*
 *	Keyboard ID codes...transmitted by the various keyboards
 *	after the IDLEKEY code.  See top of file for more details.
 */
#define KB_MS_103SD32	0x00	/* Micro Switch 103SD32-2 */
#define KB_VT100	0x01	/* Micro Switch VT100 compatible 83SD30-2 */
				/* or Keytronics VT100 compatible */


/*
 *	These are the states that the keyboard scanner can be in.
 *
 *	It starts out in AWAITIDLE state.
 */
#define AWAITIDLE	0	/* Waiting til all keys up and ID sent */
#define NORMAL		1	/* The usual (ho, hum) */
#define ABORT1		2	/* Last key down was KEYABORT1 */
#define IDLE1		3	/* Last keycode was IDLEKEY -- ID follows */
#define IDLE2		4	/* Last keycode was id -- if IDLEKEY follows,
				   don't put it in the buffer. */


/*
 *	Translation options for getkey()
 *
 *	These are how you can have your input translated.
 */
#define TR_NONE		0
#define TR_ASCII	1

/*
 *	These bits can appear in the result of TR_NONE getkey()s.
 */
#define STATEOF(key)	( key & 0x80 )	/* 0 = key down, 0x80 = key up */
#define KEYOF(key)	( key & 0x7F )	/* The key number that moved */
#define NOKEY		( -1 )		/* The argument was 0, and no key was
					   depressed.  They were all elated. */

/*
 *	These bits can appear in the result of TR_ASCII getkey()s.
 */
/* NOKEY can appear if the getkey()'s arg was 0 (no waiting) */
#define METABIT		0	/* Meta key depressed with key */
#define   METAMASK	  0x000080
#define SYSTEMBIT	1	/* System key (upper left) was down w/key */
#define   SYSTEMMASK	  0x000100
/* other "bucky" bits can be defined at will.  See "BUCKYBITS" below. */

/*
 *	This defines the bit positions used within "shiftmask" to
 *	indicate the "pressed" (1) or "released" (0) state of shift keys.
 *	Both the bit numbers, and the aggregate masks, are defined.
 *
 *	The "UPMASK" is a minor kludge.  Since whether the key is going
 *	up or down determines the translation table (just as the shift
 *	keys' positions do), we OR it with "shiftmask" to get "tempmask",
 *	which is the mask which is actually used to determine the 
 *	translation table to use.  Don't reassign 0x0080 for anything
 *	else, or we'll have to shift and such to squeeze in UPMASK,
 *	since it comes in from the hardware as 0x80.
 */
#define CAPSLOCK	0	/* Caps Lock key */
#define   CAPSMASK	  0x0001
#define SHIFTLOCK	1	/* Shift Lock key */
#define LEFTSHIFT	2	/* Left-hand shift key */
#define RIGHTSHIFT	3	/* Right-hand shift key */
#define   SHIFTMASK	  0x000E
#define LEFTCTRL	4	/* Left-hand control key */
#define RIGHTCTRL	5	/* Right-hand control key */
#define   CTRLMASK	  0x0030
/* unused...		  0x0040 */
#define   UPMASK	  0x0080
#define   CTLSMASK	  0x0100  /* Set if ^S was last keyed of ^S, ^Q;
				     determines which NOSCOLL sends. */

/*
 *	This defines the format of translation tables.
 *
 *	A translation table is 128 bytes of "entries", which are bytes
 *	(unsigned chars).  The top 4 bits of each entry are decoded by
 *	a case statement in getkey.c.  If the entry is less than 0x80, it
 *	is sent out as an ASCII character (possibly with bucky bits
 *	OR-ed in).  "Special" entries are 0x80 or greater, and
 *	invoke more complicated actions.
 *
 *	The masks IdleShifts and IdleBuckys are AND-ed with the current
 *	state of  shiftmask  and  buckybits  when a "keyboard idle" code
 *	is received.  This ensures that where we "think" the shift & bucky
 *	keys are, more accurately reflects where they really are, since the
 *	keyboard knows better than us.
 */
struct keymap {
	unsigned char	keymap [128]; /* maps keycodes to actions */
};	/* Kludge because typedefs and structs don't map very well */

struct keyboard {
	struct keymap	*Normal, *Shifted, *Caps, *Control, *Up;
	int		IdleShifts;
	int		IdleBuckys;
	};			/* All keymaps of a keyboard, plus. */


/*
	The "special" entries' top 4 bits are defined below.  Generally they are
	used with a 4-bit parameter (such as a bit number) in the low 4 bits.
	The bytes whose top 4 bits are 0x0 thru 0x7 happen to be ascii 
	characters.  They are not special cased, but just normal cased.
*/

#define	SHIFTKEYS	0x80	/* thru 0x8F.  This key helps to determine the
				   translation table used.  The bit
				   position of its bit in "shiftmask"
				   is added to the entry, eg
				   SHIFTKEYS+LEFTCTRL.  When this entry is
				   invoked, the bit in "shiftmask" is
				   toggled.  Depending which tables you put
				   it in, this works well for hold-down
				   keys or press-on, press-off keys.  */
#define	BUCKYBITS	0x90	/* thru 0x9F.  This key determines the state of
				   one of the "bucky" bits above the
				   returned ASCII character.  This is
				   basically a way to pass mode-key-up/down
				   information back to the caller with each
				   "real" key depressed.  The concept, and
				   name "bucky" (derivation unknown) comes
				   from the MIT/SAIL "TV" system...they had
				   TOP, META, CTRL, and a few other bucky
				   bits.  The bit position of its bit in
				   "buckybits", minus 7, is added to the
				   entry; eg bit 0x00000400 is BUCKYBITS+3.
				   The "-7" prevents us from messing up the
				   ASCII char, and gives us 16 useful bucky
				   bits.  When this entry is invoked,
				   the designated bit in "buckybits" is
				   toggled.  Depending which tables you put
				   it in, this works well for hold-down
				   keys or press-on, press-off keys.  */
#define FUNNY		0xA0	/* thru 0xAF.  This key does one of 16 funny
				   things based on the low 4 bits: */
#define	NOP		0xA0	/* This key does nothing. */
#define OOPS		0xA1	/* This key exists but is undefined. */
#define HOLE		0xA2	/* This key does not exist on the keyboard.
				   Its position code should never be
				   generated.  This indicates a software/
				   hardware mismatch, or bugs. */
#define	NOSCROLL	0xA3	/* This key alternately sends ^S or ^Q */
#define CTRLS		0xA4	/* This sends ^S and lets NOSCROLL know */
#define CTRLQ		0xA5	/* This sends ^Q and lets NOSCROLL know */
/* Combinations 0xA6 to 0xAF are reserved for non-parameterized functions */

#define STRING		0xB0	/* thru 0xBF.  The low-order 4 bits index
				   a table referenced via gp->KStrTab and
				   select a string to be returned, char
				   by char.  Each entry in KStrTab is a
				   pointer to a character vector.  The vector
				   contains a 1-byte length, followed by
				   data. */
/* Definitions for the individual string numbers: */
#define HOMEARROW	0x00
#define UPARROW		0x01
#define DOWNARROW	0x02
#define LEFTARROW	0x03
#define RIGHTARROW	0x04
#define PF1		0x05
#define PF2		0x06
#define PF3		0x07
#define PF4		0x08
/* Note, these might be better served by their own non-FUNNY code with the
   low 4 bits selecting which PF key.... */
/* string numbers 9 thru F are reserved.  Users making custom entries, take
   them from F downward to postpone confusion. */

/* combinations 0xC0 - 0xFF are reserved for more argument-taking entries */

#endif GETKEYPORT

/*
 *	End of keyboard.h
 */
