/*	@(#)win_input.h 1.6 83/10/26 SMI	*/

/*
 * Copyright (c) 1983 by Sun Microsystems, Inc.
 */

/*
 * The advent of personal workstation computers with high resolution
 * bitmap displays and precise pointing devices (e.g. mice) provide the
 * opportunity to generate highly interactive user interfaces.
 * This header file contains the definitions of a programming interface
 * which should support a wide variety of user/machine interfaces.
 *
 * The kernel translates a user action into an input event.
 * An input device can deliver raw device codes, codes derived from a
 * virtual input device interface or ANSI X3.64.
 * These codes are packaged into a standard inputevent structure.
 *
 * Virtual keyboard station assignments.
 */

#define	VKEY_CODES	(128)

#define	ASCII_FIRST	(0)
#define	ASCII_LAST	(127)
#define	META_FIRST	(128)
#define	META_LAST	(255)
#define	TOP_FIRST	(256)
#define	TOP_LAST	(511)

#define	MOUSE_DEVID	0x7F
#define	VKEY_FIRST	(MOUSE_DEVID << 8)

#define	VKEY_FIRSTPSEUDO	(VKEY_FIRST)
#define	LOC_MOVE		(VKEY_FIRSTPSEUDO+0) /* No neg event */
#define	LOC_STILL		(VKEY_FIRSTPSEUDO+1) /* No neg event */
#define	LOC_WINENTER		(VKEY_FIRSTPSEUDO+2) /* No neg event */
#define	LOC_WINEXIT		(VKEY_FIRSTPSEUDO+3) /* No neg event */
#define	LOC_MOVEWHILEBUTDOWN	(VKEY_FIRSTPSEUDO+4) /* No neg event */
#define	VKEY_LASTPSEUDO		(VKEY_FIRSTPSEUDO+15)

#define	VKEY_FIRSTSHIFT		(VKEY_LASTPSEUDO+1)
#define	SHIFT_LEFT		(VKEY_FIRSTSHIFT+0)
#define	SHIFT_RIGHT		(VKEY_FIRSTSHIFT+1)
#define	SHIFT_LOCK		(VKEY_FIRSTSHIFT+2)
#define	SHIFT_CAPSLOCK		(VKEY_FIRSTSHIFT+3)
#define	SHIFT_CTRL		(VKEY_FIRSTSHIFT+4)
#define	SHIFT_META		(VKEY_FIRSTSHIFT+5)
#define	SHIFT_TOP		(VKEY_FIRSTSHIFT+6)
#define	SHIFT_CMD		(VKEY_FIRSTSHIFT+7)
#define	VKEY_LASTSHIFT		(VKEY_FIRSTSHIFT+15)

#define	VKEY_FIRSTFUNC		(VKEY_LASTSHIFT+1)

#define BUT_FIRST		(VKEY_FIRSTFUNC)
#define BUT(i)			((BUT_FIRST)+(i)-1)
#define BUT_LAST		(BUT_FIRST+9)

#define KEY_LEFTFIRST		((BUT_LAST)+1)
#define KEY_LEFT(i)		((KEY_LEFTFIRST)+(i)-1)
#define KEY_LEFTLAST		((KEY_LEFTFIRST)+15)

#define KEY_RIGHTFIRST		((KEY_LEFTLAST)+1)
#define KEY_RIGHT(i)		((KEY_RIGHTFIRST)+(i)-1)
#define KEY_RIGHTLAST		((KEY_RIGHTFIRST)+15)

#define KEY_TOPFIRST		((KEY_RIGHTLAST)+1)
#define KEY_TOP(i)		((KEY_TOPFIRST)+(i)-1)
#define KEY_TOPLAST		((KEY_TOPFIRST)+15)

#define KEY_BOTTOMLEFT		((KEY_TOPLAST)+1)
#define KEY_BOTTOMRIGHT		((KEY_BOTTOMLEFT)+1)

#define	VKEY_LASTFUNC		(VKEY_FIRSTFUNC+101)

#define	VKEY_LAST		VKEY_FIRST+VKEY_CODES-1

/*
 * Common names for certain input codes
 */
#define	MS_LEFT		BUT(1)
#define	MS_MIDDLE	BUT(2)
#define	MS_RIGHT	BUT(3)

/*
 * The inputmask consists of a input code array + flags that indicates those
 * user actions to be placed in the input queue.
 *
 * Up to 16 shifted input codes can be specified whose state is reported
 * on each inputevent.  This enables different non-ASCII input mappings
 * to be done at the user program level.  The state (0==up, 1==down)
 * of im_shiftcodes[n] is reported in bit position 1<<n in ie_shiftmask.
 * im_shifts indicates how many of the 16 im_shiftcodes are valid.
 * If im_shifts==0 then ie_shiftmask contains the shiftmask used in ASCII
 * decoding.
 */

#define	BITSPERBYTE	8

struct inputmask {
	short	im_flags;
#define	IM_NEGEVENT	(0x01)		/* send input negative events too */
#define	IM_POSASCII	(0x02)		/* no neg ASCII even if IM_NEGEVENT */
#define	IM_ANSI		(0x04)		/* ansi with funcs encoded in ESC[ */
#define	IM_UNENCODED	(0x08)		/* pure device codes */
#define	IM_ASCII	(0x10)		/* enable ASCII codes 0-127 */
#define	IM_META		(0x20)		/* enable META codes 128-255 */
#define	IM_CODEARRAYSIZE (VKEY_CODES/((sizeof(char))*BITSPERBYTE))
	char	im_inputcode[IM_CODEARRAYSIZE];
	short	im_shifts;		/* # of valid entries in im_shiftcodes*/
#define	IM_SHIFTARRAYSIZE (sizeof(short)*BITSPERBYTE)
	short	im_shiftcodes[IM_SHIFTARRAYSIZE];
};

#define	win_setinputcodebit(im,code) \
	(im)->im_inputcode[((code)-VKEY_FIRST)/BITSPERBYTE] |= \
	    (1<<((BITSPERBYTE-1)-(((code)-VKEY_FIRST)%BITSPERBYTE)))
#define	win_unsetinputcodebit(im,code) \
	(im)->im_inputcode[((code)-VKEY_FIRST)/BITSPERBYTE] &= \
	    (~(1<<((BITSPERBYTE-1)-(((code)-VKEY_FIRST)%BITSPERBYTE))))
#define	win_getinputcodebit(im,code) \
	((im)->im_inputcode[((code)-VKEY_FIRST)/BITSPERBYTE] & \
	    (1<<((BITSPERBYTE-1)-(((code)-VKEY_FIRST)%BITSPERBYTE))))


/*
 * An input event structure is that which is filled in on a readrecord call.
 * ie_code contains the input code involved.  IE_NEGEVENT is used to indicate
 * a negative event.  When IM_UNENCODED was set the top byte of ie_code is a
 * device id and the bottom byte is the physical keystation number.
 *
 * ie_shiftmask contains the input code shift state that was asked for in
 * the inputmask im_shiftcodes.  This information can then be used to
 * map input codes to some other form.
 *
 * The locator position (window/screen relative) is placed in ie_loc*.
 * The time of the input event is put in ie_time (seconds & microseconds).
 * Time zone and daylight savings flag available via system call.
 */

struct	inputevent {
	short	ie_code;		/* input code */
	short	ie_flags;
#define	IE_NEGEVENT	(0x01)		/* input code is negative */
	short	ie_shiftmask;		/* input code shift state */
	short	ie_locx, ie_locy;	/* locator (usually a mouse) position */
	struct	timeval ie_time;	/* time of event */
};

#define	win_inputnegevent(ie)	((ie)->ie_flags&IE_NEGEVENT)
#define	win_inputposevent(ie)	(!((ie)->ie_flags&IE_NEGEVENT))

#define	ANSI_ENCODED	'z'
#define	win_inputdeviceid(code)			 (((code)>>8)&0Xff00)
#define	win_inputphysicalkeystation(code)	 ((code)&0Xff)

#ifdef	cplus
/*
 * C Library routines dealing this type of input
 */
void	input_imnull(struct inputmask *im);
void	input_imall(struct inputmask *im);
int	input_readevent(int fd, struct inputevent *event);
#endif

