/*	@(#)pixwin.h 1.10 84/04/04 SMI	*/

/*
 * Copyright (c) 1983 by Sun Microsystems, Inc.
 */

/*
 * Raster device independent access of windows can be done using the
 * following structures and functions.
 *
 * A pixwin is a data structure used for window image access.
 * It is the data structure that is used to multiplex a pixrect
 * in an overlapping window environment.
 *
 * This is how the pixwin_clipops are to be used:
 *
 * Pwo_getclipping gets pw_opsdata->pw_clipid,
 * frees all rectlists in pixwin, gets the right pw_clipping.
 * Pw_prretained is set to describe the rectangle of the window.
 * The rectlist is examined and the following actions are taken depending
 * on the case:
 *	Rectlist is composed of a single rect:
 *		A secondary pixrect is generated from pw_pixrect that describes
 *		the rect.  This new pixrect is pwcd_prclipping.
 *		pwcd_state is set to PWCD_SINGLERECT.
 *	Rectlist is composed of multiple rects:
 *		A secondary pixrect is generated from pw_pixrect that describes
 *		the rect of the window.  This new pixrect is pwcd_prclipping.
 *		pwcd_state is set to PWCD_MULTIRECTS.
 *	Rectlist is null:
 *		pwcd_state is set to PWCD_NULL.
 *
 * Pwo_lock compares an id gotten from the kernel with pw_opsdata->pw_clipid
 * and calls pwo_getclipping if the two ids are not equal.
 * This lock call adjusts pw_ops and pw_opshandle depending on pwcd_state:
 *	PWCD_SINGLERECT:
 *		pw_ops = pw_pixrect->pr_ops.
 *		pw_opshandle = cached secondary pixrect.
 *		This avoids rectlist clipping if do other operations before
 *		unlocking.
 *	PWCD_MULTIRECTS:
 *		pw_ops = standard pw_ops interpretations.
 *		pw_opshandle = pixwin.
 *		These procedures handle clipping to a rectlist.
 *	PWCD_NULL:
 *		pw_ops = standard minimum pw_ops interpretations.
 *		pw_opshandle = pixwin.
 *		No writing takes place except to pw_prretained.
 *	PWCD_USERDEFINE:
 *		Assumes that the user is handling the op and handle selection
 *		so does nothing.
 * If pw_lockcount is > 0 only increment this count.
 *
 * Pwo_unlock releases the display access lock.
 * Pw_ops and pw_opshandle are reset to PWCD_MULTIRECTS values to ensure
 * that the next imaging call involves a pwo_lock. 
 * If pw_lockcount is > 1 only decrement this count.
 *
 * Pwo_reset forces the release of the display access lock, frees
 * all rectlists in pixwin and resets all data to a base state.
 *
 * Here is how the basic utilities work:
 *
 * Pw_open creates a new pixwin.  Pw_close should be called when you are
 * done with the pixwin.
 *
 * Calling pw_exposed or pw_damaged sets pwo_getclipping to either
 * pw_exposed or pw_damaged, respectively, and gets the related clipping.
 *
 * Pw_damaged also sets pw_opsdata->pw_damagedid.  This is the value used
 * in the pw_donedamaged call to notify the kernel that a certain version
 * of display damage has been fixed up.  Note, that this may not be equal
 * to the pw_opsdata->pw_clipping being used when pw_donedamaged is called.
 * In this case some multiple repainting of the same area may occur but this
 * should happen infrequently.
 */

struct	pixwin {
	struct	pixrectops *pw_ops;	/* imaging ops appropriate to this pw */
	caddr_t	pw_opshandle;		/* handle used during pw_ops calls */
	int	pw_opsx;		/* left offset used during pw_ops call*/
	int	pw_opsy;		/* top offset used during pw_ops call*/
	struct	rectlist pw_fixup;	/* undefined window image after
					 * doing pw_op in which window
					 * was the source */
	struct	pixrect *pw_pixrect;	/* device containing this window */
	struct	pixrect *pw_prretained;	/* retained pixrect of window */
	struct	pixwin_clipops *pw_clipops;/* locking and clipping ops */
	struct	pixwin_clipdata *pw_clipdata;/* locking and clipping data */
	char	pw_cmsname[20];		/* name of colormap segment (cms.h) */
};

/*
 * This structure contains clipping and locking operations.
 */
struct	pixwin_clipops {
	int	(*pwco_lock)();		/* lock against window manager */
	int	(*pwco_unlock)();	/* release */
	int	(*pwco_reset)();	/* clear lock & clipping to base state,
					 * can be used on non-local goto */
	int	(*pwco_getclipping)();	/* get clipping */
};

/*
 * This structure contains clipping and locking data.
 */
struct	pixwin_clipdata {
	int	pwcd_windowfd;
	short	pwcd_state;
#define	PWCD_NULL	0
#define	PWCD_MULTIRECTS	1
#define	PWCD_SINGLERECT	2
#define	PWCD_USERDEFINE	3
	struct	rectlist pwcd_clipping;	/* current clipping */
	int	pwcd_clipid;
	int	pwcd_damagedid;
	int	pwcd_lockcount;
	struct	pixrect *pwcd_prmulti;	/* pixrect for multi rect clipping */
	struct	pixrect *pwcd_prsingle;	/* pixrect for single rect clipping */
	struct	pixwin_prlist *pwcd_prl; /* pixrect list for clipping vectors*/
	struct	rectlist pwcd_clippingsorted[RECTS_SORTS];
					/* cached current clipping sorted
					 * to facilitate pw_copy */
	struct	rect *pwcd_regionrect;	/* If !0 intersect with clipping */
};

/*
 * Vector clipping must be done at the lowest level in order to maintain
 * stepping integrity across abutting rectangle boundaries.
 * Thus, when a vector is to be clipped to multiple rects the clipper is
 * a pixrect.  The pixwin_prlist data structure is used to hold a
 * pixrect clipping list.
 */
struct	pixwin_prlist {
	struct	pixwin_prlist *prl_next;	/* Next record */
	struct	pixrect *prl_pixrect;		/* clipping pixrect */
	int	prl_x, prl_y;			/* offset of prl_pixrect from
						   window origin */
};

#define pw_rop(dpw, dx, dy, w, h, op, sp, sx, sy)			\
	(*(dpw)->pw_ops->pro_rop)((dpw)->pw_opshandle,			\
	    dx-(dpw)->pw_opsx, dy-(dpw)->pw_opsy, w, h, op, (sp), sx, sy)
#define pw_write(dpw, dx, dy, w, h, op, spr, sx, sy)			\
	(*(dpw)->pw_ops->pro_rop)((dpw)->pw_opshandle,			\
	    dx-(dpw)->pw_opsx, dy-(dpw)->pw_opsy, w, h, op, (spr), sx, sy)
#define pw_writebackground(dpw, dx, dy, w, h, op)			\
	(*(dpw)->pw_ops->pro_rop)((dpw)->pw_opshandle,		\
	    dx-(dpw)->pw_opsx, dy-(dpw)->pw_opsy, w, h, op,		\
	    (struct pixrect *)0, 0, 0)
#define pw_read(dpr, dx, dy, w, h, op, spw, sx, sy)			\
	(*(spw)->pw_ops->pro_rop)((dpr), dx, dy, w, h, op,		\
	    (spw)->pw_opshandle, sx-(spw)->pw_opsx, sy-(spw)->pw_opsy)
#define pw_copy(dpw, dx, dy, w, h, op, spw, sx, sy)			\
	(*(spw)->pw_ops->pro_rop)((dpw)->pw_opshandle,			\
	    dx-(dpw)->pw_opsx, dy-(dpw)->pw_opsy, w, h, op,		\
	    (spw)->pw_opshandle, sx-(spw)->pw_opsx, sy-(spw)->pw_opsy)
#define pw_batchrop(dpw, x, y, op, sbp, n)				\
	(*(dpw)->pw_ops->pro_batchrop)((dpw)->pw_opshandle,		\
	    x-(dpw)->pw_opsx, y-(dpw)->pw_opsy, op, sbp, n)
#define pw_stencil(dpw, x, y, w, h, op, stpr, stx, sty, spr, sy, sx)	\
	(*(dpw)->pw_ops->pro_stencil)((dpw)->pw_opshandle,              \
	    x-(dpw)->pw_opsx, y-(dpw)->pw_opsy, w, h, op,		\
	    stpr, stx, sty, spr, sy, sx)
#define pw_destroy(pw)							\
	(*(pw)->pw_ops->pro_destroy)((pw)->pw_opshandle)
#define	pw_get(pw, x, y)						\
	(*(pw)->pw_ops->pro_get)((pw)->pw_opshandle,			\
	    x-(pw)->pw_opsx, y-(pw)->pw_opsy)
#define	pw_put(pw, x, y, val)						\
	(*(pw)->pw_ops->pro_put)((pw)->pw_opshandle,			\
	    x-(pw)->pw_opsx, y-(pw)->pw_opsy, val)
#define pw_vector(pw, x0, y0, x1, y1, op, val)				\
	(*(pw)->pw_ops->pro_vector)((pw)->pw_opshandle,			\
	    x0-(pw)->pw_opsx, y0-(pw)->pw_opsy,				\
	    x1-(pw)->pw_opsx, y1-(pw)->pw_opsy, op, val)
#define	pw_region(pw, x, y, w, h)					\
	(struct pixwin *)(*(pw)->pw_ops->pro_region)(pw, x, y, w, h)
#define	pw_putattributes(pw, planes)					\
	(*(pw)->pw_ops->pro_putattributes)((pw)->pw_opshandle, planes)
#define	pw_getattributes(pw, planes)					\
	(*(pw)->pw_ops->pro_getattributes)((pw)->pw_opshandle, planes)
#define	pw_putcolormap(pw, index, count, red, green, blue)		\
	(*(pw)->pw_ops->pro_putcolormap)((pw)->pw_opshandle,		\
	    index, count, red, green, blue)
#define	pw_getcolormap(pw, index, count, red, green, blue)		\
	(*(pw)->pw_ops->pro_getcolormap)((pw)->pw_opshandle,		\
	    index, count, red, green, blue)

#define pw_lock(pixwin,rect)						\
	(*(pixwin)->pw_clipops->pwco_lock)((pixwin), (rect));
#define pw_unlock(pixwin)						\
	(*(pixwin)->pw_clipops->pwco_unlock)((pixwin));
#define pw_reset(pixwin)						\
	(*(pixwin)->pw_clipops->pwco_reset)((pixwin));
#define pw_getclipping(pixwin)						\
	(*(pixwin)->pw_clipops->pwco_getclipping)((pixwin));

extern	struct pixwin *pw_open();
extern	struct	pixfont *pw_pfsysopen();

#ifdef	cplus
/*
 * C Library routine specifically relating to pixel device functions
 */

/*
 * Pixwin_clipops types
 */
int	pwco_lock(struct pixwin *pw, struct rect *affected);
void	pwco_unlock(struct pixwin *pw);
void	pwco_reset(struct pixwin *pw);
int	pwco_getclipping(struct pixwin *pw);

/*
 * Pixwin structure utilities
 */
void	pw_open(int windowfd);
void	pw_exposed(struct pixwin *pw);
void	pw_repairretained(struct pixwin *pw);
void	pw_damaged(struct pixwin *pw);
void	pw_donedamaged(struct pixwin *pw);
void	pw_close(struct pixwin *pw);

/*
 * Pixwin color map segment utilities
 */
void	pw_setcmsname(struct pixwin *pw, char cmsname[CMS_NAMESIZE]);
void	pw_getcmsname(struct pixwin *pw, char cmsname[CMS_NAMESIZE]);
void	pw_getcmsdata(struct pixwin *pw, struct colormapseg *cms, int *planes);
void	pw_blackonwhite(struct pixwin *pw, int min, max);
void	pw_whiteonblack(struct pixwin *pw, int min, max);
void	pw_reversevideo(struct pixwin *pw, int min, max);
void	pw_cyclecolormap(struct pixwin *pw, int cycles, begin, length);
void	pw_preparesurface(struct pixwin *pw, struct rect *rect);

/*
 * Additional write functions
 */
void	pw_replrop(struct pixwin *pw, int xw, yw, width, height, int op,\
		struct pixrect *pr, int xr, yr);
void	pw_text(struct pixwin *pw, int xw, yw, int op,\
		struct pixfont *pixfont, char *s);
void	pw_ttext(struct pixwin *pw, int xw, yw, int op,\
		struct pixfont *pixfont, char *s);
void	pw_char(struct pixwin *pw, int xw, yw, int op,\
		struct pixfont *pixfont, char c);
/*
 * Font utilities for extern struct pixfont *pf_sys.
 */
struct	pixfont *pw_pfsysopen();
	pw_pfsysclose();
#endif

