/*	@(#)mbvar.h 4.9 84/02/27 SMI	*/

/*
 * Copyright (c) 1983 by Sun Microsystems, Inc.
 */

/*
 * This file contains definitions related to the kernel structures
 * for dealing with the Multibus.
 *
 * The Multibus has an mb_hd structure.
 * Each Multibus controller which is not a device has an mb_ctlr structure.
 * Each Multibus device has an mb_device structure.
 */

#ifndef LOCORE
/*
 * The Multibus structure.
 *
 * At boot time we determine the devices attached to the Multibus.
 *
 * During normal operation, resources are allocated and returned
 * to the structures here. 
 * 
 * When Multibus resources are needed and not available, or if a device
 * which can tolerate no other Multibus activity gets on the bus,
 * then device drivers may have to wait to get to the bus and are
 * queued here.
 */
struct	mb_hd {
	struct	mb_ctlr *mh_actf;	/* head of queue to transfer */
	struct	mb_ctlr *mh_actl;	/* tail of queue to transfer */
	struct	mb_driver *mh_intr[8];	/* linked list of interrupt routines */
	struct	map *mh_map;		/* DVMA resource map */
	short	mh_users;		/* transient use count */
	short	mh_xclu;		/* a bus hog is using the Multibus */
	char	mh_mrwant;		/* someone is waiting for map reg */
};

/*
 * Per-controller structure.
 * (E.g. one for each disk and tape controller)
 *
 * If a controller has devices attached, then there are
 * cross-referenced mb_device structures.
 * This structure is the one which is queued in Multibus resource wait,
 * and saves the information about Multibus resources which are used.
 * The queue of devices waiting to transfer is also attached here.
 */
struct mb_ctlr {
	struct	mb_driver *mc_driver;
	short	mc_ctlr;	/* controller index in driver */
	short	mc_alive;	/* controller exists */
	caddr_t	mc_addr;	/* address of device in i/o space */
	int	mc_intpri;	/* interrupt priority level */
	struct	mb_hd *mc_mh;
	int	mc_mbinfo;	/* Multibus resource allocation info */
	char	mc_rw;		/* READ or WRITE */
	struct	buf mc_tab;	/* queue of devices for this controller */
	struct	mb_ctlr	*mc_forw;	/* link in transfer queue */
};

/*
 * Per ``device'' structure.
 * (A controller has devices or uses and releases Multibus memory).
 * (Everything else is a ``device''.)
 *
 * If a controller has many drives attached, then there will
 * be several mb_device structures associated with a single mb_ctlr
 * structure.
 *
 * This structure contains all the information necessary to run a Multibus
 * device such as a serial line interface.  It also contains information
 * for slaves of Multibus controllers as to which device on the slave
 * this is.  A flags field here can also be given in the system specification
 * and is used to tell which mux lines are hard wired or other device
 * specific parameters.
 */
struct mb_device {
	struct	mb_driver *md_driver;
	short	md_unit;	/* unit number on the system */
	short	md_ctlr;	/* mass ctlr number; -1 if none */
	short	md_slave;	/* slave on controller */
	caddr_t	md_addr;	/* address of device in i/o space */
	int	md_intpri;	/* interrupt priority */
	short	md_dk;		/* if init 1 set to number for iostat */
	int	md_flags;	/* parameter from system specification */
	short	md_alive;	/* device exists */
	short	md_type;	/* driver specific type information */
/* if the device is connected to a controller, this is the controller */
	struct	mb_ctlr *md_mc;
	struct	mb_hd *md_hd;
};

/*
 * Per-driver structure.
 *
 * Each Multibus driver defines entries for a set of routines
 * as well as an array of types which are acceptable to it.
 * These are used at boot time by the configuration program.
 */
struct mb_driver {
	int	(*mdr_probe)();		/* see if a driver is really there */
	int	(*mdr_slave)();		/* see if a slave is there */
	int	(*mdr_attach)();	/* setup driver for a slave */
	int	(*mdr_go)();		/* routine to start transfer */
	int	(*mdr_done)();		/* routine to finish transfer */
	int	(*mdr_intr)();		/* interrupt routine */
	u_long	*mdr_ioaddr;		/* device csr addresses */
	u_long  *mdr_maddr;		/* device memory address */
	int	mdr_size;		/* amount of memory space needed */
	char	*mdr_dname;		/* name of a device */
	struct	mb_device **mdr_dinfo;	/* backpointers to mbdinit structs */
	char	*mdr_cname;		/* name of a controller */
	struct	mb_ctlr **mdr_cinfo;	/* backpointers to mbcinit structs */
	short	mdr_flags;		/* want exclusive use of Multibus */
	struct	mb_driver *mdr_link;	/* interrupt routine linked list */
};

/* Driver flags */
#define MDR_XCLU	01		/* needs exclusive use of bus */
#define MDR_BIODMA	02		/* block device does Multibus DMA */
#define MDR_SWAB	04		/* Multibus buffer must be swabbed */
#define	MDR_OBIO	010		/* device in on-board I/O space */

/* Flags to mbsetup */
#define	MB_CANTWAIT	01		/* don't block me */

/*
 * Macros to bust return word from map allocation routines.
 */
#define	MBI_NMR(i)	((int)((i)>>20)&0x1ff)
#define	MBI_MR(i)	((int)((i)>>11)&0x1ff)
#define	MBI_ADDR(i)	((int)((i)&0xfffff))
#endif

/* Convert interrupt prio to SR */
#define pritospl(n)	(SR_SMODE|((n)<<8))

/* Maximum interrupt priority used by Multibus DMA */
#define	SPLMB	4

#ifndef LOCORE
#ifdef KERNEL
/*
 * Multibus related kernel variables
 */
extern	struct	mb_hd mb_hd;

/*
 * Mbcinit and mbdinit initialize the mass storage controller and
 * device tables specifying possible devices.
 */
extern	struct	mb_ctlr mbcinit[];
extern	struct	mb_device mbdinit[];

/*
 * Multibus I/O space is mapped by MBIOmap
 * into virtual address mbio[].
 */
extern	char mbio[];		/* mb device addr space */
extern	struct pte MBIOmap[];	/* mb device addr pte's */

struct	map *iopbmap;
#define IOPBMEM		4	/* 4 pages for IOPB crud */
#define IOPBMAPSIZE	64

#define	DVMASIZE	128	/* pages, 256KB */
#define	DVMAMAPSIZE	25
extern	char DVMA[];

#endif KERNEL
#endif !LOCORE
