/*
 *	rdebug.h -- definitions for remote debugging stuff
 *
 *	$Header: rdebug.h,v 1.5 88/04/14 10:45:16 andy Locked $
 *
 *	this header file contains definitions relating to the
 *	remote debugging protocol.
 *
 *	compile-time defines for the remote debugging code:
 *		STANDALONE	compile for the stand-alone library
 *		DB_DO_PERROR	call perror(3) on errors
 *		NOREADV		use internal readv/writev
 *		DBSTATS		collect traffic and error stats
 */

#define STANDALONE

/* call perror for the user? */
#ifdef STANDALONE
#	define DB_DO_PERROR
#endif STANDLONE

#ifdef DB_DO_PERROR
#	define DBPERROR(s)	perror(s)
#else
#	define DBPERROR(s)	/* nothing */
#endif /* DB_DO_PERROR */

/*
 *	protocol version number
 *	bump this whenever the protocol is changed.
 */
#define DBVERSION	2

/*
 *	default remote device to use if environment variable is not set
 *
 */
#ifdef SIMULATOR
#define DEF_REMOTEDEV	"tt(,1,)"
#else
#define DEF_REMOTEDEV	"zs(,1,)"
#endif SIMULATOR

/*
 * The protocol used to talk to the remote debugger is fairly simple.  It
 * works over either a serial port connection or else it
 * uses udp packets to talk back and forth between the debugger and the
 * debug agent.  The packets have a simple structure containing a type
 * field and a sequence number.  Each data packet sent by either end to the
 * other end must be Acknowledged before further packets can be transmitted.
 * Data packets are tagged with a sequence number and the ack packet must
 * have the sequence number of the data packet being acknowledged.
 * If a data packet is recieved with a sequence number greater than
 * the last sequence number recieved plus one, an ack is not sent as it is assumed
 * that a previous data packet must have been lost.  The protocol will timeout
 * and retry sending data packets if an ack is not received.  After a fixed
 * number of retries the sender will quit and assume that the connection is
 * bad.  The protocol is forgiving in that if data packets with a lower
 * sequence number than the last data packet recieved are recieved, the
 * reciever will resend an ack for that packet, assuming that the earlier
 * ack was lost.  Similarly, if an ack is recieved for a data packet with
 * a sequence number lower than the last one sent, it will be ignored.
 * The protocol effectively causes the connection to be "lock step" in that
 * neither end should be able to process a packet more than one sequence number greater
 * than the other end has processed.  The protocol maintains a seperate
 * sequence number for data packets going from A to B from the sequence
 * number for data packets going from B to A.
 */

/*
 *	the stand-alone open detects a remote open by comparing the
 *	file name to REMOTEFILE
 *
 */
#define REMOTEFILE	"rdebug"

/*
 *	magic number used as a sanity check when searching for the
 *	beginning of a packet.  should never be changed.
 *	Currently only used on serial port connections.
 */
#define DBMAGIC		((unsigned char) 0x01)

#define DBPORT		2010		/* udp port number */

/* serial parameters */
#define SER_MAX_DATA	100
#define SER_READ_TOUT	xxx

/*
 * Note that in terms of the protocol description above,
 * all XXX_RESP type packets are acknowledge packets and
 * all other type packets are considered data packets.
 */
/* packet types */
#define DBP_CONN	0x10		/* connection packet */
#define DBP_CONN_RESP	0x11		/* response to DBP_CONN */
#define DBP_DATA	0x20		/* data packet */
#define DBP_DATA_RESP	0x21		/* response to DBP_DATA */
#define DBP_DATA_MORE	0x23		/* data with more to follow */
#define DBP_EXIT	0x30		/* exit packet */
#define DBP_EXIT_RESP	0x31		/* response to DBP_EXIT */
#define DBP_RESEQ	0x40		/* restart sequence numbers */
#define DBP_NAME_RESP	0x41		/* response to DBP_RESEQ */

/* error codes */
#define DBE_NOERROR	0x0		/* no error */
#define DBE_CHKSUM	0x10		/* bad checksum */
#define DBE_FRAG	0x20		/* packet fragment */
#define DBE_SEQ		0x30		/* packet out of sequence */
#define DBE_VERSION	0x40		/* unsupported protocol version */

/* packet header shape */
struct dbpacket_hdr {
	unsigned char magic[5];		/* protocol magic number */
	unsigned char version;		/* protocol version number */
	unsigned short type;		/* packet type (defined above) */
	unsigned short len;		/* length of data area */
	unsigned short seq;		/* sequence number */
	unsigned short error;		/* error code (defined above) */
	unsigned short checksum;	/* packet checksum (or zero) */
	/* data area follows... */
};

/* protocol switch */
struct dbproto {
	int (*readp)();			/* receive packet routine */
	int (*writep)();		/* send packet routine */
	int maxdata;			/* maximum packet size */
	int connret;			/* connect retries */
	int conntim;			/* connect timout */
	int dataret;			/* data retries */
	int datatim;			/* data timeout */
};

#ifdef DBSTATS
/* protocol related statistics */
struct dbstats {
	int recvd;			/* packets received */
	int sent;			/* packets sent */
	int stout;			/* send timeouts */
	int rtout;			/* receive timeouts */
	int serr;			/* send errors (out of retries) */
	int rerr;			/* receive errors (out of retries) */
	int rchsum;			/* receive checksum errors */
	int frag;			/* packets fragments received */
	int rseq;			/* receive sequence errors */
	int vers;			/* version errors */
};
#endif /* DBSTATS */

/*
 *	macros for manipulating sequence numbers
 */
#define NEXTSEQ(s)	(s+1)		/* bump a sequence number */
#define PREVSEQ(s)	(s-1)		/* un-bump a sequnce number */
#define DB_ISEQ		100		/* first inbound sequence number */
#define DB_OSEQ		100		/* first outbound sequence number */

#ifdef DBDEBUG
char *type_string();			/* convert packet type to string */
char *error_string();			/* convert error code to string */
#endif /* DBDEBUG */
