;
; North Star disk controller addresses.
;
CMMND:	EQU	300H
DRVSEL:	EQU	1H
STPOFF:	EQU	8H
STPON:	EQU	9H
NOP:	EQU	10H
RSETSF:	EQU	14H
STPOUT:	EQU	1CH
STPIN:	EQU	1DH
BSTAT:	EQU	20H
RDBYTE:	EQU	40H
MOTOR:	EQU	80H
;
; Delay time in sectors for various disk functions.
;
MOTORD:	EQU	31	; Motor up-to-speed time.
HEADD:	EQU	14	; Head load settle & sector counter sync time.
STEPD:	EQU	2	; Step time.  1 -> 20mS, 2 -> 40mS.
;
; Various numbers of things.
;
DSEG:	EQU	0040H	; Everything goes at 400 hex.
FIRST:	EQU	0001H	; First track (must be 0) and sector (1).
LAST:	EQU	0209H	; Last track (2) and sector (9) to read.
DSKSEG:	EQU	0FE80H	; "F" is for extended address modification.
;
	ORG	0
	PUT	0100H
START:
	UP			; Set auto increment for STOB.
	MOV	BX,DSKSEG	; North Star disk controller base.
	MOV	DS,BX
	MOV	SI,CMMND+MOTOR+RDBYTE+NOP
	MOV	BX,DSEG		; Memory segment to put data into.
	MOV	ES,BX
	XOR	DI,DI		; Data load address = 0.
	MOV	SS,DI
	MOV	SP,0200H	; Set Stack Pointer.
	TEST	B,[CMMND+MOTOR+NOP],10H
	JNZ	TK0LP		; Jump if motor was on (& drive selected).
	MOV	DL,MOTORD+1	; Motor was off, start it up.
	CALL	WSECTOR		; Wait for motor to come up to speed.
	MOV	AL,[CMMND+MOTOR+DRVSEL]
	MOV	DL,HEADD-1	; Select drive 1 and wait for head settle.
	CALL	WSECTOR
TK0LP:
	TEST	B,[CMMND+MOTOR+STPOUT],01H
	JNZ	TK0		; Check if we're on track zero & set direction.
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Pulse the step line.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	MOV	DL,STEPD	; Wait for required step time.
	CALL	WSECTOR
	JP	TK0LP		; Loop till we've stepped MAXSTEPS.
TK0:
	MOV	AL,[CMMND+MOTOR+STPIN]	; Set step direction.
	MOV	BX,FIRST	; Start with track zero, sector 1.
SECTRLP:
	CALL	ONESECT		; Wait for next sector.
	MOV	AL,[CMMND+MOTOR+BSTAT+NOP]	; Get sector number.
	AND	AL,0FH		; Mask to sector number.
	CMP	AL,BL		; Check if we're at the right sector.
	JNE	SECTRLP		; Loop if we aren't.
	MOV	CX,352		; Maximum loops to wait for sync byte.
RSYNCLP:
	TEST	B,[CMMND+MOTOR+NOP],04H
	LOOPZ	RSYNCLP		; Loop till sync byte is found or timeout.
	JZ	SECTRLP		; Can't find sync byte, error, try sector over.
READSECT:
	MOV	CX,256		; Byte count.
	MOV	DL,CL		; CRC = 0.
READLOOP:
	AAD			; Waste time for >= 7.5 uS.
	MOV	AL,[SI]		; Read a byte.
	STOB			; Store it.
	XOR	DL,AL		; Compute CRC.
	ROL	DL
	LOOP	READLOOP	; Loop for 256 bytes.
	AAD			; Waste time >= 7.5 uS.
	MOV	AL,[SI]		; Get CRC from disk.
	CMP	AL,DL		; Same as computed?
	JE	CRCOK		; If CRC is OK, don't retry sector.
	SUB	DI,256		; Reset data pointer to read sector again.
	JP	SECTRLP
CRCOK:
	CMP	BX,LAST		; Was that the last one?
	JNE	NEXTSECT	; Read next one if not.
	DI			; Disable interrupts for 86-DOS.
	JMP	0,DSEG		; We're done, take off.
NEXTSECT:
	INC	BL		; Increment sector count.
	CMP	BL,10		; Check if too big.
	JB	SECTRLP		; OK if less than 10.
	XOR	BL,BL		; Clear sector count.
	INC	BH		; Increment track count.
	CALL	ONESECT		; Wait one sector before stepping.
	MOV	AL,[CMMND+MOTOR+STPON]
	AAM			; Pulse the step line.
	MOV	AL,[CMMND+MOTOR+STPOFF]
	JP	SECTRLP		; Go and wait for next sector.
;
; Wait for sector routine.  ONESECT waits for the next sector.
; WSECTOR waits the number of sectors given by DL.
;
ONESECT:
	MOV	DL,1		; Wait for next sector.
WSECTOR:
	MOV	AL,[CMMND+MOTOR+RSETSF]
SECTLOOP:
	TEST	B,[CMMND+MOTOR+NOP],80H	; Test for sector hole.
	JZ	SECTLOOP	; Loop till new sector.
	DEC	DL		; Decrement sector count.
	JNZ	WSECTOR		; Loop till zero.
	RET
	END
