/*
    ASSIGN - This program will assign a winchester drive partition
	to a logical drive name. It examines the user input, reads in
	the partition table from the specified drive and calculates
	the necessary parameters. It then checks to make sure the partition
	is not already assigned and if not, assigns it.
	
	Author: RJM
	Date:	9/12/83
	File:	ASSIGN.C
	
	Copyright (C) 1983, Zenith Data Systems Corporation

	Version 1.75 - Initial version
	Version 1.76 - Add partition assignment display
	Version 1.77 - Add Unallocated partitions to display
	Version 1.78 - Add formatted partition checking also better help
	Version 1.79 - Expand number of winchesters to 8, fix help screen
		       for same. Also changed location of configuration
		       pointer to a pointer at beginning of BIOS.
	Version 1.90 - Update all utilities to V1.90
	Version 1.91 - 1/4/84 Fix small in calculation of fat sectors
	Version 2.00 - Update all utilities to V2.00
	Version 2.01 - Fix drive letters with more than two floppies
	Version 2.02 - Do different media byte for each drive
	Version 2.03 - BIOS moves to 70H in z150bios.h
	Version 2.04 - Winchester media bytes all 0xf8, sectors per track
		       are determined from controller info.
	Version 2.05 - Fixed problem of using 'unit' before it is set.
	Version 2.06 - Fixed problem of calculating size break
	Version 2.07 - No longer assumes 4 partitions, new BIOS version.
*/

/*
		RESTRICTED RIGHTS LEGEND
		------------------------
	
	    "Use, duplication, or disclosure by the
	Government is subject to restrictions as set forth
	in paragraph (b) (3) (B) of the Rights in Technical
	Data and Computer Software clause in DAR
	7-104.9(a).  Contractor/manufacturer is Zenith
	Data Systems Corporation of Hilltop Road, St.
	Joseph, Michigan 49085.
*/

#include "stdio.h"
#include "conio.h"
#include "ctype.h"
#include "z150rom.h"
#include "z150bios.h"
#include "fixed.h"

#define TRUE 1
#define FALSE 0

#define	NBI	FALSE
#define	ZENITH	TRUE
#define	SYNTREX	FALSE

#define VERSION 2
#define RELEASE 7

#define CARRY 1			/* Position of Carry flag in 8088 flag word */

/* Get a buffer that will not cross a 64K memory boundry */
char buf1[2*SECTOR_SIZE];
char *buf;			/* Pointer to correct portion of "buf1" */


/* Reserve storage for a partition table */
struct part_tbl *part_table;
char part_buf[SECTOR_SIZE];

/* Reserve a sector buffer for the boot sector */
char boot_buf[SECTOR_SIZE];

/* Define a bios parameter block */
struct bpb bpb1;

/* Local copy of the BIOS configuration pointers */
struct config2_vector c_vec;

/* Winchester drive number based at 0 */
/* actually corresponds to drive letter (drv+num_floppy+'A') */
char drv;

/* Winchester unit number */
char unit=0;

/* Partition table number to assign */
char part_num;

/* Number of floppies (set in init() ) */
char num_floppy;

/* Number of partitions available */
char num_assign;

/* Globals for head sector and cylinder of current sector */
unsigned sector, cyl, head;

/* Maximum number of heads for selected drive */
char max_hds;

/* Maximum number of sectors per track for selected drive */
unsigned max_spt;

/* Define the externals used in the z150int() module */
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;
extern unsigned rax, rbx, rcx, rdx, rsi, rdi, rsp, rbp;
extern unsigned rcs, rds, res, rss, flags;
    
/*
   This routine is used to prevent Lattice "C" from processing
   the user input string.
*/
_main(line)
char *line;
{
    line += 1;		/* skip past the first character "c" */
    main(line);
}

/*
    main() - This routine has the command line scanned and dispatches
	the command.
*/
main(line)
char *line;
{
    char flag;

    /* Greet the user */
    signon();

    /* Prepare buffer that will not cross 64K boundry */
    get_buf();
    
    while(isspace(*line)) ++line;	/* Skip white space */

    z150int(EQUIPMENT_INTR);
    num_floppy = ((ral & 0xc0) >> 6) + 1;
    if (num_floppy < 2) num_floppy = 2;		/* Any imaginarys? */

    /* Does the user want help? */
    if (*line == '?' || *line == '\0')
    {
	init();
	help();
	return;
    }
    
    /* Initialize some things */
    init();

    /* Scan the users command line setting variables unit, name and drv */
    flag =scan(line);

    if (!controller(unit))
    {
	puts("\r\nCan not communicate with winchester controller.\r\n");
	exit(0);
    }
    max_hds = rdh + 1;
    max_spt = rcl & 0x3f;

    /* Get the Partition table */
    if (!get_part(unit, &part_buf[0]))
    {
	puts("\r\nCan not read the partition table\r\n");
	exit(0);
    }
    part_table = (struct part_tbl *)&part_buf[SECTOR_SIZE-sizeof(struct part_tbl)];
        
    
    /* Did user only want partition display for a drive? */
    if (!flag)
    {
	display(part_table);
	return;
    }

    /* Ensure partition is a DOS partition */
    if (part_table->part_entry[part_num].os_id != DOS_ID)    
    {
	display(part_table);
	puts("\r\nNot a DOS partition\r\n");
	exit(0);
    }
    
    /* Assign the partition */
    assign();
    return;
}


/*
    signon() - This routine will print the signon message.
*/
signon()
{
    float ver;
    
    ver = VERSION + (float)RELEASE/100;

#if ZENITH
    printf("\r\n             ASSIGN Version %2.2f\r\n", ver);
    puts("Copyright(C) 1984, Zenith Data Systems Corporation\r\n");
#endif

#if NBI
    printf("\r\n    ASSIGN Version %2.2f\r\n", ver);
    puts("(C)Copyright NBI, Inc. 1984\r\n");
#endif

#if SYNTREX
    printf("\r\n      ASSIGN Version %2.2f\r\n", ver);
    puts("(C)Copyright SYNTREX, Inc. 1986\r\n");
#endif

    return;
}

/*
    get_buf() - This routine will prepare the "buf1" buffer and
	the "buf" pointer so it doesn't cross a 64K memory boundry.
*/
get_buf()
{
    unsigned address;
    
    buf = &buf1[0];
    address = (((unsigned)mydsreg()) << 4) + (unsigned)&buf1[0];	/* Get physical address of buffer */
    address += SECTOR_SIZE;		/* Bump address by SECTOR_SIZE */

    /* Modify buffer address if the current buffer crosses boundry */
    if (address < (unsigned) buf) buf += SECTOR_SIZE;
    return;
}


/*
    help() - This routine will print helpful information for the user.
*/
help()
{
    char let;
    int i;
    
    let = num_floppy + 'A';
    puts("\r\nThe ASSIGN utility is used to make a logical connection\r\n");
    puts("between the partitions of a winchester disk drive and the\r\n");
    puts("drive name(s)");
    for (i=0; i<num_assign; i++)
	if (i == num_assign-1 && num_assign > 1)
	    printf(" and %c:",let+i);
	else if (num_assign == 1)
	    printf(" %c:",let+i);
	else
	    printf(" %c:,",let+i);
    puts("\r\n\n");
    puts("Usage: ASSIGN [?][u:][u:p d:]\r\n\n");
    puts("Where:\r\n\n");
    puts("   ?            - Print this list\r\n");
    printf("   u:           - Unit number of winchester (0 - %d)\r\n", NUM_FIXED-1);
    printf("   p            - Partition number to ASSIGN (1 - %d)\r\n", NUM_PART);
    if (num_assign > 1)
	printf("   d:           - Drive letter (%c: - %c:)\r\n\n", let, let+num_assign-1);
    else
	printf("   d:           - Drive letter (%c:)\r\n\n",let);
    puts("If only [u:] is specified, a table of valid partitions\r\n");
    puts("for drive [u:] is printed.\r\n");
    return;
}


/*
    init() - This routine will initialize some variables needed
	for the rest of the program.
*/
init()
{
    char fixed_flags[NUM_ASSIGN+1];
    int i;

    /*
	Get the configuration information from the BIOS and ensure
	proper versions.
    */
    movbyte(BIOS2_SEG, getword(BIOS2_SEG, CONFIG_PTR),
	    mydsreg(), &c_vec, sizeof(struct config2_vector));
    if (c_vec.bios_version != BIOS2_CVER || c_vec.bios_version < 20)
    {
	puts("\r\nInvalid version of IO.SYS\r\n");
	exit(0);
    }

    if (c_vec.auto_flag) {
	movbyte(BIOS2_SEG,c_vec.assign_flags,mydsreg(),fixed_flags,NUM_ASSIGN);
	fixed_flags[NUM_ASSIGN] = 0;
	for (i=0; i<=NUM_ASSIGN; i++)
	    if (!(fixed_flags[i] & 1))
		break;
	num_assign = i;
    } else
	num_assign = NUM_ASSIGN;

    return;
}

/*
    scan() - This routine will scan an input line of the form
	"U:P d:". Where U is the winchester drive
	unit number (0 to (NUM_FIXED-1)), P is the number of the
	users partition selection and d: is the drive letter the
	user wishes to assign. U is verified and put in variable drv,
	P is put in part_num and d: is put in drv (based 
	at 0).
*/
scan(line)
char *line;
{
    char *p;
    int i;

    /* Map the command line to upper case */
    p = line;
    while(*p != '\0')
    {
	if (islower(*p)) *p = toupper(*p);
	++p;
    }
    
    /* Scan and verify unit number */
    if (*line < '0' || *line >= ('0'+NUM_FIXED) || *(line+1) != ':')
    {
	puts("\r\nInvalid winchester drive number given\r\n");
	exit(0);
    }
    
    /* Set unit number */
    unit = *line - '0';
    line +=2;				/* Skip over unit */
    
    while(isspace(*line)) ++line;	/* Skip white space */
    
    /* Check for only partition listing */
    if (*line == '\0') return(FALSE);

    /* Get the partition number */
    part_num = *line - ('0' + 1);
    if (part_num >= NUM_PART)
    {
	puts("\r\nInvalid partition selection.\r\n");
	exit(0);
    }
    

    /* Get the drive designation to assign to */

    line +=1;				/* Skip partition number */
    while(isspace(*line)) ++line;	/* Skip white space */
    drv = *line - 'A' - num_floppy;	/* Calculate drive index */
    if (drv >= num_assign || *(line+1) != ':')
    {
	puts("\r\nInvalid drive/drive not available\r\n");
	exit(0);
    }
    
    return(TRUE);
}

/*
    assign() - This routine will assign a drive letter to a
	winchester partition.
*/
assign()
{
    int i, j, index;
    char flag, *p, *bpb_offset, tmp, mbyte1, mbyte2, fmt;

    /* Do an assign on a drive */
    
    /* Calculate the BPB (Bios parameter block) for this drive */
    get_bpb(&part_table->part_entry[part_num], &bpb1);
    
    /* Ensure that this partition is not already assigned */
    for (i = 0; i < num_assign; ++i)
    {
	bpb_offset = ((char *)c_vec.fixed_bpbs) + i * sizeof(struct bpb);
	p = (char *) &bpb1;

	/* Temporarily make the media bytes the same */
	mbyte1 = bpb1.mbyte;
	mbyte2 = getbyte(BIOS2_SEG,&((struct bpb *)bpb_offset)->mbyte);
	bpb1.mbyte = 0;
	putbyte(BIOS2_SEG, &((struct bpb *)bpb_offset)->mbyte, 0);
	    
	flag = TRUE;		/* Assume BPB's are same */
	for (j = 0; j < sizeof(struct bpb); ++j)
	{
	    if (*p++ != getbyte(BIOS2_SEG, bpb_offset+j))
    
	    {
		flag = FALSE;	/* BPB's are different */
		break;
	    }
	}
	
	/* Restore the media bytes */
	bpb1.mbyte = mbyte1;
	putbyte(BIOS2_SEG, &((struct bpb *)bpb_offset)->mbyte, mbyte2);
	
	/* Was partition already used? */
	if (flag)
	{
	    display(part_table);
	    puts("\r\nPartition is already in use\r\n");
	    exit(0);
	}
	
    }

    /* Check if the partition has been formatted */
    fmt = chk_fmt(&part_table->part_entry[part_num]);
    
    /* Move the BPB to the BIOS */
    bpb_offset = ((char *)c_vec.fixed_bpbs) + drv * sizeof(struct bpb);
    movbyte(mydsreg(), &bpb1, BIOS2_SEG, bpb_offset, sizeof(struct bpb));
	
    /* Set the assigned, formatted and changed flags in the BIOS */
    tmp = getbyte(BIOS2_SEG, c_vec.assign_flags+drv);
    if (fmt) putbyte(BIOS2_SEG, c_vec.assign_flags+drv, tmp | 7);
    else putbyte(BIOS2_SEG, c_vec.assign_flags+drv, tmp | 5);
    printf("\r\nDOS partition %d assigned to drive %c:\r\n\n", part_num+1, drv+'A'+num_floppy);
    return;
}


/*
    chk_fmt() - This routine will ensure that a newly partitioned partition
	can not be accessed until it is formatted.
*/
chk_fmt(p)
struct p_ent *p;
{
    struct bpb *b;
 
    /* Get the location of the boot sector */
    head = p->begin_hcs.head;
    cyl = p->begin_hcs.cyl + (((unsigned)(p->begin_hcs.sec) & 0xc0) << 2);
    sector = p->begin_hcs.sec & 0x3f;
    
    /* Read in the boot sector */
    rbx = &boot_buf[0];
    rdl = unit + 0x80;
    if (!get_sector()) return(FALSE);
    
    /* Check if the partition is unchanged */
    b = (struct bpb *) &boot_buf[3 + 8];
    if ((unsigned)p->rel_sec != b->hidden) return(FALSE);
    if ((unsigned)p->part_size != b->secs) return(FALSE);

    return(TRUE);
}
    
    

/*
    get_bpb() - This routine will return a BPB for a partition in
	a partition table.
*/
get_bpb(p, bpb1)
struct p_ent *p;
struct bpb *bpb1;
{
    int i;
    long fatbytes;
    unsigned fatsecs;
    char clf;

#define MIN_PART ((unsigned)0x40)	/* Minimum partion size */

    /* Define the "break" points of partition sizes */
    static unsigned brk[] = {0x200, 0x800, 0x2000, 0x7fa8};

    /* Define number of directory entries for different size parts. */
    static unsigned dirs[] = {0x40, 0x70, 0x100, 0x200, 0x400};
    

    /* Check if partition is large enough */
    if ((unsigned)p->part_size < MIN_PART)
    {
	display();
	puts("\r\nPartition is too small\r\n");
	return;
    }
    
    for(i = 0; i < sizeof(brk)/sizeof(unsigned); ++i)
    {
	if ((unsigned)p->part_size <= brk[i]) break;
    }

    clf = (1 << i);		/* Calculate the cluster factor */
   
    /* Bytes in the FAT */
    fatbytes = ((p->part_size + clf - 1) / clf) + 1;
    fatbytes = (fatbytes & ((long)0xfffffffe)) + (fatbytes >> 1);

    fatsecs = (fatbytes + SECTOR_SIZE - 1) / SECTOR_SIZE;
    
    bpb1->sector_size = SECTOR_SIZE;
    bpb1->spau = clf;
    bpb1->reserved = RESERVED;
    bpb1->nfats = NFATS;
    bpb1->dirents = dirs[i];
    bpb1->secs = (unsigned)p->part_size;
    bpb1->mbyte = MBYTE;
    bpb1->fatsecs = fatsecs;
    bpb1->spt = max_spt;
    bpb1->heads = max_hds;
    bpb1->hidden = (unsigned) p->rel_sec;
    bpb1->unit = unit + 0x80;

    return;
}


/*
    display() - This routine will display the top half of the display.
*/
display(p_tbl)
struct part_tbl *p_tbl;
{
    int i, drive1, part_flags[NUM_ASSIGN];
    static char *dos = {"DOS    "}, *nondos = {"non-DOS"};
    static char *unalloc = {"Unallocated"};
    struct p_ent *p;
    unsigned s_cyl, e_cyl;
    long p_size;
    char *s;

    puts("\r\n    Partition          Start                End              Size in\r\n");
    puts("      Type            Cylinder            Cylinder          Kilobytes\r\n");
    puts("    ---------         --------            --------          ---------\r\n");
    for (i = 0; i < NUM_PART; ++i)
    {
	p = &part_table->part_entry[i];
	if (p->os_id == DOS_ID) s = dos;
	else if (p->os_id == UNALLOC_ID) s = unalloc;
	else s = nondos;
	
	if (p->os_id == UNALLOC_ID)
	{
	    printf("%1d.  %s\r\n", i+1, s);
	}
	else
	{
	    s_cyl = p->begin_hcs.cyl + ((p->begin_hcs.sec & 0xc0) << 2);
	    e_cyl = p->end_hcs.cyl + ((p->end_hcs.sec & 0xc0) << 2);
	    p_size = p->part_size * ((float) SECTOR_SIZE / 1024);
	    printf("%1d.  %s           %5d               %5d             %8ld\r\n", i+1, s, s_cyl, e_cyl, p_size);
	}
    }
    
    /* Print the currently assigned information */
    log_parts(&part_flags[0]);
    puts("\r\n");
    for (i = 0; i < num_assign; ++i)
    {
	if (part_flags[i] == -1)
	{
	    printf("  Drive %c: = Unassigned\r\n", num_floppy+i+'A');
	}
	else
	{
	    drive1 = getbyte(BIOS2_SEG, &(c_vec.fixed_bpbs+i)->unit) - 0x80;
	    printf("  Drive %c: = %1d:%d\r\n", num_floppy+i+'A', drive1, part_flags[i]);
	}
    }
	
    return;
}


/*
    get_part() - This routine will read in the Partition
    table. It returns TRUE if OK else FALSE is returned.
*/
get_part(unit, buf)
char unit, *buf;
{
    static char last_drive=0xff;	/* Last unit accessed (invalid to start) */
    int flag;
    extern unsigned head, sector, cyl;
 
    /* Is this partition table already available? */
    if (last_drive == unit) return(TRUE);
    
    last_drive = unit;		/* Flag this one as last table */
    
    rdl = unit + 0x80;
    rbx = buf;
    res = mydsreg();
    sector = 1;
    cyl = 0;
    head = 0;
    return(get_sector());
}

/*
    controller() - This routine will determine if a winchester
	controller exists in a Z150) computer. If there is one
	then TRUE is returned else FALSE is returned.
*/
controller(unit)
char unit;
{
    rah = DIO_GETPARMS;		/* Do a get drive parameters */
    rdl = 0x80+unit;		/* valid only if win. cont. is there */
    z150int(DISK_IO_INTR);
    return(!(flags & CARRY));	/* Return carry flag status */
}

/*
    encode() - This routine will encode the head, sector and cyl
	externals into the form that the Z150 ROM requires.
*/
encode()
{
    rcl = ((cyl & 0x0300) >> 2) | sector;
    rch = cyl & 0x00ff; 
    rdh = head;
    return;
}


/*
    get_sector() - This routine will read a sector from the winchester
	disk. It assumes all other parameters except rax have been set
	previously. (i.e. the externals head, sector, cyl and transfer
	address (res:rbx) are set). If the operation is sucessful TRUE
	is returned else FALSE is returned.
*/
get_sector()
{
    unsigned temp;
    
    encode();			/* Set the head, sector and cylinder */
    rah = DIO_READ;		/* Read function */
    ral = 1;			/* do 1 sector */
    temp = rbx;
    rbx = buf;
    z150int(DISK_IO_INTR);
    if (!(flags & CARRY))
    {
	movbyte(mydsreg(), buf, mydsreg(), temp, SECTOR_SIZE);
    }
    return(!(flags & CARRY));	/* Return status */
}


/*
    log_parts() - This routine will locate the partitions assigned
	to various drive letters for status information.
*/
log_parts(part_flags)
int *part_flags;
{
    struct bpb *bpb2;
    int i, j, k, part_count = 0;
    
    /* Log in the partitions known to be unassigned */
    for (i = 0; i < num_assign; ++i)
    {
	if (getbyte(BIOS2_SEG, c_vec.assign_flags+i) & 1) part_flags[i] = 0;
	else
	{
	    part_flags[i] = -1;
	    ++part_count;
	}
    }
    
    /* Loop for the number of drives */
    for (i = 0; i < NUM_FIXED; ++i)
    {
	for (j = 0; j < num_assign; ++j)
	{
	    if (part_flag[j] != -1)
	    {
		bpb2 = c_vec.fixed_bpbs + j;
		if (getbyte(BIOS2_SEG, &bpb2->unit) == (0x80 + i))
		{
		    get_part(i, &part_buf[0]);
		    for (k = 0; k < NUM_PART; ++k)
		    {
			if (part_table->part_entry[k].rel_sec ==
			    ((long) getword(BIOS2_SEG, &bpb2->hidden)))
			{
			    part_flags[j] = k+1;
			    ++part_count;
			    break;
			}
		    }
		}
	    }
	    if (part_count == num_assign) break;
	}
	if (part_count == num_assign) break;
    }
    return;
}

