/*
	PART - This program will partition a winchester disk for use
    by MS-DOS.
    
    Author: RJM
    
    Date: 8/21/83
    
    Version 1.76 - Expand number of winchesters to 8 RJM
    Version 1.77 - Make PART actually do the exit to MFM-150 RJM
    Version 1.90 - Update utilities all to V1.90
    Version 2.00 - Update all utilities to 2.00
    Version 2.01 - New SBC code to accomodate removable winchester boot.
		   Also, allocating a partition whose boot flag is set now
		   does not remove the boot flag.
    Version 2.02 - 9/3/84 Changed to use information about sectors per track
		   from the controller.
*/

/*
		RESTRICTED RIGHTS LEGEND
		------------------------
	
	    "Use, duplication, or disclosure by the
	Government is subject to restrictions as set forth
	in paragraph (b) (3) (B) of the Rights in Technical
	Data and Computer Software clause in DAR
	7-104.9(a).  Contractor/manufacturer is Zenith
	Data Systems Corporation of Hilltop Road, St.
	Joseph, Michigan 49085.
*/

#include "stdio.h"
#include "conio.h"
#include "ctype.h"
#include "z150rom.h"
#include "fixed.h"

#define	TRUE	1
#define	FALSE	0


/* Flags for correct signon message */
#define	ZENITH	TRUE
#define	NBI	FALSE


/* Define the version and release numbers for PART */
#define VERSION 2
#define RELEASE 2

/* Define the exit address to MFM-150 */
#define MFM_SEGMENT	0xf000
#define MFM_OFFSET	0xffed

/* Define some function key codes */
#define	F1	59
#define F2	60
#define F3	61

#define PAGE_BREAK	13	/* First screen line past necessary stuff */

#define CARRY	0x0001		/* Carry flag position in flag word */

char part_buf[NUM_FIXED*SECTOR_SIZE];	/* Sector buffers for fixed disks */
struct part_tbl *part_table;	/* Pointer to active partition table */
char active[NUM_FIXED];		/* Flags for active buffers in part_buf */

#define DOSF_INSTR 0xa

unsigned head, cyl, sector;	/* Variables needed for disk I/O */

char changed[NUM_FIXED];	/* Flag for partitioning changed */
char num_drive;			/* Number of winchester drives */

char buf1[2*SECTOR_SIZE];	/* Buffer for ensuring no DMA errors */
char *buf;

/* Error messages */
char *read_err = {"Error - Can not read the partition table.\r\n"};
char *write_err = {"Error - Can not write the partition table.\r\n"};
char *non_err = {"Error - Requested allocation overlaps non-DOS partition.\r\n"};
char *dos_err = {"Error - Requested allocation overlaps DOS partition.\r\n"};
char *nopart_err = {"Error - Disk is fully allocated to non-DOS partitions,\r\n        no partitioning is possible.\r\n"};
char *part_err = {"Error - Requested allocation exceeds DOS maximum\r\n        of 32 Megabytes.\r\n"};

/*
    Maximum cylinder number, maximum head number, and maximum 
    sectors per track for active disk
*/
unsigned max_cyl;
char max_hds;
unsigned max_spt;

/* Define the externals used in the z150int() module */
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;
extern unsigned rax, rbx, rcx, rdx, rsi, rdi, rsp, rbp;
extern unsigned rcs, rds, res, rss, flags;
    

/*
    main() - This is the main program for PART. It does initialization,
	sees if the user will continue and if so, puts up the main menu.
*/
main()
{
    init();
    signon();			/* Greet the user */
    puts("\r\nDo you wish to continue (Y/N)?");
    if (yes_no()) main_menu();
    exit(0);
}


/*
    init() - This routine will reset the disk system and set up
	a buffer for disk I/O so no DMA's will occur over 64K
	boundrys.
*/
init()
{
    int i;
    
    /* Reset the disk system */
    disk_reset();

    /* set up disk buffer */
    get_buf();

    for (i = 0; i < NUM_FIXED; ++i)
    {
	active[i] = FALSE;
	changed[i] = FALSE;
    }

    return;
}


/*
    signon() - Greet the user and tell him about PART. Also set up
	the number of drives.
*/
signon()
{
    float a;
    
    a = VERSION + (float)RELEASE / 100;

#if ZENITH
    printf("\r\n                 PART version %3.2f\r\n", a);
    puts("Copyright(C) 1984, Zenith Data Systems Corporation.\r\n");
#endif

#if NBI
    printf("\r\n     PART version %3.2f\r\n", a);
    puts("(C)Copyright NBI, Inc. 1984\r\n");
#endif

    /* Check if the controller exists */
    if (!controller(0))
    {
	puts("\r\nError - Can not communicate with the winchester controller\r\n");
	exit(0);
    }
    
    /* Are there any drives attached? */
    if (rdl == 0)
    {
	puts("\r\nError - No winchester drives attached\r\n");
	exit(0);
    }
    num_drive = rdl;		/* Set the number of winchester drives */
    
    /* Greet the user */
    puts("\r\n     The PART utility helps you to change the arrangement of the\r\n");
    puts("DOS partitions on your Winchester disk. PART displays a table\r\n");
    puts("showing the types of each partition, the start cylinder, the end\r\n");
    puts("cylinder and the amount of disk space allocated to each partition\r\n");
    puts("(in percentages and in kilobytes).\r\n");
    puts("\r\nCAUTION: Using PART can destroy all files on your Winchester disk.\r\n");
    puts("\r\nDo not use PART until you have transferred copies of your Winchester\r\n");
    puts("disk files to floppy disks.\r\n\n");
    
    return;
}

/*
    main_menu() - This routine will handle the dispatching of routines
    from the main menu
*/
main_menu()
{
    int i;
    char flag;
    
    while(TRUE)
    {
	cls();			/* Clear the secrren and cursor to top */
	cursor(0, 0);
	flag = FALSE;
	for (i = 0; i < num_drive; ++i)
	{
	    if (changed[i])
	    {
		flag = TRUE;
		break;
	    }
	}
	
	if (!flag)
	{
	    menu1();
	    decode1();
	}
	else
	{
	    menu2();
	    decode2();
	}
    }
    return;
}


/*
    menu1() - This routine prints out the menu for no changes done to
	the partition table.
*/
menu1()
{
    int i;
    for (i = 0; i < num_drive; ++i)
    {
	printf("  F%d - Partition winchester drive %d\r\n", i+1, i+1);
    }
    puts("\r\n");

    puts("  E  - Exit to DOS\r\n");

    puts("\r\nEnter selection <F1");
    for (i = 1; i < num_drive; ++i) printf(", F%d", i+1);
    puts(" or E>");
    
    return;
}


/*
    menu2() - This routine prints out the menu for changes done to
	the partition table.
*/
menu2()
{
    int i;
    for (i = 0; i < num_drive; ++i)
    {
	printf("  F%d - Partition winchester drive %d\r\n", i+1, i+1);
    }
    puts("\r\n");
    
    puts("  M  - Make changes, exit to MFM-150\r\n");
    puts("  A  - Abort all changes, exit to DOS\r\n");
    
    puts("\r\nEnter selection <F1");
    for (i = 1; i < num_drive; ++i) printf(", F%d", i+1);
    puts(", M or A>");
    return;
}

/*
    decode1() - This routine will get the user input and dispatch the
	selection for menu1.
*/
decode1()
{
    char flag, c;
    flag = FALSE;
    
    /* Loop until correct input */
    while(TRUE)
    {
	c = getchar();
	if (islower(c)) c = toupper(c);
	
	/* Exit command */
	if (c == 'E') exit(0);
	
	/* check for special function keys */
	if (c != 0) beep();
	else
	{
	    c = getchar();

	    if (c >= F1 && c < F1+num_drive)
	    {
		flag = TRUE;
		part(c-F1);
	    }
	    
	    /* Invalid entry */
	    else beep();
	}
	
	/* If correct input was received then exit */
	if (flag) break;
    }
    return;
}


/*
    decode2() - This routine will get the user input and dispatch the
	selection for menu2.
*/
decode2()
{
    char flag, c;
    flag = FALSE;
    
    while(TRUE)
    {
	c = getchar();
	if (islower(c)) c = toupper(c);

	/* Make changes and exit to MFM? */
	if (c == 'M') update();

	/* Abort all changes? */
	if (c == 'A') exit(0);

	/* Check for special function keys */
	if (c != 0) beep();
	else
	{
	    c = getchar();
	    
	    if (c >= F1 && c < F1+num_drive)
	    {
		flag = TRUE;
		part(c-F1);
	    }

	    /* Invalid entry */
	    else beep();
	}

	/* If correct input was received then exit */
	if (flag) break;
    }
    return;
}


/*
    part() - This routine will allow partitioning of the specified
	drive. It gets the partition table and verifies it is good.
	If the table is not good then a new one of unallocated
	partitions is created. The partitioning menu is then displayed
	and the users request is dispatched.
*/
part(drive)
char drive;
{
    int i;
    char dos_flag, all_non, alloc_err, dflt_boot;
    struct p_ent *p;
    unsigned size_parts();
 
    /* See if the partition table has already been read */
    if (!active[drive])
    {
	/* Get the partition table */
	if (!get_part(drive, &part_buf[drive*SECTOR_SIZE]))
	{
	    err(read_err);
	    return;
	}
	active[drive] = TRUE;
    }
    part_table = (struct part_tbl *) (&part_buf[drive*SECTOR_SIZE]+SECTOR_SIZE-sizeof(struct part_tbl));

    /* Set up the maximum cylinder value and maximum head value */
    controller(drive);
    max_cyl = ((rcl & 0xc0) << 2) + rch;	/* Set max. cyl. */
    max_hds = rdh + 1;				/* Set max. heads */
    max_spt = rcl & 0x3f;			/* Set max. sectors per track */
    
    /* Verify good partition table, if not then prepare one */
    check_part(drive);


    /* Check if partitioning possible on this drive */
    all_non = TRUE;
    alloc_err = FALSE;
    for (i = 0; i < NUM_PART; ++i)	/* Check if DOS part. exists */
    {
	p = &part_table->part_entry[i];
	if (p->os_id == DOS_ID) break;
	if (p->os_id == DOS_ID || p->os_id == UNALLOC_ID) all_non = FALSE;
    }
    if (p->os_id != DOS_ID)
    {
	if (all_non || (!all_non && size_parts() > max_cyl))
	{
	    alloc_err = TRUE;
	}
    }
    
    /* Do this menu until the user selects exit */
    while(TRUE)
    {
	cls();
	cursor(0, 0);
	
	/* See if deleting DOS part. is valid option (does DOS part. exist) */
	dos_flag = FALSE;
	for (i = 0; i < NUM_PART; ++i)
	{
	    if (part_table->part_entry[i].os_id == DOS_ID)
	    {
		dos_flag = TRUE;
		break;
	    }
	}

	/* display the partition information */
	display (part_table);
    
	/* Determine the default boot partition */
	dflt_boot = 0;		/* flag as none for now */
	for (i = 0; i < NUM_PART; ++i)
	{
	    if (part_table->part_entry[i].boot_flag & 0x80)
	    {
		dflt_boot = i+1;
		break;
	    }
	}

	/* Display unit, default boot part., max. cyl. and min. allocation */
	puts("\r\n");
	printf("Winchester unit = %1d\r\n", drive);
	puts("Default boot partition = ");
	if (dflt_boot == 0) puts("None\r\n");
	else printf("%d\r\n", dflt_boot);
	printf("Maximum cylinder Number = %d\r\n", max_cyl);
	puts("Minimum DOS allocation = 32 Kilobytes\r\n");
    
	if (alloc_err)
	{
	    err(nopart_err);
	    return;
	}

	menu3(dos_flag);

	if (!decode3(drive, dos_flag)) return;	/* Check for exit */
    }
    return;
}

menu3(flag)
char flag;
{

    /* Display the selection menu according to dos_flag */
    puts("\r\n");
    if (flag)
    {
	puts("  F1 - Select default boot partition\r\n");
	puts("  F2 - Allocate DOS partition\r\n");
	puts("  F3 - Delete DOS partition\r\n");
	puts("  E  - Exit\r\n");    
	puts("\r\nEnter selection <F1, F2, F3 or E>");
    }
    else
    {
	puts("  F1 - Select default boot partition\r\n");
	puts("  F2 - Allocate DOS partition\r\n");
	puts("  E  - Exit\r\n");    
	puts("\r\nEnter selection <F1, F2 or E>");
    }
    return;
}


decode3(drive, flag)
char drive;
{
    char c, flag1;
	
    /* Input and decode the user response */
    flag1 = FALSE;
    while(TRUE)
    {
	c = getchar();
		
	/* Exit? */
	if (c == 'E' || c == 'e') return(FALSE);
	    
	/* Check for function keys */
	if (c != 0) beep();
	else
	{
	    c = getchar();
	
	    /* Select default boot partition? */
	    if (c == F1)
	    {
		get_boot(drive);
		flag1 = TRUE;
	    }

	    /* Allocate partition? */
	    else if (c == F2)
	    {
		flag1 = TRUE;
		alloc(drive);
	    }
		
	    /* Delete Partition? */
	    else if (c == F3 && flag)
	    {
		flag1 = TRUE;
		delete(drive);
	    }

	    /* Invalid entry */
	    else beep();
	}
	if (flag1) break;
    }
    return(TRUE);
}

/*
    get_boot() - This routine will get the default boot partition
*/
get_boot(drive)
char drive;
{
    char c;
    int i;
    
    changed[drive] = TRUE;
    cursor(PAGE_BREAK, 0);
    clr_eop();
    puts("\r\nSelect partition (1 - 4) or RETURN for none:");
    while(TRUE)
    {
	c = getchar();
	if ((c >= '1' && c <= '4') || c == '\r') break;
	beep();
    }
    printf("%c\r\n");
    
    for (i = 0; i < NUM_PART; ++i) part_table->part_entry[i].boot_flag = 0;
    if (c != '\r') part_table->part_entry[c-'1'].boot_flag = 0x80+drive;
    return;
}


/*
    delete() - This routine will delete a DOS partition. The user
	is prompted for the number of the partition to delete.
	If it is not a DOS partition then a bell is sounded and
	the user is required to make another selection. If it is a
	DOS partition then the partition is changed to type Unallocated.
*/
delete(drive)
char drive;
{
    struct p_ent *p;
    char c;
    
    /* Partitioning has changed */
    changed[drive] = TRUE;
    
    /* Clear unnecessary stuff off the screen */
    cursor(PAGE_BREAK, 0);
    clr_eop();
    
    /* Get the users partition selection */
    puts("\r\nSelect Partition (1-4): ");
    while(TRUE)
    {
	c = getchar();
    
	/* Is it in range? */
	if (c < '1' || c > '4') beep();
	else
	{
	    p = &part_table->part_entry[c-'1'];
	
	    /* Is it the correct type of partition */
	    if (p->os_id != DOS_ID) beep();
	    else break;
	}
    }
    
    /* Delete the partition */
    p->os_id = UNALLOC_ID;
    /* Set the boot flag and the operating system ID */
    p->boot_flag = 0;
    
    /* Set the relocation sectors (number of sectors preceding the part.) */
    p->rel_sec = 0L;

    /* Set the partition size */
    p->part_size = 0L;
    
    /* Set the start head cylinder and sector */
    p->begin_hcs.head = 0; 
    p->begin_hcs.cyl = 0;
    p->begin_hcs.sec = 0;
    
    p->end_hcs.head = 0; 
    p->end_hcs.cyl = 0;
    p->end_hcs.sec = 0;
    return;
}


/*
    alloc() - This routine will allocate a partition of type DOS or
	unallocated.
*/
alloc(drive)
char drive;
{
    unsigned s_cyl, e_cyl, s1, e1, allocated, get_dec();
    char c, part_num;
    struct p_ent *p;
    long temp;
    
    cursor(PAGE_BREAK, 0);
    clr_eop();
    
    puts("\r\nSelect Partition (1-4): ");
    while(TRUE)
    {
	c = getchar();
	if (c < '1' || c > '4') beep();
	else
	{
	    p = &part_table->part_entry[c-'1'];
	    if (p->os_id != DOS_ID && p->os_id != UNALLOC_ID) beep();
	    else break;
	}
    }

    part_num = c - '1';
    p = &part_table->part_entry[part_num];

    printf("%c\r\n", c);
    while(TRUE)
    {
	puts("Start Cylinder: ");
	s_cyl = get_dec();
	if (s_cyl != 0xffff)
	    if (s_cyl <= max_cyl) break;
	beep();
	puts("\r                                   ");
	puts("                                   \r");
    }
    puts("\r\n");
	
    while(TRUE)
    {
	puts("End Cylinder: ");
	e_cyl = get_dec();
	if (e_cyl != 0xffff)
	    if (e_cyl <= max_cyl)
		if (e_cyl >= s_cyl) break;
	beep();
	puts("\r                                   ");
	puts("                                   \r");
    }
    puts("\r\n");
    
    if (!check_alloc(part_num, s_cyl, e_cyl)) return;
    
    changed[drive] = TRUE;
    /* Set the operating system ID (Note boot flag left unchanged) */
    p->os_id = DOS_ID;
    
    /* Set the relocation sectors (number of sectors preceding the part.) */
    temp = s_cyl;
    p->rel_sec = ((temp * max_spt) * max_hds);
    if (s_cyl == 0) p->rel_sec += 1;	/* if cyl. 0 account for boot sec. */

    /* Set the partition size */
    temp = e_cyl - s_cyl + 1;
    p->part_size = ((temp * max_spt) * max_hds);
    if (s_cyl == 0) p->part_size -= 1;	/* if cyl. 0 account for boot sec. */
    
    /* Set the start head cylinder and sector */
    p->begin_hcs.head = 0; 
    p->begin_hcs.cyl = (s_cyl & 0xff); 
    p->begin_hcs.sec = ((s_cyl & 0xff00) >> 2) + 1;
    if (s_cyl == 0) p->begin_hcs.sec += 1;	/* if cyl. 0 account for boot sec. */
    
    p->end_hcs.head = max_hds-1; 
    p->end_hcs.cyl = (e_cyl & 0xff); 
    p->end_hcs.sec = ((e_cyl & 0xff00) >> 2) + max_spt; 
    
    return;
}
    
check_alloc(part_num, start, end)
char part_num;
unsigned start, end;
{
    int i;
    unsigned s1, e1;
    struct p_ent *p;

    if ((((long)start-end)*max_spt*max_hds) >= MAX_PART)
    {
	err(part_err);
	return;
    }

    for (i = 0; i < NUM_PART; ++i)
    {
	if (i == part_num) continue;
	p = &part_table->part_entry[i];
	if (p->os_id == UNALLOC_ID) continue;
	s1 = p->begin_hcs.cyl + ((p->begin_hcs.sec & 0xc0) << 2);
	e1 = p->end_hcs.cyl + ((p->end_hcs.sec & 0xc0) << 2);
	if ((start >= s1 && start <= e1) || (end >=s1 && end <= e1))
	{
	    if (p->os_id == DOS_ID)
	    {
		err(dos_err);
		return(FALSE);
	    }
	    else
	    {
		err(non_err);
		return(FALSE);
	    }
	}
    }
    return(TRUE);
}

/*
    size_parts() - Determine the size of the partitions already existing
	on the disk.
*/
unsigned size_parts()
{
    unsigned tot_cyls, s_cyl, e_cyl;
    int i;
    struct p_ent *p;
    
    tot_cyls = 0;
    for (i = 0; i < NUM_PART; ++i)
    {
	p = &part_table->part_entry[i];
	if (p->os_id != UNALLOC_ID)
	{
	    s_cyl = p->begin_hcs.cyl + ((p->begin_hcs.sec & 0xc0) << 2);
	    e_cyl = p->end_hcs.cyl + ((p->end_hcs.sec & 0xc0) << 2);
	    tot_cyls += e_cyl - s_cyl + 1;
	}
    }
    return(tot_cyls);
}

	
/*
    update() - This routine will update the partition information and
	exit to the monitor.
*/
update()
{
    char drv;
    
    for (drv = 0; drv < num_drive; ++drv)
    {
	if (changed[drv])
	{
	    head = 0;
	    cyl = 0;
	    sector = 1;
	    rdl = drv + 0x80;
	    rbx = &part_buf[drv*SECTOR_SIZE];
	    if (!put_sector())
	    {
		err(write_err);
		return;
	    }
	}
    }
    jump_far(MFM_SEGMENT, MFM_OFFSET);		/* Exit to MFM-150 */
}


/*
    check_part() - This routine will check the partition for validity 
	(i.e. valid signature). If the partition is not valid, then a
	valid partition sector is created.
*/
check_part(drive)
char drive;
{
    if (part_table->signature == 0xaa55) return;
    movbyte(sbcseg(), 0, mydsreg(), &part_buf[drive*SECTOR_SIZE], SECTOR_SIZE);
    return;
}
 

/*
    err() - This routine will print an error message and wait for
	a response from the user.
*/
err(s)
char *s;
{
    puts("\r\n");
    puts(s);
    puts("\r\nHit any key to continue...");
    getchar();
    return;
}


/*
    get_dec() - This routine will return a decimal number.
*/
unsigned get_dec()
{
#define BUF_LEN 20		/* buffer for input */
    char buf[BUF_LEN], c;
    unsigned val, index;
    
    buf[0] = BUF_LEN - 2;
    
    while(TRUE)
    {
	/* Get the user string */
	bdos(DOSF_INSTR, &buf[0]);
    
	val = 0;
	index = 2;		/* Start processing at the second byte */
	while(TRUE)
	{
	    c = buf[index++];
	    if (c == '\r') break;		/* if Return then done */
	
	    /* check for valid digit */
	    if (isdigit(c))
	    {
		c -= '0';	/* Bias character to get digit */

		val = (val * 10) + c;	/* Update value */
	    }
	    else	/* Invalid decimal digit */
	    {
		return(0xffff);
	    }
	}
	if (c == '\r') break;		/* If return then done */
    }
    return(val);
}


/*
    beep() - This routine will sound a "bell" character.
*/
beep()
{
    putchar(7);
    return;
}


/*
    yes_no() - This routine will return TRUE if the user responds 'Y'
	or 'y' and will return FALSE if the user responds 'N' or 'n'.
*/
yes_no()
{
    char c1;

    while(TRUE)
    {
	c1 = getchar();
	if (islower(c1)) c1 = toupper(c1);
	if (c1 != 'N' && c1 != 'Y') beep();
	else
	{
	    putchar(c1);
	    puts("\r\n");
	    return(c1 == 'Y');
	}
    }
}


/*
    disk_reset() - This routine will reset the disk system.
*/
disk_reset()
{
    char temp;
    
    rah = DIO_RESET;
    temp = rdl;
    rdl = 0x80;
    z150int(DISK_IO_INTR);
    rdl = temp;
    return(!(flags & CARRY));
}

/*
    get_buf() - This routine will prepare the "buf1" buffer and
	the "buf" pointer so it doesn't cross a 64K memory boundry.
*/
get_buf()
{
    unsigned address;
    
    buf = &buf1[0];
    address = (((unsigned)mydsreg()) << 4) + (unsigned)&buf1[0];	/* Get physical address of buffer */
    address += SECTOR_SIZE;		/* Bump address by SECTOR_SIZE */

    /* Modify buffer address if the current buffer crosses boundry */
    if (address < (unsigned) buf) buf += SECTOR_SIZE;
    return;
}


/*
    display() - This routine will display the top half of the display.
*/
display(p_tbl)
struct part_tbl *p_tbl;
{
    int i;
    static char *dos = {"DOS        "}, *nondos = {"non-DOS    "};
    static char *unalloc = {"Unallocated"};
    struct p_ent *p;
    unsigned s_cyl, e_cyl;
    long p_size;
    char *s;
    float p_pct;

    puts("\r\n");
    puts("    Partition        Start            End            Size in     Percentage\r\n");
    puts("      Type          Cylinder        Cylinder        Kilobytes    of the Disk\r\n");
    puts("    ---------       --------        --------        ---------    -----------\r\n");
    for (i = 0; i < NUM_PART; ++i)
    {
	/* Determine the type of the partition */
	p = &p_tbl->part_entry[i];
	if (p->os_id == DOS_ID) s = dos;
	else if (p->os_id == UNALLOC_ID) s = unalloc;
	else s = nondos;

	if (p->os_id == UNALLOC_ID) printf("%1d.  %s\r\n", i+1, s);
	else
	{
	    s_cyl = p->begin_hcs.cyl + ((p->begin_hcs.sec & 0xc0) << 2);
	    e_cyl = p->end_hcs.cyl + ((p->end_hcs.sec & 0xc0) << 2);
	    p_size = p->part_size * ((float) SECTOR_SIZE / 1024);
	    p_pct = ((float)(e_cyl - s_cyl + 1) / (max_cyl+1)) * 100;
	    printf("%1d.  %s     %5d            %5d          %8ld        %3.1f%%\r\n", i+1, s, s_cyl, e_cyl, p_size, p_pct);
	}
    }
    return;
}


/*
    get_part() - This routine will read in the Partition
    table. It returns TRUE if OK else FALSE is returned.
*/
get_part(unit, buf)
char unit, *buf;
{
    int flag;
    extern unsigned head, sector, cyl;
 
    rdl = unit + 0x80;
    rbx = buf;
    res = mydsreg();
    sector = 1;
    cyl = 0;
    head = 0;
    return(get_sector());
}

/*
    controller() - This routine will determine if a winchester
	controller exists in a Z150) computer. If there is one
	then TRUE is returned else FALSE is returned.
*/
controller(unit)
char unit;
{
    rah = DIO_GETPARMS;		/* Do a get drive parameters */
    rdl = unit + 0x80;			/* valid only if win. cont. is there */
    z150int(DISK_IO_INTR);
    return(!(flags & CARRY));	/* Return carry flag status */
}

/*
    encode() - This routine will encode the head, sector and cyl
	externals into the form that the Z150 ROM requires.
*/
encode()
{
    extern unsigned head, sector, cyl;
    
    rcl = ((cyl & 0x0300) >> 2) | sector;
    rch = cyl & 0x00ff; 
    rdh = head;
    return;
}


/*
    get_sector() - This routine will read a sector from the winchester
	disk. It assumes all other parameters except rax have been set
	previously. (i.e. the externals head, sector, cyl and transfer
	address (res:rbx) are set). If the operation is sucessful TRUE
	is returned else FALSE is returned.
*/
get_sector()
{
    unsigned temp;
    int i;
    
    encode();			/* Set the head, sector and cylinder */
    temp = rbx;
    rbx = buf;
    for (i = 0; i < NUM_RETRY; ++i)
    {
	rah = DIO_READ;			/* Read function (restore in case of reset)*/
	ral = 1;			/* do 1 sector */
	z150int(DISK_IO_INTR);
	if (!(flags & CARRY)) break;
	disk_reset();
    }
    if (!(flags & CARRY))
    {
	movbyte(mydsreg(), buf, mydsreg(), temp, SECTOR_SIZE);
    }
    return(!(flags & CARRY));	/* Return status */
}


put_sector()
{
    int i;

    movbyte(mydsreg(), rbx, mydsreg(), buf, SECTOR_SIZE);
    
    encode();
    rbx = buf;
    for (i = 0; i < NUM_RETRY; ++i)
    {
	rah = DIO_WRITE;	/* Restore these in case of reset */
	ral = 1;
	z150int(DISK_IO_INTR);
	if (!(flags & CARRY)) break;
	disk_reset();
    }
    return(!(flags & CARRY));
}


/*
    clr_eop() - This routine will clear the screen from the cursor line
	down to the end of the page.
*/
clr_eop()
{
    /* Save old cursor position */
    rbh = 0;
    rah = VIO_RCP;
    z150int(VIDEO_IO_INTR);
    
    ral = 25 - rdh;
    rch = rdh;
    rcl = 0;
    rdh = 25;
    rdl = 79;
    rbh = 7;
    rah = VIO_SPU;
    z150int(VIDEO_IO_INTR);
    return;
}

