	PAGE	,132
	TITLE	Z-150 Initialization and Power-Up Diagnostics

;**********************************************************************
;
;            ----------------------------------------------
;------------ Initialization / Power-up Diagnostics Module ------------
;            ----------------------------------------------
;
;		Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************


MONITOR_SEGMENT SEGMENT WORD PUBLIC

ASSUME CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES: NOTHING

	INCLUDE ROM.LIT
	INCLUDE ROM.EXT
	INCLUDE IO.LIT
	INCLUDE INTR.LIT
	INCLUDE ../DISK/DISK.LIT
	INCLUDE ../DISK/DISK.EXT
	INCLUDE ../VIDEO/VIDEO.LIT
	INCLUDE ../VIDEO/VIDEO.EXT
	INCLUDE ../SYS/SYS.LIT
	INCLUDE ../SYS/SYS.EXT

	EXTRN	INIT_KEY_BUFF:NEAR, INIT_DISK:NEAR, SET_IO_CONFIGURATION:NEAR
	EXTRN	SET_MEMORY_SIZE:NEAR, SET_DEFAULT_VIDEO_MODE:NEAR
	EXTRN	READ_DIP_SWITCHES:NEAR, SET_INTR_PTR:NEAR, MFM_150:NEAR
	EXTRN	DISPLAY_ERROR_MSG:NEAR, DISPLAY_DISK_ERROR:NEAR
	EXTRN	BAD_RAM:NEAR, DISK_BOOT:NEAR, DELAY:NEAR
	EXTRN	ZERO_DIVIDE_INTERRUPT:FAR, SINGLE_STEP_INTERRUPT:FAR 
	EXTRN	NMI_INTERRUPT:FAR, BREAKPOINT_INTERRUPT:FAR, ROM_START:NEAR
	EXTRN	OVERFLOW_INTERRUPT:FAR, PRINT_SCREEN_INTERRUPT:FAR 
	EXTRN	TIMER_INTERRUPT:FAR, KEYBOARD_INTERRUPT:FAR
	EXTRN	RESERVED_HARDWARE_INTERRUPT:FAR, DISK_INTERRUPT:FAR 
	EXTRN	VIDEO_IO_INTERRUPT:FAR, GET_IO_CONFIG_INTERRUPT:FAR 
	EXTRN	GET_MEMORY_SIZE_INTERRUPT:FAR, DISK_IO_INTERRUPT:FAR 
	EXTRN	SERIAL_IO_INTERRUPT:FAR, KEYBOARD_IO_INTERRUPT:FAR 
	EXTRN	PRINTER_IO_INTERRUPT:FAR, BOOT_INTERRUPT:FAR, NO_BOOT_FLAG:BYTE
	EXTRN	TIME_OF_DAY_INTERRUPT:FAR, DO_NOTHING:FAR, STACK_TOP:NEAR
	EXTRN	MFM_150:NEAR, DOWNLOAD_BYTE:NEAR, WAIT_BREAK:NEAR
	EXTRN	INIT_IO_TIMERS:NEAR
	EXTRN	BAD_CPU_MSG:BYTE, BAD_ROM_MSG:BYTE
	EXTRN	BAD_KB_MSG:BYTE, BAD_TINTR_MSG:BYTE
	EXTRN	DATA_SEGMENT:WORD, CRT_MODE:BYTE, ROM_VARS:BYTE
	EXTRN	DEFAULT_IO_CONFIG:BYTE
	EXTRN	STACK_LEVEL:BYTE


	IF	NOT MORROW
	EXTRN	BAD_PARITY:NEAR
	ENDIF

	IF	CHEAP
	EXTRN	POLL_MEMORY_SIZE:NEAR, SET_DIP_SWITCHES:NEAR
	ENDIF


ROM_DATA SEGMENT
	EXTRN	EXT_ROM_INIT:FAR
ROM_DATA ENDS


;**********************************************************************
; START_UP:
;
;	Start_Up is the code executed when the Z-150 is initially powered
; up.  It executes diagnostics routines to permit the machine to be
; booted up - more extensive diagnostics may be executed from disk.
; Note that the following routines are not 'called', execution simply
; falls through from one routine to another.  This permits the diag-
; nostics to execute even if no system RAM is functioning properly.
;**********************************************************************
START_UP PROC FAR
	PUBLIC	START_UP
	CLI				;Prevent spurious interrupts
	CLD				;Set forward byte moves
	MOV	AX,ROM_DATA		;Point to the default data segment
	MOV	DS,AX			;Set it in DS
	MOV	AX,CS			;Set up the stack segment...
	MOV	SS,AX			;... to the monitor segment
	MOV	SP,OFFSET STACK_TOP	;Set the stack pointer	
;
; Enable parity generation, disable NMI
;
	MOV	AL,PARALLEL_MODE	;Get the mode for the parallel ports
	OUT	PAR_CTRL_PORT,AL	;Initialize the parallel I/O chip

	IF	NOT MORROW
	MOV	DX,MEMORY_CTRL		;Point to the memory board's ctrl port
	MOV	AL,PARITY_GEN_ENABLE	;Get code to enable memory board
	OUT	DX,AL			;Turn memory board's parity gen on
	ENDIF

	MOV	AL,SYS_PORT_INIT	;Get init value for control port
	OUT	SYS_CTRL_PORT,AL	;Disable all parity error checking

	IF	NOT MORROW
	MOV	AL,NOT NMI_ENABLE	;Get code to disable NMI interrupts
	OUT	NMI_CONTROL,AL		;Prevent spurious NMIs
	ENDIF

	MOV	DX,EXPANSION_PORT	;Point DX to the expansion interface
	MOV	AL,EXPANSION_INIT	;Get code to initialize expansion
	OUT	DX,AL			;Init expansion interface (if there)
;
; Initialize color and monochrome card video displays for 80x25 text mode...
;
	MOV	AL,01H			;Get code to init monochrome board
	MOV	DX,CRT_MODE_SELECT+MONO_CARD	;Point to the monochrome card
	OUT	DX,AL			;Initialize the monochrome card
	MOV	CURSOR_FLAG,NOT CURSOR_ENABLED ;Turn the graphics cursor off
	MOV	AL,CRT_MODE[3]		;Get the 80x25 mode byte
	MOV	DX,CRT_MODE_SELECT+COLOR_CARD	;Point to the CRT mode control port
	OUT	DX,AL			;Init the color video hardware mode
	MOV	AL,CRT_MODE[7]		;Get the monochrome mode byte
	MOV	DX,CRT_MODE_SELECT+MONO_CARD	;Point to the monochrome mode port
	OUT	DX,AL			;Init the mono video hardware mode
	MOV	SI,OFFSET ROM_PARMS_60[SIZE VID_PARMS] ;Point to 80x25 CRT parms
	MOV	DX,CRT_BASE_REGISTER+COLOR_CARD ;Point to the CRT control port
	MOV	CX,SIZE VID_PARMS	;Set length of parms in CX
	MOV	BL,0			;Place CRT ctrlr register # in BL
SU1:	MOV	AL,BL			;Get the current register number
	OUT	DX,AL			;Set up the I/O register for XFER
	INC	DX			;Now, point to CRT parameter register
	MOV	AL,CS: BYTE PTR [SI]	;Pick up a CRT parameter
	INC	SI			;Point to the next CRT parameter byte
	OUT	DX,AL			;Send it to the CRT controller
	DEC	DX			;Point back to the CRT base port
	INC	BL			;Increment the register number
	LOOP	SU1			;Continue till finished
	MOV	SI,OFFSET ROM_PARMS_60[SIZE VID_PARMS * 3] ;Point to mono parms
	MOV	DX,CRT_BASE_REGISTER+MONO_CARD ;Point to the mono control port
	MOV	CX,SIZE VID_PARMS	;Set length of parms in CX
	MOV	BL,0			;Place CRT ctrlr register # in BL
SU2:	MOV	AL,BL			;Get the current register number
	OUT	DX,AL			;Set up the I/O register for XFER
	INC	DX			;Now, point to CRT parameter register
	MOV	AL,CS: BYTE PTR [SI]	;Pick up a CRT parameter
	INC	SI			;Point to the next CRT parameter byte
	OUT	DX,AL			;Send it to the CRT controller
	DEC	DX			;Point back to the CRT base port
	INC	BL			;Increment the register number
	LOOP	SU2			;Continue till finished
	MOV	AL,ROM_VARS[SIZE VID_VARS].DEFAULT_COLORS ;Set colors
	MOV	DX,CRT_COLOR_SELECT+COLOR_CARD ;Point to the color control port
	OUT	DX,AL			;Set the color register properly
	MOV	AX,MONO_SEGMENT		;Point to the screen RAM segment
	MOV	ES,AX
	MOV	DI,OFFSET SCREEN	;Point to the start of screen RAM
	MOV	AX,(DEFAULT_ATTRIBUTE SHL 8) OR ' ' ;Set clear-screen data
	MOV	CX,8000H		;Get length of both mono + color cards
	REP	STOSW			;Clear mono and color screens to spaces
					;Fall through to processor test...
START_UP ENDP



;**********************************************************************
; PROCESSOR_TEST:
;
;	Processor_Test is called to check out the 8088 processor chip.
; It performs the following tests:
;		- Fills all registers with zeros and ones.
;		- Sets and clears flag bits
;		- Determines if conditional jumps work
;		- Tests several arithmetic/logic operations,
;		  to test out the ALU.
;**********************************************************************
PROCESSOR_TEST PROC NEAR
	PUBLIC	PROCESSOR_TEST, BAD_PROCESSOR
	STC				;Set the carry flag
	JNC	BAD_PROCESSOR		;ERROR - carry not set
	CLC				;Now, clear the carry flag
	JC	BAD_PROCESSOR		;ERROR - carry not cleared
	XOR	AX,AX			;Test ALU, set AX to 0
	JNZ	BAD_PROCESSOR		;ERROR - invalid zero flag
PT1:	MOV	DS,AX			;Copy AX into DS
	MOV	BX,DS			;Copy DS into BX
	MOV	SS,BX			;Copy BX into SS
	MOV	CX,SS			;Copy SS into CX
	MOV	ES,CX			;Copy CX into ES
	MOV	DX,ES			;Copy ES into DX
	MOV	BP,DX			;Copy DX into BP
	MOV	SP,BP			;Copy BP into SP
	MOV	SI,SP			;Copy SP into SI
	MOV	DI,SI			;Copy SI into DI
	CMP	AX,DI			;Did the copy work correctly?
	JNZ	BAD_PROCESSOR		;ERROR - bad register(s)
	DEC	AX			;Yes - decrement AX (all 1's first)
	JC	PT1			;Loop for 1's test
	TEST	AX,AX			;Set sign/zero flags
	JNS	BAD_PROCESSOR		;ERROR - sign bit not set (AX is -2)
	JZ	BAD_PROCESSOR		;ERROR - zero bit incorrect
	NEG	AX			;Now, set AX to +2
	JS	BAD_PROCESSOR		;ERROR - sign bit not cleared
	MOV	AL,000001B		;Indicate that the CPU test passed
	OUT	DIAG_LEDS,AL
	JMP	SHORT INIT_REFRESH	;Now initialize memory refresh

BAD_PROCESSOR:
	MOV	AX,ROM_DATA		;Point DS to the default data segment
	MOV	DS,AX
	MOV	AX,CS			;Set up the stack segment...
	MOV	SS,AX			;... to the monitor segment
	MOV	SP,OFFSET STACK_TOP	;Set the stack pointer	
	MOV	SI,OFFSET BAD_CPU_MSG	;Attempt to print out a failure msg
	JMP	DISPLAY_ERROR_MSG	;Transfer out to error routine
PROCESSOR_TEST ENDP



;**********************************************************************
; INIT_REFRESH:
;
;	Init_Refresh sets up the timer/counter and the DMA controller
; so that the on-board RAM will be refreshed.  
;**********************************************************************
INIT_REFRESH PROC NEAR
	PUBLIC	INIT_REFRESH
	OUT	DMA_CLEAR,AL		;Perform a DMA master clear
	MOV	AL,REFRESH_TIMER_MODE	;Get the timer mode for mem refresh
	OUT	TIMER_CTRL,AL		;Set the correct mode
	MOV	AL,REFRESH_DMA_MODE	;Get the DMA mode for memory refresh
	OUT	DMA_CONTROL,AL		;Set channel 0's mode
	MOV	AL,REFRESH_TIMER_COUNT	;Set the time counter for refresh
	OUT	TIMER_1,AL
	MOV	AX,REFRESH_DMA_COUNT	;Get the DMA count
	OUT	DMA_COUNT_0,AL		;Output the low DMA count byte
	MOV	AL,AH			;Get the high DMA count byte
	PUSH	AX			;Delay for a little while
	OUT	DMA_COUNT_0,AL		;Output high byte of the DMA count
	POP	AX			;Delay in-between DMA chip accesses
	MOV	AL,0			;Set DMA command register up
	OUT	DMA_COMMAND,AL
	PUSH	AX			;Delay for a little while
	OUT	DMA_ENABLE,AL		;Enable DMA channel 0
	OUT	DMA_SEGMENT_0,AL	;Set up page 0 as refresh page
	OUT	DMA_SEGMENT_3,AL	
	MOV	AL,DMA_MODE_1		;Disable DMA channel 1
	OUT	DMA_CONTROL,AL
	POP	AX			;Delay in-between DMA chip accesses
	MOV	AL,DMA_MODE_2		;Disable DMA channel 2
	OUT	DMA_CONTROL,AL
	PUSH	AX			;Delay for a little while
	MOV	AL,DMA_MODE_3		;Disable DMA channel 3
	OUT	DMA_CONTROL,AL
	POP	AX			;Clean up the stack
INIT_REFRESH ENDP



;**********************************************************************
; ROM_TEST:
;
;	ROM_Test adds every byte in the ROM together, and compares
; the result with 0 (a checksum placed at the last two bytes of the 
; ROM should force the generated checksum to be 0).
;**********************************************************************
ROM_TEST PROC NEAR
	PUBLIC	ROM_TEST, BAD_ROM
	MOV	AX,ROM_DATA		;Point DS to the default data segment
	MOV	DS,AX
	MOV	AX,CS			;Set up the stack segment...
	MOV	SS,AX			;... to the monitor segment
	MOV	SP,OFFSET STACK_TOP	;Set the stack pointer	
	MOV	AX,0			;Clear checksum
	MOV	BX,OFFSET ROM_START	;Point to start of the code in ROM
	MOV	CX,ROM_LENGTH SHR 1	;Number of words in ROM to CX
RT1:	ADD	AX,WORD PTR CS: [BX]	;Add a word into the checksum
	ADC	AX,0			;Add carries back into checksum
	ADD	BX,2			;Point to the next word
	LOOP	RT1			;Repeat for every word of the ROM
	CMP	AX,1			;Is the ROM OK?
	JNE	BAD_ROM			;No, take the error exit
	MOV	AL,000011B		;Yes - display on the LEDs...
	OUT	DIAG_LEDS,AL		;...that the ROM test passed
	MOV	BX,RESET_FLAG		;Get the 'warm boot' flag
	JMP	TEST_MEMORY_I		;Test out the system's RAM

BAD_ROM:
	MOV	SI,OFFSET BAD_ROM_MSG	;Point to the BAD-ROM message
	JMP	DISPLAY_ERROR_MSG	;Display the error message and quit
ROM_TEST ENDP



;**********************************************************************
; TEST_MEMORY_I:
;
;	TEST_MEMORY_I is called to test out the system's RAM.  Actually
; tested are the first 64K of RAM, the last 64K of RAM, and both the
; displayable and non-displayable video RAM.  If any errors occur,
; the RAM test will attempt to print out an error message and exit
; to the monitor.
;**********************************************************************
TEST_MEMORY_I PROC NEAR
	PUBLIC	TEST_MEMORY_I
;
; Enable parity generation and checking for memory test
;
	IF	NOT MORROW
	IN	AL,SYS_CTRL_PORT	;Read the system control port
	OR	AL,DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK
	OUT	SYS_CTRL_PORT,AL	;Clear the memory-error latch
	AND	AL,NOT (DISABLE_PARITY_CHK OR DISABLE_MEMORY_CHK)
	OUT	SYS_CTRL_PORT,AL	;Enable system parity checking
	MOV	DX,MEMORY_CTRL		;Point to the memory board's ctrl port
	MOV	AL,PARITY_GEN_ENABLE OR PARITY_CHK_ENABLE ;Enable gen, chk
	OUT	DX,AL			;Turn memory board's parity on
	ENDIF

	CMP	BX,1234H		;Is this a warm boot?
	JNE	COLD			;No, test all of memory
	JMP	MAIN			;Yes - skip the RAM test
;
; Test out the MFM-150 monitor's RAM
;
COLD:	MOV	AX,DS			;Temporarily put stack in low RAM
	MOV	SS,AX			;...in case RAM error in high memory
	MOV	AX,CS			;Point to the Z-150 monitor RAM
	MOV	ES,AX
	MOV	SI,2 * K / 2		;Set length of bank in SI
	MOV	DI,0			;Point to start of RAM
	MOV	BP,OFFSET TMI1		;Set up return address
	JMP	RAM_TEST_I		;Test out the Z-150 monitor's RAM
;
; Test out the first 64K of RAM
;
TMI1:	MOV	AX,CS			;Point stack back to high memory
	MOV	SS,AX			;...now that low memory is OK
	MOV	DI,0			;First, point to the first 64K bank
	MOV	ES,DI
	MOV	SI,8000H		;Get the word length of a 64K bank
	MOV	BP,OFFSET TMI2		;Prepare return address
	JMP	RAM_TEST_I		;Test out the first 64K bank
;
; Determine amount of RAM in system
;
	IF	CHEAP			;If cost-reduced machine...
TMI2:	CALL	POLL_MEMORY_SIZE	;Determine memory size by polling RAM
	SUB	AX,64			;Subtract 64K to get base of top segment
	MOV	CL,6			;Get shift to turn into a segment
	ELSE				;If normal Z-150, read DIP switches
TMI2:	CALL	READ_DIP_SWITCHES	;Read DIPs to get switch settings
	AND	AL,BUS_MEMORY_MASK	;Mask out all but system memory size
	MOV	AH,0			;Extend memory size to a word
	MOV	CL,11			;Shift to turn 32K blocks...
	ENDIF
	SHL	AX,CL			;...into a segment value
	MOV	ES,AX			;Point to this new segment
	TEST	AX,AX			;Is there more than 64K?
	JZ	TMI3			;No, don't bother testing it!
;
; Test out the last 64K of RAM
;
	MOV	DI,0			;Point to the first byte in the bank
	MOV	BP,OFFSET TMI3		;Prepare return address
	JMP	RAM_TEST_I		;Test the top bank of memory
;
; Fill remaining memory with 00H
;
TMI3:	MOV	AL,000111B		;Show that RAM passed testing
	OUT	DIAG_LEDS,AL
TMI4:	MOV	AX,ES			;Get the current segment
	SUB	AX,1000H		;Point to the previous 64K
	CMP	AX,1000H		;Reached bottom 64K yet?
	JB	MAIN			;Yes - bring the system up!
	MOV	ES,AX			;No, go on to the next segment
	MOV	DI,0			;Point to start of this bank
	MOV	AX,0			;Set fill data to 0
	MOV	CX,8000H		;Set length of bank = 64K bytes
	REP	STOSW			;Fill a bank with zeros
	JMP	SHORT TMI4		;Loop until all memory filled
TEST_MEMORY_I ENDP



;**********************************************************************
; MAIN:
;
;	Main is executed once the initial RAM, ROM and CPU diagnostics 
; have been performed.  It sets up the system stack, performs the 
; remaining diagnostics, and initializes the rest of the devices 
; in the system.  Once completed, it will return control to the Z-150
; monitor, or boot from disk, as selected by the system configuration
; switches.
;**********************************************************************
MAIN PROC NEAR
	PUBLIC	MAIN
	MOV	AX,ROM_DATA		;Point to the default ROM data segment
	MOV	DS,AX
	MOV	DX,RESET_FLAG		;Get the reset flag from memory into DX
	MOV	BX,IO_CONFIG		;Save the IO configuration in BX
	MOV	DI,0			;Point to the start of RAM memory
	MOV	ES,DI			;...at 0000:0000H
	MOV	CX,2000H		;Set clear count to 16K bytes
	XOR	AX,AX			;Set fill pattern to 0
	REP	STOSW			;Clear base RAM to zeros
	MOV	RESET_FLAG,DX		;Restore the reset indicator
	MOV	IO_CONFIG,BX		;Restore the previous I/O configuration
	MOV	AX,DS			;Get the data segment
	MOV	ES,AX			;Place dummy seg in ES in case of error
	MOV	DATA_SEGMENT,AX		;Record current data segment in memory
	MOV	STACK_LEVEL,0		;Flag that stack is available for use
	CALL	INIT_INTERRUPTS		;Initialize 8259 and intr vectors
	MOV	CRT_CONTROL,VIDEO_ON	;Set default value for CRT ctrl port
	CALL	SET_DEFAULT_VIDEO_MODE	;Determine power-up video mode
	MOV	AL,READ_KEYBOARD	;Get code to read DIP switch L.S. nibble
	OUT	SYS_CTRL_PORT,AL	;Leave it turned on for compatibility
	MOV	AL,000111B		;Reset LEDs with the value they were...
	OUT	DIAG_LEDS,AL		;...displaying prior to SET_MACHINE_MODE
	CALL	SET_MEMORY_SIZE		;Determine amount of system RAM
	CALL	MANUFACTURING_TEST	;Check to see if manufacturing mode
	JNC	M1			;Valid not-manufacturing-mode code
	MOV	SI,OFFSET BAD_KB_MSG	;Keyboard code bad - point to err msg
	JMP	DISPLAY_ERROR_MSG	;Display bad keycode error message
M1:	CALL	INIT_KEY_BUFF		;Initialize keyboard variables
	CALL	INIT_IO_TIMERS		;Set default device timeouts
	MOV	AL,TONE_CONTROL		;Get the init code for the timer
	OUT	TIMER_CTRL,AL		;Init the timer for use as tone gen
	CALL	TIMER_INTERRUPT_TEST	;Test out the 8253 timer and intr
	JNC	M2			;Timer passed, go init disk
	MOV	SI,OFFSET BAD_TINTR_MSG	;ERROR - timer interrupt failed
	JMP	DISPLAY_ERROR_MSG	;Exit to monitor/debugger
M2:	MOV	AL,001111B		;Indicate that timer passed test
	OUT	DIAG_LEDS,AL		;...on the diag LEDs
	CALL	READ_DIP_SWITCHES	;Read the system DIP switches
	IF	CHEAP			;If cost-reduced Z-150...
	CMP	RESET_FLAG,1234H	;Is this a warm boot?
	JE	M25			;Yes - have logical switch values
	TEST	AH,CHEAP_FLOPPY		;Is the floppy installed?
	JZ	M3			;No, don't try to talk to it!
	JMP	M26			;Yes - initialize the floppy
	ENDIF
M25:	TEST	AH,FLOPPY_ATTACHED	;Is a floppy controller in place?
	JZ	M3			;No, don't try to talk to it!
M26:	CALL	INIT_DISK		;Initialize disks and seek drive 0
	JNC	M3			;Disk init passed - continue
	JMP	DISPLAY_DISK_ERROR	;ERROR - disk initialization failure
M3:
	IF	CHEAP			;If cost-reduced Z-150
	CALL	SET_DIP_SWITCHES	;Set the DIP switch port values
	ENDIF
	CALL	SET_IO_CONFIGURATION	;Determine global I/O configuration

	IF	NOT MORROW
	MOV	AL,NMI_ENABLE		;Get code to enable NMI interrupts
	OUT	NMI_CONTROL,AL		;Enable NMI interrupts
	ENDIF

	MOV	AL,011111B		;Indicate that disk passed
	OUT	DIAG_LEDS,AL		;...on the diagnostics LEDs
	MOV	BX,Z100_EXT_ROM_SEGMENT	;Point to the Z-100 expansion ROM area
	MOV	CX,Z100_EXT_ROMS	;Get number of ROMs to check
	CALL	POLL_EXTERNAL_ROMS	;Check for Z-100 mode expansion ROMs
	MOV	BX,Z150_EXT_ROM_SEGMENT	;Point to the Z150 external ROM segment
	MOV	CX,Z150_EXT_ROMS	;Set the number of ROM slots
	CALL	POLL_EXTERNAL_ROMS	;Check for extra interface ROMs
	IN	AL,INTERRUPT_MASK	;Read the interrupt mask register
	AND	AL,TIMER_INTR_MASK	;Make sure that the timer is on...
	OUT	INTERRUPT_MASK,AL	;...in case external ROMs disabled it
	CMP	NO_BOOT_FLAG,TRUE	;Is auto-booting enabled?
	MOV	NO_BOOT_FLAG,FALSE	;[Clear the auto-boot flag value]
	JE	M6			;No, skip any possible auto-boot
	CALL	READ_DIP_SWITCHES	;Read the configuration switches
	AND	AL,BOOT_SELECT_MASK	;Mask out all but boot drive
	CMP	AL,NO_BOOT		;Are we auto-booting?
	JE	M6			;No, exit to the MFM-150 monitor
	CMP	AL,BOOT_FLOPPY		;Booting from 5-1/4" drive 0?
	JE	M4			;Yes - boot from floppy drive 0
	CMP	AL,BOOT_WINCHESTER	;Is the winchester the boot drive?
	JNE	M6			;No, exit to the ROM monitor
	OR	BOOT_DRIVE,80H		;Yes - set high bit on for winchester
	JMP	SHORT M5		;Boot from the winchester
M4:	AND	BOOT_DRIVE,7FH		;Mask out the floppy/winchester select
M5:	MOV	DL,BOOT_DRIVE		;Get the boot device as a parameter
	MOV	AL,BOOT_STRING		;...and the boot partition, too
	INT	BOOT_INTR		;Boot and execute operating system
M6:	JMP	MFM_150			;Enter the monitor/debugger
MAIN ENDP



;**********************************************************************
; RAM_TEST_I: (BASE, LENGTH)		  [DO NOT 'CALL' THIS ROUTINE!]
;
;	RAM_Test_I is called to perform a quick read/write test of
; a bank of of system RAM.  It uses a fast 'march test' algorithm to
; ensure that no addressing faults are present, and that each bit is
; capable of storing both 0 and 1.  In addition, both even and odd
; parity are used - this helps to further verify the parity generation
; and checking hardware.
;
; Input:
;    ES:DI: Pointer to base address of bank to be tested
;	SI: Length of bank to be tested (in words)
;	BP: Return address of caller (executed only if no
;	    errors occur).
;**********************************************************************
RAM_TEST_I PROC NEAR
	PUBLIC	RAM_TEST_I
	MOV	AX,01FEH		;Set initial pass data to 01FEH
	MOV	CX,SI			;Use CX as word count
	REP	STOS ES: WORD PTR [DI]	;Fill RAM with 01FEH
	MOV	DX,AX			;Save data from this pass
	NOT	AX			;Now fill RAM with 0FE01H
	SUB	DI,SI			;Point back to start of bank
	SUB	DI,SI			;(Remember that this is a word count)
	MOV	CX,SI			;Place bank length in CX again
	MOV	BX,OFFSET RTI1		;Set return address in BX
	JMP	MEM_TEST_I		;Perform a test pass
RTI1:	STD				;Now, go backwards
	SUB	DI,2			;Point to last word in bank
	MOV	AX,0			;Fill bank with 0000H
	MOV	CX,SI			;Set length of bank in CX
	MOV	BX,OFFSET RTI2		;Set return address parameter
	JMP	MEM_TEST_I		;Finish testing bank
RTI2:	CLD				;Set forward direction again
	JMP	BP			;Return to caller
RAM_TEST_I ENDP



;**********************************************************************
; MEM_TEST_I: (BASE, LENGTH, OLD, NEW)    [DO NOT 'CALL' THIS ROUTINE!]
;
;	Mem_Test_I is a special 'routine' used for testing memory
; during power-up diagnostics.  It is NOT called - rather it should
; be 'jumped' to.  When it is completed, it will return to the
; address in the BX register.  If an error occurs, control will be
; be transferred to the common error handler, which will attempt to
; initialize the video system, and display an error message.
;
; Input:
;    ES:DI: Base address of 64K segment to be tested.
;	CX: Length of bank to be tested
;	AX: Data to write into each word of RAM.
;	DX: Proper value for memory to have at start of pass.
;	BX: Return address following test completion.
; 
; Output:
;	DX: Data written to RAM memory
;
; Note: If MEM_TEST_I returns, then no error occurred. Otherwise, 
;       MEM_TEST_I will transfer control to the error handling code.
;**********************************************************************
MEM_TEST_I PROC NEAR
	PUBLIC	MEM_TEST_I
MTI1:	CMP	DX,WORD PTR ES: [DI]	;Does this word have the right data?
	JNZ	MTI3			;ERROR - improper data read
	STOS	WORD PTR ES: [DI]	;Write a word of memory with new data
	LOOP	MTI1			;Loop until whole bank tested
	MOV	DX,AX			;Save 'good' data in register DX

	IF	NOT MORROW
	IN	AL,MEMORY_STATUS	;Read the memory status port
	AND	AL,PARITY_ERR OR MEMORY_ERR ;Was there a parity error?
	JZ	MTI2			;No, return 'All OK'!
	CLC				;Flag that segment must be set
	JMP	BAD_PARITY		;ERROR - parity error occurred
	ENDIF

MTI2:	JMP	BX			;'Return' to caller
MTI3:	MOV	AX,WORD PTR ES: [DI]	;Get contents of failed memory word
	CLC				;Flag that segment must be set
	JMP	BAD_RAM			;ERROR - exit to error routine
MEM_TEST_I ENDP



;**********************************************************************
; INIT_INTERRUPTS:
;
;	Init_Interrupts initializes the interrupt vectors in low
; memory which point to I/O and interrupt service routines within
; the Z-150 monitor.  This routine also sets up the 8259A interrupt
; controller appropriately.
;**********************************************************************
INIT_INTERRUPTS PROC NEAR
	PUBLIC	INIT_INTERRUPTS
	PUSHREG	<AX,BX,CX,SI,DS>
	MOV	AX,0			;Get the number of the first vector
	MOV	BX,OFFSET Z150_INTR_TBL	;Point to the default intr table
	MOV	CX,Z150_INTERRUPTS	;Number of Z-150 intr vectors to be set
IIV1:	PUSH	CS			;Point to the ROM code segment
	POP	DS
	MOV	SI,WORD PTR CS: [BX]	;Get the intr vector
	TEST	SI,SI			;Is this a valid vector?
	JNZ	IIV2			;Yes - write it in memory
	MOV	DS,SI			;No, set segment to 0, too
IIV2:	CALL	SET_INTR_PTR		;Write the interrupt vector
	ADD	BX,2			;Point to the next interrupt vector
	INC	AX			;Increment the interrupt number
	LOOP	IIV1			;Continue till all initialized
	MOV	SI,0			;Set clear-intr-vector to 0000:0000
	MOV	DS,SI
IIV3:	CALL	SET_INTR_PTR		;Clear an interrupt vector to 0
	INC	AX			;Go on to the next interrupt
	CMP	AX,100H			;Finished initializing interrupts?
	JB	IIV3			;No, loop until rest of vectors cleared
	MOV	CX,CS			;Point DS to the code segment
	MOV	DS,CX
	CALL	READ_DIP_SWITCHES	;Read the configuration switches
	IF	CHEAP			;If cost-reduced machine...
	CMP	RESET_FLAG,1234H	;Is this a warm boot?
	JE	IIV35			;Yes - have logical switch values
	AND	AH,CHEAP_REFRESH_RATE	;Isolate the refresh rate select bit
	JE	IIV4			;Interrupt already set for 60 Hz
	JMP	IIV36			;Install 50 Hz parameters
	ENDIF
IIV35:	AND	AL,REFRESH_RATE		;Isolate the refresh rate select bit
	CMP	AL,HZ_60		;Setting up for 60 Hertz display?
	JE	IIV4			;Yes - interrupt already set
IIV36:	MOV	DS,CX			;Point DS to the code segment again
	MOV	SI,OFFSET ROM_PARMS_50	;No, point to the 50 Hertz display parms
	MOV	AL,VIDEO_PARMS_INTR	;Set AL to the video parms vector #
	CALL	SET_INTR_PTR		;Use 50 Hertz refresh display table
IIV4:	MOV	AL,INTR_MODE_1		;Init the first intr ctrlr mode byte
	OUT	INTERRUPT_CTRL,AL
	MOV	AL,HARDWARE_INTR_BASE	;Set the starting interrupt number
	OUT	INTERRUPT_PARMS,AL
	MOV	AL,INTR_MODE_2		;Init the second mode byte
	OUT	INTERRUPT_PARMS,AL
	MOV	AL,DEFAULT_INTR_MASK	;Get the default I/O interrupt config
IIV5:	OUT	INTERRUPT_MASK,AL	;Enable keyboard & timer interrupts
	POPREG	<DS,SI,CX,BX,AX>
	RET
INIT_INTERRUPTS ENDP



;**********************************************************************
; TIMER_INTERRUPT_TEST:
;
;	Timer_Interrupt_Test initializes the system real-time clock
; counter, and it then verifies its accuracy to within +/- 10%.  If
; the actual timed value is off by more than approximately 10%, or if
; no timer interrupts occur, then a failure will be declared.
;
; Output:
;	CY: Reset if the timer worked correctly, or set if an
;	    error occurred.
;**********************************************************************
TIMER_INTERRUPT_TEST PROC NEAR
	PUBLIC	TIMER_INTERRUPT_TEST
	PUSHREG	<AX,BX,CX>
	STI				;Permit timer interrupts to occur
	MOV	AL,CLOCK_TIMER_MODE	;Get the system clock channel's mode
	OUT	TIMER_CTRL,AL		;Initialize the clock timer
	MOV	TOD.LSCOUNT,0		;Set the initial time to 0 (LS WORD)
	MOV	AL,0			;Set clock period to 0 (/65536)
	OUT	TIMER_0,AL
	MOV	TOD.MSCOUNT,0		;Initialize M.S. word of system time
	OUT	TIMER_0,AL		;Timer 0 now dividing by 65536
	MOV	TOD.DAYS,0		;Clear timer overflow to 0
	MOV	BX,100			;Set maximum delay to .1 second
TINT1:	MOV	CX,1			;Set single delay to 1 millisecond
	CALL	DELAY			;Wait for a short while
	CMP	TOD.LSCOUNT,1		;Gotten an interrupt from the clock?
	JE	TINT2			;Yes - now check its accuracy
	DEC	BX			;Waited too long yet?
	JNZ	TINT1			;No, continue to wait
	JMP	TINT5			;ERROR - timer not running
TINT2:	MOV	BX,100			;Set delay to .1 second again
TINT3:	MOV	CX,1			;Delay for a millisecond...
	CALL	DELAY			;...while waiting for the clock intr
	CMP	TOD.LSCOUNT,2		;Gotten another interrupt yet?
	JE	TINT4			;Yes - See how accurate the timer is
	DEC	BX			;Waited too long yet?
	JNZ	TINT3			;No, continue to wait
	JMP	TINT5			;ERROR - timer not running
TINT4:	CMP	BX,CLOCK_MIN_TIME	;Is the timer running > 10% too fast?
	JA	TINT5			;ERROR - timer running too fast
	CMP	BX,CLOCK_MAX_TIME	;Is the timer rate > 10% too slow?
	JAE	TINT6			;No, running just fine
TINT5:	STC				;Set carry flag to indicate error
	JMP	TINT7			;Return
TINT6:	CLC				;Clear carry - no error occurred
TINT7:	POPREG	<CX,BX,AX>
	RET
TIMER_INTERRUPT_TEST ENDP



;**********************************************************************
; MANUFACTURING_TEST:
;
;	Manufacturing_Test reads the keyboard port and scans the code
; returned.  If the code is 0AAH, then the keyboard is OK, and the
; routine will return directly.  If the code is 055H, then this routine
; will download a program into memory through the keyboard port starting
; at 0000:0500.  Once a program has been downloaded, it is executed - the
; Manufacturing_Test routine will not return to the caller.  The first 
; two bytes received in manufacturing mode are the length of the code 
; to be downloaded (in bytes).
;
; Output:
;	CY: Reset if a valid start-up code of 0AAH is received from the
; 	    keyboard processor.  Set if an invalid keyboard code is
;	    received.
;**********************************************************************
MANUFACTURING_TEST PROC NEAR
	PUBLIC	MANUFACTURING_TEST
	PUSHREG	<AX,CX,DX,DI,ES>
	IN	AL,SYS_CTRL_PORT	;Read the keyboard ctrl port value
	MOV	DH,AL			;Save it in DH
	MOV	AL,INIT_KEYBOARD	;Get code to initialize the keyboard
	OUT	SYS_CTRL_PORT,AL	;Hold the keyboard reset line low
	MOV	CX,50			;Delay for roughly 20 milliseconds
	CALL	DELAY
	MOV	AL,CLEAR_KEYBOARD	;Now clear keyboard
	OUT	SYS_CTRL_PORT,AL	;...to complete power-up reset
	MOV	AL,READ_KEYBOARD	;Now, get command to read keyboard
	OUT	SYS_CTRL_PORT,AL	;Send the command to the parallel port
	CALL	DOWNLOAD_BYTE		;Read a key code from the keyboard
	JC	MT2			;ERROR - no key code returned
	CMP	AL,KEYBOARD_PRESENT	;Is this a normal power-up keycode?
	JE	MT2			;Yes - return OK
	CMP	AL,MANUFACTURING_PRESENT;Is this a manufacturing mode rqst?
	STC				;[Set carry in case it isn't]
	JNE	MT2			;ERROR - invalid keyboard code!
	CALL	DOWNLOAD_BYTE		;Get a downloaded byte
	MOV	CL,AL			;...as the first length byte
	CALL	DOWNLOAD_BYTE		;Get the second code length byte
	MOV	CH,AL			;Now have count to download in CX
	MOV	AX,INTERRUPT_SEGMENT	;Point ES to the low-memory segment
	MOV	ES,AX
	MOV	DI,OFFSET MANUFACTURING_BOOT ;Point to the mfg boot code buff
MT1:	CALL	DOWNLOAD_BYTE		;Read a downloaded object byte
	STOSB				;Store it in memory
	LOOP	MT1			;Loop until program downloaded
	JMP	MANUFACTURING_BOOT	;Execute the downloaded test program
MT2:	MOV	AL,DH			;Get the saved control port value
	OUT	SYS_CTRL_PORT,AL	;Restore the control port
	POPREG	<ES,DI,DX,CX,AX>
	RET
MANUFACTURING_TEST ENDP



;**********************************************************************
; POLL_EXTERNAL_ROMS: (BASE_SEGMENT, NUMBER_OF_ROMS)
;
;	Poll_External_ROMs is called to determine if additional
; peripheral interface ROMs are available.  The ROMs are identified
; by the first two bytes being 055H, then 0AAH.  The next byte (byte 3)
; contains the length of the ROM in 512 byte blocks.  To ensure that
; a valid ROM is installed, each ROM must pass a sumcheck test.
; Once a valid ROM is found, the monitor will perform a long call to
; the initialization code in the ROM, starting with the fourth byte.
;
; Input:
;	BX: Base segment address of expansion ROM area
;	CX: Maximum number of 2K blocks to be polled
;**********************************************************************
POLL_EXTERNAL_ROMS PROC NEAR
	PUBLIC	POLL_EXTERNAL_ROMS
	PUSHREG	<BX,CX,SI,ES>
PER1:	MOV	SI,0			;Point to the start of a ROM
	MOV	ES,BX			;Set the segment for the current ROM
	CMP	ES:[SI].FLAG,0AA55H	;Is there an interface ROM here?
	JNZ	PER2			;No, try the next 2K slot
	CALL	INSTALL_ROM		;Yes - install ROM if checksum OK
PER2:	ADD	BX,8*K SHR 4		;Point to the next 2K byte block
	LOOP	PER1			;Loop until all external ROMs polled
PER3:	POPREG	<ES,SI,CX,BX>
	RET
POLL_EXTERNAL_ROMS ENDP



;**********************************************************************
; INSTALL_ROM: (BASE_ADDR)
;
;	Install_ROM is called when an external ROM has been detected
; in the system.  It will perform a sumcheck on the contents of the ROM.
; If the sumcheck is valid, the initialization code in the ROM will
; be executed.
;**********************************************************************
INSTALL_ROM PROC NEAR
	PUBLIC	INSTALL_ROM
	PUSHREG	<AX,BX,CX,DX,SI,DI,BP,DS,ES>
	MOV	CH,BYTE PTR ES:[SI].LEN	;Get the length of this ROM (*512)
	MOV	CL,0			;Now have ROM length in words in CX
	SHL	CX,1			;Make word count into a byte count
	MOV	AL,0			;Clear sumcheck to 00H
IR1:	ADD	AL,BYTE PTR ES:[SI]	;Add a byte into the sumcheck
	INC	SI			;Point to the next byte to check
	LOOP	IR1			;Finish generating the sumcheck
	TEST	AL,AL			;Did the sumcheck pass?
	JNZ	IR2			;No, try the next ROM
	MOV	WORD PTR EXT_ROM_INIT[0],INIT	;Write the ROM's init offset
	MOV	WORD PTR EXT_ROM_INIT[2],ES	;Write the ROM's init segment
	PUSH	BX			;Save segment storage
	CALL	DWORD PTR [EXT_ROM_INIT];Execute the external ROM's init code
					;(May destroy all registers)
	POP	BX			;Restore pointer to next segment
IR2:	POPREG	<ES,DS,BP,DI,SI,DX,CX,BX,AX>
	RET
INSTALL_ROM ENDP



;**********************************************************************
;	 Z - 1 5 0   I N T E R R U P T   V E C T O R   T A B L E
;**********************************************************************
		PUBLIC	Z150_INTR_TBL
Z150_INTR_TBL	DW	OFFSET ZERO_DIVIDE_INTERRUPT
		DW	OFFSET SINGLE_STEP_INTERRUPT
		DW	OFFSET NMI_INTERRUPT
		DW	OFFSET BREAKPOINT_INTERRUPT
		DW	OFFSET OVERFLOW_INTERRUPT
		DW	OFFSET PRINT_SCREEN_INTERRUPT
		DW	OFFSET DO_NOTHING
		DW	0
		DW	OFFSET TIMER_INTERRUPT
		DW	OFFSET KEYBOARD_INTERRUPT
		DW	OFFSET RESERVED_HARDWARE_INTERRUPT
		DW	OFFSET RESERVED_HARDWARE_INTERRUPT
		DW	OFFSET RESERVED_HARDWARE_INTERRUPT
		DW	OFFSET RESERVED_HARDWARE_INTERRUPT
		DW	OFFSET DISK_INTERRUPT
		DW	OFFSET RESERVED_HARDWARE_INTERRUPT
		DW	OFFSET VIDEO_IO_INTERRUPT
		DW	OFFSET GET_IO_CONFIG_INTERRUPT
		DW	OFFSET GET_MEMORY_SIZE_INTERRUPT
		DW	OFFSET DISK_IO_INTERRUPT
		DW	OFFSET SERIAL_IO_INTERRUPT
		DW	OFFSET DO_NOTHING		;Dummy cassette I/O intr
		DW	OFFSET KEYBOARD_IO_INTERRUPT
		DW	OFFSET PRINTER_IO_INTERRUPT
		DW	OFFSET DEFAULT_IO_CONFIG	;I/O configuration tables
		DW	OFFSET BOOT_INTERRUPT
		DW	OFFSET TIME_OF_DAY_INTERRUPT
		DW	OFFSET DO_NOTHING
		DW	OFFSET DO_NOTHING
		DW	OFFSET ROM_PARMS_60		;Video parms (60 Hz)
		DW	OFFSET DISK_PARMS		;Disk parameters
		DW	0				;User-font pointer
Z150_INTERRUPTS	EQU	($ - Z150_INTR_TBL) / 2



MONITOR_SEGMENT ENDS

	END
