	PAGE	,132
	TITLE	Z-150 Serial and Parallel I/O Drivers

;**********************************************************************
;
;		  -----------------------------------
;----------------- Z-150 Serial/Parallel I/O Drivers ------------------
;		  -----------------------------------
;
;		Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME	CS:MONITOR_SEGMENT

	INCLUDE ../ROM/ROM.LIT
	INCLUDE ../ROM/INTR.LIT
	INCLUDE ../SYS/SYS.LIT
	INCLUDE IOCONFIG.LIT
	INCLUDE IOCONFIG.EXT

ESC_LENGTH	EQU	5		; Maximum escape sequence length
TO_VAL		EQU	1000		; Compatible time out count (~1.0 sec)

	EXTRN	DATA_SEGMENT:WORD, GET_INTR_PTR:NEAR, CASE:NEAR, DELAY:NEAR

;**********************************************************************
; SERIAL_IO_INTERRUPT: (FUNCTION, PARAMETERS)	    Interrupt #14H (20)
;
;	Serial_IO_Interrupt is the ROM entry point to communicate with
; the serial ports. The following routines are executed by first putting a
; function code in register AH, parameters in register AL and executing
; an INT 14H.
;
; All functions require a serial card number (0, 1) in DX.
;
; Function code 0 - Initialize serial port
;    Input:
;	AL:	Initialization parameters
;		bit 5-7 baud rate
;			000 - 110 baud
;			001 - 150 baud
;			010 - 300 baud
;			011 - 600 baud
;			100 - 1200 baud
;			101 - 2400 baud
;			110 - 4800 baud
;			111 - 9600 baud
;		bit 3-4 parity
;			X0 - No parity
;			01 - Odd parity
;			11 - Even parity
;		bit 2   Number of stop bits
;			0 - 1 stop bit
;			1 - 2 stop bits
;		bit 0-1 Word length
;			00 - 5 bits
;			01 - 6 bits
;			10 - 7 bits
;			11 - 8 bits
;   Output:
;	AX has status information
;		AH: Line status
;			bit 7 - Time out
;			bit 6 - Transmitter shift register empty
;			bit 5 - Transmitter holding register empty
;			bit 4 - Break detect
;			bit 3 - Framing error
;			bit 2 - Parity error
;			bit 1 - Overrun error
;			bit 0 - Data ready
;
;		AL: Modem status
;			bit 7 - Received line signal detect
;			bit 6 - Ring indicator
;			bit 5 - Data set ready
;			bit 4 - Clear to send
;			bit 3 - Delta receive line signal detect
;			bit 2 - Trailing edge ring detector
;			bit 1 - Delta data set ready
;			bit 0 - Delta clear to send
;
;   Function code 1 - Send character
;	Input:
;		AL has character to send
;	Output:
;		Same as above
;
;   Function code 2 - Receive character
;	Output:
;		AL: character received
;		AH: Line status (as above) except only error bits
;		    are left on (i.e. bits 7,4,3,2,1)
;
;   Function code 3 - Return status
;	Output:
;		Same as function 0
;
;**********************************************************************
SERIAL_IO_INTERRUPT PROC FAR
	PUBLIC	SERIAL_IO_INTERRUPT
	STI				;Interrupts back on
	PUSHREG	<DS,SI>			;Save registers
	PUSH	AX
	MOV	AL,IO_PARMS_INTR	;Get pointer to configuration tables
	CALL	GET_INTR_PTR
	POP	AX			;Restore parameters
	CALL	SERIAL_OP		;Do operation
	POPREG <SI,DS>
	IRET
SERIAL_IO_INTERRUPT ENDP



;**********************************************************************
; SERIAL_OP: (FUNCTION, PARAMETER, TABLE_PTR)
;
;	Serial_Op is called to actually perform one of the serial
; I/O operations.
;
;   Input:
;	AH - Function to perform
;	AL - Character to send or initialization parameters
;	DS:SI - Pointer to configuration table base
;
;**********************************************************************
SERIAL_OP PROC NEAR
	PUBLIC	SERIAL_OP
	CMP	AH,OFFSET SMAX		;Range check function value
	JAE	SOP_EXIT
	PUSHREG <ES,DI,SI,DX,BP,CX,BX>
	ADD	SI,PCT_SIZE*PRINTER_PORTS	;Adjust to serial tables
	PUSH	AX
	MOV	AL,DL			;and index to proper table
	MOV	AH,SCT_SIZE
	MUL	AH
	ADD	SI,AX
	POP	AX
	MOV	ES,DATA_SEGMENT		;Point to the current data segment
	MOV	DI,DX			;Get the port address
	SHL	DI,1
	MOV	DX,ES:SERIAL_TABLE[DI]
	OR	DX,DX			;Does port exist?
	JZ	SOP1
	SHR	DI,1			;Leave index in DI
	CALL	CASE			;Execute function
	JMP	NEAR PTR SOP1
SIOA:	DW	OFFSET S_INIT		;Initialize port
	DW	OFFSET S_OUTC		;Output character
	DW	OFFSET S_INP		;Input character
	DW	OFFSET S_STAT		;Return status
SMAX	EQU	($-SIOA)/2
SOP1:	POPREG <BX,CX,BP,DX,SI,DI,ES>
SOP_EXIT:
	RET

;	Initialize the serial port
S_INIT:	MOV	AH,AL			;Save parms
	ADD	DX,UR_LCR		;Address line control reg.
	MOV	AL,UC_DLA		;Turn on DLAB
	OUT	DX,AL
	MOV	BL,AH

;	Program the baud rate

	AND	BL,0E0H			; Isolate baud rate bits
	MOV	CL,3
	ROL	BL,CL
	SUB	BH,BH			; Use as index into baud table
	SHL	BX,1
	MOV	AL,BYTE PTR CS:BAUD_TABLE[BX+1]	;Get high divisor
	SUB	DX,UR_LCR-UR_DLM	; Divisor latch, high byte
	OUT	DX,AL
	MOV	AL,BYTE PTR CS:BAUD_TABLE[BX]	;Get low divisor
	DEC	DX			; Divisor latch low byte
	OUT	DX,AL

;	Set word length and parity

	AND	AH,1FH			;Isolate word size and parity info.
	MOV	AL,AH
	ADD	DX,UR_LCR		; Back to line control reg.
	OUT	DX,AL			; Initialize parity and word size
	SUB	DX,UR_LCR-UR_IER	; Set interrupt register
	SUB	AL,AL
	OUT	DX,AL			;All interrupts disabled
	DEC	DX			;Back to base port
	JMP	S_STAT

;	Output a character, with padding if necessary

S_OUTC:	CALL	S_OUT			;Output the users character
	CMP	AL,CR			;Check if char. was carriage return
	JNZ	S_OUTC2			;No, no padding
	MOV	CL,SCT_NUMPAD[SI]	;Yes, get the pad count
	SUB	CH,CH
	JCXZ	S_OUTC2			;No padding? Then exit
S_OUTC1:TEST	AH,1 SHL 7		;Error writing character?
	JNZ	S_OUTC2			;Yes exit
	MOV	AL,SCT_PADCHAR[SI]	;Else get pad character
	PUSH	CX
	CALL	S_OUT			; and output
	POP	CX
	LOOP	S_OUTC1
S_OUTC2:RET

;	Output a character with given handshake

S_OUT:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_PROTOCOL	;Test software or hardware handshake
	JNZ	S_OUT3			;Jump for software handshake
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_DTR	;DTR active?
	JZ	S_OUT1
	CALL	WAIT_DSR		;Wait for DSR
	JC	S_OUT2A			;Check for time out error
S_OUT1:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_RTS	;RTS active?
	JZ	S_OUT2
	CALL	WAIT_CTS		;Wait for CTS
	JC	S_OUT2A			;Check for time out error
S_OUT2:	CALL	OUTCHAR			;Output character
	JNC	S_OUT2B
S_OUT2A:JMP	TIME_OUT_ERROR
S_OUT2B:RET

;	Software handshake

S_OUT3:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_ETX_DC1	;ETX/ACK or DC1/DC3 ?
	JZ	S_OUT5			;Jump for ETX/ACK

;	DC1/DC3 Handshake

	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR
	JZ	S_OUT3B

;	Waiting for a DC1 handshake

	CALL	WAIT_DC1
	AND	BYTE PTR SCT_HANDSHAKE[SI],NOT SCT_WAIT_CHAR

;	Check for DC3

S_OUT3B:PUSH	AX
	CALL	INCHAR			;Check if character available
	JC	S_OUT4
	AND	AL,07FH			;Strip parity
	CMP	AL,DC3			;Is it ctrl-S
	JNZ	S_OUT4
	OR	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR ;If so, flag ctrl-S seen, wait for ctrl-Q
S_OUT4:	POP	AX
	CALL	OUTCHAR
	JC	TIME_OUT_ERROR
	RET

;	ETX/ACK Handshake

S_OUT5:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR	; Are we waiting for a handshake char?
	JZ	S_OUT5B			; No, continue
	CALL	WAIT_ACK
	AND	BYTE PTR SCT_HANDSHAKE[SI],NOT SCT_WAIT_CHAR	; Otherwise show not handshaking anymore
S_OUT5B: MOV	AH,AL			;Save the character
	MOV	AL,UC_DTR+UC_RTS	;Assert hardware handshake lines
	ADD	DX,UR_MCR		;Address modem control register
	OUT	DX,AL			;Assert DTR and RTS
	SUB	DX,UR_MCR		;Back to base port
	MOV	AL,AH			;Restore character
	CMP	AL,ESC			; Check for escape sequence
	JNZ	S_OUT6			; Jump if no escape sequence
	MOV	BL,ESC_LENGTH		; See if enough chars. in burst left for esc. seq.
	SUB	BL,SCT_COUNT[SI]
	JB	S_OUT6			; Jump if enough chars. left
	ADD	SCT_COUNT[SI],BL	; else make sure count is OK

S_OUT6:	CALL	OUTCHAR
	JC	TIME_OUT_ERROR		;Check for time out
	DEC	BYTE PTR SCT_COUNT[SI]	;Decrement the burst count

;	Check if burst is done

	JNZ	S_OUT7			; If zero then restore count and
	MOV	AL,SCT_FLAG[SI]		; Pick up old count
	MOV	SCT_COUNT[SI],AL	; Restore burst
	MOV	AL,ETX			;  send ETX
	CALL	OUTCHAR
	OR	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR ; Wait for an ACK
S_OUT7:
	RET

BUSY_ERROR:
	AND	AH,NOT (UC_THE+UC_TSE)	; Set device busy and set time
					; out showing could not print char.

;	Time out error, set time out flag and return status

TIME_OUT_ERROR:
	OR	AH,1 SHL 7		;Time out flag
	RET

;	Input a character

S_INP:	MOV	AL,UC_DTR+UC_RTS	;Set DTR and RTS high
	ADD	DX,UR_MCR		; in modem control reg.
	OUT	DX,AL
	SUB	DX,UR_MCR		;Back to base port
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_DEFAULT
	JZ	S_INP1			;Jump if non-compatible mode

	CALL	WAIT_DSR		;Wait for DSR signal
	JC	TIME_OUT_ERROR		;Check for time out error

;	Check if waiting for a software handshake character

S_INP1:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_PROTOCOL
	JZ	S_INP3			; Not a software handshake, continue
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR	; Are we waiting?
	JZ	S_INP3
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_ETX_DC1
	JNZ	S_INP2			; Jump for DC1/DC3 handshake
	CALL	WAIT_ACK		; Wait for an ACK character
	JMP	SHORT S_INP2A
S_INP2:	CALL	WAIT_DC1		; Wait for a DC1 character
S_INP2A: AND	BYTE PTR SCT_HANDSHAKE[SI],NOT SCT_WAIT_CHAR
	
S_INP3:	MOV	BL,ES:SERIAL_TIME_OUT[DI]	;Get the time out count
S_INP4:	MOV	BP,TO_VAL
S_INP5:	CALL	INCHAR			;Attempt to get a character
	JNC	S_INP6			;Character received? Yes, continue
	MOV	CX,1			;No, Wait a while and try again
	CALL	DELAY
	DEC	BP			;Check time out value
	JNZ	S_INP5			; and try again if OK
	DEC	BL
	JNZ	S_INP4
	JMP	TIME_OUT_ERROR
S_INP6:	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_PROTOCOL	;Is this a software handshake?
	JZ	S_INP7			;No, return
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_ETX_DC1	;Is this DC1/DC3 handshake?
	JZ	S_INP7			;No, return
	PUSH	AX			;Save character
	AND	AL,07FH			;Strip parity
	CMP	AL,DC3			;Was character DC3?
	POP	AX
	JNZ	S_INP7			;No, return
	OR	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR	;Show waiting for DC1
	OR	AH,1 SHL 7		;Show char. could not be received
	JMP	S_INP8
S_INP7:	AND	AH,UC_OR+UC_PE+UC_FE+UC_BI ;Return only error bits
S_INP8:	RET

;	Return status

S_STAT: ADD	DX,UR_LSR		;Get to line status port
	IN	AL,DX
	MOV	AH,AL
	INC	DX			;Now do the modem port
	IN	AL,DX
	SUB	DX,UR_MSR		;Back to base port

;	Check status of the selected handshake

	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_PROTOCOL
	JNZ	S_STAT4			; Do software handshake status
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_DTR
	JZ	S_STAT2			; Do RTS

;	Test state of DSR line

	TEST	AL,UC_DSR
	PUSHF				; Save status
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_DTR_POLAR	; Check polarity
	JNZ	S_STAT1			; Pick appropriate jump for polarity of line
	POPF				; Restore status
	JNZ	S_STAT9			; Show device busy
	JMP	SHORT S_STAT2
S_STAT1:
	POPF				; Restore status
	JZ	S_STAT9			; Show device busy

;	Test CTS line

S_STAT2:
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_RTS	; Check if RTS handshaking active
	JZ	S_STAT8			; No, return status
	TEST	AL,UC_CTS
	PUSHF				; Save status
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_RTS_POLAR	; Check polarity
	JNZ	S_STAT3			; Pick appropriate jump for polarity of line
	POPF				; Restore status
	JNZ	S_STAT9			; Show device busy
	JMP	SHORT S_STAT8		; Show normal status
S_STAT3:
	POPF				; Restore status
	JZ	S_STAT9			; Show device busy
	JMP	SHORT S_STAT8		; Show device not busy

;	Do status on software handshake

S_STAT4:
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_WAIT_CHAR	; Are we waiting for a char?
	JZ	S_STAT8			; No, continue
	MOV	BX,AX			; Save status
	CALL	INCHAR			; Try to read a char
	JNC	S_STAT5			; Char found, see if good handshake
	MOV	AX,BX			; else restore status and show busy
	JMP	S_STAT9
S_STAT5:
	AND	BH,NOT UC_DR		; Show that char. was taken
	AND	AL,07FH			; Strip high bit
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_ETX_DC1	;Which handshake char is needed?
	JNZ	S_STAT6			;Look for DC1
	CMP	AL,ACK			;Check for ACK
	JMP	SHORT S_STAT7
S_STAT6:
	AND	AL,07FH			;Strip parity
	CMP	AL,DC1
S_STAT7:
	MOV	AX,BX			;Restore status
	JNZ	S_STAT9			;Jump if not correct character
	AND	BYTE PTR SCT_HANDSHAKE[SI],NOT SCT_WAIT_CHAR	;Show handshake received
S_STAT8:
	RET				;Return from case
S_STAT9:
	AND	AH,NOT (UC_THE+UC_TSE)	;Show device busy
	RET
SERIAL_OP ENDP


;**	WAIT_DSR - Wait for the DSR signal before returning
;
WAIT_DSR PROC NEAR
	PUBLIC	WAIT_DSR
	PUSH	BX			;Free a register
	PUSH	AX			;Save AX
	PUSH	CX			; and CX
	PUSH	BP
	MOV	BL,ES:SERIAL_TIME_OUT[DI]	; Get time out parameter
	ADD	DX,UR_MCR		;Address modem control reg.
	MOV	AL,UC_DTR+UC_RTS	;Turn on handshake lines
	OUT	DX,AL
	ADD	DX,UR_MSR-UR_MCR	;Address modem status reg.
DSR0:	MOV	BP,TO_VAL
DSR1:	IN	AL,DX
	MOV	AH,AL
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_DTR_POLAR
	JNZ	DSR2
	XOR	AL,UC_DSR		;Flip polarity of DSR
DSR2:	TEST	AL,UC_DSR		;Signal good?
	JNZ	DSR4			;Yes then exit.
	DEC	BP			;else decrement retry count
	JZ	DSR3			;exit with 'C' set if time out
	MOV	CX,1			;Else wait 1 msec and try again
	CALL	DELAY
	JMP	DSR1
DSR3:	DEC	BL
	JNZ	DSR0
	STC				;Show time out
	JMP	SHORT	DSR5		;Exit
DSR4:	CLC				;Show good signal
DSR5:	PUSHF				;Save return flag
	SUB	DX,UR_MSR		;Back to base port
	POPF				;Restore return flag
	POP	BP
	POP	CX			;Restore CX
	POP	BX			;Restore AL
	MOV	AL,BL
	POP	BX			;Restore bx
	RET

WAIT_DSR ENDP


;**	WAIT_CTS - Wait for CTS signal active
;
WAIT_CTS PROC NEAR
	PUBLIC	WAIT_CTS
	PUSH	BX			;Free a register
	PUSH	AX			;Save AX
	PUSH	CX			;Save CX
	PUSH	BP
	MOV	BL,ES:SERIAL_TIME_OUT[DI]	;Get time out value
	ADD	DX,UR_MCR		;Address modem control reg.
	MOV	AL,UC_RTS+UC_DTR	;Assert RTS
	OUT	DX,AL
	ADD	DX,UR_MSR-UR_MCR	;Back to base port
CTS0:	MOV	BP,TO_VAL
CTS1:	IN	AL,DX
	MOV	AH,AL
	TEST	BYTE PTR SCT_HANDSHAKE[SI],SCT_RTS_POLAR
	JNZ	CTS2
	XOR	AL,UC_CTS		;Flip polarity of signal
CTS2:	TEST	AL,UC_CTS		;Is signal active?
	JNZ	CTS4
	DEC	BP			;Decrement retry count
	JZ	CTS3			;Exit with 'C' set if time out
	MOV	CX,1			;Wait 1 msec and try again
	CALL	DELAY
	JMP	CTS1
CTS3:	DEC	BL
	JNZ	CTS0
	STC				;Show time out occured
	JMP	SHORT CTS5
CTS4:	CLC				;Show good signal received
CTS5:	PUSHF				;Save return flag
	SUB	DX,UR_MSR		;Back to base port
	POPF				;Restore return flag
	POP	BP
	POP	CX			;Restore CX
	POP	BX			;Restore AL
	MOV	AL,BL
	POP	BX			;Restore BX
	RET

WAIT_CTS ENDP


;**	WAIT_ACK - Wait for ACK character
;
WAIT_ACK PROC NEAR
	PUBLIC	WAIT_ACK
	PUSH	AX			;Save AX
	MOV	AH,ACK			;Compare for ACK
	CALL	WAIT_CHAR		;Wait for the character
	POP	AX
	RET
WAIT_ACK ENDP


;**	WAIT_DC1 - Wait for DC1 (ctrl-Q) character
;
WAIT_DC1 PROC NEAR
	PUBLIC	WAIT_DC1
	PUSH	AX			;Save AX
	MOV	AH,DC1			;Compare for DC1
	CALL	WAIT_CHAR		;Wait for character
	POP	AX
	RET
WAIT_DC1 ENDP


;**	WAIT_CHAR - Wait for the character in AH to be received
;
WAIT_CHAR PROC NEAR
	PUBLIC	WAIT_CHAR
	PUSH	BX
	MOV	BL,AH
WC1:	CALL	INCHAR			;Get a character
	JC	WC1			;Try again if none
	AND	AL,7FH
	CMP	AL,BL			;Correct character?
	JNZ	WC1			;No, try again
	POP	BX
	RET
WAIT_CHAR ENDP


;**	INCHAR - Input a character and apply mapping and parity stripping
;
;	Returns - AH has line status
;
INCHAR	PROC NEAR
	PUBLIC	INCHAR
	ADD	DX,UR_LSR		;Address Line Status reg.
	IN	AL,DX
	SUB	DX,UR_LSR
	MOV	AH,AL			;Return status in AH
	TEST	AL,UC_DR		;Check for data ready
	JNZ	IC1			;If so, skip
	STC				;else show no data and return
	RET
IC1:	IN	AL,DX
	TEST	BYTE PTR SCT_ATTR[SI],SCT_SPI	;Strip parity?
	JZ	IC2
	AND	AL,7FH
IC2:	TEST	BYTE PTR SCT_ATTR[SI],SCT_MLI	;Map lower?
	JZ	IC3
	CALL	MAP_UPPER
IC3:	CLC				;Show character available
	RET
INCHAR	ENDP


;**	OUTCHAR - Send the character in AL with mapping and parity stripping
;
;	Returns - AH has line status
;
OUTCHAR PROC NEAR
	PUBLIC	OUTCHAR
	PUSH	BX			;Free a register
	PUSH	BP
	MOV	BL,ES:SERIAL_TIME_OUT[DI]
	TEST	BYTE PTR SCT_ATTR[SI],SCT_MLO	;Map to upper?
	JZ	OC1
	CALL	MAP_UPPER
OC1:	TEST	BYTE PTR SCT_ATTR[SI],SCT_SPO	;Strip parity?
	JZ	OC2
	AND	AL,7FH
OC2:	MOV	AH,AL		;Save character
OC2A:	MOV	BP,TO_VAL
OC3:	ADD	DX,UR_LSR	;Address line status reg.
	IN	AL,DX
	SUB	DX,UR_LSR	;Back to base port
	TEST	AL,UC_THE	;Ready to transmit?
	JNZ	OC4
	DEC	BP		; loop for status
	JZ	OC3A
	MOV	CX,1		; Wait a little while
	CALL	DELAY
	JMP	OC3
OC3A:	DEC	BL		; Decrement retry count
	JNZ	OC2A
	STC
	JMP	SHORT	OC5

OC4:	CLC
OC5:	XCHG	AL,AH		;Restore character and save status in AH
	OUT	DX,AL		;Send character
	POP	BP
	POP	BX		;Restore BX
	RET
OUTCHAR ENDP



	PAGE
;************************************************************************
; PRINTER_IO_INTERRUPT: (FUNCTION, PARAMETERS)	    Interrupt #17H (23)
;
;	Printer_IO_Interrupt is the ROM entry point to communicate
; with the printer. The following routines are executed by first putting
; parameters into registers, as described below, and then placing a
; function code into register AH. Finally an INT 17H instruction is
; executed.
;
; All functions require the printer number (0, 1, 2) in register DX
;
; Function code 0 - Write character to the printer
;   Output:
;	AH: Bit mapped status code
;		bit 7 - (1=) Not busy
;		bit 6 - (1=) Acknowledge
;		bit 5 - (1=) Out of paper
;		bit 4 - (1=) Selected
;		bit 3 - (1=) I/O error
;		bit 2 - Unused	(Always 0)
;		bit 1 - Unused  (Always 0)
;		bit 0 - (1=) Time out
;
; Function code 1 - Initialize printer port
;   Output:
;	Same as above
;
; Function code 2 - Read printer status into AH
;   Output:
;	Same as above
;************************************************************************
PRINTER_IO_INTERRUPT PROC NEAR
	PUBLIC	PRINTER_IO_INTERRUPT
	STI
	PUSHREG <DS,DX,BX,CX,SI,DI,BP>
	PUSH	AX
	CMP	AH,OFFSET PMAX		;Check function value
	JAE	PIO3
	PUSH	AX
	MOV	AL,IO_PARMS_INTR	;Get configuration table pointer
	CALL	GET_INTR_PTR
	MOV	AL,DL			;Index into printer table
	MOV	AH,PCT_SIZE
	MUL	AH
	MOV	BX,AX
	POP	AX
	MOV	BL,BYTE PTR PCT_REMAP[SI+BX]	;Get re-map value
	AND	BL,0FH			;Strip attributes
	OR	BL,BL			;Any re-map?
	JZ	PIO1
	DEC	BL			;Correct re-map value
	MOV	DL,BL			;In correct reg.
	SUB	DH,DH
	CALL	PARALLEL_TO_SERIAL	;Do serial operation
	JMP	SHORT PIO2
PIO1:
	PUSH	DS
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	ASSUME	DS:ROM_DATA

	MOV	DI,DX			;Save printer index
	MOV	BL,PRINTER_TIME_OUT[DI]	;Get time out value
	SHL	DI,1			;Word index
	MOV	DX,PRINTER_TABLE[DI]	;Get printer base port
	POP	DS

	ASSUME	DS:NOTHING

	OR	DX,DX			;Ensure printer exists
	JZ	PIO2

	CALL	CASE			;Dispatch command
	JMP	NEAR PTR PIO2
PCMD:	DW	OFFSET P_OUTC		;Print character
	DW	OFFSET P_INIT		;Initialize
	DW	OFFSET P_STAT		;Return status
PMAX	EQU	($-PCMD)/2

PIO2:	XOR	AH,48H			;Flip Ack and Selected bits
PIO3:	POP	BX
	MOV	AL,BL
	POPREG <BP,DI,SI,CX,BX,DX,DS>
	IRET

;	Output character with padding if necessary

P_OUTC:	CMP	AL,CR			;Check if character if carriage return
	PUSHF				;Save results
	PUSH	BX			;Save time out value
	CALL	P_OUT			;Output the users character
	POP	BX			;Restore time out value
	POPF				;Was character a CR?
	JNZ	P_OUTC2			;No, no padding
	MOV	CL,PCT_NUMPAD[SI]	;Yes, get the pad count
	SUB	CH,CH
	JCXZ	P_OUTC2			;No padding? Then exit
P_OUTC1:TEST	AH,1 SHL 0+1 SHL 3+1 SHL 5	;Error writing character?
	JNZ	P_OUTC2			;Yes exit
	MOV	AL,PCT_PADCHAR[SI]	;Else get pad character
	PUSH	CX
	PUSH	BX			;Save time out value
	CALL	P_OUT			; and output
	POP	BX			;Restore time out value
	POP	CX
	LOOP	P_OUTC1
P_OUTC2:RET


;
;	Print character in AL
;
P_OUT:	TEST	BYTE PTR PCT_REMAP[SI],PCT_SPO	;Check for strip parity
	JZ	P_OUT1			;No, skip
	AND	AL,7FH			;Yes strip parity
P_OUT1:	TEST	BYTE PTR PCT_REMAP[SI],PCT_MLO	;Check for map lower to upper
	JZ	P_OUT2			;No, skip
	CALL	MAP_UPPER		;Yes, map to upper
P_OUT2:	OUT	DX,AL			;Output character
	INC	DX			;Point to status port
P_OUT2A:MOV	BP,TO_VAL
P_OUT3:	IN	AL,DX			;Get status
	TEST	AL,1 SHL 7		;Test for not busy
	JNZ	P_OUT4			;Send strobe
	DEC	BP
	JZ	P_OUT3A
	MOV	CX,1
	CALL	DELAY			;Wait a little
	JMP	P_OUT3
P_OUT3A:DEC	BL
	JNZ	P_OUT2A
	MOV	AH,AL			;Time out, set flags
	OR	AH,1 SHL 0		;Set time out bit
	AND	AH,0F9H			;Turn off unused bits
	RET
P_OUT4:	INC	DX			;Point to port C
	MOV	AL,6 SHL 1 + 1		;Bit 6 is strobe
	OUT	DX,AL
	MOV	AL,6 SHL 1 + 0		;Set strobe low
	OUT	DX,AL
	SUB	DX,2			;Back to base port
	JMP	SHORT P_STAT

;*	Initialize printer port
;
P_INIT:	ADD	DX,2
	MOV	AL,8			;Set init line low
	OUT	DX,AL
	MOV	CX,20			;Delay for init
	CALL	DELAY
	MOV	AL,0CH			;Interrupts off
	OUT	DX,AL
	SUB	DX,2			;Back to base port
;	JMP	SHORT P_STAT		;Get status

;*	Return status
;
P_STAT:	INC	DX
	IN	AL,DX
	DEC	DX			;Back to base port
	MOV	AH,AL
	AND	AH,0F8H			;Mask off unused bits
	RET				;Exit
PRINTER_IO_INTERRUPT ENDP



;**********************************************************************
; PARALLEL_TO_SERIAL: (FUNCTION, CHARACTER, PORT_NUM, CONFIG_PTR)
;
; 	This routine will map the function number in AH to correspond to
; the equivalent serial function. Upon return from the serial I/O, the
; status will be mapped as closely as possible.
;
; 	Input:	AH - Function number
;		AL - Character to print
;		DX - Serial card number
;		DS:SI - Pointer to base of configuration table
;
; Function code 0 - Print character
;		    Map to serial function 1
;	Output:
;		AH - serial status mapped to parallel status as follows
;		bit 7 - Not busy (Transmitter register empty)
;		bit 6 - Acknowledge (0 on return from this routine)
;		bit 5 - Out of paper (0 on return from this routine)
;		bit 4 - Selected (0 on return from this routine)
;		bit 3 - I/O error (set if break, framing, parity,
;			or overrun error)
;		bit 2 - Not used (0 on return from this routine)
;		bit 1 - Not used (0 on return from this routine)
;		bit 0 - Time out
;
; Function code 1 - Initialize printer port
;		    Map to serial function 0
;	Output:
;		Same as above
;
; Function code 2 - Return printer status
;		    Map to serial function 3
;	Output:
;		Same as above
;**********************************************************************
PARALLEL_TO_SERIAL PROC NEAR
	PUBLIC	PARALLEL_TO_SERIAL
	PUSH	BX			;Save a register
	PUSH	AX			;Save the character
	CALL	CASE			;Execute appropriate function
	JMP	NEAR PTR PTS1
	DW	OFFSET PTS5		;Output character
	DW	OFFSET PTS6		;Initialize printer
	DW	OFFSET PTS7		;Return status
PTS1:	SUB	AL,AL
	XCHG	AL,AH
	TEST	AL,1 SHL 7		;Test for time out
	JZ	PTS2
	OR	AH,1 SHL 0 + 1 SHL 3	;Map to time out and error
PTS2:	TEST	AL,60H			;Test for transmitter empty
	JZ	PTS3
	OR	AH,1 SHL 7		;Map to not busy
PTS3:	XOR	AH,1 SHL 3		;Negate the I/O error bit
	POP	BX			;Restore the character
	MOV	AL,BL			; into AL
	POP	BX			;Restore BX
	RET

;*	Output character
;
PTS5:	MOV	AH,SIO_WRITE		;Map to serial output
	JMP	SHORT PTS8

;*	Initialize printer
;
PTS6:	PUSH	SI
	ADD	SI,PCT_SIZE*PRINTER_PORTS	;Address serial tables
	PUSH	AX			;Free a register
	MOV	AL,SCT_SIZE		;Find offset of correct serial table
	MUL	DL
	ADD	SI,AX			;Add into base address
	POP	AX			;Restore register
	MOV	AL,SCT_INIT[SI]		;Get initialization value
	POP	SI			;Restore configuration table pointer
	MOV	AH,SIO_INIT		;Map to serial init.
	JMP	SHORT PTS8

;*	Return status
;
PTS7:	MOV	AH,SIO_STATUS		;Map to serial status request

PTS8:	CALL	SERIAL_OP
	RET
PARALLEL_TO_SERIAL ENDP



;**********************************************************************
; MAP_UPPER: (CHAR)
;
;	Map_Upper is called to convert the character in AL into
; its upper case value.
;
; Input:
;	AL: Character to be converted (of arbitrary case)
;
; Output:
;	AL: Upper case character.
;**********************************************************************
MAP_UPPER PROC	NEAR
	PUBLIC	MAP_UPPER
	CMP	AL,'a'
	JB	MAP_EXIT
	CMP	AL,'z'
	JA	MAP_EXIT
	SUB	AL,'a'-'A'
MAP_EXIT: RET
MAP_UPPER ENDP


INIT_IO_TIMERS	PROC NEAR
	PUBLIC	INIT_IO_TIMERS
	PUSHREG	<AX,CX,SI,DI,ES>
	PUSH	DS
	POP	ES
	MOV	DI,OFFSET PRINTER_TIME_OUT
	MOV	SI,OFFSET PCT1
	MOV	CX,3
IT1:
	MOV	AL,CS:PCT_TIME_OUT[SI]
	STOSB
	ADD	SI,PCT_SIZE
	LOOP	IT1

	MOV	SI,OFFSET SCT1
	MOV	DI,OFFSET SERIAL_TIME_OUT
	MOV	CX,2
IT2:
	MOV	AL,CS:SCT_TIME_OUT[SI]
	STOSB
	ADD	SI,SCT_SIZE
	LOOP	IT2
	POPREG	<ES,DI,SI,CX,AX>
	RET
INIT_IO_TIMERS	ENDP



;**********************************************************************
;	D E F A U L T   C O N F I G U R A T I O N   T A B L E S
;**********************************************************************
;
; Printer 1
;
DEFAULT_IO_CONFIG EQU THIS BYTE
	PUBLIC	DEFAULT_IO_CONFIG
PCT1	DB	0			;No remap, no attributes.
	DB	0			;Pad character
	DB	0			;No pad count
	DB	20			;Initial time out value
;
; Printer 2
;
PCT2	DB	0
	DB	0
	DB	0
	DB	20
;
; Printer 3
;
PCT3	DB	0
	DB	0
	DB	0
	DB	20
;
; Serial port 1
;
SCT1	DB	SCT_DEFAULT+SCT_DTR+SCT_RTS+SCT_RTS_POLAR+SCT_DTR_POLAR
					;Default handshake, compatibility mode
					;DTR and RTS high polarity
	DB	0			;No attributes
	DB	0			;Pad character
	DB	0			;No pad count
	DB	0			;Flag for DC1/DC3 and ETX/ACK
	DB	0			;Count for ETX/ACK
	DB	110B SHL 5 + 00B SHL 3 + 0B SHL 2 + 11B	;4800 baud,
					;No parity, 1 stop and 8 data bits
	DB	1			;Initial time out value
;
; Serial port 2
;
SCT2	DB	SCT_DEFAULT+SCT_DTR+SCT_RTS+SCT_RTS_POLAR+SCT_DTR_POLAR
					;Default handshake, compatibility mode
					;DTR and RTS high polarity
	DB	0
	DB	0
	DB	0
	DB	0
	DB	0
	DB	110B SHL 5 + 00B SHL 3 + 0B SHL 2 + 11B
	DB	1



;**********************************************************************
;		     B A U D   R A T E   T A B L E
;**********************************************************************
BAUD_TABLE EQU THIS WORD		;Divisors for 8250 baud generator
	DW	1047			; 110 baud
	DW	768			; 150 baud
	DW	384			; 300 baud
	DW	192			; 600 baud
	DW	96			;1200 baud
	DW	48			;2400 baud
	DW	24			;4800 baud
	DW	12			;9600 baud


MONITOR_SEGMENT ENDS

	END

