#ifndef __UCHARVEC_H__
#define __UCHARVEC_H__

/*
 * Declarations for Unsigned Character (Byte) vectors
 *
 * $Id: ucvec.h,v 3.4.2.1 90/10/22 15:52:35 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 *  Defining the preprocessor directive BOUNDS_CHECK will
 *  cause bounds checking on the subscripting operator.
 */

#include "ivec.h"

class DoubleVec;
class UCharVec;

/*
 * The UCharPick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class UCharPick {
private:
  const UCharVec*	V;
  const IntVec*	X;
  UCharPick(const UCharVec* v, const IntVec* x); /* Constructor is private*/
  UCharPick(const UCharPick& p) {V=p.V; X=p.X;}
  friend		UCharVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const UCharVec&);
  void			operator=(const UCharPick&);
  void			operator=(UChar);

  inline UChar&	operator()(int i) Const;
  unsigned		length() Const	{ return X->length(); }
};

class UCharVec {
  RWBlock*		block;
  UChar*		begin;
  unsigned		npts;
  int			step;
  static int		numberPerLine;             /* For printing*/
  UCharVec(const UCharVec&, int, unsigned, int);   /* For slices*/
protected:
  void			boundsCheck(int)   Const;
  void			boundsErr(int)     Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int)     Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  UCharVec();
  UCharVec(unsigned n);
  UCharVec(unsigned n, UChar val);
  UCharVec(unsigned n, UChar val, UChar by);
  UCharVec(const UCharVec& a);
  UCharVec(const UCharPick& p);
  UCharVec(const UChar* dat, unsigned n);  /* Copy of dat will be made*/
  ~UCharVec();
  
  operator		IntVec() Const;	/* Convert to IntVec*/

  UCharVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  UChar*		data() Const	{return begin;}
  unsigned		length() Const	{return npts;}
  int			stride() Const	{return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  UCharVec		copy() Const {return deepCopy();}/* Synonym for deepCopy()*/
  UCharVec		deepCopy() Const;		/* copy with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  UCharVec&		reference(const UCharVec& v);	/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;
  
  /* Indexing:*/
  UChar&		operator[](int i) Const;	/* With bounds checking*/
  inline UChar&		operator()(int i) Const;	/* With optional bounds checking*/
  UCharPick		operator()(const IntVec& x) Const;
  inline UChar&		sub(int i) Const;		/* Assumes stride==1; use carefully*/
  
  /* Assignment:*/
  UCharVec&		operator=(const UCharVec& v); /* Must be same length as v*/
  UCharVec&		operator=(const UCharPick&);
  UCharVec&		operator=(UChar);

  /* Boolean operators:*/
  RWBoolean		operator==(const UCharVec&) Const;
  RWBoolean		operator!=(const UCharVec&) Const;
  
  /* Arithmetic operators:*/
  UCharVec&		operator++();
  UCharVec&		operator--();
  UCharVec&		operator+=(const UCharVec&);
  UCharVec&		operator+=(UChar);
  UCharVec&		operator-=(const UCharVec&);
  UCharVec&		operator-=(UChar);
  UCharVec&		operator*=(const UCharVec&);
  UCharVec&		operator*=(UChar);
  UCharVec&		operator/=(const UCharVec&);
  UCharVec&		operator/=(UChar);
  
  /* Friendly arithmetic operators:*/
  friend UCharVec	operator-(const UCharVec&);
#ifndef NO_UNARY_PLUS
  friend UCharVec	operator+(const UCharVec&);
#endif
  friend UCharVec	operator*(const UCharVec&,const UCharVec&);
  friend UCharVec	operator/(const UCharVec&,const UCharVec&);
  friend UCharVec	operator+(const UCharVec&,const UCharVec&);
  friend UCharVec	operator-(const UCharVec&,const UCharVec&);
  friend UCharVec	operator*(const UCharVec&,UChar);
  friend UCharVec	operator*(UChar,const UCharVec&);
  friend UCharVec	operator/(const UCharVec&,UChar);
  friend UCharVec	operator/(UChar,const UCharVec&);
  friend UCharVec	operator+(const UCharVec&,UChar);
  friend UCharVec	operator+(UChar,const UCharVec&);
  friend UCharVec	operator-(const UCharVec&,UChar);
  friend UCharVec	operator-(UChar,const UCharVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions:*/
  UCharVec		apply(mathFunTy) Const;
  friend UCharVec	abs(const UCharVec&);
  friend UCharVec	cumsum(const UCharVec&);
  friend UChar		dot(const UCharVec&,const UCharVec&);
  friend int		max(const UCharVec&);
  friend int		min(const UCharVec&);
  friend UChar		prod(const UCharVec&);
  friend UCharVec	reverse(const UCharVec&);
  friend UChar		sum(const UCharVec&);
#endif
};

/* Input / Output of vectors of unsigned characters:*/
ostream&	operator<<(ostream&, const UCharVec&);
istream&	operator>>(istream&, UCharVec&);

/******************* I N L I N E S **************************/

Inline void	
  UCharVec::boundsCheck(int i) Const{
  if(i<0 || i>npts) boundsErr(i);
}

Inline UChar&	
  UCharVec::operator[](int i) Const{
  boundsCheck(i); return begin[i*step];
}

inline UChar&	
  UCharVec::operator()(int i) Const {
#ifdef BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

inline UChar&
UCharVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline UCharVec
  UCharVec::slice(int start, unsigned n, int str) Const {
	return UCharVec(*this, start, n, str);
}

#ifndef NO_UNARY_PLUS
  Inline UCharVec	operator+(const UCharVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline UCharVec	operator*(UChar a, const UCharVec& b)	{return b*a;}
  inline UCharVec	operator+(UChar a, const UCharVec& b)	{return b+a;}
#endif

Inline UCharVec	abs(const UCharVec& a)    {return a;}

/********************  Pick inlines *****************************/
 
Inline
UCharPick::UCharPick(const UCharVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline UChar&
UCharPick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __UCHARVEC_H__ */
