#ifndef __SCHARVEC_H__
#define __SCHARVEC_H__

/*
 * Declarations for Signed Character (Byte) vectors
 *
 * $Id: scvec.h,v 3.4.2.1 90/10/22 15:52:32 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 *  Defining the preprocessor directive BOUNDS_CHECK will
 *  cause bounds checking on the subscripting operator.
 */

#include "ivec.h"

class DoubleVec;
class SCharVec;

/*
 * The SCharPick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class SCharPick {
private:
  const SCharVec*	V;
  const IntVec*	X;
  SCharPick(const SCharVec* v, const IntVec* x); /* Constructor is private*/
  SCharPick(const SCharPick& p) {V=p.V; X=p.X;}
  friend		SCharVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const SCharVec&);
  void			operator=(const SCharPick&);
  void			operator=(SChar);

  inline SChar&	operator()(int i) Const;
  unsigned		length() Const	{ return X->length(); }
};

class SCharVec {
  RWBlock*		block;
  SChar*		begin;
  unsigned		npts;
  int			step;
  static int		numberPerLine;             /* For printing*/
  SCharVec(const SCharVec&, int, unsigned, int);   /* For slices*/
protected:
  void			boundsCheck(int)   Const;
  void			boundsErr(int)     Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int)     Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  SCharVec();
  SCharVec(unsigned n);
  SCharVec(unsigned n, SChar val);
  SCharVec(unsigned n, SChar val, SChar by);
  SCharVec(const SCharVec& a);
  SCharVec(const SCharPick& p);
  SCharVec(const SChar* dat, unsigned n);  /* Copy of dat will be made*/
  ~SCharVec();
  
  operator		IntVec()    Const;  /* Convert to IntVec*/

  /* Slices:*/
  SCharVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  SChar*		data()   Const {return begin;}
  unsigned		length() Const {return npts;}
  int			stride() Const {return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  SCharVec		copy() Const {return deepCopy();}/* Synonym for deepCopy()*/
  SCharVec		deepCopy() Const;		/* copy with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  SCharVec&		reference(const SCharVec& v);	/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;
  
  /* Indexing:*/
  SChar&		operator[](int i) Const;	/* With bounds checking*/
  inline SChar&		operator()(int i) Const;	/* With optional bounds checking*/
  SCharPick		operator()(const IntVec& x) Const;
  inline SChar&		sub(int i) Const;		/* Assumes stride==1; use carefully*/
  
  /* Assignment:*/
  SCharVec&		operator=(const SCharVec& v); /* Must be same length as v*/
  SCharVec&		operator=(const SCharPick&);
  SCharVec&		operator=(SChar);

  /* Boolean operators:*/
  RWBoolean		operator==(const SCharVec&) Const;
  RWBoolean		operator!=(const SCharVec&) Const;
  
  /* Arithmetic operators:*/
  SCharVec&		operator++();
  SCharVec&		operator--();
  SCharVec&		operator+=(const SCharVec&);
  SCharVec&		operator+=(SChar);
  SCharVec&		operator-=(const SCharVec&);
  SCharVec&		operator-=(SChar);
  SCharVec&		operator*=(const SCharVec&);
  SCharVec&		operator*=(SChar);
  SCharVec&		operator/=(const SCharVec&);
  SCharVec&		operator/=(SChar);
  
  /* Friendly arithmetic operators:*/
  friend SCharVec	operator-(const SCharVec&);
#ifndef NO_UNARY_PLUS
  friend SCharVec	operator+(const SCharVec&);
#endif
  friend SCharVec	operator*(const SCharVec&,const SCharVec&);
  friend SCharVec	operator/(const SCharVec&,const SCharVec&);
  friend SCharVec	operator+(const SCharVec&,const SCharVec&);
  friend SCharVec	operator-(const SCharVec&,const SCharVec&);
  friend SCharVec	operator*(const SCharVec&,SChar);
  friend SCharVec	operator*(SChar,const SCharVec&);
  friend SCharVec	operator/(const SCharVec&,SChar);
  friend SCharVec	operator/(SChar,const SCharVec&);
  friend SCharVec	operator+(const SCharVec&,SChar);
  friend SCharVec	operator+(SChar,const SCharVec&);
  friend SCharVec	operator-(const SCharVec&,SChar);
  friend SCharVec	operator-(SChar,const SCharVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions:*/
  SCharVec		apply(mathFunTy) Const;
  friend SCharVec	abs(const SCharVec&);
  friend SCharVec	cumsum(const SCharVec&);
  friend SCharVec	delta(const SCharVec&);
  friend SChar		dot(const SCharVec&,const SCharVec&);
  friend int		max(const SCharVec&);
  friend int		min(const SCharVec&);
  friend SChar		prod(const SCharVec&);
  friend SCharVec	reverse(const SCharVec&);
  friend SChar		sum(const SCharVec&);
#endif
  
};

/* Other (related) declarations:*/
/* Note that these input and output >numeric< equivalents, not*/
/* characters.*/
ostream&		operator<<(ostream&, const SCharVec&);
istream&		operator>>(istream&, SCharVec&);

SCharVec    toChar(const IntVec&);

/******************* I N L I N E S **************************/

Inline void	
  SCharVec::boundsCheck(int i) Const{
  if(i<0 || i>npts) boundsErr(i);
}

Inline SChar&	
  SCharVec::operator[](int i) Const{
  boundsCheck(i); return begin[i*step];
}

inline SChar&	
  SCharVec::operator()(int i) Const {
#ifdef BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

inline SChar&
SCharVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline SCharVec
  SCharVec::slice(int start, unsigned n, int str) Const {
	return SCharVec(*this, start, n, str);
}

#ifndef NO_UNARY_PLUS
  Inline SCharVec	operator+(const SCharVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline SCharVec	operator*(SChar a, const SCharVec& b)	{return b*a;}
  inline SCharVec	operator+(SChar a, const SCharVec& b)	{return b+a;}
#endif

/********************  Pick inlines *****************************/
 
Inline
SCharPick::SCharPick(const SCharVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline SChar&
SCharPick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __SCHARVEC_H__ */
