#ifndef __IGEMATRIX_H__
#define __IGEMATRIX_H__

/*
 * Declarations for int general matricies.
 *
 * $Id: igemat.h,v 3.4.2.1 90/10/22 15:52:18 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * This class is derived from class IntVec.  Data is stored
 * in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "ivec.h"
class DGEMatrix;
class IGEMatrix;
class FGEMatrix;
class SCGEMatrix;

/*
 * The IGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class IGEPick {
private:
  const IGEMatrix*	M;
  const IntVec*		rowPick;
  const IntVec*		colPick;
  IGEPick(const IGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  IGEPick(const IGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		IGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const IGEMatrix&);
  void			operator=(const IGEPick&);
  void			operator=(int);

  int&			operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class IGEMatrix : private IntVec {
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void 			assertColRange(unsigned) Const;
  void 			assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const IntVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const IGEMatrix&) Const;
  void			assertProduct(const IntVec&) Const;
public:
  IGEMatrix();
  IGEMatrix(unsigned rows, unsigned cols);
  IGEMatrix(unsigned rows, unsigned cols, int initval);
  IGEMatrix(const int* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  IGEMatrix(const IntVec& v, unsigned, unsigned); /* Reference to v will be made*/
  IGEMatrix(const IGEMatrix& m);	   /* Reference to m will be made*/
  IGEMatrix(const IGEPick& p);

  operator		DGEMatrix() Const;	/* Convert to DGEMatrix*/

  int*			data() Const	{return IntVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  IGEMatrix&		reference(const IGEMatrix& m); /* Reference self to m*/
  IGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  IGEMatrix		copy() Const	{return deepCopy();} /* Synonym for deepCopy()*/
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			printOn(ostream& s) Const;
  void			readFrom(istream&);
  void			storeOn(ostream&) Const;
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&) Const;
  void			readFrom(RWFile*); /* Internal binary formatting*/
  void			storeOn(RWFile*) Const;
  unsigned		binaryStoreSize() Const;

  IntVec		operator[](unsigned j) Const;	/* Return a col as a slice*/
  IntVec		col(unsigned j) Const;		/* Return a col as a slice*/
  IntVec		row(unsigned i) Const;		/* Return a row as a slice*/
  IntVec		diagonal(int idiag=0) Const;	/* Return a diagonal as a slice*/
  inline int&		operator()(int i, int j) Const; /* Subscripting*/
  inline int&		sub(int, int) Const;		/* Assumes stride==1; use with care*/
  IGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  IGEMatrix		product(const IGEMatrix&) Const; /* Inner product*/
  IntVec		product(const IntVec&) Const;

/* Assignment operators --- self must be same size as m*/
  IGEMatrix&		operator=(const IGEMatrix& m);
  IGEMatrix&		operator=(const IGEPick& p);
  IGEMatrix&		operator=(int);
  IGEMatrix&		operator+=(const IGEMatrix& m);
  IGEMatrix&		operator+=(int);
  IGEMatrix&		operator-=(const IGEMatrix& m);
  IGEMatrix&		operator-=(int);
  IGEMatrix&		operator*=(const IGEMatrix& m);
  IGEMatrix&		operator*=(int);
  IGEMatrix&		operator/=(const IGEMatrix& m);
  IGEMatrix&		operator/=(int);

  /* Boolean operators:*/
  RWBoolean		operator==(const IGEMatrix&) Const;
  RWBoolean		operator!=(const IGEMatrix&) Const;

/* Increment/decrement operators*/
  IGEMatrix&		operator++();
  IGEMatrix&		operator--();

/* Friendly arithmetic operators.*/
  friend IGEMatrix	operator-(const IGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend IGEMatrix	operator+(const IGEMatrix&);	/* Unary plus*/
#endif
  friend IGEMatrix	operator*(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator/(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator+(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator-(const IGEMatrix&, const IGEMatrix&);
  friend IGEMatrix	operator*(const IGEMatrix&, int);
  friend IGEMatrix	operator*(int, const IGEMatrix&);
  friend IGEMatrix	operator/(const IGEMatrix&, int);
  friend IGEMatrix	operator/(int, const IGEMatrix&);
  friend IGEMatrix	operator+(const IGEMatrix&, int);
  friend IGEMatrix	operator+(int, const IGEMatrix&);
  friend IGEMatrix	operator-(const IGEMatrix&, int);
  friend IGEMatrix	operator-(int, const IGEMatrix&);

  friend IGEMatrix	abs(const IGEMatrix&);
  friend IGEMatrix	cumsum(const IGEMatrix&);
  friend IGEMatrix	delta(const IGEMatrix&);
  friend int		dot(const IGEMatrix&,const IGEMatrix&);
  friend int		prod(const IGEMatrix&);
  friend int		sum(const IGEMatrix&);
  friend SCGEMatrix	toChar(const IGEMatrix&);
};

/* Other (related) declarations:*/
ostream&		operator<<(ostream& s, const IGEMatrix& m);
istream& 		operator>>(istream& s, IGEMatrix& m);
IGEMatrix		transpose(const IGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned IGEMatrix::cols() Const { return ncols;}
Inline unsigned IGEMatrix::rows() Const { return nrows;}
Inline void IGEMatrix::deepenShallowCopy(){IntVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline IGEMatrix operator+(const IGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline IGEMatrix operator*(int d, const IGEMatrix& m){ return m*d; }
  inline IGEMatrix operator+(int d, const IGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline IntVec
IGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return IntVec::slice(j*nrows,nrows,1);
}

Inline IntVec
IGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return IntVec::slice(j*nrows,nrows,1);
}

Inline IntVec
IGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return IntVec::slice(i, ncols, nrows);
}

Inline IntVec
IGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return IntVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline int&
IGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return IntVec::operator()(j*nrows+i);
}

inline int&
IGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return IntVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/
 
Inline
IGEPick::IGEPick(const IGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline int&
IGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __IGEMATRIX_H__ */
