#ifndef __FLOATVEC_H__
#define __FLOATVEC_H__

/*
 * Declarations for Float Precision Vectors
 *
 * $Id: fvec.h,v 3.4.2.1 90/10/22 15:52:15 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * Defining the preprocessor directive BOUNDS_CHECK will
 * cause bounds checking on the subscripting operator.
 */

#include "ivec.h"

class FloatVec;
class DoubleVec;

/*
 * The FloatPick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class FloatPick {
private:
  const FloatVec*	V;
  const IntVec*	X;
  FloatPick(const FloatVec* v, const IntVec* x); /* Constructor is private*/
  FloatPick(const FloatPick& p) {V=p.V; X=p.X;}
  friend		FloatVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const FloatVec&);
  void			operator=(const FloatPick&);
  void			operator=(float);

  inline float&	operator()(int i) Const;
  unsigned		length() Const	{ return X->length(); }
};

class FloatVec {
  RWBlock*		block;
  float*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine;			/* For printing*/
  FloatVec(const FloatVec&, int, unsigned, int);	/* For slices*/
protected:
  void			boundsCheck(int) Const;
  void			boundsErr(int) Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int) Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  FloatVec();
  FloatVec(unsigned n);
  FloatVec(unsigned n, float val);
  FloatVec(unsigned n, float val, float by);
  FloatVec(const FloatVec& a);
  FloatVec(const FloatPick& p);
  FloatVec(const float* dat, unsigned n);  /* Copy of dat will be made*/
  ~FloatVec();

  operator		DoubleVec() Const;	/* Conversion to DoubleVec*/

  FloatVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  float*		data()   Const	{return begin;}
  unsigned		length() Const	{return npts;}
  int			stride() Const	{return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  FloatVec		copy() Const  {return deepCopy();}/* Synonym for deepCopy()*/
  FloatVec		deepCopy() Const;		/* copy with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  FloatVec&		reference(const FloatVec& v);	/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;
  
  /* Indexing:*/
  float&		operator[](int i) Const;	/* With bounds checking*/
  inline float&		operator()(int i) Const;	/* With optional bounds checking*/
  FloatPick		operator()(const IntVec& x) Const;
  inline float&		sub(int i) Const;		/* Assumes stride==1; use carefully*/
  
  /* Assignment:*/
  FloatVec&		operator=(const FloatVec& v); /* Must be same length as v*/
  FloatVec&		operator=(const FloatPick&);
  FloatVec&		operator=(float);

  /* Boolean operators:*/
  RWBoolean		operator==(const FloatVec&) Const;
  RWBoolean		operator!=(const FloatVec&) Const;
  
  /* Arithmetic operators:*/
  FloatVec&		operator++();
  FloatVec&		operator--();
  FloatVec&		operator+=(const FloatVec&);
  FloatVec&		operator+=(float);
  FloatVec&		operator-=(const FloatVec&);
  FloatVec&		operator-=(float);
  FloatVec&		operator*=(const FloatVec&);
  FloatVec&		operator*=(float);
  FloatVec&		operator/=(const FloatVec&);
  FloatVec&		operator/=(float);
  
  /* Friendly arithmetic operators:*/
  friend FloatVec	operator-(const FloatVec&);
#ifndef NO_UNARY_PLUS
  friend FloatVec	operator+(const FloatVec&);
#endif
  friend FloatVec	operator*(const FloatVec&,const FloatVec&);
  friend FloatVec	operator/(const FloatVec&,const FloatVec&);
  friend FloatVec	operator+(const FloatVec&,const FloatVec&);
  friend FloatVec	operator-(const FloatVec&,const FloatVec&);
  friend FloatVec	operator*(const FloatVec&,float);
  friend FloatVec	operator*(float,const FloatVec&);
  friend FloatVec	operator/(const FloatVec&,float);
  friend FloatVec	operator/(float,const FloatVec&);
  friend FloatVec	operator+(const FloatVec&,float);
  friend FloatVec	operator+(float,const FloatVec&);
  friend FloatVec	operator-(const FloatVec&,float);
  friend FloatVec	operator-(float,const FloatVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions:*/
  FloatVec		apply(mathFunTy) Const;
  friend FloatVec	abs(const FloatVec&);
  friend FloatVec	acos(const FloatVec&);
  friend FloatVec	asin(const FloatVec&);
  friend FloatVec	atan(const FloatVec&);
  friend FloatVec	atan2(const FloatVec&,const FloatVec&);
  friend FloatVec	ceil(const FloatVec&);
  friend FloatVec	cos(const FloatVec&);
  friend FloatVec	cosh(const FloatVec&);
  friend FloatVec	cumsum(const FloatVec&);
  friend FloatVec	delta(const FloatVec&);
  friend float		dot(const FloatVec&,const FloatVec&);
  friend FloatVec	exp(const FloatVec&); 
  friend FloatVec	floor(const FloatVec&);
  friend FloatVec	log(const FloatVec&);
  friend FloatVec	log10(const FloatVec&);
  friend int	  	max(const FloatVec&);
  friend int	  	min(const FloatVec&);
  friend float		mean(const FloatVec&);
  friend float		prod(const FloatVec&);
  friend FloatVec	pow(const FloatVec&,const FloatVec&);
  friend FloatVec	reverse(const FloatVec&);
  friend FloatVec	sin(const FloatVec&);
  friend FloatVec	sinh(const FloatVec&);
  friend FloatVec	sqrt(const FloatVec&);
  friend float		sum(const FloatVec&);
  friend FloatVec	tan(const FloatVec&);
  friend FloatVec	tanh(const FloatVec&);
  friend float		variance(const FloatVec&);
#endif
  
};

/* Other (related) declarations:*/
FloatVec		expandEven(const FloatVec&);
FloatVec		expandOdd(const FloatVec&);
FloatVec    toFloat(const DoubleVec&);
ostream&		operator<<(ostream&, const FloatVec&);
istream&		operator>>(istream&, FloatVec&);

/******************* I N L I N E S **************************/

Inline void
  FloatVec::boundsCheck(int i) Const{
	if(i<0 || i>npts) boundsErr(i);
  }

Inline float&
  FloatVec::operator[](int i) Const{
	boundsCheck(i); return begin[i*step];
  }

inline float&
  FloatVec::operator()(int i) Const{
#ifdef BOUNDS_CHECK    
	boundsCheck(i);
#endif
	return begin[i*step];
  }

inline float&
FloatVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline FloatVec
  FloatVec::slice(int start, unsigned n, int str) Const{
	return FloatVec(*this, start, n, str);
  }

#ifndef NO_UNARY_PLUS
  Inline FloatVec	operator+(const FloatVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline FloatVec	operator*(float a, const FloatVec& b)	{return b*a;}
  inline FloatVec	operator+(float a, const FloatVec& b)	{return b+a;}
#endif

#ifndef NO_VECTOR_MATHFUN
Inline FloatVec abs(const FloatVec& V)	{ return V.apply(::fabs); }
Inline FloatVec acos(const FloatVec& V){ return V.apply(::acos); }
Inline FloatVec asin(const FloatVec& V){ return V.apply(::asin); }
Inline FloatVec atan(const FloatVec& V){ return V.apply(::atan); }
Inline FloatVec ceil(const FloatVec& V){ return V.apply(::ceil); }
Inline FloatVec cos(const FloatVec& V)	{ return V.apply(::cos); }
Inline FloatVec cosh(const FloatVec& V){ return V.apply(::cosh); }
Inline FloatVec exp(const FloatVec& V)	{ return V.apply(::exp); }
Inline FloatVec floor(const FloatVec& V){ return V.apply(::floor); }
Inline FloatVec log(const FloatVec& V)	{ return V.apply(::log); }
Inline FloatVec log10(const FloatVec& V)	{ return V.apply(::log10); }
Inline FloatVec sin(const FloatVec& V)	{ return V.apply(::sin); }
Inline FloatVec sinh(const FloatVec& V){ return V.apply(::sinh); }
Inline FloatVec sqrt(const FloatVec& V){ return V.apply(::sqrt); }
Inline FloatVec tan(const FloatVec& V)	{ return V.apply(::tan); }
Inline FloatVec tanh(const FloatVec& V){ return V.apply(::tanh); }
Inline float    mean(const FloatVec& V){ return sum(V)/V.length(); }
#endif
 
/********************  Pick inlines *****************************/
 
Inline
FloatPick::FloatPick(const FloatVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline float&
FloatPick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __FLOATVEC_H__ */
