#ifndef __FGEMATRIX_H__
#define __FGEMATRIX_H__

/*
 * Declarations for float precision general matricies.
 *
 * $Id: fgemat.h,v 3.4.2.1 90/10/22 15:52:12 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * This class is derived from class FloatVec.  Data is stored
 * in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "fvec.h"

class DGEMatrix;
class FGEMatrix;
class IGEMatrix;

/*
 * The FGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class FGEPick {
private:
  const FGEMatrix*	M;
  const IntVec*	rowPick;
  const IntVec*	colPick;
  FGEPick(const FGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  FGEPick(const FGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		FGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const FGEMatrix&);
  void			operator=(const FGEPick&);
  void			operator=(float);

  float&		operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class FGEMatrix : private FloatVec {
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void 		assertColRange(unsigned) Const;
  void 		assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const FloatVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const FGEMatrix&) Const;
  void			assertProduct(const FloatVec&) Const;
public:
  FGEMatrix();
  FGEMatrix(unsigned rows, unsigned cols);
  FGEMatrix(unsigned rows, unsigned cols, float initval);
  FGEMatrix(const float* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  FGEMatrix(const FloatVec& v, unsigned, unsigned); /* Reference to v will be made*/
  FGEMatrix(const FGEMatrix& m);	   /* Reference to m will be made*/
  FGEMatrix(const FGEPick& p);

  operator  DGEMatrix() Const;    /*  conversion to DGEMatrix*/

  float*		data() Const	{return FloatVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  FGEMatrix&		reference(const FGEMatrix& m); /* Reference self to m*/
  FGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  FGEMatrix		copy() Const  {return deepCopy();} /* Synonym for deepCopy()*/
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			printOn(ostream& s) Const;
  void			readFrom(istream&);
  void			storeOn(ostream&) Const;
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&) Const;
  void			readFrom(RWFile*); /* Internal binary formatting*/
  void			storeOn(RWFile*) Const;
  unsigned		binaryStoreSize() Const;

  FloatVec		operator[](unsigned j) Const;	/* Return a col as a slice*/
  FloatVec		col(unsigned j) Const;		/* Return a col as a slice*/
  FloatVec		row(unsigned i) Const;		/* Return a row as a slice*/
  FloatVec		diagonal(int idiag=0) Const;	/* Return a diagonal as a slice*/
  inline float&		operator()(int i, int j) Const; /* Subscripting*/
  inline float&		sub(int, int) Const;		/* Assumes stride==1; use with care*/
  FGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  FGEMatrix		product(const FGEMatrix&) Const; /* Inner product*/
  FloatVec		product(const FloatVec&) Const;

/* Assignment operators --- self must be same size as m*/
  FGEMatrix&		operator=(const FGEMatrix& m);
  FGEMatrix&		operator=(const FGEPick& p);
  FGEMatrix&		operator=(float);
  FGEMatrix&		operator+=(const FGEMatrix& m);
  FGEMatrix&		operator+=(float);
  FGEMatrix&		operator-=(const FGEMatrix& m);
  FGEMatrix&		operator-=(float);
  FGEMatrix&		operator*=(const FGEMatrix& m);
  FGEMatrix&		operator*=(float);
  FGEMatrix&		operator/=(const FGEMatrix& m);
  FGEMatrix&		operator/=(float);

  /* Boolean operators:*/
  RWBoolean		operator==(const FGEMatrix&) Const;
  RWBoolean		operator!=(const FGEMatrix&) Const;

/* Increment/decrement operators*/
  FGEMatrix&		operator++();
  FGEMatrix&		operator--();

/* Friendly arithmetic operators.*/
  friend FGEMatrix	operator-(const FGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend FGEMatrix	operator+(const FGEMatrix&);	/* Unary plus*/
#endif
  friend FGEMatrix	operator*(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator/(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator+(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator-(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	operator*(const FGEMatrix&, float);
  friend FGEMatrix	operator*(float, const FGEMatrix&);
  friend FGEMatrix	operator/(const FGEMatrix&, float);
  friend FGEMatrix	operator/(float, const FGEMatrix&);
  friend FGEMatrix	operator+(const FGEMatrix&, float);
  friend FGEMatrix	operator+(float, const FGEMatrix&);
  friend FGEMatrix	operator-(const FGEMatrix&, float);
  friend FGEMatrix	operator-(float, const FGEMatrix&);

/* Friendly math functions:*/

  friend FGEMatrix	abs(const FGEMatrix&);
  friend FGEMatrix	acos(const FGEMatrix&);
  friend FGEMatrix	asin(const FGEMatrix&);
  friend FGEMatrix	atan(const FGEMatrix&);
  friend FGEMatrix	atan2(const FGEMatrix&,const FGEMatrix&);
  friend FGEMatrix	ceil(const FGEMatrix&);
  friend FGEMatrix	cos(const FGEMatrix&);
  friend FGEMatrix	cosh(const FGEMatrix&);
  friend FGEMatrix	cumsum(const FGEMatrix&);
  friend FGEMatrix	delta(const FGEMatrix&);
  friend float		dot(const FGEMatrix&, const FGEMatrix&);
  friend FGEMatrix	exp(const FGEMatrix&); 
  friend FGEMatrix	floor(const FGEMatrix&);
  friend FGEMatrix	log(const FGEMatrix&);
  friend FGEMatrix	log10(const FGEMatrix&);
  friend float		mean(const FGEMatrix&);
  friend float		prod(const FGEMatrix&);
  friend FGEMatrix	pow(const FGEMatrix&,const FGEMatrix&);
  friend FGEMatrix	sin(const FGEMatrix&);
  friend FGEMatrix	sinh(const FGEMatrix&);
  friend FGEMatrix	sqrt(const FGEMatrix&);
  friend float		sum(const FGEMatrix&);
  friend FGEMatrix	tan(const FGEMatrix&);
  friend FGEMatrix	tanh(const FGEMatrix&);
  friend float		variance(const FGEMatrix&);
  friend IGEMatrix	toInt(const FGEMatrix&);
};

/* Other (related) declarations:*/
istream& 		operator>>(istream& s, FGEMatrix& m);
ostream&		operator<<(ostream& s, const FGEMatrix& m);
FGEMatrix		transpose(const FGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned FGEMatrix::cols() Const { return ncols;}
Inline unsigned FGEMatrix::rows() Const { return nrows;}
Inline void FGEMatrix::deepenShallowCopy(){FloatVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline FGEMatrix operator+(const FGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline FGEMatrix operator*(float d, const FGEMatrix& m){ return m*d; }
  inline FGEMatrix operator+(float d, const FGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline FloatVec
FGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return FloatVec::slice(j*nrows,nrows,1);
}

Inline FloatVec
FGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return FloatVec::slice(j*nrows,nrows,1);
}

Inline FloatVec
FGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return FloatVec::slice(i, ncols, nrows);
}

Inline FloatVec
FGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return FloatVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline float&
FGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return FloatVec::operator()(j*nrows+i);
}

inline float&
FGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return FloatVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/

Inline
FGEPick::FGEPick(const FGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline float&
FGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __FGEMATRIX_H__ */
