#ifndef __DGEMATRIX_H__
#define __DGEMATRIX_H__

/*
 * Declarations for double precision general matricies.
 *
 * $Id: dgemat.h,v 3.4.2.1 90/10/22 15:52:07 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 * 
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 * 
 ***************************************************************************
 *
 */

/*
 * Class DGEMatrix is derived from class DoubleVec.  Data is stored
 * in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "dvec.h"

class FGEMatrix;
class DGEMatrix;
class IGEMatrix;

/*
 * The DGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class DGEPick {
private:
  const DGEMatrix*		M;
  const IntVec*		rowPick;
  const IntVec*		colPick;
  DGEPick(const DGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  DGEPick(const DGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		DGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const DGEMatrix&);
  void			operator=(const DGEPick&);
  void			operator=(double);

  double&		operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class DGEMatrix : private DoubleVec {
  friend class CGEMatrix;
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void 		assertColRange(unsigned) Const;
  void 		assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const DoubleVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const DGEMatrix&) Const;
  void			assertProduct(const DoubleVec&) Const;
public:
  DGEMatrix();
  DGEMatrix(unsigned rows, unsigned cols);
  DGEMatrix(unsigned rows, unsigned cols, double initval);
  DGEMatrix(const double* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  DGEMatrix(const DoubleVec& v, unsigned, unsigned); /* Reference to v will be made*/
  DGEMatrix(const DGEMatrix& m);	   /* Reference to m will be made*/
  DGEMatrix(const DGEPick& p);

  double*		data() Const	{return DoubleVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  DGEMatrix&		reference(const DGEMatrix& m); /* Reference self to m*/
  DGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  DGEMatrix		copy() Const	{return deepCopy();} /* Synonym for deepCopy()*/
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			printOn(ostream& s) Const;
  void			readFrom(istream&);
  void			storeOn(ostream&) Const;
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&) Const;
  void			readFrom(RWFile*); /* Internal binary formatting*/
  void			storeOn(RWFile*) Const;
  unsigned		binaryStoreSize() Const;

  DoubleVec		operator[](unsigned j) Const;	/* Return a col as a slice*/
  DoubleVec		col(unsigned j) Const;		/* Return a col as a slice*/
  DoubleVec		row(unsigned i) Const;		/* Return a row as a slice*/
  DoubleVec		diagonal(int idiag=0) Const;	/* Return a diagonal as a slice*/
  inline double&	operator()(int i, int j) Const; /* Subscripting*/
  inline double&	sub(int, int) Const;		/* Assumes stride==1; use with care*/
  DGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  DGEMatrix		product(const DGEMatrix&) Const; /* Inner product*/
  DoubleVec		product(const DoubleVec&) Const;

/* Assignment operators --- self must be same size as m*/
  DGEMatrix&		operator=(const DGEMatrix& m);
  DGEMatrix&		operator=(const DGEPick& p);
  DGEMatrix&		operator=(double);
  DGEMatrix&		operator+=(const DGEMatrix& m);
  DGEMatrix&		operator+=(double);
  DGEMatrix&		operator-=(const DGEMatrix& m);
  DGEMatrix&		operator-=(double);
  DGEMatrix&		operator*=(const DGEMatrix& m);
  DGEMatrix&		operator*=(double);
  DGEMatrix&		operator/=(const DGEMatrix& m);
  DGEMatrix&		operator/=(double);

  /* Boolean operators:*/
  RWBoolean		operator==(const DGEMatrix&) Const;
  RWBoolean		operator!=(const DGEMatrix&) Const;

/* Increment/decrement operators*/
  DGEMatrix&		operator++();
  DGEMatrix&		operator--();

/* Friendly arithmetic operators.*/
  friend DGEMatrix	operator-(const DGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend DGEMatrix	operator+(const DGEMatrix&);	/* Unary plus*/
#endif
  friend DGEMatrix	operator*(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator/(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator+(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator-(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator*(const DGEMatrix&, double);
  friend DGEMatrix	operator*(double, const DGEMatrix&);
  friend DGEMatrix	operator/(const DGEMatrix&, double);
  friend DGEMatrix	operator/(double, const DGEMatrix&);
  friend DGEMatrix	operator+(const DGEMatrix&, double);
  friend DGEMatrix	operator+(double, const DGEMatrix&);
  friend DGEMatrix	operator-(const DGEMatrix&, double);
  friend DGEMatrix	operator-(double, const DGEMatrix&);

  friend DGEMatrix	abs(const DGEMatrix&);
  friend DGEMatrix	acos(const DGEMatrix&);
  friend DGEMatrix	asin(const DGEMatrix&);
  friend DGEMatrix	atan(const DGEMatrix&);
  friend DGEMatrix	atan2(const DGEMatrix&,const DGEMatrix&);
  friend DGEMatrix	ceil(const DGEMatrix&);
  friend DGEMatrix	cos(const DGEMatrix&);
  friend DGEMatrix	cosh(const DGEMatrix&);
  friend DGEMatrix	cumsum(const DGEMatrix&);
  friend DGEMatrix	delta(const DGEMatrix&);
  friend double	dot(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	exp(const DGEMatrix&); 
  friend DGEMatrix	floor(const DGEMatrix&);
  friend DGEMatrix	log(const DGEMatrix&);
  friend DGEMatrix	log10(const DGEMatrix&);
  friend double	mean(const DGEMatrix&);
  friend double	prod(const DGEMatrix&);
  friend DGEMatrix	pow(const DGEMatrix&,const DGEMatrix&);
  friend DGEMatrix	sin(const DGEMatrix&);
  friend DGEMatrix	sinh(const DGEMatrix&);
  friend double	sum(const DGEMatrix&);
  friend DGEMatrix	sqrt(const DGEMatrix&);
  friend DGEMatrix	tan(const DGEMatrix&);
  friend DGEMatrix	tanh(const DGEMatrix&);
  friend double	variance(const DGEMatrix&);
  friend FGEMatrix	toFloat(const DGEMatrix&);
  friend IGEMatrix	toInt(const DGEMatrix&);
};

/* Other (related) declarations:*/
ostream&		operator<<(ostream& s, const DGEMatrix& m);
istream& 		operator>>(istream& s, DGEMatrix& m);
DGEMatrix		transpose(const DGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned DGEMatrix::cols() Const { return ncols;}
Inline unsigned DGEMatrix::rows() Const { return nrows;}
Inline void DGEMatrix::deepenShallowCopy(){DoubleVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline DGEMatrix operator+(const DGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline DGEMatrix operator*(double d, const DGEMatrix& m){ return m*d; }
  inline DGEMatrix operator+(double d, const DGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline DoubleVec
DGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return DoubleVec::slice(j*nrows,nrows,1);
}

Inline DoubleVec
DGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return DoubleVec::slice(j*nrows,nrows,1);
}

Inline DoubleVec
DGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return DoubleVec::slice(i, ncols, nrows);
}

Inline DoubleVec
DGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return DoubleVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline double&
DGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DoubleVec::operator()(j*nrows+i);
}

inline double&
DGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DoubleVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/
 
Inline
DGEPick::DGEPick(const DGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline double&
DGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __DGEMATRIX_H__ */
