#ifndef __CGEMATRIX_H__
#define __CGEMATRIX_H__

/*
 * Declarations for Double Precision Complex GEneral matrices.
 *
 * $Id: cgemat.h,v 3.4.2.1 90/10/22 15:51:59 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989	This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * Matrix of DComplex.  This class is derived from 
 * class DComplexVec.  Data is stored in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "cvec.h"
class DGEMatrix;
class CGEMatrix;

/*
 * The CGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class CGEPick {
private:
  const CGEMatrix*		M;
  const IntVec*		rowPick;
  const IntVec*		colPick;
  CGEPick(const CGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  CGEPick(const CGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		CGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const CGEMatrix&);
  void			operator=(const CGEPick&);
  void			operator=(DComplex);

  DComplex&		operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class CGEMatrix : private DComplexVec {
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const DComplexVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const CGEMatrix&) Const;
  void			assertProduct(const DComplexVec&) Const;
public:
  CGEMatrix();
  CGEMatrix(unsigned rows, unsigned cols);
  CGEMatrix(unsigned rows, unsigned cols, DComplex initval);
  CGEMatrix(const DGEMatrix& re); /* conversion from DGEMatrix*/
  CGEMatrix(const DGEMatrix& re, const DGEMatrix& im); /* conversion from DGEMatrix*/
  CGEMatrix(const DComplex* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  CGEMatrix(const DComplexVec& v, unsigned, unsigned); /* Reference to v will be made*/
  CGEMatrix(const CGEMatrix& m);	   /* Reference to m will be made*/
  CGEMatrix(const CGEPick& p);

  DComplex*		data() Const	{return DComplexVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  unsigned		binaryStoreSize() Const;
  CGEMatrix		copy() Const {return deepCopy();} /* Synonym for deepCopy()*/
  CGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			printOn(ostream& s) Const;
  void			readFrom(RWFile*);
  void			readFrom(fileDescTy&);
  void			readFrom(istream&);
  CGEMatrix&		reference(const CGEMatrix& m);/* Reference self to m*/
  void			resize(unsigned,unsigned);
  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;

  DComplexVec		operator[](unsigned j) Const;   /* Return a col as a slice*/
  DComplexVec		col(unsigned j) Const;          /* Return a col as a slice*/
  DComplexVec		row(unsigned i) Const;          /* Return a row as a slice*/
  DComplexVec		diagonal(int idiag=0) Const;    /* Return a diagonal as a slice*/
  inline DComplex&	operator()(int i, int j) Const; /* Subscripting*/
  inline DComplex&	sub(int, int) Const;		/* Assumes stride==1; use with care*/
  CGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  CGEMatrix		product(const CGEMatrix&) Const;/* Inner product*/
  DComplexVec		product(const DComplexVec&) Const;

/* Assignment   operators --- self must be same size as m*/
  CGEMatrix&		operator=(const CGEMatrix& m);
  CGEMatrix&		operator=(const CGEPick& p);
  CGEMatrix&		operator=(DComplex);
  CGEMatrix&		operator+=(const CGEMatrix& m);
  CGEMatrix&		operator+=(DComplex);
  CGEMatrix&		operator-=(const CGEMatrix& m);
  CGEMatrix&		operator-=(DComplex);
  CGEMatrix&		operator*=(const CGEMatrix& m);
  CGEMatrix&		operator*=(DComplex);
  CGEMatrix&		operator/=(const CGEMatrix& m);
  CGEMatrix&		operator/=(DComplex);

  /* Boolean operators:*/
  RWBoolean		operator==(const CGEMatrix&) Const;
  RWBoolean		operator!=(const CGEMatrix&) Const;

/* Increment/decrement operators*/
/*CGEMatrix&		operator++();*/
/*CGEMatrix&		operator--();*/

/* Friendly arithmetic operators.*/
  friend CGEMatrix	operator-(const CGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend CGEMatrix	operator+(const CGEMatrix&);	/* Unary plus*/
#endif
  friend CGEMatrix	operator*(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator/(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator+(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator-(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	operator*(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator*(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator/(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator/(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator+(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator+(DComplex, const CGEMatrix&);
  friend CGEMatrix	operator-(const CGEMatrix&, DComplex);
  friend CGEMatrix	operator-(DComplex, const CGEMatrix&);

/* friendly arithmetic functions*/
  friend DGEMatrix	abs(const CGEMatrix&);
  friend CGEMatrix	cos(const CGEMatrix&);
  friend CGEMatrix	cosh(const CGEMatrix&);
  friend CGEMatrix	cumsum(const CGEMatrix&);
  friend CGEMatrix	delta(const CGEMatrix&);
  friend DComplex	dot(const CGEMatrix&, const CGEMatrix&);
  friend CGEMatrix	exp(const CGEMatrix&); 
  friend CGEMatrix	log(const CGEMatrix&);
  friend DComplex	mean(const CGEMatrix&);
  friend DComplex	prod(const CGEMatrix&);
  friend CGEMatrix	pow(const CGEMatrix&,const CGEMatrix&);
  friend CGEMatrix	sin(const CGEMatrix&);
  friend CGEMatrix	sinh(const CGEMatrix&);
  friend CGEMatrix	sqrt(const CGEMatrix&);
  friend DComplex	sum(const CGEMatrix&);
  friend double	variance(const CGEMatrix&);

/* Complex specific functions:*/
  friend DGEMatrix	arg(const CGEMatrix& V);
  friend CGEMatrix	conj(const CGEMatrix& V);
  friend DGEMatrix	imag(const CGEMatrix& V);
  friend DGEMatrix	norm(const CGEMatrix& V);
  friend DGEMatrix	real(const CGEMatrix& V);
};

/* Other (related) declarations:*/
istream& 		operator>>(istream& s, CGEMatrix& m);
ostream&		operator<<(ostream& s, const CGEMatrix& m);
CGEMatrix		transpose(const CGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned CGEMatrix::cols() Const { return ncols;}
Inline unsigned CGEMatrix::rows() Const { return nrows;}
Inline void CGEMatrix::deepenShallowCopy(){DComplexVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline CGEMatrix operator+(const CGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline CGEMatrix operator*(DComplex d, const CGEMatrix& m){ return m*d; }
  inline CGEMatrix operator+(DComplex d, const CGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline DComplexVec
CGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return DComplexVec::slice(j*nrows,nrows,1);
}

Inline DComplexVec
CGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return DComplexVec::slice(j*nrows,nrows,1);
}

Inline DComplexVec
CGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return DComplexVec::slice(i, ncols, nrows);
}

Inline DComplexVec
CGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return DComplexVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline DComplex&
CGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DComplexVec::operator()(j*nrows+i);
}

inline DComplex&
CGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DComplexVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/
 
Inline
CGEPick::CGEPick(const CGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline DComplex&
CGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __CGEMATRIX_H__ */
