/*
 * 
 * $Copyright
 * Copyright 1993, 1994, 1995  Intel Corporation
 * INTEL CONFIDENTIAL
 * The technical data and computer software contained herein are subject
 * to the copyright notices; trademarks; and use and disclosure
 * restrictions identified in the file located in /etc/copyright on
 * this system.
 * Copyright$
 * 
 */
 
/*****************************************************************************
 *              Copyright (c) 1990 San Diego Supercomputer Center.
 *              All rights reserved.  The SDSC software License Agreement
 *              specifies the terms and conditions for redistribution.
 *
 * File:        jrec.c
 *
 * Abstract:    This file contains jrec, which reads in the accounting
 *              data and produce job record report
 *****************************************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <ctype.h>
#include <memory.h>
#include <pwd.h>
#include <errno.h>
#include "mac_def.h"
#include "filename.h"
#include "nx/nxacct.h"
#include "conf.h"
#include "mac.h"
#include "actable.h"

#define INLENGTH 127
#define NTYPES 11
#define DESC_LEN 32
#define NAME_LEN 16
#define LOGIN_LEN 9
#define DATE_LEN 9
#define TIME_LEN 9

extern void filltm();
extern time_t mktime();
void addrec();
void updrec();
void chkrec();
void delrec();
void endrec();
void write_down();

struct procrec 
{
    int acid;
    int uid;
    int part_id;
    int part_size;
    int node_type;
    int pgid;
    char qname[32];
    int qtype;
    time_t starttime;
    float pre_cpu_hours;
    float cpu_hours;
    float cpu_rate;
    float pre_under_hours;
    float under_hours;
    float under_rate;
    float pre_idle_hours;
    float idle_hours;
    float idle_rate;
    long submit_time;
    long reqst_secs;
} *procrecs;

/* 1 is occupied and 0 is empty */
int *procflgs;
int nrecs;
int nproc, tmp_nproc;

struct down_info {
	time_t	start;
	time_t	end;
	char	type[6];
	char	comment[64];
};

struct ulink {
      uid_t uid;
      struct ulink *next;
};
struct ulink *ulink_root=NULL;
int ucount=0;

FILE *fd;
int jflg, wflg, skip, v1_1=1;
char starthms[TIME_LEN],endhms[TIME_LEN];
struct tm starttm;
struct tm endtm;
char startdate[DATE_LEN],enddate[DATE_LEN];
time_t starttime,endtime;
time_t lasttime, boottime;
int acct_name_len=7;
char *acct_name_buf;
int verbose=0;
struct actable *actab;
int naccts, agids[MAX_AGIDS];

/*===========================================================================*
 * Function:    jrec
 *
 * Abstract:    This program reads log files which contain accounting
 * 		data generated by MACD and output the job records in 2 forms:
 *     		with -j option, the output is more human-readable
 *    		without -j option, the output is in arep-input-format
 * 		that is more efficient for arep-process to
 * 		produce usage summary report
 *
 * Arguments:   argc
 *		argv
 * execution line:
 *        	jrec -v -j -h site_name -n total#nodes 
 *        	-f input_fname -s start_date -e end_date > file_name
 *
 * Return value:
 *              None
 *
 * Notes:	Target machine:      INTEL's front-end PC 
 *
 *===========================================================================*/

main(argc,argv)
int argc;
char *argv[];
{
    int c, nfiles; 
    int nflg=0,hflg=0,sflg=0,eflg=0,errflg=0,fflg=0,dflg=0,pflg=0,preflg=1;
    int i, siflg=0, si_cnt=0, si;
    int sched_boot=0, unsched_boot=0;
    long dtstamp=0;        /*   0: up status
                    		>0: last down time-stamp */
    long downtime=0;       /* total down time */
    long sched_downtime=0; /* scheduled down time */
    long curtime;
    char cubeln[INLENGTH+1], keyword[16];
    int dn_days, dn_hrs, dn_mins, dn_secs, remain;
    int sched_dn_days, sched_dn_hrs, sched_dn_mins, sched_dn_secs;
    char timeln[INLENGTH+1];
    char **fname, *ptr, *site, *infptr, *logpath;
    char dumstr[128], *down_file;
    int chktime=0;
    struct down_info last_down;
    FILE *down_fd;
    struct macsconf *conf;
    extern char *optarg, *malloc(), *calloc();
    extern int optind, opterr;
    extern char **lognames();
    extern void bcopy();

    /*
     * parse command line arguement
     */
    if (argc < 2 ) errflg++;
    else while((c=getopt(argc,argv,"vjd:h:n:f:s:e:p:")) != -1) {
        switch(c) {
            case 'v':
                verbose = 1;
                (void) fprintf (stderr, 
                "Jrec starts\nJrec: Parsing arguments ...\n");
                break;
            case 'j':
                jflg++;
                break;
            case 'd':
                dflg++;
                down_file = (char *)malloc (strlen(optarg)+1);
                (void) strcpy(down_file,optarg);
                break;
            case 'h':
                hflg++;
                site = (char *)malloc (strlen(optarg)+1);
		(void) strcpy(site,optarg);
                break;
            case 'n':
                nflg++;
                nrecs = atoi(optarg);
		if (nrecs < 100) nrecs = 100;
                break;
            case 's':
                sflg++;
                ptr=strchr(optarg,'@');
                if (ptr) {
                    (void) strncpy(starthms,ptr+1,TIME_LEN-1);
                    starthms[TIME_LEN-1] = '\0';
                    *ptr = '\0';
                }
                else (void) strncpy(starthms,"00:00:00",TIME_LEN);
                (void) strncpy(startdate,optarg,DATE_LEN-1);
                startdate[DATE_LEN-1] = '\0';
                break;
            case 'e':
                eflg++;
                ptr=strchr(optarg,'@');
                if (ptr) {
                    (void) strncpy(endhms,ptr+1,TIME_LEN-1);
                    endhms[TIME_LEN-1] = '\0';
                    *ptr = '\0';
                }
                else (void) strncpy(endhms,"23:59:59",TIME_LEN);
                (void) strncpy(enddate,optarg,DATE_LEN-1);
                enddate[DATE_LEN-1] = '\0';
                break;
            case 'f':
                fflg++;
                infptr = optarg;
                if (infptr == NULL) errflg++;
                break;
            case 'p':
                pflg++;
                logpath = optarg;
                if (logpath == NULL) errflg++;
                break;
            default:
                errflg++;
        }
    }
    if (!fflg && (!sflg || !eflg)) {
        (void) fprintf (stderr,
            "The start_date and end_date must be specified\n");
        errflg++;
    }
    if (dflg && (down_fd = fopen (down_file, "a+")) == NULL) {
	(void) fprintf (stderr,
	    "Failed open file %s for down-time report\n", down_file);
	errflg++;
    }
    if (errflg) {
        (void) fprintf(stderr,"Usage:  \n");
        (void) fprintf(stderr, "jrec [-vj]\n");
        (void) fprintf (stderr, "\t[-d down-time-report_filename]\n"); 
        (void) fprintf (stderr, "\t[-h site_name]\n"); 
        (void) fprintf (stderr, "\t[-n total#nodes]\n"); 
        (void) fprintf (stderr, "\t[-s start_date]\n"); 
        (void) fprintf (stderr, "\t[-e end_date]\n"); 
        (void) fprintf (stderr, "\t[-f file_name]\n"); 
        (void) fprintf (stderr, "\t[-p log_dir_path]\n"); 
        (void) fprintf(stderr, "date format: MM/DD/YY[@hh:mm:ss]\n");
        exit(1);
    }

    /*
     * initialization
     */
    if (verbose) (void) fprintf (stderr, "Jrec:  Intialize ...\n");
    if (getuid() != 0) {
	if ((naccts = getagidfromuid(agids, getuid())) <= 0) {
	    (void) fprintf (stderr, "Permission denied\n");
	    exit(1);
        }
	if (verbose) {
	    (void) fprintf (stderr, "Will process accounts ");
	    for (i=0; i<naccts; i++) (void) fprintf (stderr, "%d,",agids[i]);
	    (void) fprintf (stderr, "\n");
	}
    }
    else if (verbose) (void) fprintf (stderr, "Will process all accounts\n");

    (void) tzset();
    (void) memset (&starttm, '\0', sizeof (struct tm));
    (void) memset (&endtm, '\0', sizeof (struct tm));
    (void) memset (&last_down, '\0', sizeof (struct down_info));

#ifdef DEBUG
    (void) fprintf(stderr, "hflg = %d  sflg = %d  eflg = %d\n",
        hflg,sflg,eflg);
    if (sflg) {
        (void) fprintf(stderr, "startdate = %s\n",startdate);
        (void) fprintf(stderr, "starthms = %s\n",starthms);
    }
    if (eflg) {
        (void) fprintf(stderr, "enddate = %s\n",enddate);
        (void) fprintf(stderr, "endhms = %s\n",endhms);
    }
    (void) fflush (stderr);
#endif

    /*
     * get or set time strings and variables
     */
    (void) tzset();
    if (sflg) {
        (void) filltm(startdate,starthms,&starttm);
        starttime = mktime(&starttm);
        boottime = lasttime = starttime;
    }
    if (eflg) {
        time_t time_now = time(0);
        (void) filltm(enddate,endhms,&endtm);
        endtime = mktime(&endtm);
        if (endtime > time_now) {
            struct tm *tm_now = localtime (&time_now);
            endtime = time_now;
            (void) sprintf (enddate, "%2.2d/%2.2d/%2.2d",
                tm_now->tm_mon+1, tm_now->tm_mday, tm_now->tm_year);
                        (void) sprintf (endhms, "%2.2d:%2.2d:%2.2d",
                                tm_now->tm_hour, tm_now->tm_min, tm_now->tm_sec);
        }
    }

#ifdef DEBUG
    (void) fprintf (stderr, 
        "starting tm struct and resulting time_t conversion\n");
    dumptm(&starttm);
    (void) fprintf(stderr, 
        "ending tm struct and resulting time_t conversion\n");
    dumptm(&endtm);
    (void) fflush (stderr);
#endif

    /*
     * get configuration info
     */
    if ((conf = readconf()) == NULL) {
        switch (errno) {
            case ENOENT: (void) fprintf(stderr,
                             "Fail in opening %s. JREC abort\n",
                             MACS_CONF_FILE);
                         break;
            case ENOMEM: (void) fprintf(stderr,
                             "Fail in malloc() in readconf().  JREC abort\n");
                         break;
            case EINVAL: (void) fprintf(stderr,
                             "Invalid enforcement parameter in %s.  JREC abort\n",
                             MACS_CONF_FILE);
                         break;
            default: (void) fprintf(stderr,
                        "Unknown error returned from readconf().  JREC abort\n");
        }
        exit(-1);
    }
    actab = get_actab();
    if (actab == NULL) {
        (void) fprintf (stderr,
            "Fail in reading %s file\n", MACS_ACTABLE_FILE);
        exit(-1);
    }
    if (actab->n_actlevel > 0)
        for (i=0, acct_name_len=0; i<actab->n_actlevel; i++)
            acct_name_len += actab->actlevel[i];
    acct_name_buf = (char *)malloc (acct_name_len +1);
    if (!nflg) {
        if (actab->n_bins > 0) nrecs = actab->bins[actab->n_bins-1];
        else {
            (void) fprintf (stderr,
                "Total number of nodes Unknown, use -n option\n");
            exit(1);
        }
	if (nrecs < 100) nrecs = 100;
    }

#ifdef DEBUG
    (void) fprintf (stderr, "nrecs=%d\n", nrecs);
    (void) fflush (stderr);
#endif

    /*
     * allocate process table and initialize process entry flags
     */
    procrecs = (struct procrec *) calloc (nrecs, sizeof(struct procrec));
    if (procrecs == NULL) {
        (void) fprintf (stderr, "Fail allocate procrec table\n");
        exit(1);
    }
    procflgs = (int *) calloc (nrecs, sizeof(int));
    if (procflgs == NULL) {
        (void) fprintf (stderr, "Fail allocate procflgs array\n");
        exit(1);
    }
    
    /*
     * open input file specified in command line
     */
    if (fflg) {
        fd = fopen (infptr, "r");
        if (fd <= (FILE *)NULL) {
            (void) fprintf (stderr, "Fail open %s\n", infptr);
                exit(1);
        }
        chktime = 1;
	preflg = 1;
        goto fproc;
    }

    /*
     * get all the file names needed
     */
    if (!pflg && (logpath = malloc (strlen(MACD_LOG_PATH) + 2)) != NULL)
            (void) sprintf (logpath, "%s/", MACD_LOG_PATH);
    fname = lognames (&starttime, endtime, &nfiles, logpath);

    {
    /* 
     * starttime may have been changed 
     */
    struct tm *tm_ptr = localtime(&starttime);
    (void) sprintf(startdate,"%2.2d/%2.2d/%2.2d",
                tm_ptr->tm_mon + 1,tm_ptr->tm_mday,tm_ptr->tm_year);
        (void) sprintf(starthms,"%2.2d:%2.2d:%2.2d",
                tm_ptr->tm_hour,tm_ptr->tm_min,tm_ptr->tm_sec);
    (void) bcopy (tm_ptr, &starttm, sizeof(struct tm));
    }
    boottime = lasttime = starttime;

#ifdef DEBUG
    (void) fprintf (stderr, "starttime=%d startdate=%s starthms=%s\n",
        starttime, startdate, starthms);
    (void) fflush (stderr);
#endif

    /* 
     * something wrong, nfiles indicate the problem
     */
    if (fname == NULL)
        switch (nfiles) {
            case -1:
                (void) fprintf (stderr, "Invalid input parameters:\n");
                (void) fprintf (stderr, 
                "    starttime (%d) >= endtime (%d), jrec abort\n",
                starttime, endtime);
                exit(-1);
            case -2:    
                (void) fprintf (stderr, 
                "Error in opening pipe to /bin/ls -1, jrec abort\n");
                exit(-1);
            case -3:    
                (void) fprintf (stderr, 
                "No valid logfile names found, jrec abort\n");
                exit(-1);
            case -4:    
                (void) fprintf (stderr, "No data for the time period\n");
                exit(0);
            case -5:    
                (void) fprintf (stderr, 
			"Invalid argument logpath, empty or NULL\n");
                exit(0);
            default:
                (void) fprintf (stderr, 
                "Cannot obtain log file names, jrec abort\n");
                exit(-1);
        }
    if (!nfiles) {
        (void) fprintf (stderr, "0 log files obtained, jrec exit\n");
        exit(0);
    }

#ifdef DEBUG
    (void) fprintf (stderr, "nfiles=%d\n");
    for (i=0; i<nfiles; i++)
    (void) fprintf (stderr, "filename[%d]=%s\n",i,fname[i]);
    (void) fflush (stderr);
#endif

    /*
     * output header
     */
    if (hflg) {
	time_t now = time(0);
        if (verbose) (void) fprintf (stderr, "Jrec:  Output header ...\n");
	(void) fprintf(stdout, "%s Intel PARAGON Job Usage Report\n", site);
	(void) fprintf(stdout, "- produced on %s\n\n", ctime(&now));
        
        if (sflg && eflg) (void) fprintf(stdout,"%s %s to %s %s\n\n",
            startdate,starthms,enddate,endhms);

        (void) strncpy (acct_name_buf, "account name", acct_name_len);
        for (i=12; i<acct_name_len; i++) acct_name_buf[i] = ' ';
        acct_name_buf[acct_name_len] = '\0';
    
        if (jflg) {
            (void) fprintf(stdout,
                "%s  login   partition# size type   cpu-hours ", acct_name_buf);
            (void) fprintf(stdout, " rate idle cpu-hrs rate  under-used  rate");
#ifndef INTELv1r1
            (void) fprintf(stdout, " total charge    job submitted      job started   ");
            (void) fprintf(stdout, "     job ended     st reqst cpuhrs queue name\n");
#else
            (void) fprintf(stdout, " total charge    job started    ");
            (void) fprintf(stdout, "     job ended     st queue name\n");
#endif
            for (i=0; i<acct_name_len; i++) acct_name_buf[i] = '-';
            (void) fprintf(stdout,
                "%s -------- ---------- ---- ---- ------------", acct_name_buf);
            (void) fprintf(stdout, " ---- ------------ ---- ------------ ----");
#ifndef INTELv1r1
            (void) fprintf(stdout, " ------------- ----------------- -----------------");
            (void) fprintf(stdout, " ----------------- -- ------------ ----------\n");
#else
            (void) fprintf(stdout, " ------------- -----------------");
            (void) fprintf(stdout, " ----------------- -- ----------\n");
#endif
        } else {
            (void) fprintf(stdout, "%s  login   type size   cpu-hours", 
                acct_name_buf);
            (void) fprintf(stdout, "  idle cpu-hrs under-used ");
            (void) fprintf(stdout, "  total charge\n");
            for (i=0; i<acct_name_len; i++) acct_name_buf[i] = '-';
            (void) fprintf(stdout, "%s -------- ---- ---- ------------",
                acct_name_buf);
            (void) fprintf(stdout, " ------------ ------------");
            (void) fprintf(stdout, " -------------\n");
        }
    }
    if (dflg) {
	time_t now = time(0);
        if (verbose) (void) fprintf (stderr, "jrec:  output header in %s\n",
	    down_file);
	if (hflg) {
	    (void) fprintf(down_fd, "%s Intel PARAGON Down-Time Report\n", site);
	} else {
	    (void) fprintf(down_fd, "Intel PARAGON Down-Time Report\n");
	}
	(void) fprintf(down_fd, "- produced on %s\n\n", ctime(&now));
            
        if (sflg && eflg) (void) fprintf(down_fd,"%s %s to %s %s\n\n",
            startdate,starthms,enddate,endhms);
    
        (void) fprintf(down_fd,
            "       down              up           duration   type      comments\n");
        (void) fprintf(down_fd,
            "----------------- ----------------- ------------ ----- -------------------------\n");
    }

    /*
     * process log files one-by-one
     */
    for (preflg=1, i=0; i<nfiles; i++, chktime=0) {
        /*
         * need check records against the time for
         * the first file and the last file
         */
        if (i==0) chktime=1;
        if (i==nfiles-1) chktime = 1;

        fd = fopen (fname[i], "r");
        if (fd <= (FILE *)NULL) {
            (void) fprintf (stderr, "Fail open %s\n", fname[i]);
                continue;
        }
#ifdef DEBUG
        (void) fprintf (stderr, "fname=%s, fd=%d\n", fname[i], fd);
        (void) fflush (stderr);
#endif
        if (verbose) (void) fprintf (stderr,
            "Jrec:  Process file %s ...\n", fname[i]);

        /*
	 * process one line at a time 
	 */
	skip = 0;
fproc:  while (1) {
	    if (!skip && fgets (timeln, INLENGTH, fd) == NULL) break;
	    else if (skip) {
		(void) strncpy (timeln, cubeln, INLENGTH);
	    }
	    skip = 0;
            wflg = 1;
            /*
	     * skip GOTSMD and MACDSTAT|MACDSYNC after MACDUP 
	     */
	    (void) strncpy (keyword, timeln, 6);
	    keyword[6] = '\0';
	    if (strcmp (keyword, "MACDUP") == 0) {
		if (fgets (timeln, INLENGTH, fd) <= NULL) break;
		(void) strncpy (keyword, timeln, 6);
		keyword[6] = '\0';
		if (strcmp (keyword, "GOTSMD") == 0) {
		    if (fgets (timeln, INLENGTH, fd) <= NULL) break;
		    (void) strncpy (keyword, timeln, 8);
		    keyword[8] = '\0';
		    if ((strcmp (keyword, "MACDSTAT") == 0) ||
			(strcmp (keyword, "MACDSYNC") == 0)) {
			if (fgets (timeln, INLENGTH, fd) <= NULL) break;
		    }
		}
		else {
		    (void) strncpy (keyword, timeln, 8);
		    keyword[8] = '\0';
		    if ((strcmp (keyword, "MACDSTAT") == 0) ||
			(strcmp (keyword, "MACDSYNC") == 0)) {
			if (fgets (timeln, INLENGTH, fd) <= NULL) break;
		    }
		}
	    }

	    /*
	     * OLDJOB & RUBOFJOB entries are used for update, but
	     * when OLDJOB entries are in the very begining of the
	     * very first log file, then they are used for
	     * debitting the usage of previous period
	     */
	    if (preflg) {
	        if (timeln[0] != 'R' && timeln[0] != 'O') preflg = 0;
		else if (timeln[0] == 'O') timeln[0] = 'o';
	    }

            /*
             * get time_t
             */
            if (sscanf (timeln, "%s : (%d) %s %s %s %s %s",
                dumstr, &curtime, dumstr, dumstr, dumstr, dumstr, dumstr) < 2) {
#ifdef DEBUG
                (void) fprintf (stderr, "curtime=%d %s\n", curtime, timeln);
                (void) fflush (stderr);
#endif
                continue;
            }
            if (curtime > time(0) + 86400) {
                /* time in entry is more than 24 hours (86400 seconds) 
		   in the future -- skip this entry */
                (void) fprintf (stderr, "Invalid date -- entry ignored: %s", timeln);
                (void) fflush (stderr);
		continue;
            }
            if (starttime == 0) {
                /* !sflg && 1st data line */
                lasttime = boottime = starttime = curtime;
            }

            /*
             * check time
             */
            if (chktime) {
                if (sflg && curtime < starttime) wflg = 0;
                if (eflg && curtime > endtime) goto fend;
            }

	    /*
	     * shouldn't palsed more than sync_interval minutes,
	     * must be down or hang or other problems, mark it down.
	    if (dtstamp == 0 && 
		(curtime - lasttime - 60) > (conf->sync_interval * 60))
		dtstamp = lasttime;
	     */

            /*
             * if the previous status is not down
             * and this is not OLDJOB line, then
             * update last up-time
             */
	    if (dtstamp == 0) { 
		if (timeln[0] == 'H' || timeln[0] == 'G')
		    continue;
                if (timeln[0] != 'O' && timeln[0] != 'P') lasttime = curtime;
	    }

#ifdef DEBUG
            if (nproc) {
                (void) fprintf (stderr, "case %c\n", timeln[0]);
                (void) fflush (stderr);
            }
#endif

            /*
             * deal with different lines
             */
            switch (timeln[0]) {
                case 'I':    /* scheduled interrupt */
                    {
                    int si; 
		    char si_type[6], si_comment[64];
                    (void) sscanf (timeln,
                        "INTRRUPT : %s %s %s %s %s %s - Scheduled=%d Type=%s %[^\n]",
                        dumstr, dumstr, dumstr, dumstr, dumstr, dumstr, &si, si_type, si_comment);

                    if (dflg && (si == siflg || si != SI_START) 
			&& last_down.start) {
			/* update type and comment */
			if (strlen(si_type) > 0) {
			    (void) strncpy (last_down.type, si_type, 5);
                            last_down.type[5] = '\0';
			}
			if (strlen(si_comment) > 0) {
                            (void) strncpy (last_down.comment, si_comment, 24);
                            last_down.comment[24] = '\0';
			}
#ifdef DEBUG
                        (void) fprintf (stderr,
                            "2 scheduled interrupt=%d in row, last time is %s\n",
                            si, dumstr);
                        (void) fflush (stderr);
#endif
                    }
		    if (dflg && ((siflg && si != 1) || (!siflg && si == 1))
			&& last_down.start) {
			/* si status change.  if start!=0 in last_down
			then there is a down time record ready for output. */
			if (last_down.end <= last_down.start || 
				last_down.end > lasttime) 
			    last_down.end = lasttime;
			if (wflg && last_down.end > starttime) {
			    if (last_down.start < starttime)
				last_down.start = starttime;
			    (void) write_down (down_fd, &last_down);
			}
			last_down.start = last_down.end = 0;
		    }

                    if (si == SI_START) {
                        if (!siflg) {
                            if (wflg) si_cnt++;
                            siflg = 1;
                        }
			if (dflg) {
			    last_down.type[0] = '\0';
			    last_down.comment[0] = '\0';
			    (void) strncpy (last_down.type, si_type, 5);
                            last_down.type[5] = '\0';
                            (void) strncpy (last_down.comment, si_comment, 24);
                            last_down.comment[24] = '\0';
			}
                    } else if (si == SI_END && siflg)  siflg = 0;

                    break;
                    }

 /*                case 'M':    MACDSYNC
                    if (dtstamp) {
                        boottime = curtime;
			if (dflg) {
			   if (last_down.start != dtstamp) 
			      last_down.start = dtstamp;
			   last_down.end = lasttime;
			}
                        if (wflg) { 
			   if (dtstamp < starttime) dtstamp = starttime;
                           downtime += lasttime - dtstamp;
                           if (siflg)
                              sched_downtime += lasttime - dtstamp;
                        }
#ifdef DEBUG
                        (void) fprintf (stderr,
                            "case %c: unsched_boot = %d, downtime = %d\n",
                            timeln[0], unsched_boot, downtime);
                        (void) fflush (stderr);
#endif
                        dtstamp = 0;
                    }
                    break; */

                case 'P':    /* PARABOOT */
		    if (curtime == starttime) break;
                    if (dtstamp == 0) {
                        dtstamp = lasttime;
			if (dflg) {
                            if (wflg && last_down.start 
				&& last_down.end > starttime) {
			        if (last_down.start < starttime)
				    last_down.start = starttime;
				(void) write_down (down_fd, &last_down);
			    }
			    last_down.start = lasttime;
			}
                        if (nproc) endrec (NULL, 0);
                    }
                    lasttime = curtime;
		    if (dflg) last_down.end = lasttime;
                    if (wflg) {
                        if (siflg) sched_boot++;
                        else unsched_boot++;
                    }
                    boottime = curtime;
                    if (wflg) { 
		       if (dtstamp < starttime) dtstamp = starttime;
                       downtime += lasttime - dtstamp;
                       if (siflg) sched_downtime += lasttime - dtstamp;
                    }
#ifdef DEBUG
                    (void) fprintf (stderr,
                        "case %c: unsched_boot = %d, downtime = %d lasttime = %d dtstamp = %d\n",
                        timeln[0], unsched_boot, downtime, lasttime, dtstamp);
                        (void) fflush (stderr);
#endif
                    dtstamp = 0;
                    break;

                case 'S':    /* STARTJOB */
                case 'B':    /* BEGINAPP */
                case 'E':    /* ENDAPP */
                case 'J':    /* JOBDONE */
                    if (dtstamp) {
			if (dflg) {
			   if (last_down.start != dtstamp) 
			      last_down.start = dtstamp;
			   last_down.end = lasttime;
			}
                        if (wflg) { 
			   if (dtstamp < starttime) dtstamp = starttime;
                           downtime += curtime - dtstamp;
                           if (siflg) sched_downtime += curtime - dtstamp;
                        }
#ifdef DEBUG
                        (void) fprintf (stderr,
                            "case %c: unsched_boot = %d, downtime = %d\n",
                            timeln[0], unsched_boot, downtime);
                        (void) fflush (stderr);
#endif
                        dtstamp = 0;
                    }
                    lasttime = curtime;
                    if (fgets (cubeln, INLENGTH, fd) == NULL) {
                        (void) fprintf(stderr,
                            "Jrec: Error getting input line\n");
                        continue;
                    }

                    if (timeln[0] == 'S') addrec (cubeln);
                    else if (timeln[0] == 'E' || timeln[0] == 'B') 
			updrec (timeln, cubeln);
                    else if (timeln[0] == 'J') endrec (cubeln, 0);
                    else if (timeln[0] == 'F') addrec ((char *)-1);
		    else if ((lasttime - boottime) < 5);
		    else endrec((char *)-1, 0);
                    break;

                case 'o':	/* usage of previous period */
                    if (fgets (cubeln, INLENGTH, fd) == NULL) {
                        (void) fprintf(stderr,
                            "Jrec: Error getting input line\n");
                        continue;
                    }
                    chkrec (cubeln, -1);
                    break;

                case 'O':	/* OLDJOB */
                    if (fgets (cubeln, INLENGTH, fd) == NULL) {
                        (void) fprintf(stderr,
                            "Jrec: Error getting input line\n");
                        continue;
                    }
                    chkrec (cubeln, 1);
                    break;

                case 'R':    /* RUBOFJOB */
                    if (fgets (cubeln, INLENGTH, fd) == NULL) {
                        (void) fprintf(stderr,
                            "Jrec: Error getting input line\n");
                        continue;
                    }
                    delrec (cubeln);
                    break;
                    
                default:    /* skip */
#ifdef DEBUG
                    if (verbose) {
                        (void) fprintf (stderr,
                        "jrec:  skipped line: %s\n", 
                            timeln);
                        (void) fflush (stderr);
                    }
#endif
                    break;
            }    /* switch */
        }    /* while */
        (void) fclose (fd);
        if (fflg) goto fend;
    }    /* for */

fend:   
    /*
     * end all the existing jobs
     */
    if (eflg) {
        if (dtstamp && endtime > dtstamp)  {
	    if (dtstamp < starttime) dtstamp = starttime;
            downtime += endtime - dtstamp;
            if (wflg && siflg) sched_downtime += endtime - dtstamp;
	    if (dflg) {
		if (last_down.start != dtstamp) last_down.start = dtstamp;
		last_down.end = endtime;
	    }
        }
    }
    if (nproc) endrec (NULL, 1);
    if (dflg && wflg && last_down.start && last_down.end > starttime) {
	if (last_down.start < starttime) last_down.start = starttime;
	(void) write_down (down_fd, &last_down);
    }

    /*
     * compute and output the time information
     */
    if (verbose) (void) fprintf (stderr, "Jrec:  Compute down time ...\n");
    if (!sflg) {
        struct tm *tm_ptr = localtime (&starttime);
        (void) sprintf(startdate,"%2.2d/%2.2d/%2.2d",
            tm_ptr->tm_mon + 1,tm_ptr->tm_mday,tm_ptr->tm_year);
        (void) sprintf(starthms,"%2.2d:%2.2d:%2.2d",
            tm_ptr->tm_hour,tm_ptr->tm_min,tm_ptr->tm_sec);
    }
    if (!eflg) {
        struct tm *tm_ptr = localtime (&curtime);
        endtime = curtime;
        (void) sprintf(enddate,"%2.2d/%2.2d/%2.2d",
            tm_ptr->tm_mon + 1,tm_ptr->tm_mday,tm_ptr->tm_year);
        (void) sprintf(endhms,"%2.2d:%2.2d:%2.2d",
            tm_ptr->tm_hour,tm_ptr->tm_min,tm_ptr->tm_sec);
    }
    if (jflg || dflg) {
        dn_days = downtime / (24 * 60 * 60);
        remain = downtime - (24 * 60 * 60) * dn_days;
        dn_hrs = remain / 3600;
        remain = remain - dn_hrs * 3600;
        dn_mins = remain / 60;
        dn_secs = remain - dn_mins * 60;
        if (jflg) (void) fprintf (stdout, "%s %s to %s %s - down ",
            startdate, starthms, enddate, endhms); 
        if (dflg) (void) fprintf (down_fd, "\n%s %s to %s %s - down ",
            startdate, starthms, enddate, endhms); 
        if (dn_days) {
	    if (jflg) (void) fprintf (stdout, " %d days", dn_days); 
	    if (dflg) (void) fprintf (down_fd, " %d days", dn_days); 
	}
        if (dn_hrs) {
	    if (jflg) (void) fprintf (stdout, " %d hours", dn_hrs); 
	    if (dflg) (void) fprintf (down_fd, " %d hours", dn_hrs); 
	}
        if (dn_mins) {
	    if (jflg) (void) fprintf (stdout, " %d minutes", dn_mins); 
	    if (dflg) (void) fprintf (down_fd, " %d minutes", dn_mins); 
	}
        if (jflg) (void) fprintf (stdout, " %d seconds\n", dn_secs); 
        if (dflg) (void) fprintf (down_fd, " %d seconds\n", dn_secs); 
        sched_dn_days = sched_downtime / (24 * 60 * 60);
        remain = sched_downtime - (24 * 60 * 60) * sched_dn_days;
        sched_dn_hrs = remain / 3600;
        remain = remain - sched_dn_hrs * 3600;
        sched_dn_mins = remain / 60;
        sched_dn_secs = remain - sched_dn_mins * 60;
        if (jflg) (void) fprintf (stdout, "scheduled down ");
        if (dflg) (void) fprintf (down_fd, "scheduled down ");
        if (sched_dn_days) {
            if (jflg) (void) fprintf (stdout, " %d days", sched_dn_days); 
            if (dflg) (void) fprintf (down_fd, " %d days", sched_dn_days); 
	}
        if (sched_dn_hrs) {
            if (jflg) (void) fprintf (stdout, " %d hours", sched_dn_hrs); 
            if (dflg) (void) fprintf (down_fd, " %d hours", sched_dn_hrs); 
	}
        if (sched_dn_mins) {
            if (jflg) (void) fprintf (stdout, " %d minutes", sched_dn_mins); 
            if (dflg) (void) fprintf (down_fd, " %d minutes", sched_dn_mins); 
	}
        if (jflg) (void) fprintf (stdout, " %d seconds\n", sched_dn_secs); 
        if (dflg) (void) fprintf (down_fd, " %d seconds\n", sched_dn_secs); 
        if (jflg) (void) fprintf (stdout,
            "%d scheduled interrupts, rebooted %d times\n",
            si_cnt, sched_boot);
        if (dflg) (void) fprintf (down_fd,
            "%d scheduled interrupts, rebooted %d times\n",
            si_cnt, sched_boot);
        if (jflg) (void) fprintf (stdout,
            "%d unscheduled reboots\n", unsched_boot);
        if (dflg) (void) fprintf (down_fd, 
	    "%d unscheduled reboots\n%d users\n", unsched_boot, ucount);
    }
    if (!jflg) {
        (void) fprintf (stdout, 
            "(%010d) %s %s to (%010d) %s %s - down %d seconds, scheduled down %d seconds, ",
            starttime,startdate,starthms, endtime, enddate, 
            endhms, downtime, sched_downtime); 
        (void) fprintf (stdout,
            "scheduled %d, reboot %d; unscheduled reboot %d.  Users %d\n",
            si_cnt, sched_boot, unsched_boot, ucount);
    }

    (void) fclose (stdout);
    if (dflg) (void) fclose (down_fd);
    if (verbose) (void) fprintf (stderr, "Jrec:  --- end jrec ---\n");
}

/*===========================================================================*
 * Function:    addrec
 *
 * Abstract:    get job information from the input line and add a job entry 
 *		in the process table
 *
 * Arguments:   cubeln - input line
 *
 * Return value:
 *              None
 *
 * Notes:
 *===========================================================================*/
void addrec (cubeln)
char *cubeln;
{
    int i, j, k, l;
    struct tm *tmp_tm;
    struct ulink *ptr, *nptr;
    float cpu_charge;
    char date[9], hms[9], user_name[16];
    struct nxacct *acct_ent;
    struct passwd *user_ent;

    /*
     * Any room in the process table for add?
     */
    for (i=0; i<nrecs && procflgs[i]; i++);
    if (i>=nrecs) {
    (void) fprintf (stderr, "nrecs=%d i=%d\n", nrecs, i);
    (void) fprintf (stderr,
        "Addrec: Internal job table overflow.  Abort.\n");
        exit(-1);
    }
    (void) memset (&(procrecs[i]), '\0', sizeof (struct procrec));

    if (!cubeln) {
        (void) fprintf (stderr, "Addrec: Invalid argument value, cubeln=NULL\n");
        return;
    }
    /*
     * parse and read in the job information from the input line 
     */
    else if (sscanf(cubeln, " Acct=%d User=%d Part=%d Size=%d Type=%d Queue=%s\n",
        &(procrecs[i].acid), &(procrecs[i].uid),
        &(procrecs[i].part_id), &(procrecs[i].part_size), 
        &(procrecs[i].node_type), procrecs[i].qname) != 6) {
        (void) fprintf (stderr, "Addrec: Error parsing input line\n%s",
            cubeln);
        return;
    }
    if (fgets (cubeln, INLENGTH, fd) == NULL) {
        if (!v1_1) (void) fprintf(stderr, "Addrec: Error getting input line\n");
    }
    else if (sscanf(cubeln, " Subm=%d Reqst=%d", 
	&(procrecs[i].submit_time), &(procrecs[i].reqst_secs)) != 2) {
        if (!v1_1) (void) fprintf (stderr, "Addrec:  Error parsing input line\n%s",
                cubeln);
	procrecs[i].submit_time = procrecs[i].reqst_secs = 0;
	v1_1 = skip = 1;
    }
    else v1_1 = 0;

    if (getuid() != 0) {
	for (j=0; j<naccts; j++) if (procrecs[i].acid == agids[j]) break;
        if (j >= naccts) return;
    }

    /*
     * if new user, remember it and increment user count 
     */
    if (ulink_root == NULL) {
      ulink_root = (struct ulink *) calloc (1, sizeof (struct ulink));
      ulink_root->uid = procrecs[i].uid;
      ucount = 1;
    } else {
        for (nptr=ulink_root; nptr!=NULL; ptr=nptr, nptr=nptr->next)
          if (nptr->uid == procrecs[i].uid) break;
        if (nptr==NULL) {
          ptr->next = (struct ulink *) calloc (1, sizeof (struct ulink));
          ptr->next->uid = procrecs[i].uid;
          ucount++;
	}
    }

    /*
     * If a duplicate entry exist, if cpu_hours is zero,
     * delete the previous one -- must due to BEGINAPP was 
     * logged before STARTJOB; if cpu_hours is positive,
     * then assume the previous job ended.
     */
    tmp_nproc = nproc;
    if (nproc) for (j=0,l=0; j<nrecs && l<tmp_nproc; j++) {
        if (i==j) continue;
        if (procflgs[j] == 0) continue;
	l++;
        if (procrecs[j].acid != procrecs[i].acid) continue;
        if (procrecs[j].uid != procrecs[i].uid) continue;
        if (procrecs[j].part_id != procrecs[i].part_id) continue;
 
        /*
         * a match is found, but cpu_hours <= 0.0, discard the old one
         */
	if ((procrecs[j].cpu_hours - procrecs[j].pre_cpu_hours) <= 0.0){
	    procflgs[j] = 0;
	    nproc--;
	    break;
	}

        /*
         * a match is found, but cpu_hours >= 0.0, write out.
         */
	if (!wflg) {
	    procflgs[j] = 0;
	    nproc--;
	    break;
	}
	if (strlen(procrecs[j].qname) &&
		strcmp (procrecs[j].qname, "UNKNOWN") != 0) { 
            for (k=0; k<actab->n_dedicateq; k++)
            if (strcmp (procrecs[j].qname, actab->dedicateq[k]) == 0) {
                procrecs[j].qtype = DEDICATE;
                break;
            }
            if (procrecs[j].qtype == 0) procrecs[j].qtype = BATCH;
        }
        if ((acct_ent = nx_getaid (procrecs[j].acid)) == NULL)
            (void) strncpy (acct_name_buf, "Unknown",8);
        else (void) strncpy (acct_name_buf,
            acct_ent->acct_name, acct_name_len);
        acct_name_buf[acct_name_len] = '\0';
        for (k=strlen(acct_name_buf); k<acct_name_len; k++)
            acct_name_buf[k] = ' ';
        if ((user_ent = getpwuid (procrecs[j].uid)) == NULL)
            (void) strncpy (user_name, "Unknown",8);
        else {
            (void) strncpy (user_name, user_ent->pw_name, 8);
            user_name[8] = '\0';
        }
	procrecs[j].cpu_hours -= procrecs[j].pre_cpu_hours;
	if (procrecs[j].cpu_hours < 0.0) procrecs[j].cpu_hours = 0.0;
	procrecs[j].under_hours -= procrecs[j].pre_under_hours;
	if (procrecs[j].under_hours < 0.0) procrecs[j].under_hours = 0.0;
	procrecs[j].idle_hours -= procrecs[j].pre_idle_hours;
	if (procrecs[j].idle_hours < 0.0) procrecs[j].idle_hours = 0.0;
	cpu_charge = procrecs[j].cpu_hours * procrecs[j].cpu_rate 
		+ procrecs[j].under_hours * procrecs[j].under_rate 
		+ procrecs[j].idle_hours * procrecs[j].idle_rate;
        if (jflg) {
            long subm_t=0;
            float reqst_t=0.0;
            (void) fprintf (stdout,"%s %8.8s", acct_name_buf,user_name);
            (void) fprintf(stdout," %10d", procrecs[j].part_id);
            (void) fprintf(stdout," %4d", procrecs[j].part_size);
            (void) fprintf(stdout," %4d", procrecs[j].qtype);
            (void) fprintf(stdout," %12.3f", procrecs[j].cpu_hours);
            (void) fprintf(stdout," %4.2f", procrecs[j].cpu_rate);
            (void) fprintf(stdout," %12.3f", procrecs[j].idle_hours);
            (void) fprintf(stdout," %4.2f", procrecs[j].idle_rate);
            (void) fprintf(stdout," %12.3f", procrecs[j].under_hours);
            (void) fprintf(stdout," %4.2f", procrecs[j].under_rate);
            (void) fprintf(stdout," %13.3f", cpu_charge);
            subm_t = procrecs[i].submit_time;
            if (subm_t <= 0 && cubeln > NULL)
                subm_t = procrecs[j].submit_time;
            if (subm_t > 0) {
                tmp_tm = localtime (&subm_t);
            (void) fprintf(stdout,
                " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                tmp_tm->tm_mon+1, tmp_tm->tm_mday,
                tmp_tm->tm_year, tmp_tm->tm_hour,
                tmp_tm->tm_min, tmp_tm->tm_sec);
            }
            else if (procrecs[i].qtype) 
		(void) fprintf(stdout, " Unknown  Unknown ");
            else (void) fprintf(stdout, " N/A      N/A     ");
            if (procrecs[j].starttime) {
                tmp_tm = localtime (&(procrecs[j].starttime));
                (void) fprintf(stdout,
                " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                tmp_tm->tm_mon+1, tmp_tm->tm_mday,
                tmp_tm->tm_year, tmp_tm->tm_hour,
                tmp_tm->tm_min, tmp_tm->tm_sec);
            } else (void) fprintf(stdout, " Unknown  Unknown ");
            (void) fprintf(stdout, " Unknown  Unknown ");
	    (void) fprintf(stdout, "  1");	/* completed */
#ifndef INTELv1r1
            reqst_t = procrecs[i].reqst_secs / 3600.;
            if (reqst_t <= 0.0) reqst_t = procrecs[j].reqst_secs / 3600.;
            if (reqst_t > 0.0) (void) fprintf(stdout, " %12.3f", reqst_t);
            else if (procrecs[i].qtype) (void) fprintf(stdout, " Unknown     ");
            else (void) fprintf(stdout, " N/A         ");
#endif
            (void) fprintf(stdout," %s\n", procrecs[j].qname);
        } else (void) fprintf (stdout,
            "%s %8s %1d %4d %12.3f %12.3f %12.3f %13.3f\n",
            acct_name_buf,user_name,
            procrecs[j].qtype,
            procrecs[j].part_size,
            procrecs[j].cpu_hours, 
            procrecs[j].idle_hours, 
            procrecs[j].under_hours, 
            cpu_charge);
 
        procflgs[j] = 0;
        nproc--;
	break;
    }

    /*
     * set the queue type and start time, cpu usages for the job
     */
    procrecs[i].qtype = 0;
    if (strlen(procrecs[i].qname) &&
	strcmp (procrecs[i].qname, "UNKNOWN") != 0) { 
        for (j=0; j<actab->n_dedicateq; j++)
            if (strcmp (procrecs[i].qname, actab->dedicateq[j]) == 0) {
                procrecs[i].qtype = DEDICATE;
                break;
            }
        if (procrecs[i].qtype == 0) procrecs[i].qtype = BATCH;
    }
    procrecs[i].starttime = lasttime;

    /*
     * set non-empty flag for the new entry and increment process count
     */
    procflgs[i] = 1;
    nproc++;
#ifdef DEBUG
    (void) fprintf (stderr, "addrec: procflgs=(");
    for (j=0; j<nrecs; j++) if(procflgs[j]) (void) fprintf (stderr,"%d,",j);
    (void) fprintf (stderr, ")\n");
    (void) fprintf (stderr, "\naddrec:  add record %d\n", i);
    (void) fprintf (stderr,"acid= %d\n",procrecs[i].acid);
    (void) fprintf (stderr,"uid = %d\n",procrecs[i].uid);
    (void) fprintf (stderr,"part_id= %d\n",procrecs[i].part_id);
    (void) fprintf (stderr,"part_size = %d\n",procrecs[i].part_size);
    (void) fprintf (stderr,"node_type = %d\n",procrecs[i].node_type);
    (void) fprintf (stderr,"qname = %s\n",procrecs[i].qname);
    (void) fprintf (stderr,"starttime = %d\n",procrecs[i].starttime);
    (void) fprintf (stderr,"pre_cpu_hours = %f\n",procrecs[i].pre_cpu_hours);
    (void) fprintf (stderr,"cpu_hours = %f\n",procrecs[i].cpu_hours);
    (void) fprintf (stderr,"pre_under_hours = %f\n",procrecs[i].pre_under_hours);
    (void) fprintf (stderr,"under_hours = %f\n",procrecs[i].under_hours);
    (void) fprintf (stderr,"pre_idle_hours = %f\n",procrecs[i].pre_idle_hours);
    (void) fprintf (stderr,"idle_hours = %f\n",procrecs[i].idle_hours);
    (void) fflush (stderr);
#endif
}

/*===========================================================================*
 * Function:    updrec
 *
 * Abstract:    This function is called for every ENDAPP.  It looks for
 *		a matching JOB entry in the job records.  If one is found,
 *		increment cpu time of the job, otherwise, add an entry to it 
 *
 * Arguments:   cubeln - input line
 *
 * Return value:
 *              None
 *
 * Notes:
 *===========================================================================*/
void updrec (timeln, cubeln)
char *timeln, *cubeln;
{
    long cpu_seconds;
    time_t app_start_time=0;
    int i, j, k, pgid, part_size, acid, uid, part_id, type;
    float cpu_hours, cpu_rate=0.0;
    char qname[32];
    struct tm *tmp_tm;
    struct ulink *ptr, *nptr;

    /*
     * read in and parse the job information
     */
    if (sscanf(cubeln,
        " Acct=%d User=%d Part=%d Pgid=%d Size=%d Time=%d Rate=%f",
        &acid, &uid, &part_id, &pgid, &part_size, 
        &cpu_seconds, &cpu_rate) < 6) {
        (void) fprintf (stderr,
            "Updrec:  Error parsing input line\n%s",
            cubeln);
        return;
    }
    cpu_hours = (double)cpu_seconds / 3600.;

    if (getuid() != 0) {
	for (j=0; j<naccts; j++) if (acid == agids[j]) break;
        if (j >= naccts) return;
    }

    /*
     * if new user, remember it and increment user count 
     */
    if (ulink_root == NULL) {
      ulink_root = (struct ulink *) calloc (1, sizeof (struct ulink));
      ulink_root->uid = uid;
      ucount = 1;
    } else {
        for (nptr=ulink_root; nptr!=NULL; ptr=nptr, nptr=nptr->next)
          if (nptr->uid == uid) break;
        if (nptr==NULL) {
          ptr->next = (struct ulink *) calloc (1, sizeof (struct ulink));
          ptr->next->uid = uid;
          ucount++;
        }
    }

    /*
     * compare to entries in process table one-by-one
     */
    tmp_nproc = nproc;
    if (nproc) for (i=0, k=0; i<nrecs && k<tmp_nproc; i++) {
        if (procflgs[i] == 0) continue;
	k++;
        if (acid != procrecs[i].acid) continue;
        if (lasttime < procrecs[i].starttime) continue;
        if (uid != procrecs[i].uid) continue;
        if (part_id != procrecs[i].part_id) continue;
	if (procrecs[i].qtype == INTERACTIVE && procrecs[i].pgid != pgid)
	    continue;

        /*
	 * a match is found
	 */
        procrecs[i].cpu_hours += cpu_hours;
	if (procrecs[i].part_size < part_size) 
	    procrecs[i].part_size = part_size;
	if (timeln[0] != 'E' || procrecs[i].qtype != INTERACTIVE) return;
	if ((cpu_hours -= procrecs[i].pre_cpu_hours) <= 0.0) 
	    cpu_hours = 0.0;
	app_start_time = procrecs[i].starttime;
        procflgs[i] = 0;
        nproc--;
    }

    if (timeln[0] == 'E') {
        if (wflg) {    /* in reporting time interval */
	    struct nxacct *acct_ent;
	    struct passwd *user_ent;
	    extern struct passwd *getpwuid();
	    struct tm *tmp_tm;
	    char acct_name[NAME_LEN], user_name[NAME_LEN];

            if ((acct_ent = nx_getaid (acid)) == NULL)
                (void) strncpy (acct_name_buf, "Unknown",8);
            else (void) strncpy (acct_name_buf,
                acct_ent->acct_name, acct_name_len);
            acct_name_buf[acct_name_len] = '\0';
            for (j=strlen(acct_name_buf); j<acct_name_len; j++) 
                acct_name_buf[j] = ' ';
            if ((user_ent = getpwuid (uid)) == NULL)
                (void) strncpy (user_name, "Unknown",8);
            else {
		(void) strncpy (user_name, user_ent->pw_name, 8);
                user_name[8] = '\0';
	    }
            if (jflg) {
                (void) fprintf (stdout,"%s %8.8s",
                    acct_name_buf,user_name);
                (void) fprintf(stdout," %10d", part_id);
                (void) fprintf(stdout," %4d", part_size);
                (void) fprintf(stdout," %4d", 0);
                (void) fprintf(stdout," %12.3f", cpu_hours);
                (void) fprintf(stdout," %4.2f", cpu_rate);
                (void) fprintf(stdout," %12.3f", 0.0);
                (void) fprintf(stdout," %4.2f", 0.0);
                (void) fprintf(stdout," %12.3f", 0.0);
                (void) fprintf(stdout," %4.2f", 0.0);
                (void) fprintf(stdout," %13.3f", cpu_hours * cpu_rate);
#ifndef INTELv1r1
		(void) fprintf(stdout," N/A      N/A     ");
#endif
                if (app_start_time) {
		    tmp_tm = localtime (&app_start_time);
                    (void) fprintf(stdout,
                    " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                    tmp_tm->tm_mon+1, tmp_tm->tm_mday, 
                    tmp_tm->tm_year, tmp_tm->tm_hour, 
                    tmp_tm->tm_min, tmp_tm->tm_sec);
		} else (void) fprintf(stdout, " Unknown  Unknown ");
                tmp_tm = localtime (&lasttime);
                (void) fprintf(stdout,
                    " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                    tmp_tm->tm_mon+1, tmp_tm->tm_mday, 
                    tmp_tm->tm_year, tmp_tm->tm_hour, 
                    tmp_tm->tm_min, tmp_tm->tm_sec);
		(void) fprintf(stdout, "  1");	/* completed */
#ifndef INTELv1r1
		(void) fprintf(stdout," N/A         ");
#endif
                (void) fprintf(stdout," %s\n", "UNKNOWN");
	    }
            else (void) fprintf (stdout, 
                "%s %8s %1d %4d %12.3f %12.3f %12.3f %13.3f\n",
                acct_name_buf,user_name, INTERACTIVE, part_size,
                cpu_hours, 0.0, 0.0, cpu_hours * cpu_rate);
        }
	return;
    }
        
    /*
     * no match BEGINAPP, add a new entry 
     */
    for (i=0; i<nrecs && procflgs[i]; i++);
    if (i>=nrecs) {
    (void) fprintf (stderr, "nrecs=%d i=%d\n", nrecs, i);
    (void) fprintf (stderr,
        "Updrec: Internal job table overflow.  Abort.\n");
        exit(-1);
    }
    (void) memset (&(procrecs[i]), '\0', sizeof (struct procrec));

    procrecs[i].acid = acid;
    procrecs[i].uid = uid;
    procrecs[i].part_id = part_id;
    procrecs[i].part_size = part_size;
    procrecs[i].pgid = pgid;
    (void) strncpy (procrecs[i].qname, "UNKNOWN", 8);
    procrecs[i].qtype = INTERACTIVE;
    procrecs[i].cpu_hours = cpu_hours;
    procrecs[i].cpu_rate = cpu_rate;
    if (timeln[0] == 'B') procrecs[i].starttime = lasttime;
    else procrecs[i].starttime = 0;

    procflgs[i] = 1;
    nproc++;
#ifdef DEBUG
    (void) fprintf (stderr, "updrec: procflgs=(");
    for (j=0; j<nrecs; j++) if(procflgs[j]) (void) fprintf (stderr,"%d,",j);
    (void) fprintf (stderr, ")\n");
    (void) fprintf (stderr, "\nupdrec:  add record %d\n", i);
    (void) fprintf (stderr,"acid= %d\n",procrecs[i].acid);
    (void) fprintf (stderr,"uid = %d\n",procrecs[i].uid);
    (void) fprintf (stderr,"part_id= %d\n",procrecs[i].part_id);
    (void) fprintf (stderr,"pgid= %d\n",procrecs[i].pgid);
    (void) fprintf (stderr,"part_size = %d\n",procrecs[i].part_size);
    (void) fprintf (stderr,"qname = %s\n",procrecs[i].qname);
    (void) fprintf (stderr,"starttime = %d\n",procrecs[i].starttime);
    (void) fflush (stderr);
#endif

    return;
}

/*===========================================================================*
 * Function:    endrec
 *
 * Abstract:    This function log the job information when it's ended
 * 		if cubeln==NULL then lasttime is long integer time in seconds.
 * 		This happens when the machine is down, all jobs are ended,
 * 		or the end of the reporting period has been reached.
 * 		All records should be written out using the down-time as the
 * 		end time.
 * 		if cubeln!=NULL then look for a matching getcube info in
 * 		the job records.  If one is found, write out the record, and
 * 		set record flag to 0;  If none is found, this record has no
 * 		getcube entry and will be threw away to avoid double accounting
 *
 * Arguments:   cubeln - input line
 *		lastcall- 0/1
 *
 * Return value:
 *              None
 *
 * Notes:
 *===========================================================================*/

void endrec (cubeln, lastcall)
char *cubeln;
int lastcall;
{
    int i, j, k, size, acid, uid, part_id, type, qtype=0;
    char date[9], hms[9], bootdate[9], boothms[9], qname[32],
        acct_name[NAME_LEN], user_name[NAME_LEN];
    long cpu_seconds=0, under_seconds=0, idle_seconds=0; 
    long reqst_secs=0, submit_time=0;
    float cpu_hours=0.0, cpu_rate=0.0, 
        under_hours=0.0, under_rate=0.0, 
        idle_hours=0.0, idle_rate=0.0;
    struct tm *tmp_tm;
    struct ulink *ptr, *nptr;
    struct nxacct *acct_ent;
    struct passwd *user_ent;
    extern struct passwd *getpwuid();

    /*
     * JOBDONE is encounterred, parse job information
     */
    if (cubeln != NULL) {
        if (sscanf(cubeln,
            " Acct=%d User=%d Part=%d Size=%d Type=%d Queue=%s\n",
            &acid, &uid, &part_id, &size, &type, qname) != 6) {
            (void) fprintf (stderr,
                "Endrec:  Error parsing input line\n%s",
                cubeln);
            return;
        }
        if (fgets (cubeln, INLENGTH, fd) == NULL) {
                  (void) fprintf(stderr,
            "Endrec: Error getting input line\n");
            return;
        }
        if (sscanf(cubeln, 
            " Time=%d Rate=%f Under=%d Rate=%f Idle=%d Rate=%f",
            &cpu_seconds, &cpu_rate, &under_seconds, &under_rate,
            &idle_seconds, &idle_rate) != 6) {
            (void) fprintf (stderr,
                "Endrec:  Error parsing input line\n%s",
                cubeln);
            return;
        }
        cpu_hours = (float)cpu_seconds / 3600.;
        idle_hours = (float)idle_seconds / 3600.;
        under_hours = (float)under_seconds / 3600.;
        if (fgets (cubeln, INLENGTH, fd) == NULL) {
            if (!v1_1) (void) fprintf(stderr,
            "Endrec: Error getting input line\n");
        }
        else if (sscanf(cubeln, 
            " Subm=%d Reqst=%d", &submit_time, &reqst_secs) != 2) {
            if (!v1_1) (void) fprintf (stderr,
                "Endrec:  Error parsing input line\n%s", cubeln);
	    submit_time = reqst_secs = 0;
	    v1_1 = skip = 1;
        }
	else v1_1 = 0;
        if (getuid() != 0) {
	    for (j=0; j<naccts; j++) if (acid == agids[j]) break;
            if (j >= naccts) return;
        }

        /*
         * if new user, remember it and increment user count 
         */
        if (ulink_root == NULL) {
          ulink_root = (struct ulink *) calloc (1, sizeof (struct ulink));
          ulink_root->uid = uid;
          ucount = 1;
        } else {
            for (nptr=ulink_root; nptr!=NULL; ptr=nptr, nptr=nptr->next)
              if (nptr->uid == uid) break;
            if (nptr==NULL) {
              ptr->next = (struct ulink *) calloc (1, sizeof (struct ulink));
              ptr->next->uid = uid;
              ucount++;
            }
        }
#ifdef DEBUG
        (void) fprintf (stderr,
            "endrec:  Acct=%d User=%d Part=%d Size=%d Type=%d Queue=%s",
            acid, uid, part_id, size, type, qname);
        (void) fprintf (stderr,
            "endrec:  Time=%f Rate=%f Under=%f Rate=%f Idle=%f Rate=%f",
            cpu_hours, cpu_rate, under_hours, under_rate,
            idle_hours, idle_rate);
#endif
    }

    /*
     * fill-in time strings
     */
    tmp_tm = localtime (&lasttime);
    if (jflg) {
        (void) sprintf (date, "%2.2d/%2.2d/%2.2d",
            tmp_tm->tm_mon+1, tmp_tm->tm_mday, tmp_tm->tm_year);
        (void) sprintf (hms, "%2.2d:%2.2d:%2.2d",
            tmp_tm->tm_hour, tmp_tm->tm_min, tmp_tm->tm_sec);
    }

    /*
     * compare to job entries in process table one-by-one
     */
    tmp_nproc = nproc;
    if (nproc) for (i=0, k=0; i<nrecs && k<tmp_nproc; i++) {
        if (procflgs[i] == 0) continue;
	k++;
        if (cubeln > NULL) {
	    if (acid != procrecs[i].acid) continue;
	    if (lasttime < procrecs[i].starttime) continue;
	    if (uid != procrecs[i].uid) continue;
	    if (part_id != procrecs[i].part_id) continue;
	    if (wflg && strlen (procrecs[i].qname) <= 0
                        && strlen (qname) > 0) {
                    (void) strncpy (procrecs[i].qname, qname, 31);
                    procrecs[i].qname[31] = '\0';
            }
        }
	if (!wflg) {
            procflgs[i] = 0;
            nproc--;
	    continue;
	}

        if (cubeln > NULL) {	/* match */
	    procrecs[i].part_size = size;
	    procrecs[i].node_type = type;
	    procrecs[i].cpu_hours = cpu_hours;
	    procrecs[i].under_hours = under_hours;
	    procrecs[i].idle_hours = idle_hours;
	    procrecs[i].cpu_rate = cpu_rate;
	    procrecs[i].under_rate = under_rate;
	    procrecs[i].idle_rate = idle_rate;
	}

	/* match or end of all jobs, any cpu usage to report? */
	procrecs[i].cpu_hours -= procrecs[i].pre_cpu_hours;
	procrecs[i].under_hours -= procrecs[i].pre_under_hours;
	procrecs[i].idle_hours -= procrecs[i].pre_idle_hours;
	if (procrecs[i].cpu_hours <= 0.0 && procrecs[i].idle_hours <= 0.0) {
            procflgs[i] = 0;
            nproc--;
	    continue;
	}
	if (procrecs[i].under_hours < 0.0) procrecs[i].under_hours = 0.0;
	if (procrecs[i].idle_hours < 0.0) procrecs[i].idle_hours = 0.0;

	if (strlen(procrecs[i].qname) &&
		strcmp (procrecs[i].qname, "UNKNOWN") != 0) { 
            for (j=0; j<actab->n_dedicateq; j++)
            if (strcmp (procrecs[i].qname, actab->dedicateq[j]) == 0) {
                procrecs[i].qtype = DEDICATE;
                break;
            }
            if (procrecs[i].qtype == 0) procrecs[i].qtype = BATCH;
        }

        if ((acct_ent = nx_getaid (procrecs[i].acid)) == NULL)
            (void) strncpy (acct_name_buf, "Unknown",8);
        else (void) strncpy (acct_name_buf,
            acct_ent->acct_name, acct_name_len);
        acct_name_buf[acct_name_len] = '\0';
        for (j=strlen(acct_name_buf); j<acct_name_len; j++) 
            acct_name_buf[j] = ' ';
        if ((user_ent = getpwuid (procrecs[i].uid)) == NULL)
            (void) strncpy (user_name, "Unknown",8);
        else {
	    (void) strncpy (user_name, user_ent->pw_name, 8);
	    user_name[8] = '\0';
	}
        if (jflg) {
	    long subm_t=0; 
	    float reqst_t=0.0;
            (void) fprintf (stdout,"%s %8.8s", acct_name_buf,user_name);
            (void) fprintf(stdout," %10d", procrecs[i].part_id);
            (void) fprintf(stdout," %4d", procrecs[i].part_size);
            (void) fprintf(stdout," %4d", procrecs[i].qtype);
            (void) fprintf(stdout," %12.3f", procrecs[i].cpu_hours);
            (void) fprintf(stdout," %4.2f", procrecs[i].cpu_rate);
            (void) fprintf(stdout," %12.3f", procrecs[i].idle_hours);
            (void) fprintf(stdout," %4.2f", procrecs[i].idle_rate);
            (void) fprintf(stdout," %12.3f", procrecs[i].under_hours);
            (void) fprintf(stdout," %4.2f", procrecs[i].under_rate);
            (void) fprintf(stdout," %13.3f", 
		procrecs[i].cpu_hours * procrecs[i].cpu_rate +
                procrecs[i].under_hours * procrecs[i].under_rate + 
		procrecs[i].idle_hours * procrecs[i].idle_rate);
	    subm_t = submit_time;
	    if (subm_t <= 0 && cubeln > NULL) 
		subm_t = procrecs[i].submit_time;
	    if (subm_t > 0) {
                tmp_tm = localtime (&subm_t);
                (void) fprintf(stdout,
                    " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                    tmp_tm->tm_mon+1, tmp_tm->tm_mday, 
                    tmp_tm->tm_year, tmp_tm->tm_hour, 
                    tmp_tm->tm_min, tmp_tm->tm_sec);
	    }
	    else if (procrecs[i].qtype)
		(void) fprintf(stdout, " Unknown  Unknown ");
	    else (void) fprintf(stdout, " N/A      N/A     ");
            if (procrecs[i].starttime) {
		tmp_tm = localtime (&(procrecs[i].starttime));
                (void) fprintf(stdout,
                    " %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d",
                    tmp_tm->tm_mon+1, tmp_tm->tm_mday, 
                    tmp_tm->tm_year, tmp_tm->tm_hour, 
                    tmp_tm->tm_min, tmp_tm->tm_sec);
	    } else (void) fprintf(stdout, " Unknown  Unknown ");
            (void) fprintf(stdout, " %8.8s %8.8s", date,hms);
            if (cubeln == NULL && !lastcall) 
		(void) fprintf(stdout, "  0");	/* incomplete */
	    else (void) fprintf(stdout, "  1");	/* completed */
#ifndef INTELv1r1
	    reqst_t = procrecs[i].reqst_secs / 3600.;
	    if (reqst_t <= 0.0) reqst_t = reqst_secs / 3600.;
            if (reqst_t > 0.0) (void) fprintf(stdout, " %12.3f", reqst_t);
	    else if (procrecs[i].qtype)
		(void) fprintf(stdout, " Unknown     ");
	    else (void) fprintf(stdout, " N/A         ");
#endif
            (void) fprintf(stdout," %s\n", procrecs[i].qname);
        } else
            (void) fprintf (stdout, 
                "%s %8s %1d %4d %12.3f %12.3f %12.3f %13.3f\n",
                    acct_name_buf,user_name,
                    procrecs[i].qtype, procrecs[i].part_size,
                    procrecs[i].cpu_hours,
                    procrecs[i].idle_hours, procrecs[i].under_hours,
                    procrecs[i].cpu_hours * procrecs[i].cpu_rate +
                    procrecs[i].idle_hours * procrecs[i].idle_rate +
                    procrecs[i].under_hours * procrecs[i].under_rate);

        procflgs[i] = 0;
        nproc--;
#ifdef DEBUG
    (void) fprintf (stderr, "endrec: procflgs=(");
    for (j=0; j<nrecs; j++) if(procflgs[j]) (void) fprintf (stderr,"%d,",j);
    (void) fprintf (stderr, ")\n");
        (void) fprintf (stderr, "endrec: close record %d\n", i);
        (void) fprintf (stderr,"acid= %d\n",procrecs[i].acid);
        (void) fprintf (stderr,"uid = %d\n",procrecs[i].uid);
        (void) fprintf (stderr,"part_id = %d\n",procrecs[i].part_id);
        (void) fprintf (stderr,"part_size = %d\n", procrecs[i].part_size);
        (void) fprintf (stderr,"node_type = %d\n",procrecs[i].node_type);
        (void) fprintf (stderr,"qname = %s\n",procrecs[i].qname);
        (void) fflush (stderr);
#endif

        if (cubeln != NULL) return;
    }

    return; /* Threw away unmatched JOBDONE to avoid double counting */
}

/*===========================================================================*
 * Function:    chkrec
 *
 * Abstract:    This function is called for every OLDJOB.  It tries to find 
 *		a match in the process table and update the usage.  
 *		When there is no match, assume it's finished before, skip.
 *		Normally, sign is 1, only when this function is called to
 *		debit usage of previous period, the sign is -1
 *
 * Arguments:   cubeln - input line
 *		sign - sign for usage
 *
 * Return value:
 *              None
 *
 * Notes:
 *===========================================================================*/

void chkrec (cubeln, sign)
char *cubeln;
int sign;
{
    int i, j, k, acid, uid, part_id, part_size, node_type, pgid, qtype;
    char qname[32];
    time_t cpu_seconds, under_seconds, idle_seconds;
    float cpu_rate, under_rate, idle_rate,
        cpu_hours, under_hours, idle_hours, cpu_charge;
    struct tm *tmp_tm;

    /*
     * read in and parse the job information
     */
    if ((i = sscanf(cubeln,
        " Acct=%d User=%d Part=%d Size=%d Type=%d Queue=%s Pgid=%d\n",
        &acid, &uid, &part_id, &part_size, &node_type, qname, &pgid)) < 5) {
        (void) fprintf (stderr,
            "Chkrec:  Error parsing input line\n%s",
            cubeln);
        return;
    }
    if (i == 5)
        strcpy (qname, "UNKNOWN");

    if (i <= 6)
	pgid = -1;

    /*
     * get the next line and parse it
     */
    if (fgets (cubeln, INLENGTH, fd) == NULL) {
             (void) fprintf(stderr,
            "Chkrec: Error getting input line\n");
        return;
    }
    if (sscanf(cubeln, 
        " Time=%d Rate=%f Under=%d Rate=%f Idle=%d Rate=%f\n", 
        &cpu_seconds, &cpu_rate, &under_seconds, &under_rate,
        &idle_seconds, &idle_rate) != 6) {
        (void) fprintf (stderr,
            "Chkrec:  Error parsing input line\n%s",
            cubeln);
        return;
    }
    if (getuid() != 0) {
	for (j=0; j<naccts; j++) if (acid == agids[j]) break;
        if (j >= naccts) return;
    }

    cpu_hours = (float)cpu_seconds / 3600.;
    under_hours = (float)under_seconds / 3600.;
    idle_hours = (float)idle_seconds / 3600.;

    qtype = 0;
    if (strlen(qname) && strcmp (qname, "UNKNOWN") != 0) { 
        for (j=0; j<actab->n_dedicateq; j++)
            if (strcmp (qname, actab->dedicateq[j]) == 0) {
                qtype = DEDICATE;
                break;
            }
        if (qtype == 0) qtype = BATCH;
    }

    /*
     * compare process table entries one-by-one
     */
    tmp_nproc = nproc;
    if (nproc) for (i=0, k=0; i<nrecs && k<tmp_nproc; i++) {
        if (procflgs[i] == 0) {
            continue;
        }
	k++;
        if (acid != procrecs[i].acid) continue;
        if (uid != procrecs[i].uid) continue;
        if (part_id != procrecs[i].part_id) continue;
	if (qtype == INTERACTIVE && pgid != procrecs[i].pgid)
	    continue;
	/* match, update */
        if (part_size != procrecs[i].part_size && part_size > 0) 
            procrecs[i].part_size = part_size;
        if (node_type != procrecs[i].node_type && node_type > 0)
            procrecs[i].node_type = node_type;
	if (strlen (procrecs[i].qname) <= 0 && strlen (qname) > 0)
	    (void) strncpy (procrecs[i].qname, qname, 31);
	procrecs[i].qname[31] = '\0';
	procrecs[i].qtype = qtype;
	procrecs[i].pgid = pgid;
	if (cpu_hours > procrecs[i].cpu_hours) 
            procrecs[i].cpu_hours = cpu_hours;
	if (under_hours > procrecs[i].under_hours) 
            procrecs[i].under_hours = under_hours;
	if (idle_hours > procrecs[i].idle_hours) 
            procrecs[i].idle_hours = idle_hours;
	procrecs[i].cpu_rate = cpu_rate;
	procrecs[i].under_rate = under_rate;
	procrecs[i].idle_rate = idle_rate;
        return;
    }

    if (sign < 0) {	/* debit usage of previous period */
        /*
         * Any room in the process table for add?
         */
        for (i=0; i<nrecs && procflgs[i]; i++);
        if (i>=nrecs) {
        (void) fprintf (stderr, "nrecs=%d i=%d\n", nrecs, i);
        (void) fprintf (stderr,
            "Chkrec: Internal job table overflow.  Abort.\n");
            exit(-1);
        }
        (void) memset (&(procrecs[i]), '\0', sizeof (struct procrec));
    
        procrecs[i].acid = acid;
        procrecs[i].uid = uid;
        procrecs[i].part_id = part_id;
        procrecs[i].part_size = part_size;
	if (strlen (qname) > 0)
	    (void) strncpy (procrecs[i].qname, qname, 31);
	procrecs[i].qname[31] = '\0';
	procrecs[i].qtype = qtype;
        procrecs[i].pgid = pgid;
        procrecs[i].pre_cpu_hours = cpu_hours;
        procrecs[i].pre_under_hours = under_hours;
        procrecs[i].pre_idle_hours = idle_hours;
        procrecs[i].cpu_rate = cpu_rate;
        procrecs[i].under_rate = under_rate;
        procrecs[i].idle_rate = idle_rate;
    
        /*
         * set non-empty flag for the new entry and increment process count
         */
        procflgs[i] = 1;
        nproc++;
        if (verbose) (void) fprintf (stderr,
            "Chkrec: Usage of previous accounting period Acct=%d User=%d Part=%d, added in\n",
            acid, uid, part_id);
        return;
    }

    if (verbose) (void) fprintf (stderr,
        "Chkrec: No entry for existing job Acct=%d User=%d Part=%d, assume done\n",
        acid, uid, part_id);
    return;
}

/*===========================================================================*
 * Function:    delrec
 *
 * Abstract:    This function is called for every RUBOFJOB.  It looks for
 *              a matching JOB entry in the job records.  Only when there
 *		is a match, the job record is deleted
 *
 * Arguments:   cubeln - input line
 *
 * Return value:
 *              None
 *
 * Notes:
 *===========================================================================*/

void delrec (cubeln)
char *cubeln;
{
    int i, j, k, acid, uid, part_id, part_size, node_type;
    char qname[32];

    /*
     * read in and parse the job information
     */
    if (sscanf(cubeln,
        " Acct=%d User=%d Part=%d Size=%d Type=%d Queue=%s\n",
        &acid, &uid, &part_id, &part_size, &node_type, qname) != 6) {
        (void) fprintf (stderr,
            "Delrec:  Error parsing input line\n%s",
            cubeln);
        return;
    }

    if (getuid() != 0) {
	for (j=0; j<naccts; j++) if (acid == agids[j]) break;
        if (j >= naccts) return;
    }

    /*
     * compare to process table entry one-by-one
     */
    tmp_nproc = nproc;
    if (nproc) for (i=0,k=0; i<nrecs && k<tmp_nproc; i++) {
        if (procflgs[i] == 0) continue;
        if (acid != procrecs[i].acid) continue;
        if (uid != procrecs[i].uid) continue;
        if (part_id != procrecs[i].part_id) continue;
        if (part_size != procrecs[i].part_size) continue;
        if (node_type != procrecs[i].node_type) continue;
        if (strcmp(qname, procrecs[i].qname) != 0) continue;
        procflgs[i] = 0;
        nproc--;
        return;
    }
    return;
}

/*
 * write a line of down time record into output file
 */
void write_down (fd, last_down)
FILE *fd;
struct down_info *last_down;
{
char buf[128];
int days, hrs, mins, secs;
struct tm start_tm, end_tm, *tm_ptr;

    if ( (secs = last_down->end - last_down->start) < 0) {
	(void) fprintf (stderr, "down=%d up=%d, invalid time range\n",
		last_down->start, last_down->end);
	return;
    }

    tm_ptr = localtime (&(last_down->start));
    (void) bcopy (tm_ptr, &start_tm, sizeof(struct tm));
    tm_ptr = localtime (&(last_down->end));
    (void) bcopy (tm_ptr, &end_tm, sizeof(struct tm));
    days = secs / (24 * 60 * 60);
    secs -= days * (24 * 60 * 60);
    hrs = secs / 3600;
    secs -= hrs * 3600;
    mins = secs / 60;
    secs -= mins * 60;
    (void) fprintf (fd, 
	"%2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d %2.2d/%2.2d/%2.2d %2.2d:%2.2d:%2.2d %3.3d %2.2d:%2.2d:%2.2d %-5s %-24s\n", 
	start_tm.tm_mon+1, start_tm.tm_mday, start_tm.tm_year, 
	start_tm.tm_hour, start_tm.tm_min, start_tm.tm_sec,
	end_tm.tm_mon+1, end_tm.tm_mday, end_tm.tm_year, 
	end_tm.tm_hour, end_tm.tm_min, end_tm.tm_sec,
	days, hrs, mins, secs, last_down->type, last_down->comment);
    (void) memset (last_down, '\0', sizeof (struct down_info));
    return;
}
