static char *HPUX_ID = "@(#)37.6   86/08/26";
/* HPUX_ID: @(#)mkfs.c  37.6     86/06/23  */

/*
 * make file system for cylinder-group style file systems
 * add fs_clean        4/12/85
 * SYNOPSIS: either specify all optional arguments or none
 */

/*
 * The following constants set the options used for the number
 * of sectors (fs_nsect), and number of tracks (fs_ntrak).
 */
#define DFLNSECT        32
#define DFLNTRAK        16

/*
 * The following two constants set the option block and fragment sizes.
 * Both constants must be a power of 2 and meet the following constraints:
 *      MINBSIZE <= DESBLKSIZE <= MAXBSIZE
 *      DEV_BSIZE <= DESFRAGSIZE <= DESBLKSIZE
 *      DESBLKSIZE / DESFRAGSIZE <= 8
 */
#define DESBLKSIZE      8192
#define DESFRAGSIZE     1024

/*
 * Cylinder groups may have up to MAXCPG cylinders. The actual
 * number used depends upon how much information can be stored
 * on a single cylinder. The option is to used 16 cylinders
 * per group.
 */
#define DESCPG          16      /* desired fs_cpg */

/*
 * MINFREE gives the minimum acceptable percentage of file system
 * blocks which may be free. If the freelist drops below this level
 * only the superuser may continue to allocate blocks. This may
 * be set to 0 if no reserve of free blocks is deemed necessary,
 * however throughput drops by fifty percent if the file system
 * is run at between 90% and 100% full; thus the option value of
 * fs_minfree is 10%.
 */
#define MINFREE         10

/*
 * ROTDELAY gives the minimum number of milliseconds to initiate
 * another disk transfer on the same cylinder. It is used in
 * determining the rotationally optimal layout for disk blocks
 * within a file; the option of fs_rotdelay is 4ms.
 */
#define ROTDELAY        4

/*
 * MAXCONTIG sets the option for the maximum number of blocks
 * that may be allocated sequentially. Since UNIX drivers are
 * not capable of scheduling multi-block transfers, this options
 * to 1 (ie no contiguous blocks are allocated).
 */
#define MAXCONTIG       1

/*
 * MAXBLKPG determines the maximum number of data blocks which are
 * placed in a single cylinder group. This is currently a function
 * of the block and fragment size of the file system.
 */
#define MAXBLKPG(fs)    ((fs)->fs_fsize / sizeof(daddr_t))

/*
 * Each file system has a number of inodes statically allocated.
 * We allocate one inode slot per NBPI bytes, expecting this
 * to be far more than we will ever need.
 */
#define NBPI            2048

/*
 * Disks are assumed to rotate at 60HZ, unless otherwise specified.
 */
#define DEFHZ           60

#ifndef STANDALONE
#include <stdio.h>
#endif

#if ( !defined(spectrum) || !defined(HFS) ) && !defined(STANDALONE)
#include <a.out.h>
#endif

#include <sys/types.h>
#include <sys/param.h>
#include <sys/inode.h>
#include <sys/ino.h>
#include <sys/fs.h>
#include <sys/sysmacros.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <ustat.h>
#include <ctype.h>
#ifdef WKSTN
/* could be used by everyone */
#include <fcntl.h>
#define PAWS_UID    17
#define BASIC_UID   18
#define WS_GID       9
#endif

#ifdef WKSTN
#define UMASK           0777
#else
#define UMASK           0755
#endif
#define MAXINOPB        (MAXBSIZE / sizeof(struct dinode))
#define POWEROF2(num)   (((num) & ((num) - 1)) == 0)

union {
	struct fs fs;
	char pad[MAXBSIZE];
} fsun;
#define sblock  fsun.fs
struct fs *fs;

struct  csum *fscs;

union {
	struct cg cg;
	char pad[MAXBSIZE];
} cgun;
#define acg     cgun.cg

#if !defined(spectrum) && !defined(HFS)
struct exec head;      /* header of object file of BOOT program */
#endif

#ifdef WKSTN
/* avoid a 256K array used only as a source of zeroes! */
struct  dinode zino[MAXBSIZE/sizeof(struct dinode)];
/* for listing file */
char *listing_name;
FILE *listing_file;
#else
struct  dinode zino[MAXIPG];
#endif
char    *fsys, *proto;
char token[BUFSIZ];

time_t  utime;
FILE *fin;
int     fsi,
	fso,
	ino = 3;
short   errs;

long getnum();

daddr_t alloc();
daddr_t proto_alloc();

main(argc, argv)
	int argc;
	char *argv[];
{
	long cylno, rpos, blk, i, j, inos, fssize, n, warn = 0;
	int f, c;
	char buf[BBSIZE];
	struct stat statarea;
	struct ustat ustatarea;
	short option;

#ifdef WKSTN
	if (argc == 12) {
		int fd;

		listing_name = argv[11];
		argc = 11;
		fd = creat_text(listing_name, 0666);
		if (fd != -1)
			listing_file = fdopen(fd, "w");
		if (fd == -1 || listing_file == NULL) {
			fprintf(stderr, "Cannot open listing file %s\n",
				listing_name);
			exit(1);
		}
	}
#endif
	option = 0;
#ifndef STANDALONE
	time(&utime);

	/* check invocation */
	if (argc < 3 || (argc >3 && argc != 11)) {
		printf("usage: mkfs special size \n");
		printf("            [nsect ntrak bsize fsize cpg minfree rps nbpi]\n");
		printf("   or  mkfs special proto \n");
		printf("            [nsect ntrak bsize fsize cpg minfree rps nbpi]\n");
		exit(1);
	}
	if (argc == 11)
		option = 1;

	/* check validity of the specified special file; this file must exist */
	fsys = argv[1];
#ifdef WKSTN
	fso = open(fsys, O_RDWR|O_CREAT, 0666);
	if(fso < 0) {
		printf("%s: cannot create\n", fsys);
		exit(1);
	}
	fsi = fso;
#else
	if (stat(fsys, &statarea) < 0)
	{
		printf(" %s: cannot stat\n", fsys);
		exit(1);
	}
	if ((statarea.st_mode & S_IFMT) == S_IFBLK)
		if (ustat(statarea.st_rdev, &ustatarea) >= 0){
			printf("*** MOUNTED FILE SYSTEM\n");
			exit(1);
		}

	fso = creat(fsys, 0666); /*create new file or rewrite old file*/
	if(fso < 0) {
		printf("%s: cannot create\n", fsys);
		exit(1);
	}

	fsi = open(fsys, 0);
	if(fsi < 0) {
		printf("%s: cannot open\n", fsys);
		exit(1);
	}
#endif

	/* check which form of mkfs command is used */
	proto = argv[2];
#ifdef WKSTN
	fin = NULL;
	n = 0;
	fssize = atoi(proto);
#else
	fin = fopen(proto, "r");
	if (fin == NULL)     /* first form is specified */
	{
		fssize = n = 0;
		for (f = 0; c = proto[f]; f++)
			if (isdigit(c))
				n = n*10 + (c- '0');
			else
				printf("%s:  cannot open\n", proto), exit(1);
		fssize = n;
		n = 0;
	}
	else            /* second form is used: proto */
	{
	       /* get name of boot program and copy it to
		* the boot area. Boot program "" is ignored.
		*/
		getstr();
#if !defined(spectrum) && !defined(HFS)
		if (strcmp(token, "\"\""))
		{
			f = open(token, 0);   /* open boot program */
			if (f <= 0)
			{
				printf("%s:  cannot open\n", token);
				goto f1;
			}
			read(f, ((char *)&head), sizeof head);
			if (head.a_magic.file_type != EXEC_MAGIC ||
			    head.a_magic.system_id != HP98x6_ID){
				printf("%s:  bad format\n", token);
				goto f1;
			}
			c = head.a_text + head.a_data;
			if ( c > BBSIZE)
			{
				printf("%s: too big\n", token);
				goto f1;
			}
			/* read boot program */
			read(f, buf, c);
			/* write boot-block to file system */
			lseek(fso, 0L, 0);
			if (write(fso, buf, BBSIZE) != BBSIZE)
			{
				printf("write error: boot-block\n");
				exit(1);
			}
f1:                     close(f);
		}
#endif
		fssize = getnum();
	}  /* end of else proto */
#endif not WKSTN
#else STANDALONE
	{
		static char protos[60];
		char fsbuf[100];

		printf("file sys size: ");
		gets(protos);
		fssize = atoi(protos);
		do {
			printf("file system: ");
			gets(fsbuf);
			fso = open(fsbuf, 1);
			fsi = open(fsbuf, 0);
		} while (fso < 0 || fsi < 0);
	}
	argc = 0;
#endif STANDALONE

	/* check the specified size of file system and verify that
	 * the special device file is big enough to contain the
	 * new file system
	 */
	if (fssize <= 0)
		printf("preposterous size %d\n", fssize), exit(1);
	wtfs(fssize - 1, DEV_BSIZE, (char *)&sblock);

	/*
	 * collect and verify the sector and track info...
	 */
	if (!option)
	{
		sblock.fs_nsect = DFLNSECT;
		sblock.fs_ntrak = DFLNTRAK;
	}
	else
	{
		sblock.fs_nsect = atoi(argv[3]);
		sblock.fs_ntrak = atoi(argv[4]);
	}
	if (sblock.fs_nsect <= 0)
	     printf("preposterous nsect %d\n",sblock.fs_nsect),exit(1);
	if (sblock.fs_ntrak <= 0)
		printf("preposterous ntrak %d\n",sblock.fs_ntrak),exit(1);
	sblock.fs_spc = sblock.fs_ntrak * sblock.fs_nsect;

	/*
	 * collect and verify the block and fragment sizes
	 */
	if (!option)
	{
		sblock.fs_bsize=DESBLKSIZE;
		sblock.fs_fsize=DESFRAGSIZE;
	}
	else
	{
		sblock.fs_bsize= atoi(argv[5]);
		sblock.fs_fsize= atoi(argv[6]);
	}
	if (!POWEROF2(sblock.fs_bsize)) {
		printf("block size must be a power of 2, not %d\n",
		    sblock.fs_bsize);
		exit(1);
	}
	if (!POWEROF2(sblock.fs_fsize)) {
		printf("fragment size must be a power of 2, not %d\n",
		    sblock.fs_fsize);
		exit(1);
	}
	if (sblock.fs_fsize < DEV_BSIZE) {
		printf("fragment size %d is too small, minimum is %d\n",
		    sblock.fs_fsize, DEV_BSIZE);
		exit(1);
	}
	if (sblock.fs_bsize < MINBSIZE) {
		printf("block size %d is too small, minimum is %d\n",
		    sblock.fs_bsize, MINBSIZE);
		exit(1);
	}
	if (sblock.fs_fsize > MAXBSIZE){
		printf("fragment size %d is too big, maximum is %d\n",
		    sblock.fs_fsize, MAXBSIZE);
		exit(1);
	}
	if (sblock.fs_bsize >MAXBSIZE){
		printf("block size %d is too big, maximum is %d\n",
		    sblock.fs_bsize, MAXBSIZE);
		exit(1);
	}
	if (sblock.fs_bsize < sblock.fs_fsize) {
		printf("block size (%d) cannot be smaller than fragment size (%d)\n",
		    sblock.fs_bsize, sblock.fs_fsize);
		exit(1);
	}

	sblock.fs_bmask = ~(sblock.fs_bsize - 1);
	sblock.fs_fmask = ~(sblock.fs_fsize - 1);
	for (sblock.fs_bshift = 0, i = sblock.fs_bsize; i > 1; i >>= 1)
		sblock.fs_bshift++;
	for (sblock.fs_fshift = 0, i = sblock.fs_fsize; i > 1; i >>= 1)
		sblock.fs_fshift++;
	sblock.fs_frag = numfrags(&sblock, sblock.fs_bsize);
	if (sblock.fs_frag > MAXFRAG) {
		printf("fragment size %d is too small, minimum with block size %d is %d\n",
		    sblock.fs_fsize, sblock.fs_bsize,
		    sblock.fs_bsize / MAXFRAG);
		exit(1);
	}
	for (sblock.fs_fragshift = 0, i = sblock.fs_frag; i > 1; i >>= 1)
		sblock.fs_fragshift++;
	sblock.fs_nindir = sblock.fs_bsize / sizeof(daddr_t);
	sblock.fs_inopb = sblock.fs_bsize / sizeof(struct dinode);
	sblock.fs_nspf = sblock.fs_fsize / DEV_BSIZE;
	for (sblock.fs_fsbtodb = 0, i = sblock.fs_nspf; i > 1; i >>= 1)
		sblock.fs_fsbtodb++;
	sblock.fs_sblkno =
	    roundup(howmany(BBSIZE + SBSIZE, sblock.fs_fsize), sblock.fs_frag);
	sblock.fs_cblkno = (daddr_t)(sblock.fs_sblkno +
	    roundup(howmany(SBSIZE, sblock.fs_fsize), sblock.fs_frag));
	sblock.fs_iblkno = sblock.fs_cblkno + sblock.fs_frag;
	sblock.fs_cgoffset = roundup(
	    howmany(sblock.fs_nsect, sblock.fs_fsize / DEV_BSIZE),
	    sblock.fs_frag);
	for (sblock.fs_cgmask = 0xffffffff, i = sblock.fs_ntrak; i > 1; i >>= 1)
		sblock.fs_cgmask <<= 1;
	if (!POWEROF2(sblock.fs_ntrak))
		sblock.fs_cgmask <<= 1;
	for (sblock.fs_cpc = NSPB(&sblock), i = sblock.fs_spc;
	     sblock.fs_cpc > 1 && (i & 1) == 0;
	     sblock.fs_cpc >>= 1, i >>= 1)
		/* void */;
	if (sblock.fs_cpc > MAXCPG) {
		printf("maximum block size with nsect %d and ntrak %d is %d\n",
		    sblock.fs_nsect, sblock.fs_ntrak,
		    sblock.fs_bsize / (sblock.fs_cpc / MAXCPG));
		exit(1);
	}
	/*
	 * collect and verify the number of cylinders per group
	 */
#ifdef WKSTN
	if (!option || atoi(argv[7]) == 0)   /* default value */
#else
	if (!option)   /* default value */
#endif
	{
		sblock.fs_cpg = MAX(sblock.fs_cpc, DESCPG);
		sblock.fs_fpg = (sblock.fs_cpg * sblock.fs_spc) / NSPF(&sblock);
		while (sblock.fs_fpg / sblock.fs_frag > MAXBPG(&sblock) &&
		    sblock.fs_cpg > sblock.fs_cpc) {
			sblock.fs_cpg -= sblock.fs_cpc;
			sblock.fs_fpg =
			    (sblock.fs_cpg * sblock.fs_spc) / NSPF(&sblock);
		}
	}
	else  /* take the given value */
	{
		sblock.fs_cpg = atoi(argv[7]);
		sblock.fs_fpg = (sblock.fs_cpg * sblock.fs_spc) / NSPF(&sblock);
	}
	if (sblock.fs_cpg < 1) {
		printf("cylinder groups must have at least 1 cylinder\n");
		exit(1);
	}
	if (sblock.fs_cpg > MAXCPG) {
		printf("cylinder groups are limited to %d cylinders\n", MAXCPG);
		exit(1);
	}
	if (sblock.fs_cpg % sblock.fs_cpc != 0) {
		printf("cylinder groups must have a multiple of %d cylinders\n",
		    sblock.fs_cpc);
		exit(1);
	}
	/*
	 * Now have size for file system and nsect and ntrak.
	 * Determine number of cylinders and blocks in the file system.
	 */
	sblock.fs_size = fssize = dbtofsb(&sblock, fssize);
	sblock.fs_ncyl = fssize * NSPF(&sblock) / sblock.fs_spc;
	if (fssize * NSPF(&sblock) > sblock.fs_ncyl * sblock.fs_spc) {
		sblock.fs_ncyl++;
		warn = 1;
	}
	if (sblock.fs_ncyl < 1) {
		printf("file systems must have at least one cylinder\n");
		exit(1);
	}
	/*
	 * determine feasability/values of rotational layout tables
	 */
	if (sblock.fs_ntrak == 1) {
		sblock.fs_cpc = 0;
		goto next;
	}
	if (sblock.fs_spc * sblock.fs_cpc > MAXBPC * NSPB(&sblock) ||
	    sblock.fs_nsect > (1 << NBBY) * NSPB(&sblock)) {
		printf("%s %s %d %s %d.%s",
		    "Warning: insufficient space in super block for\n",
		    "rotational layout tables with nsect", sblock.fs_nsect,
		    "and ntrak", sblock.fs_ntrak,
		    "\nFile system performance may be impaired.\n");
		sblock.fs_cpc = 0;
		goto next;
	}
	/*
	 * calculate the available blocks for each rotational position
	 */
	for (cylno = 0; cylno < MAXCPG; cylno++)
		for (rpos = 0; rpos < NRPOS; rpos++)
			sblock.fs_postbl[cylno][rpos] = -1;
	blk = sblock.fs_spc * sblock.fs_cpc / NSPF(&sblock);
	for (i = 0; i < blk; i += sblock.fs_frag)
		/* void */;
	for (i -= sblock.fs_frag; i >= 0; i -= sblock.fs_frag) {
		cylno = cbtocylno(&sblock, i);
		rpos = cbtorpos(&sblock, i);
		blk = i / sblock.fs_frag;
		if (sblock.fs_postbl[cylno][rpos] == -1)
			sblock.fs_rotbl[blk] = 0;
		else
			sblock.fs_rotbl[blk] =
			    sblock.fs_postbl[cylno][rpos] - blk;
		sblock.fs_postbl[cylno][rpos] = blk;
	}
next:
	/*
	 * Validate specified/determined cpg.
	 */
	if (sblock.fs_spc > MAXBPG(&sblock) * NSPB(&sblock)) {
		printf("too many sectors per cylinder (%d sectors)\n",
		    sblock.fs_spc);
		while(sblock.fs_spc > MAXBPG(&sblock) * NSPB(&sblock)) {
			sblock.fs_bsize <<= 1;
			if (sblock.fs_frag < MAXFRAG)
				sblock.fs_frag <<= 1;
			else
				sblock.fs_fsize <<= 1;
		}
		printf("nsect %d, and ntrak %d, requires block size of %d,\n",
		    sblock.fs_nsect, sblock.fs_ntrak, sblock.fs_bsize);
		printf("\tand fragment size of %d\n", sblock.fs_fsize);
		exit(1);
	}
	if (sblock.fs_fpg > MAXBPG(&sblock) * sblock.fs_frag) {
		printf("cylinder group too large (%d cylinders); ",
		    sblock.fs_cpg);
		printf("max: %d cylinders per group\n",
		    MAXBPG(&sblock) * sblock.fs_frag /
		    (sblock.fs_fpg / sblock.fs_cpg));
		exit(1);
	}
	sblock.fs_cgsize = fragroundup(&sblock,
	    sizeof(struct cg) + howmany(sblock.fs_fpg, NBBY));
	/*
	 * Compute/validate number of cylinder groups.
	 */
	sblock.fs_ncg = sblock.fs_ncyl / sblock.fs_cpg;
	if (sblock.fs_ncyl % sblock.fs_cpg)
		sblock.fs_ncg++;
	if ((sblock.fs_spc * sblock.fs_cpg) % NSPF(&sblock)) {
		printf("mkfs: nsect %d, ntrak %d, cpg %d is not tolerable\n",
		    sblock.fs_nsect, sblock.fs_ntrak, sblock.fs_cpg);
		printf("as this would have cyl groups whose size\n");
		printf("is not a multiple of %d; choke!\n", sblock.fs_fsize);
		exit(1);
	}
	/*
	 * Compute number of inode blocks per cylinder group.
	 * Start with one inode per NBPI bytes; adjust as necessary.
	 */
	inos = MAX(NBPI, sblock.fs_fsize);
	if (option)   /*get the given value*/
	{
		i = atoi(argv[10]);
		if (i <= 0)
			/* LAF 870429 changed wording below */
			printf("%s: improper data bytes per Inode reset to %d\n", argv[10], inos);
		else
			inos = i;
	}
	i = sblock.fs_iblkno + MAXIPG / INOPF(&sblock);
	inos = (fssize - sblock.fs_ncg * i) * sblock.fs_fsize / inos /
	    INOPB(&sblock);
	if (inos <= 0)
		inos = 1;
	sblock.fs_ipg = ((inos / sblock.fs_ncg) + 1) * INOPB(&sblock);
	if (sblock.fs_ipg > MAXIPG)
		sblock.fs_ipg = MAXIPG;
	sblock.fs_dblkno = sblock.fs_iblkno + sblock.fs_ipg / INOPF(&sblock);
	i = MIN(~sblock.fs_cgmask, sblock.fs_ncg - 1);
	if (cgdmin(&sblock, i) - cgbase(&sblock, i) >= sblock.fs_fpg) {
		printf("inode blocks/cyl group (%d) >= data blocks (%d)\n",
		    cgdmin(&sblock, i) - cgbase(&sblock, i) / sblock.fs_frag,
		    sblock.fs_fpg / sblock.fs_frag);
		printf("number of cylinders per cylinder group must be increased\n");
		exit(1);
	}
	j = sblock.fs_ncg - 1;
	if ((i = fssize - j * sblock.fs_fpg) < sblock.fs_fpg &&
	    cgdmin(&sblock, j) - cgbase(&sblock, j) > i) {
#ifdef WKSTN
		if (sblock.fs_ncg >= 2) {
#endif
		printf("Warning: inode blocks/cyl group (%d) >= data blocks (%d) in last\n",
		    (cgdmin(&sblock, j) - cgbase(&sblock, j)) / sblock.fs_frag,
		    i / sblock.fs_frag);
		printf("    cylinder group. This implies %d sector(s) cannot be allocated.\n",
		    i * NSPF(&sblock));
#ifdef WKSTN
		}
#endif
		sblock.fs_ncg--;
		sblock.fs_ncyl -= sblock.fs_ncyl % sblock.fs_cpg;
		sblock.fs_size = fssize = sblock.fs_ncyl * sblock.fs_spc /
		    NSPF(&sblock);
		warn = 0;
	}
#ifdef WKSTN
	if (sblock.fs_ncg <= 0) {
		printf("Device size too small\n");
		exit(1);
	}
#endif
	if (warn) {
		printf("Warning: %d sector(s) in last cylinder unallocated\n",
		    sblock.fs_spc -
		    (fssize * NSPF(&sblock) - (sblock.fs_ncyl - 1)
		    * sblock.fs_spc));
	}
	/*
	 * fill in remaining fields of the super block
	 */
	sblock.fs_csaddr = cgdmin(&sblock, 0);
	sblock.fs_cssize =
	    fragroundup(&sblock, sblock.fs_ncg * sizeof(struct csum));
	i = sblock.fs_bsize / sizeof(struct csum);
	sblock.fs_csmask = ~(i - 1);
	for (sblock.fs_csshift = 0; i > 1; i >>= 1)
		sblock.fs_csshift++;
	i = sizeof(struct fs) +
		howmany(sblock.fs_spc * sblock.fs_cpc, NSPB(&sblock));
	sblock.fs_sbsize = fragroundup(&sblock, i);
	fscs = (struct csum *)calloc(1, sblock.fs_cssize);
	sblock.fs_magic = FS_MAGIC;
	sblock.fs_clean = FS_CLEAN;
	sblock.fs_rotdelay = ROTDELAY;
	if (!option)
		sblock.fs_minfree = MINFREE;
	else
	{
		sblock.fs_minfree = atoi(argv[8]);
		if (sblock.fs_minfree < 0 || sblock.fs_minfree > 99) {
			printf("%s: improper minfree reset to %d%%\n", argv[8],
				MINFREE);   /*LAF 870429 "bogus"->"improper"*/
			sblock.fs_minfree = MINFREE;
		}
	}
	sblock.fs_maxcontig = MAXCONTIG;
	sblock.fs_maxbpg = MAXBLKPG(&sblock);
	if (!option)
		sblock.fs_rps = DEFHZ;
	else
		sblock.fs_rps = atoi(argv[9]);
	sblock.fs_cgrotor = 0;
	sblock.fs_cstotal.cs_ndir = 0;
	sblock.fs_cstotal.cs_nbfree = 0;
	sblock.fs_cstotal.cs_nifree = 0;
	sblock.fs_cstotal.cs_nffree = 0;
	sblock.fs_fmod = 0;
	sblock.fs_ronly = 0;
	/*
	 * Dump out summary information about file system.
	 */
	printf("%s:\t%d sectors in %d cylinders of %d tracks, %d sectors\n",
	    fsys, sblock.fs_size * NSPF(&sblock), sblock.fs_ncyl,
	    sblock.fs_ntrak, sblock.fs_nsect);
	printf("\t%.1fMb in %d cyl groups (%d c/g, %.2fMb/g, %d i/g)\n",
	    (float)sblock.fs_size * sblock.fs_fsize * 1e-6, sblock.fs_ncg,
	    sblock.fs_cpg, (float)sblock.fs_fpg * sblock.fs_fsize * 1e-6,
	    sblock.fs_ipg);
	/*
	 * Now build the cylinders group blocks and
	 * then print out indices of cylinder groups.
	 */
#ifdef WKSTN
	printf("Alternate superblocks at:");
#else
	printf("super-block backups (for fsck -b#) at:");
#endif
	for (cylno = 0; cylno < sblock.fs_ncg; cylno++) {
		initcg(cylno);
		if (cylno % 10 == 0)
			printf("\n");
		printf(" %d,", fsbtodb(&sblock, cgsblock(&sblock, cylno)));
	}
	printf("\n");
#ifdef WKSTN
	/* make a listing file if requested */
	if (listing_file != NULL) {

	    fprintf(listing_file,
		"%s:\n%d sectors in %d cylinders of %d tracks, %d sectors\n",
		fsys, sblock.fs_size * NSPF(&sblock), sblock.fs_ncyl,
		sblock.fs_ntrak, sblock.fs_nsect);
	    fprintf(listing_file,
		"%.1fMb in %d cyl groups (%d c/g, %.2fMb/g, %d i/g)\n",
		(float)sblock.fs_size * sblock.fs_fsize * 1e-6, sblock.fs_ncg,
		sblock.fs_cpg, (float)sblock.fs_fpg * sblock.fs_fsize * 1e-6,
		sblock.fs_ipg);
	    fprintf(listing_file, "Alternate superblocks at:");
	    for (cylno = 0; cylno < sblock.fs_ncg; cylno++) {
		    if (cylno % 10 == 0)
			    fprintf(listing_file, "\n");
		    fprintf(listing_file,
			" %d,", fsbtodb(&sblock, cgsblock(&sblock, cylno)));
	    }
	    fprintf(listing_file, "\n");
	    fclose(listing_file);

	}
#endif
	/*
	 * Construct the initial file system with lost+found directory;
	 * Now if mkfs proto, construct the file system according
	 * to the information contained in the proto file;
	 * then write out the super block
	 */
	fsinit();
#ifndef WKSTN
	if (fin)
	{
		fs = &sblock;
		descend((struct inode *)0);
	}
#endif
	sblock.fs_time = utime;
	wtfs(SBLOCK, SBSIZE, (char *)&sblock);
	for (i = 0; i < sblock.fs_cssize; i += sblock.fs_bsize)
		wtfs(fsbtodb(&sblock, sblock.fs_csaddr + numfrags(&sblock, i)),
			sblock.fs_cssize - i < sblock.fs_bsize ?
			    sblock.fs_cssize - i : sblock.fs_bsize,
			((char *)fscs) + i);
	/*
	 * Write out the duplicate super blocks
	 */
	for (cylno = 0; cylno < sblock.fs_ncg; cylno++)
		wtfs(fsbtodb(&sblock, cgsblock(&sblock, cylno)),
		    SBSIZE, (char *)&sblock);
#ifndef STANDALONE
#ifndef WKSTN
	exit(0);
#endif
#endif
}

/*
 * Initialize a cylinder group.
 */
initcg(cylno)
int cylno;
{
	daddr_t cbase, d, dlower, dupper, dmax;
	long i, j, s;
	register struct csum *cs;

	/*
	 * Determine block bounds for cylinder group.
	 * Allow space for super block summary information in first
	 * cylinder group.
	 */
	cbase = cgbase(&sblock, cylno);
	dmax = cbase + sblock.fs_fpg;
	if (dmax > sblock.fs_size)
		dmax = sblock.fs_size;
	dlower = cgsblock(&sblock, cylno) - cbase;
	dupper = cgdmin(&sblock, cylno) - cbase;
	cs = fscs + cylno;
	acg.cg_time = utime;
	acg.cg_magic = CG_MAGIC;
	acg.cg_cgx = cylno;
	if (cylno == sblock.fs_ncg - 1)
		acg.cg_ncyl = sblock.fs_ncyl % sblock.fs_cpg;
	else
		acg.cg_ncyl = sblock.fs_cpg;
	acg.cg_niblk = sblock.fs_ipg;
	acg.cg_ndblk = dmax - cbase;
	acg.cg_cs.cs_ndir = 0;
	acg.cg_cs.cs_nffree = 0;
	acg.cg_cs.cs_nbfree = 0;
	acg.cg_cs.cs_nifree = 0;
	acg.cg_rotor = 0;
	acg.cg_frotor = 0;
	acg.cg_irotor = 0;
	for (i = 0; i < sblock.fs_frag; i++) {
		acg.cg_frsum[i] = 0;
	}
	for (i = 0; i < sblock.fs_ipg; ) {
		for (j = INOPB(&sblock); j > 0; j--) {
			clrbit(acg.cg_iused, i);
			i++;
		}
		acg.cg_cs.cs_nifree += INOPB(&sblock);
	}
	if (cylno == 0)
		for (i = 0; i < ROOTINO; i++) {
			setbit(acg.cg_iused, i);
			acg.cg_cs.cs_nifree--;
		}
	while (i < MAXIPG) {
		clrbit(acg.cg_iused, i);
		i++;
	}
	lseek(fso, fsbtodb(&sblock, cgimin(&sblock, cylno)) * DEV_BSIZE, 0);
#ifdef WKSTN
	/* this code OK for real mkfs, where zino is bigger */
	for (i = sblock.fs_ipg*sizeof(struct dinode); i > 0; i -= sizeof(zino)){
	    j = MIN(sizeof(zino), i);
	    if (write(fso, (char *)zino, j) != j)
		printf("write error %ld\n", numfrags(&sblock, lseek(fso,0L,1)));
	}
#else
	if (write(fso, (char *)zino, sblock.fs_ipg * sizeof (struct dinode)) !=
	    sblock.fs_ipg * sizeof (struct dinode))
		printf("write error %ld\n", numfrags(&sblock, lseek(fso,0L,1)));
#endif
	for (i = 0; i < MAXCPG; i++) {
		acg.cg_btot[i] = 0;
		for (j = 0; j < NRPOS; j++)
			acg.cg_b[i][j] = 0;
	}
	if (cylno == 0) {
		/*
		 * reserve space for summary info and Boot block
		 */
		dupper += howmany(sblock.fs_cssize, sblock.fs_fsize);
		for (d = 0; d < dlower; d += sblock.fs_frag)
			clrblock(&sblock, acg.cg_free, d/sblock.fs_frag);
	} else {
		for (d = 0; d < dlower; d += sblock.fs_frag) {
			setblock(&sblock, acg.cg_free, d/sblock.fs_frag);
			acg.cg_cs.cs_nbfree++;
			acg.cg_btot[cbtocylno(&sblock, d)]++;
			acg.cg_b[cbtocylno(&sblock, d)][cbtorpos(&sblock, d)]++;
		}
		sblock.fs_dsize += dlower;
	}
	sblock.fs_dsize += acg.cg_ndblk - dupper;
	for (; d < dupper; d += sblock.fs_frag)
		clrblock(&sblock, acg.cg_free, d/sblock.fs_frag);
	if (d > dupper) {
		acg.cg_frsum[d - dupper]++;
		for (i = d - 1; i >= dupper; i--) {
			setbit(acg.cg_free, i);
			acg.cg_cs.cs_nffree++;
		}
	}
	while ((d + sblock.fs_frag) <= dmax - cbase) {
		setblock(&sblock, acg.cg_free, d/sblock.fs_frag);
		acg.cg_cs.cs_nbfree++;
		acg.cg_btot[cbtocylno(&sblock, d)]++;
		acg.cg_b[cbtocylno(&sblock, d)][cbtorpos(&sblock, d)]++;
		d += sblock.fs_frag;
	}
	if (d < dmax - cbase) {
		acg.cg_frsum[dmax - cbase - d]++;
		for (; d < dmax - cbase; d++) {
			setbit(acg.cg_free, d);
			acg.cg_cs.cs_nffree++;
		}
		for (; d % sblock.fs_frag != 0; d++)
			clrbit(acg.cg_free, d);
	}
	for (d /= sblock.fs_frag; d < MAXBPG(&sblock); d ++)
		clrblock(&sblock, acg.cg_free, d);
	sblock.fs_cstotal.cs_ndir += acg.cg_cs.cs_ndir;
	sblock.fs_cstotal.cs_nffree += acg.cg_cs.cs_nffree;
	sblock.fs_cstotal.cs_nbfree += acg.cg_cs.cs_nbfree;
	sblock.fs_cstotal.cs_nifree += acg.cg_cs.cs_nifree;
	*cs = acg.cg_cs;
	wtfs(fsbtodb(&sblock, cgtod(&sblock, cylno)),
		sblock.fs_bsize, (char *)&acg);
}

/*
 * initialize the file system
 */
struct inode node;
#define PREDEFDIR 3
struct direct root_dir[] = {
	{ ROOTINO, sizeof(struct direct), 1, ".", "" },
	{ ROOTINO, sizeof(struct direct), 2, "..", "" },
	{ LOSTFOUNDINO, sizeof(struct direct), 10, "lost+found", "" },
};
struct direct lost_found_dir[] = {
	{ LOSTFOUNDINO, sizeof(struct direct), 1, ".", "" },
	{ ROOTINO, sizeof(struct direct), 2, "..", "" },
	{ 0, DIRBLKSIZ, 0, 0, "" },
};
char dbuf[MAXBSIZE];

fsinit()
{
	int i;

	/*
	 * initialize the node
	 */
	node.i_atime = utime;
	node.i_mtime = utime;
	node.i_ctime = utime;
	/*
	 * create the lost+found directory
	 */
	(void)makedir(lost_found_dir, 2);
	for (i = DIRBLKSIZ; i < sblock.fs_bsize; i += DIRBLKSIZ)
		memcpy(&dbuf[i], &lost_found_dir[2], sizeof(struct direct));
		/* DIRSIZ(&lost_found_dir[2])); */
	node.i_number = LOSTFOUNDINO;
	node.i_mode = IFDIR | UMASK;
#ifdef WKSTN
#ifdef PAWS
	node.i_uid = PAWS_UID;
#else
	node.i_uid = BASIC_UID;
#endif
	node.i_gid = WS_GID;
#endif WKSTN
	node.i_nlink = 2;
	node.i_size = sblock.fs_bsize;
	node.i_db[0] = alloc(node.i_size, node.i_mode);
	node.i_blocks = btodb(fragroundup(&sblock, node.i_size));
	wtfs(fsbtodb(&sblock, node.i_db[0]), node.i_size, dbuf);
	iput(&node);

	/*
	 * create the root directory
	 */
	node.i_number = ROOTINO;
	node.i_mode = IFDIR | UMASK;
#ifdef WKSTN
#ifdef PAWS
	node.i_uid = PAWS_UID;
#else
	node.i_uid = BASIC_UID;
#endif
	node.i_gid = WS_GID;
#endif WKSTN
	node.i_nlink = PREDEFDIR;
	node.i_size = makedir(root_dir, PREDEFDIR);
	node.i_db[0] = alloc(sblock.fs_fsize, node.i_mode);
	node.i_blocks = btodb(fragroundup(&sblock, node.i_size));
	wtfs(fsbtodb(&sblock, node.i_db[0]), sblock.fs_fsize, dbuf);
	iput(&node);
}

/*
 * construct a set of directory entries in "dbuf".
 * return size of directory.
 */
makedir(protodir, entries)
	register struct direct *protodir;
	int entries;
{
	char *cp;
	int i, spcleft;

	spcleft = DIRBLKSIZ;
	for (cp = dbuf, i = 0; i < entries - 1; i++) {
		protodir[i].d_reclen = sizeof(struct direct);
		/* DIRSIZ(&protodir[i]); */
		memcpy(cp, &protodir[i],  protodir[i].d_reclen);
		cp += protodir[i].d_reclen;
		spcleft -= protodir[i].d_reclen;
	}
	protodir[i].d_reclen = spcleft;
	memcpy(cp, &protodir[i], sizeof(struct direct));  /* DIRSIZ(&protodir[i])); */
	cp += sizeof(struct direct);  /* DIRSIZ(&protodir[i]); */
	return (cp - dbuf);
}

/*
 * allocate a block or frag
 */
daddr_t
alloc(size, mode)
	int size;
	int mode;
{
	int i, frag;
	daddr_t d;

	rdfs(fsbtodb(&sblock, cgtod(&sblock, 0)), sblock.fs_cgsize,
	    (char *)&acg);
	if (acg.cg_magic != CG_MAGIC) {
		printf("cg 0: bad magic number\n");
		return (0);
	}
	if (acg.cg_cs.cs_nbfree == 0) {
		printf("first cylinder group ran out of space\n");
		return (0);
	}
	for (d = 0; d < acg.cg_ndblk; d += sblock.fs_frag)
		if (isblock(&sblock, acg.cg_free, d / sblock.fs_frag))
			goto goth;
	printf("internal error: can't find block in cyl 0\n");
	return (0);
goth:
	clrblock(&sblock, acg.cg_free, d / sblock.fs_frag);
	acg.cg_cs.cs_nbfree--;
	sblock.fs_cstotal.cs_nbfree--;
	fscs[0].cs_nbfree--;
	if (mode & IFDIR) {
		acg.cg_cs.cs_ndir++;
		sblock.fs_cstotal.cs_ndir++;
		fscs[0].cs_ndir++;
	}
	acg.cg_btot[cbtocylno(&sblock, d)]--;
	acg.cg_b[cbtocylno(&sblock, d)][cbtorpos(&sblock, d)]--;
	if (size != sblock.fs_bsize) {
		frag = howmany(size, sblock.fs_fsize);
		fscs[0].cs_nffree += sblock.fs_frag - frag;
		sblock.fs_cstotal.cs_nffree += sblock.fs_frag - frag;
		acg.cg_cs.cs_nffree += sblock.fs_frag - frag;
		acg.cg_frsum[sblock.fs_frag - frag]++;
		for (i = frag; i < sblock.fs_frag; i++)
			setbit(acg.cg_free, d + i);
	}
	wtfs(fsbtodb(&sblock, cgtod(&sblock, 0)), sblock.fs_cgsize,
	    (char *)&acg);
	return (d);
}

/*
 * Allocate an inode on the disk
 */
iput(ip)
	register struct inode *ip;
{
	struct dinode buf[MAXINOPB];
	daddr_t d;
	int c;

	c = itog(&sblock, ip->i_number);
	rdfs(fsbtodb(&sblock, cgtod(&sblock, 0)), sblock.fs_cgsize,
	    (char *)&acg);
	if (acg.cg_magic != CG_MAGIC) {
		printf("cg 0: bad magic number\n");
		exit(1);
	}
	acg.cg_cs.cs_nifree--;
	setbit(acg.cg_iused, ip->i_number);
	wtfs(fsbtodb(&sblock, cgtod(&sblock, 0)), sblock.fs_cgsize,
	    (char *)&acg);
	sblock.fs_cstotal.cs_nifree--;
	fscs[0].cs_nifree--;
	if(ip->i_number >= sblock.fs_ipg * sblock.fs_ncg) {
		printf("fsinit: inode value out of range (%d).\n",
		    ip->i_number);
		exit(1);
	}
	d = fsbtodb(&sblock, itod(&sblock, ip->i_number));
	rdfs(d, sblock.fs_bsize, buf);
	buf[itoo(&sblock, ip->i_number)].di_ic = ip->i_ic;
	wtfs(d, sblock.fs_bsize, buf);
}

/*
 * read a block from the file system
 */
rdfs(bno, size, bf)
	daddr_t bno;
	int size;
	char *bf;
{
	int n;

	if (lseek(fsi, bno * DEV_BSIZE, 0) < 0) {
		printf("seek error: %ld\n", bno);
		perror("rdfs");
		exit(1);
	}
	n = read(fsi, bf, size);
	if(n != size) {
		printf("read error: %ld\n", bno);
		perror("rdfs");
		exit(1);
	}
}

/*
 * write a block to the file system
 */
wtfs(bno, size, bf)
	daddr_t bno;
	int size;
	char *bf;
{
	int n;

	if (lseek(fso, bno * DEV_BSIZE, 0) < 0) {
		printf("seek error: %ld\n", bno);
		perror("wtfs");
		exit(1);
	}
	n = write(fso, bf, size);
	if(n != size) {
		printf("write error: %ld\n", bno);
		perror("wtfs");
		exit(1);
	}
}

/*
 * check if a block is available
 */
isblock(fs, cp, h)
	struct fs *fs;
	unsigned char *cp;
	int h;
{
	unsigned char mask;

	switch (fs->fs_frag) {
	case 8:
		return (cp[h] == 0xff);
	case 4:
		mask = 0x0f << ((h & 0x1) << 2);
		return ((cp[h >> 1] & mask) == mask);
	case 2:
		mask = 0x03 << ((h & 0x3) << 1);
		return ((cp[h >> 2] & mask) == mask);
	case 1:
		mask = 0x01 << (h & 0x7);
		return ((cp[h >> 3] & mask) == mask);
	default:
#ifdef STANDALONE
		printf("isblock bad fs_frag %d\n", fs->fs_frag);
#else
		fprintf(stderr, "isblock bad fs_frag %d\n", fs->fs_frag);
#endif
		return;
	}
}

/*
 * take a block out of the map
 */
clrblock(fs, cp, h)
	struct fs *fs;
	unsigned char *cp;
	int h;
{
	switch ((fs)->fs_frag) {
	case 8:
		cp[h] = 0;
		return;
	case 4:
		cp[h >> 1] &= ~(0x0f << ((h & 0x1) << 2));
		return;
	case 2:
		cp[h >> 2] &= ~(0x03 << ((h & 0x3) << 1));
		return;
	case 1:
		cp[h >> 3] &= ~(0x01 << (h & 0x7));
		return;
	default:
#ifdef STANDALONE
		printf("clrblock bad fs_frag %d\n", fs->fs_frag);
#else
		fprintf(stderr, "clrblock bad fs_frag %d\n", fs->fs_frag);
#endif
		return;
	}
}

/*
 * put a block into the map
 */
setblock(fs, cp, h)
	struct fs *fs;
	unsigned char *cp;
	int h;
{
	switch (fs->fs_frag) {
	case 8:
		cp[h] = 0xff;
		return;
	case 4:
		cp[h >> 1] |= (0x0f << ((h & 0x1) << 2));
		return;
	case 2:
		cp[h >> 2] |= (0x03 << ((h & 0x3) << 1));
		return;
	case 1:
		cp[h >> 3] |= (0x01 << (h & 0x7));
		return;
	default:
#ifdef STANDALONE
		printf("setblock bad fs_frag %d\n", fs->fs_frag);
#else
		fprintf(stderr, "setblock bad fs_frag %d\n", fs->fs_frag);
#endif
		return;
	}
}
#ifndef WKSTN
descend(par)
	struct inode *par;
{
	struct inode in;
	int ibc = 0;
	int i, f, c, tmp;
	struct dinode *dip, inos[MAXBSIZE / sizeof (struct dinode)];
	daddr_t ib[MAXBSIZE / sizeof (daddr_t)];
	char buf[MAXBSIZE];
	int entries = (DEV_BSIZE / sizeof (daddr_t));

	getstr();
	in.i_mode = gmode(token[0], "-bcd", IFREG, IFBLK, IFCHR, IFDIR);
	in.i_mode |= gmode(token[1], "-u", 0, ISUID, 0, 0);
	in.i_mode |= gmode(token[2], "-g", 0, ISGID, 0, 0);
	for (i = 3; i < 6; i++) {
		c = token[i];
		if (c < '0' || c > '7') {
			printf("%c/%s: bad octal mode digit\n", c, token);
			errs++;
			c = 0;
		}
		in.i_mode |= (c-'0')<<(15-3*i);
	}
	in.i_uid = (u_short)getnum(); in.i_gid = (u_short)getnum();
	for (i = 0; i < fs->fs_bsize; i++)
		buf[i] = 0;
	for (i = 0; i < NINDIR(fs); i++)
		ib[i] = (daddr_t)0;
	in.i_nlink = 1;
	in.i_size = 0;
	for (i = 0; i < NDADDR; i++)
		in.i_db[i] = (daddr_t)0;
	for (i = 0; i < NIADDR; i++)
		in.i_ib[i] = (daddr_t)0;
	if (par != (struct inode *)0) {
		ialloc(&in);
	} else {
		par = &in;
		i = itod(fs, ROOTINO);
		rdfs(fsbtodb(fs, i), fs->fs_bsize, (char *)inos);
		dip = &inos[ROOTINO % INOPB(fs)];
		in.i_number = ROOTINO;
		in.i_nlink = dip->di_nlink;
		in.i_size = dip->di_size;
		in.i_db[0] = dip->di_db[0];
		rdfs(fsbtodb(fs, in.i_db[0]), fs->fs_bsize, buf);
	}

	switch (in.i_mode&IFMT) {

	case IFREG:
		getstr();
		f = open(token, 0);
		if (f < 0) {
			printf("%s: cannot open\n", token);
			errs++;
			break;
		}
		while ((i = read(f, buf, (int)fs->fs_bsize)) > 0) {
			in.i_size += i;
			newblk(buf, &ibc, ib, (int)blksize(fs, &in, ibc));
		}
		if (ibc <= NDADDR)
			in.i_blocks = btodb( fragroundup(fs,in.i_size) );
		else {
			tmp = ibc - NDADDR;
			ibc++;
			if ( tmp > entries ) {
				tmp -= entries;
				ibc += ((tmp + entries -1) / entries) + 1;
			}
			if ( tmp > entries * entries ) {
				tmp -= entries*entries;
				i = (tmp+(entries*entries))-1 / entries*entries;
				ibc += i + (i+entries-1)/entries + 1;
			}
			in.i_blocks = ibc * NSPB(fs);
		}
		close(f);
		break;

	case IFBLK:
	case IFCHR:
		/*
		 * special file
		 * content is maj/min types
		 */
		i = getnum() & 0xffffff;
		f = getnum() & 0xffffff;
		in.i_rdev = makedev(i,f);
		in.i_blocks = 0;
		break;

	case IFDIR:
		/*
		 * directory
		 * put in extra links
		 * call recursively until
		 * name of "$" found
		 */

		if (in.i_number != ROOTINO) {
			par->i_nlink++;
			in.i_nlink++;
			entry(&in, in.i_number, ".", buf);
			entry(&in, par->i_number, "..", buf);
		}
		for (;;) {
			getstr();
			if (token[0]=='$' && token[1]=='\0')
				break;
			entry(&in, (ino_t)(ino+1), token, buf);
			descend(&in);
		}
		in.i_blocks = btodb (fragroundup(fs, in.i_size));
		if (in.i_number != ROOTINO)
			newblk(buf, &ibc, ib, (int)blksize(fs, &in, 0));
		else
			wtfs(fsbtodb(fs, in.i_db[0]), (int)fs->fs_bsize, buf);
		break;
	}
	proto_iput(&in, &ibc, ib);
}

/*ARGSUSED*/
gmode(c, s, m0, m1, m2, m3)
	char c, *s;
{
	int i;

	if ( c == s[0] )
		return(m0);
	else if ( c == s[1] )
		return(m1);
	else if ( c == s[2] )
		return(m2);
	else if ( c == s[3] )
		return(m3);
	else {
		printf("%c/%s: bad mode\n", c, token);
		errs++;
		return(0);
	}
}

long
getnum()
{
	int i, c;
	long n;

	getstr();
	n = 0;
	i = 0;
	for (i = 0; c=token[i]; i++) {
		if (c<'0' || c>'9') {
			printf("%s: bad number\n", token);
			errs++;
			return((long)0);
		}
		n = n*10 + (c-'0');
	}
	return(n);
}

getstr()
{
	int i, c;

loop:
	switch (c = getc(fin)) {

	case ' ':
	case '\t':
	case '\n':
		goto loop;

	case EOF:
		printf("Unexpected EOF\n");
		exit(1);

	case ':':
		while (getc(fin) != '\n')
			;
		goto loop;

	}
	i = 0;
	do {
		token[i++] = c;
		c = getc(fin);
	} while (c != ' ' && c != '\t' && c != '\n' && c != '\0');
	token[i] = 0;
}

entry(ip, inum, str, buf)
	struct inode *ip;
	ino_t inum;
	char *str;
	char *buf;
{
	register struct direct *dp, *odp;
	int i, oldsize, newsize, spacefree;

	odp = dp = (struct direct *)buf;
	while ((int)dp - (int)buf < ip->i_size) {
		odp = dp;
		dp = (struct direct *)((int)dp + dp->d_reclen);
	}
	if (odp != dp)
		oldsize = sizeof(struct direct);
	else
		oldsize = 0;
	spacefree = odp->d_reclen - oldsize;
	dp = (struct direct *)((int)odp + oldsize);
	dp->d_ino = inum;
	dp->d_namlen = (strlen(str) > DIRSIZ ? DIRSIZ : strlen(str));
	for (i=0; i<PADSIZ; i++)
		dp->d_pad[i] = 0;
	newsize = sizeof(struct direct);
	if (spacefree >= newsize) {
		odp->d_reclen = oldsize;
		dp->d_reclen = spacefree;
	} else {
		dp = (struct direct *)((int)odp + odp->d_reclen);
		if ((int)dp - (int)buf >= fs->fs_bsize) {
			printf("directory too large\n");
			exit(1);
		}
		dp->d_ino = inum;
		dp->d_namlen = (strlen(str) > DIRSIZ ? DIRSIZ : strlen(str));
		dp->d_reclen = DIRBLKSIZ;
	}
	if (strlen(str) > DIRSIZ)
		strncpy(dp->d_name, str, DIRSIZ) ;
	else
	{
		strncpy( dp->d_name, str, strlen(str)) ;
		for (i=strlen(str)+1; i<DIRSIZ; i++)
			dp->d_name[i] = 0;
	}
	for (i=0; i<PADSIZ; i++)
		dp->d_pad[i] = 0;
	ip->i_size = (int)dp - (int)buf + newsize;
}

newblk(buf, aibc, ib, size)
	int *aibc;
	char *buf;
	daddr_t *ib;
	int size;
{
	int i;
	daddr_t bno;

	bno = proto_alloc(size);
	wtfs(fsbtodb(fs, bno), (int)fs->fs_bsize, buf);
	for (i = 0; i < fs->fs_bsize; i++)
		buf[i] = 0;
	ib[(*aibc)++] = bno;
	if (*aibc >= NINDIR(fs)) {
		printf("indirect block full\n");
		errs++;
		*aibc = 0;
	}
}

proto_iput(ip, aibc, ib)
	struct inode *ip;
	int *aibc;
	daddr_t *ib;
{
	daddr_t d;
	int i;
	struct dinode buf[MAXBSIZE / sizeof (struct dinode)];

	ip->i_atime = ip->i_mtime = ip->i_ctime = time((long *)0);
	switch (ip->i_mode&IFMT) {

	case IFDIR:
	case IFREG:
		for (i = 0; i < *aibc; i++) {
			if (i >= NDADDR)
				break;
			ip->i_db[i] = ib[i];
		}
		if (*aibc > NDADDR) {
			ip->i_ib[0] = proto_alloc((int)fs->fs_bsize);
			for (i = 0; i < NINDIR(fs) - NDADDR; i++) {
				ib[i] = ib[i+NDADDR];
				ib[i+NDADDR] = (daddr_t)0;
			}
			wtfs(fsbtodb(fs, ip->i_ib[0]),
			    (int)fs->fs_bsize, (char *)ib);
		}
		break;

	case IFBLK:
	case IFCHR:
		break;

	default:
		printf("bad mode %o\n", ip->i_mode);
		exit(1);
	}
	d = fsbtodb(fs, itod(fs, ip->i_number));
	rdfs(d, (int)fs->fs_bsize, (char *)buf);
	buf[itoo(fs, ip->i_number)].di_ic = ip->i_ic;
	wtfs(d, (int)fs->fs_bsize, (char *)buf);
}

daddr_t
proto_alloc(size)
	int size;
{
	int i, frag;
	daddr_t d;
	static int cg = 0;

again:
#ifdef DEBUG
printf("proto_alloc: cg=%d, cgtod=%ld\n", cg, cgtod(&sblock, cg));
#endif
	rdfs(fsbtodb(&sblock, cgtod(&sblock, cg)), (int)sblock.fs_cgsize,
	    (char *)&acg);
#ifdef DEBUG
printf("proto_alloc: fsbtodb:cgtod=%ld\n", fsbtodb(&sblock, cgtod(&sblock, cg)));
#endif
	if (acg.cg_magic != CG_MAGIC) {
		printf("cg %d: bad magic number\n", cg);
		return (0);
	}
	if (acg.cg_cs.cs_nbfree == 0) {
		cg++;
		if (cg >= fs->fs_ncg) {
			printf("ran out of space\n");
			return (0);
		}
		goto again;
	}
	for (d = 0; d < acg.cg_ndblk; d += sblock.fs_frag)
		if (isblock(&sblock, (u_char *)acg.cg_free, d / sblock.fs_frag))
			goto goth;
	printf("internal error: can't find block in cyl %d\n", cg);
	return (0);
goth:
	clrblock(&sblock, (u_char *)acg.cg_free, d / sblock.fs_frag);
	acg.cg_cs.cs_nbfree--;
	sblock.fs_cstotal.cs_nbfree--;
	fscs[cg].cs_nbfree--;
	acg.cg_btot[cbtocylno(&sblock, d)]--;
	acg.cg_b[cbtocylno(&sblock, d)][cbtorpos(&sblock, d)]--;
	if (size != sblock.fs_bsize) {
		frag = howmany(size, sblock.fs_fsize);
		fscs[cg].cs_nffree += sblock.fs_frag - frag;
		sblock.fs_cstotal.cs_nffree += sblock.fs_frag - frag;
		acg.cg_cs.cs_nffree += sblock.fs_frag - frag;
		acg.cg_frsum[sblock.fs_frag - frag]++;
		for (i = frag; i < sblock.fs_frag; i++)
			setbit(acg.cg_free, d + i);
	}
	wtfs(fsbtodb(&sblock, cgtod(&sblock, cg)), (int)sblock.fs_cgsize,
	    (char *)&acg);
	return (acg.cg_cgx * fs->fs_fpg + d);
}

/*
 * Allocate an inode on the disk
 */
ialloc(ip)
	register struct inode *ip;
{
	struct dinode buf[MAXBSIZE / sizeof (struct dinode)];
	daddr_t d;
	int c;

	ip->i_number = ++ino;
	c = itog(&sblock, ip->i_number);
	rdfs(fsbtodb(&sblock, cgtod(&sblock, c)), (int)sblock.fs_cgsize,
	    (char *)&acg);
	if (acg.cg_magic != CG_MAGIC) {
		printf("cg %d: bad magic number\n", c);
		exit(1);
	}
	if (ip->i_mode & IFDIR) {
		acg.cg_cs.cs_ndir++;
		sblock.fs_cstotal.cs_ndir++;
		fscs[c].cs_ndir++;
	}
	acg.cg_cs.cs_nifree--;
	setbit(acg.cg_iused, ip->i_number);
	wtfs(fsbtodb(&sblock, cgtod(&sblock, c)), (int)sblock.fs_cgsize,
	    (char *)&acg);
	sblock.fs_cstotal.cs_nifree--;
	fscs[c].cs_nifree--;
	if(ip->i_number >= sblock.fs_ipg * sblock.fs_ncg) {
		printf("fsinit: inode value out of range (%d).\n",
		    ip->i_number);
		exit(1);
	}
	return (ip->i_number);
}
#endif not WKSTN


