static char *version = "@(#) $Revision: 3.2 $   $Date: 89/02/27 11:16:10 $";
/*
 * Print source code size metrics report.
 * See the manual entry for details.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <varargs.h>

char	*malloc();
char	*realloc();

#define	PROC				/* null; easy to find procs */
#define	FALSE	  0
#define	TRUE	  1
#define	CHNULL	  ('\0')
#define	CHNEWLINE ('\n')
#define	CPNULL	  ((char *) NULL)
#define	FILENULL  ((FILE *) NULL)
#define RCSSUFF	  ",v"
#define	REG	  register
#define DEFREL	  "0"


/*********************************************************************
 * USAGE AND INVOCATION:
 */

char *usage[] = {
    "usage: %s [-ab] [-n newrel] [-o oldrel] [-p unchanged,touched]",
    "       [-s command] [-S] [files...]",
    "-a lines purely added to r files accrue under TOUCHED (default: NEW)",
    "-b ignore whitespace in r files (diff -b, treat as a single blank)",
    "-n global default new release (and optional level) number (default: 0)",
    "-o global default old release (and optional level) number (default: 0)",
    "-p percentages for UNCHANGED and TOUCHED for u files (default: 100,0)",
    "-s command to use to strip reused sources before comparing",
    "-S strip using cstrip, pstrip, or shstrip according to file basename",
    CPNULL,
};

char	*myname;			/* how program was invoked	*/

int	aflag = FALSE;			/* -a (added lines) option	*/
int	bflag = FALSE;			/* -b (ignore whitespace) opt	*/
int	Sflag = FALSE;			/* -S (auto-strip) option	*/

char	*newrel = DEFREL;		/* from -n (newrel) option	*/
char	*oldrel = DEFREL;		/* from -o (oldrel) option	*/

float	percent[] = { 1, 0, 0 };	/* values from -p option / 100	*/

#define	PCT_UNCHANGED	0		/* indices for percent[]	*/
#define	PCT_TOUCHED	1
#define	PCT_NEW		2

char	*scommand = CPNULL;		/* from -s (strip command) opt	*/


/*********************************************************************
 * MISCELLANEOUS GLOBAL VALUES:
 */

char	*defargv[] = { "-" };		/* default argument list	*/

char	*oldtempfile = "/tmp/csoldXXXXXX";  /* temporary file versions	*/
char	*newtempfile = "/tmp/csnewXXXXXX";

char	*cstrip  = "cstrip";		/* C source code stripper	*/
char	*pstrip  = "pstrip";		/* Pascal source code stripper	*/
char	*shstrip = "shstrip";		/* shell script stripper	*/

#define	CMD  0				/* for calling RunCommand()	*/
#define	PIPE 1

char	*PickSCommand();
long	RunCommand();
long	CountLines();
int	ExitNicely();


/*********************************************************************
 * TOTALS DATA STRUCTURE:
 *
 * The array accrues total line counts for printing.
 * The (extra) last row is used for grand totals.
 */

#define	ROW_NEW		0		/* row indices */
#define	ROW_UNCHANGED	1
#define	ROW_ADDED	2
#define	ROW_DELETED	3
#define	ROW_MODIFIED	4
#define	ROW_UNKNOWN	5
#define	ROW_TOTAL	6
#define	ROWMAX		ROW_TOTAL

#define	COL_UNCHANGED	0		/* column indices */
#define	COL_TOUCHED	1
#define	COL_NEW		2
#define	COLMAX		COL_NEW

long	total [ROWMAX + 1] [COLMAX + 1];	/* rows, columns */


/*********************************************************************
 * MACRO PROCEDURES:
 *
 * Basename is either the full string or past last '/', if any.
 * File is SCCS if basename starts with "s.".
 */

char	*cpBNTemp;			/* temporary */

#define	BASENAME(cp)	\
	(((cpBNTemp = strrchr (cp, '/')) == CPNULL) ? cp : (cpBNTemp + 1))

#define	SCCS(filename)	(strncmp (BASENAME (filename), "s.", 2) == 0)


/************************************************************************
 * M A I N
 *
 * Parse arguments, initialize, read data, and print report.
 */

PROC main (argc, argv)
REG	int	argc;
REG	char	**argv;
{
extern	int	optind;			/* from getopt()	*/
extern	char	*optarg;		/* from getopt()	*/
REG	int	option;			/* option "letter"	*/
REG	FILE	*filep;			/* open input file	*/
REG	char	*filename;		/* name to use		*/

/*
 * PARSE ARGUMENTS:
 */

	myname = *argv;

	while ((option = getopt (argc, argv, "abn:o:p:s:S")) != EOF)
	{
	    switch (option)
	    {
	    case 'a':	aflag = TRUE;		break;
	    case 'b':	bflag = TRUE;		break;
	    case 'S':	Sflag = TRUE;		break;

	    case 'n':	newrel	 = optarg;	break;
	    case 'o':	oldrel	 = optarg;	break;
	    case 's':	scommand = optarg;	break;

	    case 'p':	GetPercents (optarg);	break;

	    default:	Usage();
	    }
	}

	if ((scommand != CPNULL) && Sflag)
	    Error ("both -s and -S not allowed at same time");

	argc -= optind;			/* skip options	*/
	argv += optind;

/*
 * PREPARE TEMPORARY FILES AND SIGNALS:
 *
 * Temporary file names are initialized here, but the files themselves are
 * created when first needed.  Set signals IF they're not already ignored.
 */

	mktemp (oldtempfile);
	mktemp (newtempfile);

	if (signal (SIGHUP,  SIG_IGN) != SIG_IGN)
	    signal (SIGHUP, ExitNicely);

	if (signal (SIGINT,  SIG_IGN) != SIG_IGN)
	    signal (SIGINT, ExitNicely);

	if (signal (SIGQUIT,  SIG_IGN) != SIG_IGN)
	    signal (SIGQUIT, ExitNicely);

	signal (SIGTERM, ExitNicely);		/* unconditional */

/*
 * READ FROM LIST OF FILES OR STDIN:
 */

	if (argc < 1)				/* no file names */
	{
	    argc = 1;
	    argv = defargv;
	}

	while (argc-- > 0)
	{
	    if (strcmp (*argv, "-") == 0)	/* read stdin */
	    {
		filename = "<stdin>";
		filep	 = stdin;
	    }
	    else if ((filep = fopen ((filename = *argv), "r")) == FILENULL)
		Error ("can't open \"%s\" to read it", filename);

	    ReadFile (filename, filep);

	    if (filep != stdin)
		fclose (filep);			/* assume it works */

	    argv++;
	}

/*
 * PRINT REPORT:
 */

	PrintReport();
	ExitNicely (0);

} /* main */


/************************************************************************
 * U S A G E
 *
 * Print usage messages (char *usage[]) to stderr and exit nonzero.
 * Each message is followed by a newline.
 */

PROC Usage()
{
REG	int	which = 0;		/* current line */

	while (usage [which] != CPNULL)
	{
	    fprintf (stderr, usage [which++], myname);
	    fputc   (CHNEWLINE, stderr);
	}

	ExitNicely (1);

} /* Usage */


/************************************************************************
 * E R R O R
 *
 * Print an error message to stderr and exit nonzero.  Message is preceded
 * by "<myname>: " using global char *myname, and followed by a newline.
 */

/* VARARGS */
PROC Error (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

	ExitNicely (1);

} /* Error */


/************************************************************************
 * W A R N
 *
 * Print a warning message to stderr.  Message is preceded
 * by "<myname>: " using global char *myname, and followed by a newline.
 */

/* VARARGS */
PROC Warn (message, arg1, arg2, arg3, arg4)
	char	*message;
	int	arg1, arg2, arg3, arg4;
{
	fprintf (stderr, "%s: warning: ", myname);
	fprintf (stderr, message, arg1, arg2, arg3, arg4);
	fputc   (CHNEWLINE, stderr);

} /* Warn */


/************************************************************************
 * E X I T   N I C E L Y
 *
 * Remove temporary files if any (ignore errors), and exit with a given
 * value (maybe a signal number if called due to a signal).
 */

PROC ExitNicely (exitval)
	int	exitval;
{
	unlink (oldtempfile);
	unlink (newtempfile);

	exit (exitval);

} /* ExitNicely */


PROC int RCS(fname)
	char	*fname;
{
	int	rc = FALSE;

	if (strlen(fname) > 1)
	    if (strcmp(fname + strlen(fname) - 2, RCSSUFF) == 0)
		rc = TRUE;

	return(rc);
}


/************************************************************************
 * G E T   P E R C E N T S
 *
 * Given a string which should contain "<number>,<number>", being two non-
 * negative integers adding to less than 101, set the values in global
 * float percent[] according to the given values.  Does not insure that the
 * integers are valid, only that comma is present and atoi() return values
 * don't add to more than 100.  Converts the comma to a CHNULL.
 */

PROC GetPercents (string)
	char	*string;	/* string to parse */
{
REG	char	*cp = string;	/* place in string */
	float	sum;		/* of first two	   */

	while (*cp != ',')
	    if (*cp++ == CHNULL)	/* no comma */
		Error ("missing comma in -p percentage values");

	*cp = CHNULL;

	percent [PCT_UNCHANGED] = atoi (string) / 100.0;  /* first number   */
	percent [PCT_TOUCHED  ] = atoi (cp + 1) / 100.0;  /* rest of string */

	if ((sum = (percent [PCT_UNCHANGED] + percent [PCT_TOUCHED])) > 1)
	    Error ("-p percentages add to more than 100");

	percent [PCT_NEW] = 1 - sum;

} /* GetPercents */


/************************************************************************
 * R E A D   F I L E
 *
 * Given a filename (for error messages) and an open stream pointer, read
 * data lines from the stream until exhausted, accumulating statistics for
 * each one, then check for error.
 *
 * char line[] is static so it's only allocated once.
 */

PROC ReadFile (filename, filep)
	char	*filename;	/* for errors	*/
REG	FILE	*filep;		/* open stream	*/
{
static	char	line [BUFSIZ];	/* input line	*/
REG	char	*cp;		/* pos in line	*/
REG	int	linenum = 0;	/* in file	*/

	while (fgets (line, BUFSIZ, filep) != CPNULL)
	{
	    if ((cp = strchr (line, CHNEWLINE)) != CPNULL)
		*cp = CHNULL;			/* terminate at newline */

	    DoInputLine (filename, ++linenum, line);
	}

	if (ferror (filep))
	    Error ("read failed from \"%s\"", filename);

} /* ReadFile */


/************************************************************************
 * D O   I N P U T   L I N E
 *
 * Given a filename and linenumber (for error messages) and an input line,
 * parse the line and accumulate statistics for one file.  Error out if
 * anything is noticeably wrong with the line.
 */

PROC DoInputLine (filename, linenum, line)
	char	*filename;		/* for errors	*/
	int	linenum;		/* for errors	*/
	char	*line;			/* to parse	*/
{
REG	char	*tokensep = " \t";	/* token separator	*/
	char	*ltp;			/* linetype pointer	*/
REG	char	linetype;		/* 'n' | 'r' | 'u'	*/
	char	*dofilename;		/* file to size		*/
	char	*qual1, *qual2;		/* fields 3 and 4	*/

/*
 * PARSE LINE INTO TOKENS:
 */

	if (((ltp = strtok (line, tokensep)) == CPNULL)
	 || (((linetype = *ltp) != 'n')
	  && (linetype != 'r') && (linetype != 'u')))
	{
	    Warn ("missing or invalid type-letter in file \"%s\", line %d",
		   filename, linenum);
	}

	else if ((dofilename = strtok (CPNULL, tokensep)) == CPNULL)
	    Warn ("missing filename in file \"%s\", line %d",
		filename, linenum);

	else if (((qual1 = strtok (CPNULL, tokensep)) != CPNULL)
	 && ((qual2 = strtok (CPNULL, tokensep)) != CPNULL)
	 && ((linetype != 'r') || (strtok (CPNULL, tokensep) != CPNULL)))
	{
	    Warn ("too many tokens in file \"%s\", line %d",
		filename, linenum);
	}

/*
 * HANDLE DIFFERENT LINE TYPES:
 */

	else if (linetype == 'r')
	    DoReused (filename, linenum, dofilename, qual1, qual2);
	else
	    DoNewUnknown (linetype, dofilename, qual1);

} /* DoInputLine */


/************************************************************************
 * D O   N E W   U N K N O W N
 *
 * Given a line type, a filename to size, and a qualifier string (may be
 * null), accumulate statistics for one new or reused-unknown file,
 * possibly comment-stripped first, into global long total[], possibly
 * using global float percent[].
 */

PROC DoNewUnknown (linetype, dofilename, qual1)
	char	linetype;	/* 'n' | 'u'	*/
REG	char	*dofilename;	/* file to size	*/
	char	*qual1;		/* qualifier	*/
{
REG	long	lines;		/* in the file	*/
	char	*scmd = PickSCommand (dofilename);

/*
 * COUNT LINES IN FILE (retrieve files from SCCS or RCS as necessary).:
 */

	if (SCCS (dofilename))
	{
	    GetSccsFile (newtempfile, dofilename, qual1, newrel);
	    dofilename = newtempfile;
	}
	else if (RCS (dofilename))
	{
	    GetRcsFile (newtempfile, dofilename, qual1, newrel);
	    dofilename = newtempfile;
	}
	else if (scmd != CPNULL)		/* need to strip file */
	{
	    StripFile (scmd, dofilename, newtempfile);
	    dofilename = newtempfile;
	}

	lines = CountLines (dofilename);

/*
 * SAVE LINE COUNT IN APPROPRIATE PLACE:
 */

	if (linetype == 'n')			/* new file */
	    (total [ROW_NEW] [COL_NEW]) += lines;
	else					/* unknown, distribute count */
	{
	    (total [ROW_UNKNOWN] [COL_UNCHANGED]) +=
		(long) ((lines * percent [PCT_UNCHANGED]) + 0.5);  /* round */

	    (total [ROW_UNKNOWN] [COL_TOUCHED]) +=
		(long) ((lines * percent [PCT_TOUCHED]) + 0.5);

	    (total [ROW_UNKNOWN] [COL_NEW]) +=
		(long) ((lines * percent [PCT_NEW]) + 0.5);
	}

} /* DoNewUnknown */


/************************************************************************
 * D O   R E U S E D
 *
 * Given a filename and linenumber (for error messages), a filename to size
 * and qualifier strings (second or both may be null), accumulate statistics
 * for two versions of one file into global long total[].  Warn if
 * anything goes wrong.
 */

PROC DoReused (filename, linenum, dofilename, qual1, qual2)
	char	*filename;	/* for errors	*/
	int	linenum;	/* for errors	*/
REG	char	*dofilename;	/* file to size	*/
	char	*qual1, *qual2;	/* qualifiers	*/
{
REG	char	*oldfilename;	/* files to diff */
REG	char	*newfilename;

	char	*scmd = PickSCommand (dofilename);
	int	onequal;	/* exactly one qualifier given? */
	int	error = FALSE;

	long	unchanged = 0;	/* line count	*/
	long	addpure	  = 0;	/* pure inserts	*/
	long	addmod	  = 0;	/* part of mods	*/
	long	deleted	  = 0;	/* line count	*/
	long	modified  = 0;	/* line count	*/

	onequal = ((qual1 != CPNULL) && (qual2 == CPNULL));

/*
 * SCCS FILE: PREPARE FILES AND FILENAMES:
 *
 * Files are also run through a strip command if needed.
 */

	if (SCCS (dofilename))
	{
	    if (onequal)
	    {
		Warn (
	    "wrong number of qualifiers for SCCS r file, file \"%s\", line %d",
			filename, linenum);
		error = TRUE;
	    }
	    else
	    {
		newfilename = newtempfile;
		oldfilename = oldtempfile;

		GetSccsFile (newfilename, dofilename, qual1, newrel);
		GetSccsFile (oldfilename, dofilename, qual2, oldrel);
	    }
	}

/*
 * RCS FILE: PREPARE FILES AND FILENAMES:
 *
 * Files are also run through a strip command if needed.
 */

	else if (RCS (dofilename))
	{
	    if (onequal)
	    {
		Warn (
	    "wrong number of qualifiers for RCS r file, file \"%s\", line %d",
			filename, linenum);
		error = TRUE;
	    }
	    else
	    {
		newfilename = newtempfile;
		oldfilename = oldtempfile;

		GetRcsFile (newfilename, dofilename, qual1, newrel);
		GetRcsFile (oldfilename, dofilename, qual2, oldrel);
	    }
	}

/*
 * NON-SCCS/RCS FILE: PREPARE FILES AND FILENAMES:
 *
 * Files are specially run through a strip command if needed.
 */

	else
	{
	    if (! onequal)
	    {
		Warn (
	"wrong number of qualifiers for non-SCCS r file, file \"%s\", line %d",
			filename, linenum);
		error = TRUE;      
	    }
	    else 
	    {
		if (scmd == CPNULL)			/* no strip command */
		{
		    newfilename = dofilename;	/* use given names */
		    oldfilename = qual1;
		}
		else				/* do strip files */
		{
		    StripFile (scmd, dofilename, (newfilename = newtempfile));
		    StripFile (scmd, qual1,      (oldfilename = oldtempfile));
		}
	    }
	}

	/* now newfilename and oldfilename are set */

	if (! error) {
/*
 * COMPARE OLD AND NEW FILES:
 */

	    CountDiffs (oldfilename, newfilename,
			& addpure, & addmod, & deleted, & modified);

	    unchanged = CountLines (oldfilename) - deleted - modified;

/*
 * SAVE STATISTICS IN APPROPRIATE PLACES:
 */

	    (total [ROW_UNCHANGED] [COL_UNCHANGED])			+= unchanged;
	    (total [ROW_ADDED    ] [aflag ? COL_TOUCHED : COL_NEW])	+= addpure;
	    (total [ROW_ADDED    ] [COL_TOUCHED])			+= addmod;
	    (total [ROW_DELETED  ] [COL_TOUCHED])			+= deleted;
	    (total [ROW_MODIFIED ] [COL_TOUCHED])			+= modified;
	}

} /* DoReused */


/************************************************************************
 * G E T   S C C S   F I L E
 *
 * Given a temporary filename, an SCCS filename (pathname), a release [and
 * level] number, and a default number to use if null, do a get(1) and
 * possible comment strip of the file into the temp file, possibly at the
 * top of the given (or default) release/level.  Terminate if the get fails.
 */

PROC GetSccsFile (tempfilename, sccsfilename, release, defrel)
REG	char	*tempfilename;	/* where to write data	*/
REG	char	*sccsfilename;	/* file to get		*/
REG	char	*release;	/* number or null	*/
	char	*defrel;	/* default release num	*/
{
	char	*scmd = PickSCommand (sccsfilename);
	char	*relopt;
REG	int	result;		/* of get command	*/

	if (release == CPNULL)			/* not given   */
	    release = defrel;			/* use default */


/*
 * SET UP RELEASE OPTION STRING.
 */

	if (atof (release) == 0)		/* no number in effect */
	{
	    relopt = malloc(5 + strlen(" ") + 1);
	    sprintf (relopt, "%.*s", 5 + strlen(" ") + 1, " ");
	}
	else					/* release number known */
	{
	    relopt = malloc(5 + strlen(release) + 1);
	    sprintf (relopt, "-t -r%.*s", 5 + strlen(release) + 1, release);
	}

	/* now result is set to the return from a command line */

/*
 * EXECUTE COMMAND, PIPING RESULTS THROUGH STRIPPER IF NECESSARY.
 */

	if (scmd == CPNULL)			/* no strip command */
	    result = RunCommand (CMD, "get -k -s -p", relopt, sccsfilename, 
				 " > ", tempfilename, CPNULL);
	else
	    result = RunCommand (CMD, "get -k -s -p", relopt, sccsfilename, 
				 "|", scmd, ">", tempfilename, CPNULL);

/*
 * CHECK RESULTS:
 */

	if (result)
	    Error ("get %sof file \"%s\" into file \"%s\" failed",
		  (scmd == CPNULL) ? "" : "and strip ",
		  sccsfilename, tempfilename);

} /* GetSccsFile */


/************************************************************************
 * G E T   R C S   F I L E
 *
 * Given a temporary filename, an RCS filename (pathname), a release [and
 * level] number, and a default number to use if null, do a co(1) and
 * possible comment strip of the file into the temp file, possibly at the
 * top of the given (or default) release/level.  Terminate if the co fails.
 */

PROC GetRcsFile (tempfilename, rcsfilename, release, defrel)
REG	char	*tempfilename;	/* where to write data	*/
REG	char	*rcsfilename;	/* file to get		*/
REG	char	*release;	/* number or null	*/
	char	*defrel;	/* default release num	*/
{
	char	*scmd;
	char	*tempname;
REG	int	result;		/* of get command	*/
   	char 	*relopt; 	/* option string for co(1)  */

	tempname = malloc(strlen(rcsfilename));
	strncpy(tempname, rcsfilename, (strlen(rcsfilename) - 2));
	tempname[strlen(rcsfilename) - 2] = CHNULL;
	scmd = PickSCommand(tempname);

/*
 * SET UP RELEASE OPTION STRING.
 */

	if (release != CPNULL)			/* release given */
	{
	    relopt = malloc(5 + strlen(release) + 1);
	    sprintf (relopt, "-r%.*s", 5 + strlen(release) + 1, release);
	}
	else if (strcmp(defrel, DEFREL) != 0)   /* default release given */
	{
	    relopt = malloc(5 + strlen(defrel) + 1);
	    sprintf (relopt, "-r%.*s", 5 + strlen(defrel) + 1, defrel);
	}
	else					/* use top level, i.e relopt is blank */
	{
	    relopt = malloc(5 + strlen(" ") + 1);
	    sprintf (relopt, "%.*s", 5 + strlen(" ") + 1, " ");
	}

/*
 * EXECUTE COMMAND, PIPING RESULTS THROUGH STRIPPER IF NECESSARY.
 */

	if (scmd == CPNULL)			/* no strip command */
	    result = RunCommand (CMD, "co  -q -p ", relopt, rcsfilename, 
				 " > ", tempfilename, CPNULL);
	else
	    result = RunCommand (CMD, "co  -q -p ", relopt, rcsfilename,
				 "|", scmd, ">", tempfilename, CPNULL);

/*
 * CHECK RESULTS:
 */

	if (result)
	    Error ("co %sof file \"%s\" into file \"%s\" failed",
		  (scmd == CPNULL) ? "" : "and strip ",
		  rcsfilename, tempfilename);

} /* GetRcsFile */


/************************************************************************
 * S T R I P   F I L E
 *
 * Given the names of a strip command (non-null) and source and temporary
 * files, strip the source file into the temp file using the command.
 * Error out if it fails.
 */

PROC StripFile (scommand, sourcefile, tempfile)
	char	*scommand;
	char	*sourcefile;
	char	*tempfile;
{
	if (RunCommand (CMD, scommand, "<", sourcefile, ">", tempfile, CPNULL))
	{
	    Error ("strip from file \"%s\" to file \"%s\" failed",
		    sourcefile, tempfile);
	}
} /* StripFile */


/************************************************************************
 * P I C K   S   C O M M A N D
 *
 * Given the name of a source file, and global char* scommand, cstrip, pstrip,
 * and shstrip, return the name of a strip command to use for the file, or
 * CPNULL if none is needed.
 *
 * Stripping is needed if scommand != CPNULL (a name was given by the user) or,
 * if Sflag is true, if the basename ends in ".c", ".h", ".p", or ".sh", or if
 * it contains no "."  and the file is executable (probably a shell script).
 */

PROC char * PickSCommand (filename)
	char	*filename;
{
	char	*basename;	/* of filename */
	int	length;		/* of basename */

	if (scommand != CPNULL)
	    return (scommand);		/* use given command */

	if (! Sflag)
	    return (CPNULL);		/* no stripping needed */

	length = strlen (basename = BASENAME (filename));

	if ((length >= 2)
	 && ((strcmp (basename + length - 2, ".c") == 0)
	  || (strcmp (basename + length - 2, ".h") == 0)))
	{
	    return (cstrip);		/* use C code stripper */
	}

	if ((length >= 2) && (strcmp (basename + length - 2, ".p") == 0))
	{
	    return (pstrip);		/* use Pascal code stripper */
	}

	if (((length >= 3) && (strcmp (basename + length - 3, ".sh") == 0))
	 || (strchr (basename, '.') == CPNULL))
	{
	    return (shstrip);		/* use shell script stripper */
	}

	return (CPNULL);		/* no stripping needed */

} /* PickSCommand */


/************************************************************************
 * C O U N T   L I N E S
 *
 * Given a filename, open the file and count and return how many lines
 * (newline characters) it contains.  Error out if open or read fails.
 */

PROC long CountLines (filename)
	char	*filename;	/* file to count  */
{
REG	FILE	*filep;		/* for opening it */
REG	char	ch;		/* read from file */
REG	long	lines = 0;	/* count of lines */

	if ((filep = fopen (filename, "r")) == FILENULL)
	    Error ("can't open file \"%s\" to count lines", filename);

	while ((ch = getc (filep)) != EOF)	/* count lines */
	    if (ch == CHNEWLINE)
		lines++;

	if (ferror (filep))
	    Error ("read from file \"%s\" failed", filename);

	fclose (filep);				/* assume it works */
	return (lines);

} /* CountLines */


/************************************************************************
 * C O U N T   D I F F S
 *
 * Given old and new filenames to compare, and pointers to four values
 * to return, diff the two files, analyze the results, and return values
 * through the pointers.  Error out if diff can't be run or it fails.
 */

PROC CountDiffs (oldfilename, newfilename,
		 addpurep, addmodp, deletedp, modifiedp)

	char	*oldfilename;	/* for diff		*/
	char	*newfilename;	/* for diff		*/
	long	*addpurep;	/* purely inserted	*/
	long	*addmodp;	/* inserted with modify	*/
	long	*deletedp;	/* deleted lines	*/
	long	*modifiedp;	/* modified lines	*/
{
REG	FILE	*pipep;		/* to read pipeline	*/
	int	result;		/* of pipeline		*/

	if ((pipep = (FILE *) RunCommand (PIPE, "diff", (bflag ? "-b" : ""),
					  oldfilename, newfilename, CPNULL))
	    == FILENULL)
	{
	    Error ("can't run diff on files \"%s\" and \"%s\"", 
		      oldfilename, newfilename);
	}

	CountDiffLines (pipep, addpurep, addmodp, deletedp, modifiedp);

	if (ferror (pipep))
	    Error ("read from diff command failed");

	if (((result = pclose (pipep)) < 0)	/* pclose() failed	  */
	 || (result && (result != (1 << 8))))	/* not exit(0) or exit(1) */
	{
	    Error ("diff of files \"%s\" and \"%s\" failed",
		   oldfilename, newfilename);
	}

} /* CountDiffs */


/************************************************************************
 * C O U N T   D I F F   L I N E S
 *
 * Given a read pipeline from a diff command, and pointers to four values
 * to return, analyze the diff output, and return values through the
 * pointers.  It's up to the caller to check for I/O errors.
 *
 * Each diff output line must begin with one of:
 *
 *	-	separator line (ignored)
 *	[0-9]	mode line, contains "a", "d", or "c"
 *		(meaning an add, delete, or change section follows)
 *	<	deletion
 *	>	insertion
 *
 * Lines in "delete" sections must start with "<"; they accrue to "del".
 * Lines in "add"    sections must start with ">"; they accrue to "add".
 * Lines in "change" sections start with either; they accrue accordingly.
 *
 * NOT VERY ROBUST:  Does not check diff output for validity.
 * Also, expects the first line to be a digit (mode) line.
 */

PROC CountDiffLines (pipep, addpurep, addmodp, deletedp, modifiedp)
REG	FILE	*pipep;		/* to read input	*/
	long	*addpurep;	/* purely inserted	*/
	long	*addmodp;	/* inserted with modify	*/
	long	*deletedp;	/* deleted lines	*/
	long	*modifiedp;	/* modified lines	*/
{
static	char	line [BUFSIZ];	/* only allocate once	*/
REG	char	chtype;		/* first char of line	*/
REG	char	chmode = '?';	/* of current section	*/
REG	int	add, del;	/* line counts		*/

/*
 * READ LINES UNTIL EOF:
 */

	while (fgets (line, BUFSIZ, pipep) != CPNULL)
	{
	    if ((chtype = line [0]) == '-')	/* separator line */
		continue;			/* ignore it	  */

/*
 * START OF NEW SECTION:
 */

	    if ((chtype >= '0') && (chtype <= '9'))
	    {
		REG char *cp = line;

		/* save previous values, if any (none the first time) */

		SaveDiffCounts (chmode, add, del,
				addpurep, addmodp, deletedp, modifiedp);

		add = del = 0;			/* for next section */

		do {				/* find new mode */
		    chmode = *(++cp);
		} while ((chmode != 'a') && (chmode != 'd') && (chmode != 'c'));

		continue;			/* next input line */
	    }

/*
 * ACCRUE CHANGE COUNTS:
 *
 * At this point we must have an insert (">") or delete ("<") line.
 * Assume that all lines in "add" sections are insertions and all in "delete"
 * sections are deletions.
 */

	    if ((chmode == 'a') || (chtype == '>'))
		add++;
	    else
		del++;

	} /* while */

/*
 * SAVE COUNTS FOR LAST SECTION:
 */

	SaveDiffCounts (chmode, add, del,
			addpurep, addmodp, deletedp, modifiedp);

} /* CountDiffLines */


/************************************************************************
 * S A V E   D I F F   C O U N T S
 *
 * Given a section mode letter, added and deleted line counts for the
 * section, and pointers to four totals values, accrue the line counts to
 * the totals as appropriate.  chmode may be invalid (when just starting
 * the first section), in which case, save nothing.
 *
 * Lines in "delete" sections accrue to deleted.
 * Lines in "add"    sections accrue to addpure.
 * Lines in "change" sections accrue min(del,add) to modified and the
 * excess to either addmod or deleted, as appropriate.
 */

PROC SaveDiffCounts (chmode, add, del, addpurep, addmodp, deletedp, modifiedp)
REG	char	chmode;		/* section mode letter	*/
REG	int	add;		/* lines added		*/
REG	int	del;		/* lines deleted	*/
	long	*addpurep;	/* purely inserted	*/
	long	*addmodp;	/* inserted with modify	*/
	long	*deletedp;	/* deleted lines	*/
	long	*modifiedp;	/* modified lines	*/
{
	if (chmode == 'a')		/* an add section */
	    *addpurep += add;

	else if (chmode == 'd')		/* a delete section */
	    *deletedp += del;

	else if (chmode == 'c')		/* a change section */
	{
	    if (del < add)		/* excess additions */
	    {
		*modifiedp += del;
		*addmodp   += add - del;
	    }
	    else			/* excess deletions (if any) */
	    {
		*modifiedp += add;
		*deletedp  += del - add;
	    }
	}

} /* SaveDiffCounts */


/************************************************************************
 * R U N   C O M M A N D
 *
 * Given a series of arguments terminated by CPNULL, build a command line
 * in malloc'd space starting with the second argument (type char *),
 * concatenated with blank separators.  Call the command line using
 * system(3) or popen(3) (according to the first argument, an int with
 * value CMD or PIPE), free the space, and return the return value from
 * system() (a wait(2) return value) or popen() (a FILE *).  In the popen()
 * case, the pipeline is left open for reading.
 *
 * HARDWARE DEPENDENT, might fail if sizeof (FILE *) > sizeof (long).
 */

/* VARARGS */
PROC long RunCommand (va_alist)
	va_dcl
{
REG	va_list	argp;		/* to traverse args	 */
REG	char	*cp;		/* one arg string	 */
	int	runtype;	/* type of call to do	 */
REG	char	*command;	/* command line to run	 */
REG	int	size = 1;	/* of cmd including null */
REG	int	newsize;	/* of new part		 */
	long	result;		/* of running command	 */

/*
 * INITIALIZE:
 */

	if ((command = malloc (size)) == CPNULL)
	    Error ("malloc failed");

	*command = CHNULL;

	va_start (argp);

	runtype = va_arg (argp, int);

/*
 * BUILD COMMAND LINE:
 *
 * Tack on each new part after a blank separator.
 */

	while ((cp = va_arg (argp, char *)) != CPNULL)
	{
	    if ((command =
		    realloc (command, size + 1 + (newsize = strlen (cp))))
		== CPNULL)
	    {
		Error ("realloc failed");
	    }

	    command [size - 1] = ' ';		/* separator	   */
	    strcpy (command + size, cp);	/* concat new part */
	    size += 1 + newsize;
	}

	va_end (argp);

/*
 * RUN COMMAND LINE:
 */

	result = (runtype == CMD) ? system (command)
				  : (long) popen (command, "r");

	free (command);
	return (result);

} /* RunCommand */


/************************************************************************
 * P R I N T   R E P O R T
 *
 * Given global long total[], print a report using those totals.
 * For printing the first section, assumes that certain values in total[]
 * are zero; does not bother to add them.
 */

#define	TITLEWIDTH 17			/* size of row titles	 */
#define	NUMWIDTH   11			/* size of number fields */

char	*rowtitle[] = {			/* for printing second section */
		"New code",
		"Reused, unchanged",
		"        added",
		"        deleted",
		"        modified",
		"Reused, unknown",
		"TOTAL",
};

PROC PrintReport()
{
	long	new, unchanged, added, deleted, modified;
REG	int	row, col;		/* current entry */

/*
 * PRINT FIRST SECTION:
 */

    puts ("             -------------- REUSED CODE ---------------    CURRENT");
    puts ("   NEW CODE  UNCHANGED      ADDED    DELETED   MODIFIED      TOTAL");

	new	  = total [ROW_NEW]	  [COL_NEW];

	unchanged = total [ROW_UNCHANGED] [COL_UNCHANGED] +
		    total [ROW_UNKNOWN]   [COL_UNCHANGED];

	added	  = total [ROW_ADDED]	  [COL_TOUCHED]	+
		    total [ROW_ADDED]	  [COL_NEW]	+
		    total [ROW_UNKNOWN]   [COL_NEW];

	deleted	  = total [ROW_DELETED]   [COL_TOUCHED];

	modified  = total [ROW_MODIFIED]  [COL_TOUCHED]	+
		    total [ROW_UNKNOWN]   [COL_TOUCHED];

	PrintNumber (new);
	PrintNumber (unchanged);
	PrintNumber (added);
	PrintNumber (deleted);
	PrintNumber (modified);
	PrintNumber (new + unchanged + added + modified);  /* no deleted */

/*
 * PRINT SECOND SECTION:
 */

	puts ("\n\nSOURCE LINES       UNCHANGED    TOUCHED        NEW");

	for (row = 0; row <= ROWMAX; row++)	/* does totals line too */
	{
	    printf ("%-*s", TITLEWIDTH, rowtitle [row]);

	    for (col = 0; col <= COLMAX; col++)
	    {
		PrintNumber (total [row] [col]);
		total [ROWMAX] [col] += total [row] [col];
	    }

	    putchar (CHNEWLINE);
	}

} /* PrintReport */


/************************************************************************
 * P R I N T   N U M B E R
 *
 * Given a number, print it in width NUMWIDTH if non-zero, else print a
 * dot in the field.
 */

PROC PrintNumber (number)
	long	number;
{
	if (number)	printf ("%*ld", NUMWIDTH, number);
	else		printf ("%*s",  NUMWIDTH, ".");

} /* PrintNumber */
